"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayMonitoring = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const common_1 = require("../../common");
const dashboard_1 = require("../../dashboard");
const ApiGatewayMetricFactory_1 = require("./ApiGatewayMetricFactory");
const DefaultLatencyTypesToRender = [
    common_1.LatencyType.P50,
    common_1.LatencyType.P90,
    common_1.LatencyType.P99,
];
class ApiGatewayMonitoring extends common_1.Monitoring {
    constructor(scope, props) {
        var _b, _c, _d, _e;
        super(scope, props);
        // used when humanReadableName is not provided by user
        const fallbackNameArray = [props.api.restApiName];
        fallbackNameArray.push((_b = props.apiStage) !== null && _b !== void 0 ? _b : "prod");
        if (props.apiMethod) {
            fallbackNameArray.push(props.apiMethod);
        }
        if (props.apiResource) {
            fallbackNameArray.push(props.apiResource);
        }
        const namingStrategy = new dashboard_1.MonitoringNamingStrategy({
            ...props,
            namedConstruct: props.api,
            fallbackConstructName: fallbackNameArray
                .join("-")
                .replace(/[^a-zA-Z0-9-_]/g, ""),
            humanReadableName: (_c = props.humanReadableName) !== null && _c !== void 0 ? _c : fallbackNameArray.join(" "),
        });
        this.title = namingStrategy.resolveHumanReadableName();
        this.alarmFactory = this.createAlarmFactory(namingStrategy.resolveAlarmFriendlyName());
        this.errorAlarmFactory = new common_1.ErrorAlarmFactory(this.alarmFactory);
        this.tpsAlarmFactory = new common_1.TpsAlarmFactory(this.alarmFactory);
        this.latencyAlarmFactory = new common_1.LatencyAlarmFactory(this.alarmFactory);
        this.tpsAnnotations = [];
        this.latencyAnnotations = [];
        this.errorCountAnnotations = [];
        this.errorRateAnnotations = [];
        const metricFactory = new ApiGatewayMetricFactory_1.ApiGatewayMetricFactory(scope.createMetricFactory(), props);
        this.tpsMetric = metricFactory.metricTps();
        this.latencyMetrics = {};
        this.latencyTypesToRender = [
            ...((_d = props.latencyTypesToRender) !== null && _d !== void 0 ? _d : DefaultLatencyTypesToRender),
        ];
        this.error4XXCountMetric = metricFactory.metric4XXErrorCount();
        this.error4XXRateMetric = metricFactory.metric4XXErrorRate();
        this.fault5XXCountMetric = metricFactory.metric5XXFaultCount();
        this.fault5XXRateMetric = metricFactory.metric5XXFaultRate();
        const latencyAlarmDefinitions = {
            [common_1.LatencyType.P50]: props.addLatencyP50Alarm,
            [common_1.LatencyType.P70]: props.addLatencyP70Alarm,
            [common_1.LatencyType.P90]: props.addLatencyP90Alarm,
            [common_1.LatencyType.P95]: props.addLatencyP95Alarm,
            [common_1.LatencyType.P99]: props.addLatencyP99Alarm,
            [common_1.LatencyType.P999]: props.addLatencyP999Alarm,
            [common_1.LatencyType.P9999]: props.addLatencyP9999Alarm,
            [common_1.LatencyType.P100]: props.addLatencyP100Alarm,
            [common_1.LatencyType.TM50]: props.addLatencyTM50Alarm,
            [common_1.LatencyType.TM70]: props.addLatencyTM70Alarm,
            [common_1.LatencyType.TM90]: props.addLatencyTM90Alarm,
            [common_1.LatencyType.TM95]: props.addLatencyTM95Alarm,
            [common_1.LatencyType.TM99]: props.addLatencyTM99Alarm,
            [common_1.LatencyType.TM999]: props.addLatencyTM999Alarm,
            [common_1.LatencyType.TM9999]: props.addLatencyTM9999Alarm,
            [common_1.LatencyType.AVERAGE]: props.addLatencyAverageAlarm,
        };
        Object.values(common_1.LatencyType).forEach((latencyType) => {
            this.latencyMetrics[latencyType] =
                metricFactory.metricLatencyInMillis(latencyType);
        });
        for (const [latencyType, alarmDefinition] of Object.entries(latencyAlarmDefinitions)) {
            for (const disambiguator in alarmDefinition) {
                const alarmProps = alarmDefinition[disambiguator];
                const latencyTypeEnum = latencyType;
                const metric = this.latencyMetrics[latencyTypeEnum];
                const createdAlarm = this.latencyAlarmFactory.addLatencyAlarm(metric, latencyTypeEnum, alarmProps, disambiguator);
                this.latencyAnnotations.push(createdAlarm.annotation);
                this.latencyTypesToRender.push(latencyTypeEnum);
                this.addAlarm(createdAlarm);
            }
        }
        for (const disambiguator in props.add5XXFaultCountAlarm) {
            const alarmProps = props.add5XXFaultCountAlarm[disambiguator];
            const createdAlarm = this.errorAlarmFactory.addErrorCountAlarm(this.fault5XXCountMetric, common_1.ErrorType.FAULT, alarmProps, disambiguator);
            this.errorCountAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
        for (const disambiguator in props.add4XXErrorCountAlarm) {
            const alarmProps = props.add4XXErrorCountAlarm[disambiguator];
            const createdAlarm = this.errorAlarmFactory.addErrorCountAlarm(this.error4XXCountMetric, common_1.ErrorType.ERROR, alarmProps, disambiguator);
            this.errorCountAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
        for (const disambiguator in props.add4XXErrorRateAlarm) {
            const alarmProps = props.add4XXErrorRateAlarm[disambiguator];
            const createdAlarm = this.errorAlarmFactory.addErrorRateAlarm(this.error4XXRateMetric, common_1.ErrorType.ERROR, alarmProps, disambiguator);
            this.errorRateAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
        for (const disambiguator in props.add5XXFaultRateAlarm) {
            const alarmProps = props.add5XXFaultRateAlarm[disambiguator];
            const createdAlarm = this.errorAlarmFactory.addErrorRateAlarm(this.fault5XXRateMetric, common_1.ErrorType.FAULT, alarmProps, disambiguator);
            this.errorRateAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
        for (const disambiguator in props.addLowTpsAlarm) {
            const alarmProps = props.addLowTpsAlarm[disambiguator];
            const createdAlarm = this.tpsAlarmFactory.addMinTpsAlarm(this.tpsMetric, alarmProps, disambiguator);
            this.tpsAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
        for (const disambiguator in props.addHighTpsAlarm) {
            const alarmProps = props.addHighTpsAlarm[disambiguator];
            const createdAlarm = this.tpsAlarmFactory.addMaxTpsAlarm(this.tpsMetric, alarmProps, disambiguator);
            this.tpsAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
        (_e = props.useCreatedAlarms) === null || _e === void 0 ? void 0 : _e.consume(this.createdAlarms());
    }
    summaryWidgets() {
        return [
            this.createTitleWidget(),
            this.createTpsWidget(common_1.ThirdWidth, common_1.DefaultSummaryWidgetHeight),
            this.createLatencyWidget(common_1.ThirdWidth, common_1.DefaultSummaryWidgetHeight),
            this.createErrorRateWidget(common_1.ThirdWidth, common_1.DefaultSummaryWidgetHeight),
        ];
    }
    widgets() {
        return [
            this.createTitleWidget(),
            this.createTpsWidget(common_1.QuarterWidth, common_1.DefaultGraphWidgetHeight),
            this.createLatencyWidget(common_1.QuarterWidth, common_1.DefaultGraphWidgetHeight),
            this.createErrorCountWidget(common_1.QuarterWidth, common_1.DefaultGraphWidgetHeight),
            this.createErrorRateWidget(common_1.QuarterWidth, common_1.DefaultGraphWidgetHeight),
        ];
    }
    createTitleWidget() {
        return new dashboard_1.MonitoringHeaderWidget({
            family: "API Gateway Endpoint",
            title: this.title,
        });
    }
    createTpsWidget(width, height) {
        return new aws_cloudwatch_1.GraphWidget({
            width,
            height,
            title: "TPS",
            left: [this.tpsMetric],
            leftYAxis: common_1.CountAxisFromZero,
            leftAnnotations: this.tpsAnnotations,
        });
    }
    createLatencyWidget(width, height) {
        const left = Array.from(new Set(this.latencyTypesToRender))
            .sort()
            .map((type) => this.latencyMetrics[type]);
        return new aws_cloudwatch_1.GraphWidget({
            width,
            height,
            title: "Latency",
            left,
            leftYAxis: common_1.TimeAxisMillisFromZero,
            leftAnnotations: this.latencyAnnotations,
        });
    }
    createErrorCountWidget(width, height) {
        return new aws_cloudwatch_1.GraphWidget({
            width,
            height,
            title: "Errors",
            left: [this.error4XXCountMetric, this.fault5XXCountMetric],
            leftYAxis: common_1.CountAxisFromZero,
            leftAnnotations: this.errorCountAnnotations,
        });
    }
    createErrorRateWidget(width, height) {
        return new aws_cloudwatch_1.GraphWidget({
            width,
            height,
            title: "Errors (rate)",
            left: [this.error4XXRateMetric, this.fault5XXRateMetric],
            leftYAxis: common_1.RateAxisFromZero,
            leftAnnotations: this.errorRateAnnotations,
        });
    }
}
exports.ApiGatewayMonitoring = ApiGatewayMonitoring;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayMonitoring[_a] = { fqn: "cdk-monitoring-constructs.ApiGatewayMonitoring", version: "1.6.2" };
//# sourceMappingURL=data:application/json;base64,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