"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomMonitoring = exports.AxisPosition = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const common_1 = require("../../common");
const dashboard_1 = require("../../dashboard");
var AxisPosition;
(function (AxisPosition) {
    AxisPosition["LEFT"] = "left";
    AxisPosition["RIGHT"] = "right";
})(AxisPosition = exports.AxisPosition || (exports.AxisPosition = {}));
/**
 * Custom monitoring is a construct allowing you to monitor your own custom metrics.
 * The entire construct consists of metric groups.
 * Each metric group represents a single graph widget with multiple metrics.
 * Each metric inside the metric group represents a single metric inside a graph.
 * The widgets will be sized automatically to waste as little space as possible.
 */
class CustomMonitoring extends common_1.Monitoring {
    constructor(scope, props) {
        var _b;
        super(scope, props);
        const namingStrategy = new dashboard_1.MonitoringNamingStrategy({ ...props });
        this.title = namingStrategy.resolveHumanReadableName();
        this.description = props.description;
        this.descriptionWidgetHeight = props.descriptionWidgetHeight;
        const alarmFactory = this.createAlarmFactory(namingStrategy.resolveAlarmFriendlyName());
        this.customAlarmFactory = new common_1.CustomAlarmFactory(alarmFactory);
        this.anomalyDetectingAlarmFactory = new common_1.AnomalyDetectingAlarmFactory(alarmFactory);
        this.metricGroups = props.metricGroups.map((metricGroup) => {
            const metricGroupWithAnnotation = {
                metricGroup,
                annotations: [],
                rightAnnotations: [],
                titleAddons: [],
            };
            if (metricGroup.horizontalAnnotations) {
                metricGroupWithAnnotation.annotations.push(...metricGroup.horizontalAnnotations);
            }
            if (metricGroup.horizontalRightAnnotations) {
                metricGroupWithAnnotation.rightAnnotations.push(...metricGroup.horizontalRightAnnotations);
            }
            metricGroup.metrics.forEach((metric) => {
                if (this.hasAlarm(metric) && this.hasAnomalyDetection(metric)) {
                    throw new Error("Adding both a regular alarm and an anomoly detection alarm at the same time is not supported");
                }
                if (this.hasAlarm(metric)) {
                    this.setupAlarm(metricGroupWithAnnotation, metric);
                }
                else if (this.hasAnomalyDetection(metric)) {
                    this.setupAnomalyDetectionAlarm(metricGroupWithAnnotation, metric);
                }
            });
            return metricGroupWithAnnotation;
        });
        (_b = props.useCreatedAlarms) === null || _b === void 0 ? void 0 : _b.consume(this.createdAlarms());
    }
    summaryWidgets() {
        return this.getAllWidgets(true);
    }
    widgets() {
        return this.getAllWidgets(false);
    }
    getAllWidgets(summary) {
        const filteredMetricGroups = summary
            ? this.metricGroups.filter((group) => { var _b; return (_b = group.metricGroup.important) !== null && _b !== void 0 ? _b : false; })
            : this.metricGroups;
        if (filteredMetricGroups.length < 1) {
            // short-circuit if there are no metrics specified
            return [];
        }
        const rows = [];
        // header and description
        rows.push(new aws_cloudwatch_1.Row(new dashboard_1.MonitoringHeaderWidget({ title: this.title })));
        if (this.description && !summary) {
            rows.push(new aws_cloudwatch_1.Row(this.createDescriptionWidget(this.description, this.descriptionWidgetHeight)));
        }
        // graphs
        rows.push(new aws_cloudwatch_1.Row(...this.createCustomMetricGroupWidgets(filteredMetricGroups, summary)));
        return rows;
    }
    createDescriptionWidget(markdown, descriptionWidgetHeight) {
        return new aws_cloudwatch_1.TextWidget({
            markdown,
            width: common_1.FullWidth,
            height: descriptionWidgetHeight !== null && descriptionWidgetHeight !== void 0 ? descriptionWidgetHeight : 1,
        });
    }
    createCustomMetricGroupWidgets(annotatedGroups, summary) {
        const widgets = [];
        const metricGroupWidgetWidth = common_1.recommendedWidgetWidth(annotatedGroups.length);
        annotatedGroups.forEach((annotatedGroup) => {
            var _b;
            const metrics = annotatedGroup.metricGroup.metrics;
            const left = this.toMetrics(metrics.filter((metric) => { var _b; return ((_b = metric.position) !== null && _b !== void 0 ? _b : AxisPosition.LEFT) == AxisPosition.LEFT; }));
            const right = this.toMetrics(metrics.filter((metric) => {
                var _b;
                return ((_b = metric.position) !== null && _b !== void 0 ? _b : AxisPosition.LEFT) ==
                    AxisPosition.RIGHT;
            }));
            const hasOneMetricOnly = metrics.length === 1;
            const hasAnomalyDetection = metrics.filter((metric) => this.hasAnomalyDetection(metric)).length > 0;
            const useAnomalyDetectionWidget = hasOneMetricOnly && hasAnomalyDetection;
            let title = annotatedGroup.metricGroup.title;
            if (annotatedGroup.titleAddons.length > 0) {
                title = `${title} (${annotatedGroup.titleAddons.join(", ")})`;
            }
            const graphWidgetProps = {
                title,
                width: metricGroupWidgetWidth,
                height: summary ? common_1.DefaultSummaryWidgetHeight : common_1.DefaultGraphWidgetHeight,
                left,
                right,
                leftAnnotations: annotatedGroup.annotations,
                rightAnnotations: annotatedGroup.rightAnnotations,
                leftYAxis: annotatedGroup.metricGroup.graphWidgetAxis,
                rightYAxis: annotatedGroup.metricGroup.graphWidgetRightAxis,
            };
            const widget = useAnomalyDetectionWidget
                ? new AnomalyDetectionGraphWidget(graphWidgetProps)
                : common_1.createGraphWidget((_b = annotatedGroup.metricGroup.graphWidgetType) !== null && _b !== void 0 ? _b : common_1.GraphWidgetType.LINE, graphWidgetProps);
            widgets.push(widget);
        });
        return widgets;
    }
    toMetrics(metrics) {
        const metricFactory = this.createMetricFactory();
        return metrics.map((metric) => {
            var _b;
            if (this.hasAlarm(metric)) {
                // metric with alarm
                return metricFactory.adaptMetricPreservingPeriod(metric.metric);
            }
            else if (this.hasAnomalyDetection(metric)) {
                // metric with anomaly detection
                return metricFactory.createMetricAnomalyDetection(metric.metric, metric.anomalyDetectionStandardDeviationToRender, `Expected (stdev = ${metric.anomalyDetectionStandardDeviationToRender})`, undefined, 
                // needs to be unique in the whole widget and start with lowercase
                AnomalyDetectionMetricIdPrefix +
                    common_1.getHashForMetricExpressionId(metric.alarmFriendlyName), (_b = 
                // preserve the most specific metric period
                metric.period) !== null && _b !== void 0 ? _b : metric.metric.period);
            }
            else if (this.isSearch(metric)) {
                // metric search
                return metricFactory.createMetricSearch(metric.searchQuery, metric.dimensionsMap, metric.statistic, metric.namespace, metric.label, metric.period);
            }
            else {
                // general metric
                return metricFactory.adaptMetricPreservingPeriod(metric);
            }
        });
    }
    hasAlarm(metric) {
        // type guard
        return metric.addAlarm !== undefined;
    }
    hasAnomalyDetection(metric) {
        // type guard
        return (metric
            .anomalyDetectionStandardDeviationToRender !== undefined);
    }
    isSearch(metric) {
        // type guard
        return metric.searchQuery !== undefined;
    }
    setupAlarm(metricGroup, metric) {
        var _b;
        if (this.isSearch(metric)) {
            throw new Error("Alarming on search queries is not supported by CloudWatch");
        }
        for (const disambiguator in metric.addAlarm) {
            const alarmProps = metric.addAlarm[disambiguator];
            const createdAlarm = this.customAlarmFactory.addCustomAlarm(metric.metric, metric.alarmFriendlyName, disambiguator, alarmProps);
            const targetAnnotations = ((_b = metric.position) !== null && _b !== void 0 ? _b : AxisPosition.LEFT) == AxisPosition.LEFT
                ? metricGroup.annotations
                : metricGroup.rightAnnotations;
            targetAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
    }
    setupAnomalyDetectionAlarm(metricGroup, metric) {
        var _b;
        if (this.isSearch(metric)) {
            throw new Error("Alarming on search queries is not supported by CloudWatch");
        }
        const alarmStDevs = new Set();
        const metricFactory = this.createMetricFactory();
        for (const disambiguator in metric.addAlarmOnAnomaly) {
            const alarmProps = metric.addAlarmOnAnomaly[disambiguator];
            if (alarmProps.alarmWhenAboveTheBand ||
                alarmProps.alarmWhenBelowTheBand) {
                const anomalyMetric = metricFactory.createMetricAnomalyDetection(
                // Because the metric was provided to us, we use metricFactory.overrideNamespace() to
                // confirm it aligns with any namespace overrides requested for this MonitoringFacade
                metricFactory.adaptMetricPreservingPeriod(metric.metric), alarmProps.standardDeviationForAlarm, `Band (stdev ${alarmProps.standardDeviationForAlarm})`, undefined, 
                // expression ID needs to be unique across the whole widget; needs to start with a lowercase letter
                AnomalyDetectionAlarmIdPrefix +
                    common_1.getHashForMetricExpressionId(metric.alarmFriendlyName + "_" + disambiguator), (_b = 
                // preserve the most-specific metric period
                metric.period) !== null && _b !== void 0 ? _b : metric.metric.period);
                const createdAlarm = this.anomalyDetectingAlarmFactory.addAlarmWhenOutOfBand(anomalyMetric, metric.alarmFriendlyName, disambiguator, alarmProps);
                // no need to add annotation since the bands are rendered automatically
                this.addAlarm(createdAlarm);
                alarmStDevs.add(alarmProps.standardDeviationForAlarm);
            }
        }
        if (alarmStDevs.size > 0) {
            const alarmStDevsString = Array.from(alarmStDevs).sort().join(", ");
            metricGroup.titleAddons.push(`alarms with stdev ${alarmStDevsString}`);
        }
    }
}
exports.CustomMonitoring = CustomMonitoring;
_a = JSII_RTTI_SYMBOL_1;
CustomMonitoring[_a] = { fqn: "cdk-monitoring-constructs.CustomMonitoring", version: "1.6.2" };
const AnomalyDetectionAlarmIdPrefix = "alarm_";
const AnomalyDetectionMetricIdPrefix = "anomaly_";
const AnomalyBandMetricIdSuffix = "_band";
/**
 * INTERNAL - PLEASE DO NOT USE
 * This is a hacky solution to make band visible in GraphWidget (default widget only renders lines, not the band).
 * The class makes assumptions about the internal JSON structure but found no other way :(.
 * Ideally, we want to remove this hack once the anomaly detection rendering in CDK gets improved
 */
class AnomalyDetectionGraphWidget extends aws_cloudwatch_1.GraphWidget {
    constructor(props) {
        super(props);
    }
    toJson() {
        var _b, _c, _d, _e;
        const json = super.toJson();
        if (json.length !== 1 || !((_c = (_b = json === null || json === void 0 ? void 0 : json[0]) === null || _b === void 0 ? void 0 : _b.properties) === null || _c === void 0 ? void 0 : _c.metrics)) {
            throw new Error("The JSON is expected to have exactly one element with properties.metrics property.");
        }
        const metrics = json[0].properties.metrics;
        if (metrics.length < 2) {
            throw new Error("The number of metrics must be at least two (metric + anomaly detection math).");
        }
        const anomalyDetectionMetricPart = (_d = metrics[0]) === null || _d === void 0 ? void 0 : _d.value;
        if (!anomalyDetectionMetricPart ||
            anomalyDetectionMetricPart.length !== 1) {
            throw new Error("First metric must be a math expression.");
        }
        const evaluatedMetricPart = (_e = metrics[1]) === null || _e === void 0 ? void 0 : _e.value;
        if (!evaluatedMetricPart ||
            evaluatedMetricPart.length < 1 ||
            !evaluatedMetricPart[evaluatedMetricPart.length - 1].id) {
            throw new Error("Second metric must have an ID.");
        }
        // band rendering requires ID to be set
        anomalyDetectionMetricPart[0].id =
            evaluatedMetricPart[evaluatedMetricPart.length - 1].id +
                AnomalyBandMetricIdSuffix;
        // band rendering requires the evaluated metric to be visible
        evaluatedMetricPart[evaluatedMetricPart.length - 1].visible = true;
        return json;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiQ3VzdG9tTW9uaXRvcmluZy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIkN1c3RvbU1vbml0b3JpbmcudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFDQSwrREFVb0M7QUFFcEMseUNBaUJzQjtBQUN0QiwrQ0FHeUI7QUFFekIsSUFBWSxZQUdYO0FBSEQsV0FBWSxZQUFZO0lBQ3RCLDZCQUFhLENBQUE7SUFDYiwrQkFBZSxDQUFBO0FBQ2pCLENBQUMsRUFIVyxZQUFZLEdBQVosb0JBQVksS0FBWixvQkFBWSxRQUd2QjtBQWdLRDs7Ozs7O0dBTUc7QUFDSCxNQUFhLGdCQUFpQixTQUFRLG1CQUFVO0lBUTlDLFlBQVksS0FBc0IsRUFBRSxLQUE0Qjs7UUFDOUQsS0FBSyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztRQUVwQixNQUFNLGNBQWMsR0FBRyxJQUFJLG9DQUF3QixDQUFDLEVBQUUsR0FBRyxLQUFLLEVBQUUsQ0FBQyxDQUFDO1FBQ2xFLElBQUksQ0FBQyxLQUFLLEdBQUcsY0FBYyxDQUFDLHdCQUF3QixFQUFFLENBQUM7UUFFdkQsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDO1FBQ3JDLElBQUksQ0FBQyx1QkFBdUIsR0FBRyxLQUFLLENBQUMsdUJBQXVCLENBQUM7UUFFN0QsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixDQUMxQyxjQUFjLENBQUMsd0JBQXdCLEVBQUUsQ0FDMUMsQ0FBQztRQUNGLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLDJCQUFrQixDQUFDLFlBQVksQ0FBQyxDQUFDO1FBQy9ELElBQUksQ0FBQyw0QkFBNEIsR0FBRyxJQUFJLHFDQUE0QixDQUNsRSxZQUFZLENBQ2IsQ0FBQztRQUVGLElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsQ0FBQyxXQUFXLEVBQUUsRUFBRTtZQUN6RCxNQUFNLHlCQUF5QixHQUFxQztnQkFDbEUsV0FBVztnQkFDWCxXQUFXLEVBQUUsRUFBRTtnQkFDZixnQkFBZ0IsRUFBRSxFQUFFO2dCQUNwQixXQUFXLEVBQUUsRUFBRTthQUNoQixDQUFDO1lBRUYsSUFBSSxXQUFXLENBQUMscUJBQXFCLEVBQUU7Z0JBQ3JDLHlCQUF5QixDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQ3hDLEdBQUcsV0FBVyxDQUFDLHFCQUFxQixDQUNyQyxDQUFDO2FBQ0g7WUFDRCxJQUFJLFdBQVcsQ0FBQywwQkFBMEIsRUFBRTtnQkFDMUMseUJBQXlCLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUM3QyxHQUFHLFdBQVcsQ0FBQywwQkFBMEIsQ0FDMUMsQ0FBQzthQUNIO1lBRUQsV0FBVyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRTtnQkFDckMsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxJQUFJLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxNQUFNLENBQUMsRUFBRTtvQkFDN0QsTUFBTSxJQUFJLEtBQUssQ0FDYiw4RkFBOEYsQ0FDL0YsQ0FBQztpQkFDSDtnQkFFRCxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEVBQUU7b0JBQ3pCLElBQUksQ0FBQyxVQUFVLENBQUMseUJBQXlCLEVBQUUsTUFBTSxDQUFDLENBQUM7aUJBQ3BEO3FCQUFNLElBQUksSUFBSSxDQUFDLG1CQUFtQixDQUFDLE1BQU0sQ0FBQyxFQUFFO29CQUMzQyxJQUFJLENBQUMsMEJBQTBCLENBQUMseUJBQXlCLEVBQUUsTUFBTSxDQUFDLENBQUM7aUJBQ3BFO1lBQ0gsQ0FBQyxDQUFDLENBQUM7WUFFSCxPQUFPLHlCQUF5QixDQUFDO1FBQ25DLENBQUMsQ0FBQyxDQUFDO1FBRUgsTUFBQSxLQUFLLENBQUMsZ0JBQWdCLDBDQUFFLE9BQU8sQ0FBQyxJQUFJLENBQUMsYUFBYSxFQUFFLEVBQUU7SUFDeEQsQ0FBQztJQUVELGNBQWM7UUFDWixPQUFPLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVELE9BQU87UUFDTCxPQUFPLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVPLGFBQWEsQ0FBQyxPQUFnQjtRQUNwQyxNQUFNLG9CQUFvQixHQUFHLE9BQU87WUFDbEMsQ0FBQyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUN0QixDQUFDLEtBQUssRUFBRSxFQUFFLHdCQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsU0FBUyxtQ0FBSSxLQUFLLEdBQUEsQ0FDaEQ7WUFDSCxDQUFDLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQztRQUV0QixJQUFJLG9CQUFvQixDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDbkMsa0RBQWtEO1lBQ2xELE9BQU8sRUFBRSxDQUFDO1NBQ1g7UUFFRCxNQUFNLElBQUksR0FBVSxFQUFFLENBQUM7UUFFdkIseUJBQXlCO1FBQ3pCLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxvQkFBRyxDQUFDLElBQUksa0NBQXNCLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3RFLElBQUksSUFBSSxDQUFDLFdBQVcsSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUNoQyxJQUFJLENBQUMsSUFBSSxDQUNQLElBQUksb0JBQUcsQ0FDTCxJQUFJLENBQUMsdUJBQXVCLENBQzFCLElBQUksQ0FBQyxXQUFXLEVBQ2hCLElBQUksQ0FBQyx1QkFBdUIsQ0FDN0IsQ0FDRixDQUNGLENBQUM7U0FDSDtRQUVELFNBQVM7UUFDVCxJQUFJLENBQUMsSUFBSSxDQUNQLElBQUksb0JBQUcsQ0FDTCxHQUFHLElBQUksQ0FBQyw4QkFBOEIsQ0FBQyxvQkFBb0IsRUFBRSxPQUFPLENBQUMsQ0FDdEUsQ0FDRixDQUFDO1FBRUYsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRU8sdUJBQXVCLENBQzdCLFFBQWdCLEVBQ2hCLHVCQUFnQztRQUVoQyxPQUFPLElBQUksMkJBQVUsQ0FBQztZQUNwQixRQUFRO1lBQ1IsS0FBSyxFQUFFLGtCQUFTO1lBQ2hCLE1BQU0sRUFBRSx1QkFBdUIsYUFBdkIsdUJBQXVCLGNBQXZCLHVCQUF1QixHQUFJLENBQUM7U0FDckMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLDhCQUE4QixDQUNwQyxlQUFtRCxFQUNuRCxPQUFnQjtRQUVoQixNQUFNLE9BQU8sR0FBa0IsRUFBRSxDQUFDO1FBQ2xDLE1BQU0sc0JBQXNCLEdBQUcsK0JBQXNCLENBQ25ELGVBQWUsQ0FBQyxNQUFNLENBQ3ZCLENBQUM7UUFFRixlQUFlLENBQUMsT0FBTyxDQUFDLENBQUMsY0FBYyxFQUFFLEVBQUU7O1lBQ3pDLE1BQU0sT0FBTyxHQUFHLGNBQWMsQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDO1lBQ25ELE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQ3pCLE9BQU8sQ0FBQyxNQUFNLENBQ1osQ0FBQyxNQUFNLEVBQUUsRUFBRSxXQUNULE9BQUEsT0FBRSxNQUFjLENBQUMsUUFBUSxtQ0FBSSxZQUFZLENBQUMsSUFBSSxDQUFDLElBQUksWUFBWSxDQUFDLElBQUksQ0FBQSxFQUFBLENBQ3ZFLENBQ0YsQ0FBQztZQUNGLE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQzFCLE9BQU8sQ0FBQyxNQUFNLENBQ1osQ0FBQyxNQUFNLEVBQUUsRUFBRTs7Z0JBQ1QsT0FBQSxPQUFFLE1BQWMsQ0FBQyxRQUFRLG1DQUFJLFlBQVksQ0FBQyxJQUFJLENBQUM7b0JBQy9DLFlBQVksQ0FBQyxLQUFLLENBQUE7YUFBQSxDQUNyQixDQUNGLENBQUM7WUFDRixNQUFNLGdCQUFnQixHQUFHLE9BQU8sQ0FBQyxNQUFNLEtBQUssQ0FBQyxDQUFDO1lBQzlDLE1BQU0sbUJBQW1CLEdBQ3ZCLE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUM7WUFDMUUsTUFBTSx5QkFBeUIsR0FBRyxnQkFBZ0IsSUFBSSxtQkFBbUIsQ0FBQztZQUMxRSxJQUFJLEtBQUssR0FBRyxjQUFjLENBQUMsV0FBVyxDQUFDLEtBQUssQ0FBQztZQUU3QyxJQUFJLGNBQWMsQ0FBQyxXQUFXLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtnQkFDekMsS0FBSyxHQUFHLEdBQUcsS0FBSyxLQUFLLGNBQWMsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUM7YUFDL0Q7WUFFRCxNQUFNLGdCQUFnQixHQUFxQjtnQkFDekMsS0FBSztnQkFDTCxLQUFLLEVBQUUsc0JBQXNCO2dCQUM3QixNQUFNLEVBQUUsT0FBTyxDQUFDLENBQUMsQ0FBQyxtQ0FBMEIsQ0FBQyxDQUFDLENBQUMsaUNBQXdCO2dCQUN2RSxJQUFJO2dCQUNKLEtBQUs7Z0JBQ0wsZUFBZSxFQUFFLGNBQWMsQ0FBQyxXQUFXO2dCQUMzQyxnQkFBZ0IsRUFBRSxjQUFjLENBQUMsZ0JBQWdCO2dCQUNqRCxTQUFTLEVBQUUsY0FBYyxDQUFDLFdBQVcsQ0FBQyxlQUFlO2dCQUNyRCxVQUFVLEVBQUUsY0FBYyxDQUFDLFdBQVcsQ0FBQyxvQkFBb0I7YUFDNUQsQ0FBQztZQUVGLE1BQU0sTUFBTSxHQUFHLHlCQUF5QjtnQkFDdEMsQ0FBQyxDQUFDLElBQUksMkJBQTJCLENBQUMsZ0JBQWdCLENBQUM7Z0JBQ25ELENBQUMsQ0FBQywwQkFBaUIsT0FDZixjQUFjLENBQUMsV0FBVyxDQUFDLGVBQWUsbUNBQUksd0JBQWUsQ0FBQyxJQUFJLEVBQ2xFLGdCQUFnQixDQUNqQixDQUFDO1lBRU4sT0FBTyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN2QixDQUFDLENBQUMsQ0FBQztRQUVILE9BQU8sT0FBTyxDQUFDO0lBQ2pCLENBQUM7SUFFTyxTQUFTLENBQUMsT0FBdUI7UUFDdkMsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixFQUFFLENBQUM7UUFFakQsT0FBTyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUU7O1lBQzVCLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsRUFBRTtnQkFDekIsb0JBQW9CO2dCQUNwQixPQUFPLGFBQWEsQ0FBQywyQkFBMkIsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUM7YUFDakU7aUJBQU0sSUFBSSxJQUFJLENBQUMsbUJBQW1CLENBQUMsTUFBTSxDQUFDLEVBQUU7Z0JBQzNDLGdDQUFnQztnQkFDaEMsT0FBTyxhQUFhLENBQUMsNEJBQTRCLENBQy9DLE1BQU0sQ0FBQyxNQUFNLEVBQ2IsTUFBTSxDQUFDLHlDQUF5QyxFQUNoRCxxQkFBcUIsTUFBTSxDQUFDLHlDQUF5QyxHQUFHLEVBQ3hFLFNBQVM7Z0JBQ1Qsa0VBQWtFO2dCQUNsRSw4QkFBOEI7b0JBQzVCLHFDQUE0QixDQUFDLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQztnQkFDeEQsMkNBQTJDO2dCQUMzQyxNQUFNLENBQUMsTUFBTSxtQ0FBSSxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FDdEMsQ0FBQzthQUNIO2lCQUFNLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsRUFBRTtnQkFDaEMsZ0JBQWdCO2dCQUNoQixPQUFPLGFBQWEsQ0FBQyxrQkFBa0IsQ0FDckMsTUFBTSxDQUFDLFdBQVcsRUFDbEIsTUFBTSxDQUFDLGFBQWEsRUFDcEIsTUFBTSxDQUFDLFNBQVMsRUFDaEIsTUFBTSxDQUFDLFNBQVMsRUFDaEIsTUFBTSxDQUFDLEtBQUssRUFDWixNQUFNLENBQUMsTUFBTSxDQUNkLENBQUM7YUFDSDtpQkFBTTtnQkFDTCxpQkFBaUI7Z0JBQ2pCLE9BQU8sYUFBYSxDQUFDLDJCQUEyQixDQUFDLE1BQU0sQ0FBQyxDQUFDO2FBQzFEO1FBQ0gsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8sUUFBUSxDQUFDLE1BQW9CO1FBQ25DLGFBQWE7UUFDYixPQUFRLE1BQWdDLENBQUMsUUFBUSxLQUFLLFNBQVMsQ0FBQztJQUNsRSxDQUFDO0lBRU8sbUJBQW1CLENBQ3pCLE1BQW9CO1FBRXBCLGFBQWE7UUFDYixPQUFPLENBQ0osTUFBMkM7YUFDekMseUNBQXlDLEtBQUssU0FBUyxDQUMzRCxDQUFDO0lBQ0osQ0FBQztJQUVPLFFBQVEsQ0FBQyxNQUFvQjtRQUNuQyxhQUFhO1FBQ2IsT0FBUSxNQUE2QixDQUFDLFdBQVcsS0FBSyxTQUFTLENBQUM7SUFDbEUsQ0FBQztJQUVPLFVBQVUsQ0FDaEIsV0FBNkMsRUFDN0MsTUFBNkI7O1FBRTdCLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUN6QixNQUFNLElBQUksS0FBSyxDQUNiLDJEQUEyRCxDQUM1RCxDQUFDO1NBQ0g7UUFFRCxLQUFLLE1BQU0sYUFBYSxJQUFJLE1BQU0sQ0FBQyxRQUFRLEVBQUU7WUFDM0MsTUFBTSxVQUFVLEdBQUcsTUFBTSxDQUFDLFFBQVEsQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUNsRCxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsa0JBQWtCLENBQUMsY0FBYyxDQUN6RCxNQUFNLENBQUMsTUFBTSxFQUNiLE1BQU0sQ0FBQyxpQkFBaUIsRUFDeEIsYUFBYSxFQUNiLFVBQVUsQ0FDWCxDQUFDO1lBQ0YsTUFBTSxpQkFBaUIsR0FDckIsT0FBQyxNQUFNLENBQUMsUUFBUSxtQ0FBSSxZQUFZLENBQUMsSUFBSSxDQUFDLElBQUksWUFBWSxDQUFDLElBQUk7Z0JBQ3pELENBQUMsQ0FBQyxXQUFXLENBQUMsV0FBVztnQkFDekIsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxnQkFBZ0IsQ0FBQztZQUNuQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1lBQ2hELElBQUksQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDLENBQUM7U0FDN0I7SUFDSCxDQUFDO0lBRU8sMEJBQTBCLENBQ2hDLFdBQTZDLEVBQzdDLE1BQXdDOztRQUV4QyxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDekIsTUFBTSxJQUFJLEtBQUssQ0FDYiwyREFBMkQsQ0FDNUQsQ0FBQztTQUNIO1FBRUQsTUFBTSxXQUFXLEdBQUcsSUFBSSxHQUFHLEVBQVUsQ0FBQztRQUN0QyxNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsbUJBQW1CLEVBQUUsQ0FBQztRQUVqRCxLQUFLLE1BQU0sYUFBYSxJQUFJLE1BQU0sQ0FBQyxpQkFBaUIsRUFBRTtZQUNwRCxNQUFNLFVBQVUsR0FBRyxNQUFNLENBQUMsaUJBQWlCLENBQUMsYUFBYSxDQUFDLENBQUM7WUFDM0QsSUFDRSxVQUFVLENBQUMscUJBQXFCO2dCQUNoQyxVQUFVLENBQUMscUJBQXFCLEVBQ2hDO2dCQUNBLE1BQU0sYUFBYSxHQUFHLGFBQWEsQ0FBQyw0QkFBNEI7Z0JBQzlELHFGQUFxRjtnQkFDckYscUZBQXFGO2dCQUNyRixhQUFhLENBQUMsMkJBQTJCLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxFQUN4RCxVQUFVLENBQUMseUJBQXlCLEVBQ3BDLGVBQWUsVUFBVSxDQUFDLHlCQUF5QixHQUFHLEVBQ3RELFNBQVM7Z0JBQ1QsbUdBQW1HO2dCQUNuRyw2QkFBNkI7b0JBQzNCLHFDQUE0QixDQUMxQixNQUFNLENBQUMsaUJBQWlCLEdBQUcsR0FBRyxHQUFHLGFBQWEsQ0FDL0M7Z0JBQ0gsMkNBQTJDO2dCQUMzQyxNQUFNLENBQUMsTUFBTSxtQ0FBSSxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FDdEMsQ0FBQztnQkFFRixNQUFNLFlBQVksR0FDaEIsSUFBSSxDQUFDLDRCQUE0QixDQUFDLHFCQUFxQixDQUNyRCxhQUFhLEVBQ2IsTUFBTSxDQUFDLGlCQUFpQixFQUN4QixhQUFhLEVBQ2IsVUFBVSxDQUNYLENBQUM7Z0JBRUosdUVBQXVFO2dCQUN2RSxJQUFJLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQyxDQUFDO2dCQUM1QixXQUFXLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO2FBQ3ZEO1NBQ0Y7UUFFRCxJQUFJLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxFQUFFO1lBQ3hCLE1BQU0saUJBQWlCLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDcEUsV0FBVyxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMscUJBQXFCLGlCQUFpQixFQUFFLENBQUMsQ0FBQztTQUN4RTtJQUNILENBQUM7O0FBNVRILDRDQTZUQzs7O0FBRUQsTUFBTSw2QkFBNkIsR0FBRyxRQUFRLENBQUM7QUFDL0MsTUFBTSw4QkFBOEIsR0FBRyxVQUFVLENBQUM7QUFDbEQsTUFBTSx5QkFBeUIsR0FBRyxPQUFPLENBQUM7QUFFMUM7Ozs7O0dBS0c7QUFDSCxNQUFNLDJCQUE0QixTQUFRLDRCQUFXO0lBQ25ELFlBQVksS0FBdUI7UUFDakMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ2YsQ0FBQztJQUVELE1BQU07O1FBQ0osTUFBTSxJQUFJLEdBQUcsS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDO1FBQzVCLElBQUksSUFBSSxDQUFDLE1BQU0sS0FBSyxDQUFDLElBQUksY0FBQyxJQUFJLGFBQUosSUFBSSx1QkFBSixJQUFJLENBQUcsQ0FBQywyQ0FBRyxVQUFVLDBDQUFFLE9BQU8sQ0FBQSxFQUFFO1lBQ3hELE1BQU0sSUFBSSxLQUFLLENBQ2Isb0ZBQW9GLENBQ3JGLENBQUM7U0FDSDtRQUNELE1BQU0sT0FBTyxHQUFVLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDO1FBQ2xELElBQUksT0FBTyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDdEIsTUFBTSxJQUFJLEtBQUssQ0FDYiwrRUFBK0UsQ0FDaEYsQ0FBQztTQUNIO1FBQ0QsTUFBTSwwQkFBMEIsU0FBVSxPQUFPLENBQUMsQ0FBQyxDQUFDLDBDQUFFLEtBQUssQ0FBQztRQUM1RCxJQUNFLENBQUMsMEJBQTBCO1lBQzNCLDBCQUEwQixDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQ3ZDO1lBQ0EsTUFBTSxJQUFJLEtBQUssQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO1NBQzVEO1FBQ0QsTUFBTSxtQkFBbUIsU0FBVSxPQUFPLENBQUMsQ0FBQyxDQUFDLDBDQUFFLEtBQUssQ0FBQztRQUNyRCxJQUNFLENBQUMsbUJBQW1CO1lBQ3BCLG1CQUFtQixDQUFDLE1BQU0sR0FBRyxDQUFDO1lBQzlCLENBQUMsbUJBQW1CLENBQUMsbUJBQW1CLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFDdkQ7WUFDQSxNQUFNLElBQUksS0FBSyxDQUFDLGdDQUFnQyxDQUFDLENBQUM7U0FDbkQ7UUFDRCx1Q0FBdUM7UUFDdkMsMEJBQTBCLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUM5QixtQkFBbUIsQ0FBQyxtQkFBbUIsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRTtnQkFDdEQseUJBQXlCLENBQUM7UUFDNUIsNkRBQTZEO1FBQzdELG1CQUFtQixDQUFDLG1CQUFtQixDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxPQUFPLEdBQUcsSUFBSSxDQUFDO1FBQ25FLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztDQUNGIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgRHVyYXRpb24gfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcbmltcG9ydCB7XG4gIERpbWVuc2lvbnNNYXAsXG4gIEdyYXBoV2lkZ2V0LFxuICBHcmFwaFdpZGdldFByb3BzLFxuICBIb3Jpem9udGFsQW5ub3RhdGlvbixcbiAgSU1ldHJpYyxcbiAgSVdpZGdldCxcbiAgUm93LFxuICBUZXh0V2lkZ2V0LFxuICBZQXhpc1Byb3BzLFxufSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWNsb3Vkd2F0Y2hcIjtcblxuaW1wb3J0IHtcbiAgQW5vbWFseURldGVjdGluZ0FsYXJtRmFjdG9yeSxcbiAgQW5vbWFseURldGVjdGlvblRocmVzaG9sZCxcbiAgQmFzZU1vbml0b3JpbmdQcm9wcyxcbiAgQ3VzdG9tQWxhcm1GYWN0b3J5LFxuICBDdXN0b21UaHJlc2hvbGQsXG4gIERlZmF1bHRHcmFwaFdpZGdldEhlaWdodCxcbiAgRGVmYXVsdFN1bW1hcnlXaWRnZXRIZWlnaHQsXG4gIEZ1bGxXaWR0aCxcbiAgR3JhcGhXaWRnZXRUeXBlLFxuICBNZXRyaWNTdGF0aXN0aWMsXG4gIE1ldHJpY1dpdGhBbGFybVN1cHBvcnQsXG4gIE1vbml0b3JpbmcsXG4gIE1vbml0b3JpbmdTY29wZSxcbiAgY3JlYXRlR3JhcGhXaWRnZXQsXG4gIGdldEhhc2hGb3JNZXRyaWNFeHByZXNzaW9uSWQsXG4gIHJlY29tbWVuZGVkV2lkZ2V0V2lkdGgsXG59IGZyb20gXCIuLi8uLi9jb21tb25cIjtcbmltcG9ydCB7XG4gIE1vbml0b3JpbmdIZWFkZXJXaWRnZXQsXG4gIE1vbml0b3JpbmdOYW1pbmdTdHJhdGVneSxcbn0gZnJvbSBcIi4uLy4uL2Rhc2hib2FyZFwiO1xuXG5leHBvcnQgZW51bSBBeGlzUG9zaXRpb24ge1xuICBMRUZUID0gXCJsZWZ0XCIsXG4gIFJJR0hUID0gXCJyaWdodFwiLFxufVxuXG4vKipcbiAqIEN1c3RvbSBtZXRyaWMgd2l0aCBhbiBhbGFybSBkZWZpbmVkLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEN1c3RvbU1ldHJpY1dpdGhBbGFybSB7XG4gIC8qKlxuICAgKiBtZXRyaWMgdG8gYWxhcm0gb25cbiAgICovXG4gIHJlYWRvbmx5IG1ldHJpYzogTWV0cmljV2l0aEFsYXJtU3VwcG9ydDtcbiAgLyoqXG4gICAqIGFsYXJtIGZyaWVuZGx5IG5hbWVcbiAgICovXG4gIHJlYWRvbmx5IGFsYXJtRnJpZW5kbHlOYW1lOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBhbGFybSBkZWZpbml0aW9uc1xuICAgKi9cbiAgcmVhZG9ubHkgYWRkQWxhcm06IFJlY29yZDxzdHJpbmcsIEN1c3RvbVRocmVzaG9sZD47XG4gIC8qKlxuICAgKiBheGlzIChyaWdodCBvciBsZWZ0KSBvbiB3aGljaCB0byBncmFwaCBtZXRyaWNcbiAgICogZGVmYXVsdDogQXhpc1Bvc2l0aW9uLkxFRlRcbiAgICovXG4gIHJlYWRvbmx5IHBvc2l0aW9uPzogQXhpc1Bvc2l0aW9uO1xufVxuXG4vKipcbiAqIEN1c3RvbSBtZXRyaWMgd2l0aCBhbm9tYWx5IGRldGVjdGlvbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDdXN0b21NZXRyaWNXaXRoQW5vbWFseURldGVjdGlvbiB7XG4gIC8qKlxuICAgKiBtZXRyaWMgdG8gYWxhcm0gb25cbiAgICovXG4gIHJlYWRvbmx5IG1ldHJpYzogTWV0cmljV2l0aEFsYXJtU3VwcG9ydDtcbiAgLyoqXG4gICAqIGFub21hbHkgZGV0ZWN0aW9uIHBlcmlvZFxuICAgKiBAZGVmYXVsdCBtZXRyaWMgcGVyaW9kIChpZiBkZWZpbmVkKSBvciBnbG9iYWwgZGVmYXVsdFxuICAgKi9cbiAgcmVhZG9ubHkgcGVyaW9kPzogRHVyYXRpb247XG4gIC8qKlxuICAgKiBhbGFybSBmcmllbmRseSBuYW1lXG4gICAqL1xuICByZWFkb25seSBhbGFybUZyaWVuZGx5TmFtZTogc3RyaW5nO1xuICAvKipcbiAgICogc3RhbmRhcmQgZGV2aWF0aW9uIGZvciB0aGUgYW5vbWFseSBkZXRlY3Rpb24gdG8gYmUgcmVuZGVyZWQgb24gdGhlIGdyYXBoIHdpZGdldFxuICAgKi9cbiAgcmVhZG9ubHkgYW5vbWFseURldGVjdGlvblN0YW5kYXJkRGV2aWF0aW9uVG9SZW5kZXI6IG51bWJlcjtcbiAgLyoqXG4gICAqIGFkZHMgYWxhcm0gb24gYSBkZXRlY3RlZCBhbm9tYWx5XG4gICAqL1xuICByZWFkb25seSBhZGRBbGFybU9uQW5vbWFseT86IFJlY29yZDxzdHJpbmcsIEFub21hbHlEZXRlY3Rpb25UaHJlc2hvbGQ+O1xufVxuXG4vKipcbiAqIEN1c3RvbSBtZXRyaWMgc2VhcmNoLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEN1c3RvbU1ldHJpY1NlYXJjaCB7XG4gIC8qKlxuICAgKiBtZXRyaWMgbmFtZXNwYWNlXG4gICAqIEBkZWZhdWx0IG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IG5hbWVzcGFjZT86IHN0cmluZztcbiAgLyoqXG4gICAqIHNlYXJjaCBxdWVyeSAoY2FuIGJlIGVtcHR5KVxuICAgKi9cbiAgcmVhZG9ubHkgc2VhcmNoUXVlcnk6IHN0cmluZztcbiAgLyoqXG4gICAqIGN1c3RvbSBsYWJlbCBmb3IgdGhlIG1ldHJpY3NcbiAgICogQGRlZmF1bHQgXCIgXCJcbiAgICovXG4gIHJlYWRvbmx5IGxhYmVsPzogc3RyaW5nO1xuICAvKipcbiAgICogc2VhcmNoIGRpbWVuc2lvbnMgKGNhbiBiZSBlbXB0eSlcbiAgICovXG4gIHJlYWRvbmx5IGRpbWVuc2lvbnNNYXA6IERpbWVuc2lvbnNNYXA7XG4gIC8qKlxuICAgKiBtZXRyaWMgc3RhdGlzdGljXG4gICAqL1xuICByZWFkb25seSBzdGF0aXN0aWM6IE1ldHJpY1N0YXRpc3RpYztcbiAgLyoqXG4gICAqIG1ldHJpYyBwZXJpb2RcbiAgICogQGRlZmF1bHQgZ2xvYmFsIGRlZmF1bHRcbiAgICovXG4gIHJlYWRvbmx5IHBlcmlvZD86IER1cmF0aW9uO1xuICAvKipcbiAgICogYXhpcyAocmlnaHQgb3IgbGVmdCkgb24gd2hpY2ggdG8gZ3JhcGggbWV0cmljXG4gICAqIGRlZmF1bHQ6IEF4aXNQb3NpdGlvbi5MRUZUXG4gICAqL1xuICByZWFkb25seSBwb3NpdGlvbj86IEF4aXNQb3NpdGlvbjtcbn1cblxuLyoqXG4gKiBFYWNoIGN1c3RvbSBtZXRyaWMgY2FuIGJlIG9mIGZvdXIgdHlwZXM6XG4gKiBAc2VlIE1ldHJpY1dpdGhBbGFybVN1cHBvcnQgZm9yIGEgc3RhbmRhcmQgbWV0cmljXG4gKiBAc2VlIEN1c3RvbU1ldHJpY1NlYXJjaCBmb3IgYSBzZWFyY2hcbiAqIEBzZWUgQ3VzdG9tTWV0cmljV2l0aEFsYXJtIGZvciBhIG1ldHJpYyB3aXRoIGFuIGFsYXJtXG4gKiBAc2VlIEN1c3RvbU1ldHJpY1dpdGhBbm9tYWx5RGV0ZWN0aW9uIGZvciBhIG1ldHJpYyB3aXRoIGFuIGFub21hbHkgZGV0ZWN0aW5nIGFsYXJtXG4gKi9cbmV4cG9ydCB0eXBlIEN1c3RvbU1ldHJpYyA9XG4gIHwgTWV0cmljV2l0aEFsYXJtU3VwcG9ydFxuICB8IEN1c3RvbU1ldHJpY1NlYXJjaFxuICB8IEN1c3RvbU1ldHJpY1dpdGhBbGFybVxuICB8IEN1c3RvbU1ldHJpY1dpdGhBbm9tYWx5RGV0ZWN0aW9uO1xuXG4vKipcbiAqIEN1c3RvbSBtZXRyaWMgZ3JvdXAgcmVwcmVzZW50cyBhIHNpbmdsZSB3aWRnZXQuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ3VzdG9tTWV0cmljR3JvdXAge1xuICAvKipcbiAgICogdGl0bGUgb2YgdGhlIHdob2xlIGdyb3VwXG4gICAqL1xuICByZWFkb25seSB0aXRsZTogc3RyaW5nO1xuICAvKipcbiAgICogdHlwZSBvZiB0aGUgd2lkZ2V0XG4gICAqIEBkZWZhdWx0IGxpbmVcbiAgICovXG4gIHJlYWRvbmx5IGdyYXBoV2lkZ2V0VHlwZT86IEdyYXBoV2lkZ2V0VHlwZTtcbiAgLyoqXG4gICAqIG9wdGlvbmFsIGF4aXNcbiAgICogQGRlZmF1bHQgdW5kZWZpbmVkXG4gICAqL1xuICByZWFkb25seSBncmFwaFdpZGdldEF4aXM/OiBZQXhpc1Byb3BzO1xuICAvKipcbiAgICogb3B0aW9uYWwgcmlnaHQgYXhpc1xuICAgKiBkZWZhdWx0OiB1bmRlZmluZWRcbiAgICovXG4gIHJlYWRvbmx5IGdyYXBoV2lkZ2V0UmlnaHRBeGlzPzogWUF4aXNQcm9wcztcbiAgLyoqXG4gICAqIEZsYWcgaW5kaWNhdGluZywgd2hldGhlciB0aGlzIGlzIGFuIGltcG9ydGFudCBtZXRyaWMgZ3JvdXAgdGhhdCBzaG91bGQgYmUgaW5jbHVkZWQgaW4gdGhlIHN1bW1hcnkgYXMgd2VsbC5cbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGltcG9ydGFudD86IGJvb2xlYW47XG4gIC8qKlxuICAgKiBsaXN0IG9mIG1ldHJpY3MgaW4gdGhlIGdyb3VwIChjYW4gYmUgZGVmaW5lZCBpbiBkaWZmZXJlbnQgd2F5cywgc2VlIHRoZSB0eXBlIGRvY3VtZW50YXRpb24pXG4gICAqL1xuICByZWFkb25seSBtZXRyaWNzOiBDdXN0b21NZXRyaWNbXTtcbiAgLyoqXG4gICAqIG9wdGlvbmFsIGN1c3RvbSBob3Jpem9udGFsIGFubm90YXRpb25zIHdoaWNoIHdpbGwgYmUgZGlzcGxheWVkIG92ZXIgdGhlIG1ldHJpY3Mgb24gdGhlIGxlZnQgYXhpc1xuICAgKiAoaWYgdGhlcmUgYXJlIGFueSBhbGFybXMsIGFueSBleGlzdGluZyBhbm5vdGF0aW9ucyB3aWxsIGJlIG1lcmdlZCB0b2dldGhlcilcbiAgICovXG4gIHJlYWRvbmx5IGhvcml6b250YWxBbm5vdGF0aW9ucz86IEhvcml6b250YWxBbm5vdGF0aW9uW107XG4gIC8qKlxuICAgKiBvcHRpb25hbCBjdXN0b20gaG9yaXpvbnRhbCBhbm5vdGF0aW9ucyB3aGljaCB3aWxsIGJlIGRpc3BsYXllZCBvdmVyIHRoZSBtZXRyaWNzIG9uIHRoZSByaWdodCBheGlzXG4gICAqIChpZiB0aGVyZSBhcmUgYW55IGFsYXJtcywgYW55IGV4aXN0aW5nIGFubm90YXRpb25zIHdpbGwgYmUgbWVyZ2VkIHRvZ2V0aGVyKVxuICAgKi9cbiAgcmVhZG9ubHkgaG9yaXpvbnRhbFJpZ2h0QW5ub3RhdGlvbnM/OiBIb3Jpem9udGFsQW5ub3RhdGlvbltdO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIEN1c3RvbU1vbml0b3JpbmdQcm9wcyBleHRlbmRzIEJhc2VNb25pdG9yaW5nUHJvcHMge1xuICByZWFkb25seSBkZXNjcmlwdGlvbj86IHN0cmluZztcbiAgcmVhZG9ubHkgZGVzY3JpcHRpb25XaWRnZXRIZWlnaHQ/OiBudW1iZXI7XG4gIHJlYWRvbmx5IG1ldHJpY0dyb3VwczogQ3VzdG9tTWV0cmljR3JvdXBbXTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBDdXN0b21NZXRyaWNHcm91cFdpdGhBbm5vdGF0aW9ucyB7XG4gIHJlYWRvbmx5IG1ldHJpY0dyb3VwOiBDdXN0b21NZXRyaWNHcm91cDtcbiAgcmVhZG9ubHkgYW5ub3RhdGlvbnM6IEhvcml6b250YWxBbm5vdGF0aW9uW107XG4gIHJlYWRvbmx5IHJpZ2h0QW5ub3RhdGlvbnM6IEhvcml6b250YWxBbm5vdGF0aW9uW107XG4gIHJlYWRvbmx5IHRpdGxlQWRkb25zOiBzdHJpbmdbXTtcbn1cblxuLyoqXG4gKiBDdXN0b20gbW9uaXRvcmluZyBpcyBhIGNvbnN0cnVjdCBhbGxvd2luZyB5b3UgdG8gbW9uaXRvciB5b3VyIG93biBjdXN0b20gbWV0cmljcy5cbiAqIFRoZSBlbnRpcmUgY29uc3RydWN0IGNvbnNpc3RzIG9mIG1ldHJpYyBncm91cHMuXG4gKiBFYWNoIG1ldHJpYyBncm91cCByZXByZXNlbnRzIGEgc2luZ2xlIGdyYXBoIHdpZGdldCB3aXRoIG11bHRpcGxlIG1ldHJpY3MuXG4gKiBFYWNoIG1ldHJpYyBpbnNpZGUgdGhlIG1ldHJpYyBncm91cCByZXByZXNlbnRzIGEgc2luZ2xlIG1ldHJpYyBpbnNpZGUgYSBncmFwaC5cbiAqIFRoZSB3aWRnZXRzIHdpbGwgYmUgc2l6ZWQgYXV0b21hdGljYWxseSB0byB3YXN0ZSBhcyBsaXR0bGUgc3BhY2UgYXMgcG9zc2libGUuXG4gKi9cbmV4cG9ydCBjbGFzcyBDdXN0b21Nb25pdG9yaW5nIGV4dGVuZHMgTW9uaXRvcmluZyB7XG4gIHByb3RlY3RlZCByZWFkb25seSB0aXRsZTogc3RyaW5nO1xuICBwcm90ZWN0ZWQgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG4gIHByb3RlY3RlZCByZWFkb25seSBkZXNjcmlwdGlvbldpZGdldEhlaWdodD86IG51bWJlcjtcbiAgcHJvdGVjdGVkIHJlYWRvbmx5IGN1c3RvbUFsYXJtRmFjdG9yeTogQ3VzdG9tQWxhcm1GYWN0b3J5O1xuICBwcm90ZWN0ZWQgcmVhZG9ubHkgYW5vbWFseURldGVjdGluZ0FsYXJtRmFjdG9yeTogQW5vbWFseURldGVjdGluZ0FsYXJtRmFjdG9yeTtcbiAgcHJvdGVjdGVkIHJlYWRvbmx5IG1ldHJpY0dyb3VwczogQ3VzdG9tTWV0cmljR3JvdXBXaXRoQW5ub3RhdGlvbnNbXTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogTW9uaXRvcmluZ1Njb3BlLCBwcm9wczogQ3VzdG9tTW9uaXRvcmluZ1Byb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIHByb3BzKTtcblxuICAgIGNvbnN0IG5hbWluZ1N0cmF0ZWd5ID0gbmV3IE1vbml0b3JpbmdOYW1pbmdTdHJhdGVneSh7IC4uLnByb3BzIH0pO1xuICAgIHRoaXMudGl0bGUgPSBuYW1pbmdTdHJhdGVneS5yZXNvbHZlSHVtYW5SZWFkYWJsZU5hbWUoKTtcblxuICAgIHRoaXMuZGVzY3JpcHRpb24gPSBwcm9wcy5kZXNjcmlwdGlvbjtcbiAgICB0aGlzLmRlc2NyaXB0aW9uV2lkZ2V0SGVpZ2h0ID0gcHJvcHMuZGVzY3JpcHRpb25XaWRnZXRIZWlnaHQ7XG5cbiAgICBjb25zdCBhbGFybUZhY3RvcnkgPSB0aGlzLmNyZWF0ZUFsYXJtRmFjdG9yeShcbiAgICAgIG5hbWluZ1N0cmF0ZWd5LnJlc29sdmVBbGFybUZyaWVuZGx5TmFtZSgpXG4gICAgKTtcbiAgICB0aGlzLmN1c3RvbUFsYXJtRmFjdG9yeSA9IG5ldyBDdXN0b21BbGFybUZhY3RvcnkoYWxhcm1GYWN0b3J5KTtcbiAgICB0aGlzLmFub21hbHlEZXRlY3RpbmdBbGFybUZhY3RvcnkgPSBuZXcgQW5vbWFseURldGVjdGluZ0FsYXJtRmFjdG9yeShcbiAgICAgIGFsYXJtRmFjdG9yeVxuICAgICk7XG5cbiAgICB0aGlzLm1ldHJpY0dyb3VwcyA9IHByb3BzLm1ldHJpY0dyb3Vwcy5tYXAoKG1ldHJpY0dyb3VwKSA9PiB7XG4gICAgICBjb25zdCBtZXRyaWNHcm91cFdpdGhBbm5vdGF0aW9uOiBDdXN0b21NZXRyaWNHcm91cFdpdGhBbm5vdGF0aW9ucyA9IHtcbiAgICAgICAgbWV0cmljR3JvdXAsXG4gICAgICAgIGFubm90YXRpb25zOiBbXSxcbiAgICAgICAgcmlnaHRBbm5vdGF0aW9uczogW10sXG4gICAgICAgIHRpdGxlQWRkb25zOiBbXSxcbiAgICAgIH07XG5cbiAgICAgIGlmIChtZXRyaWNHcm91cC5ob3Jpem9udGFsQW5ub3RhdGlvbnMpIHtcbiAgICAgICAgbWV0cmljR3JvdXBXaXRoQW5ub3RhdGlvbi5hbm5vdGF0aW9ucy5wdXNoKFxuICAgICAgICAgIC4uLm1ldHJpY0dyb3VwLmhvcml6b250YWxBbm5vdGF0aW9uc1xuICAgICAgICApO1xuICAgICAgfVxuICAgICAgaWYgKG1ldHJpY0dyb3VwLmhvcml6b250YWxSaWdodEFubm90YXRpb25zKSB7XG4gICAgICAgIG1ldHJpY0dyb3VwV2l0aEFubm90YXRpb24ucmlnaHRBbm5vdGF0aW9ucy5wdXNoKFxuICAgICAgICAgIC4uLm1ldHJpY0dyb3VwLmhvcml6b250YWxSaWdodEFubm90YXRpb25zXG4gICAgICAgICk7XG4gICAgICB9XG5cbiAgICAgIG1ldHJpY0dyb3VwLm1ldHJpY3MuZm9yRWFjaCgobWV0cmljKSA9PiB7XG4gICAgICAgIGlmICh0aGlzLmhhc0FsYXJtKG1ldHJpYykgJiYgdGhpcy5oYXNBbm9tYWx5RGV0ZWN0aW9uKG1ldHJpYykpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgICBcIkFkZGluZyBib3RoIGEgcmVndWxhciBhbGFybSBhbmQgYW4gYW5vbW9seSBkZXRlY3Rpb24gYWxhcm0gYXQgdGhlIHNhbWUgdGltZSBpcyBub3Qgc3VwcG9ydGVkXCJcbiAgICAgICAgICApO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKHRoaXMuaGFzQWxhcm0obWV0cmljKSkge1xuICAgICAgICAgIHRoaXMuc2V0dXBBbGFybShtZXRyaWNHcm91cFdpdGhBbm5vdGF0aW9uLCBtZXRyaWMpO1xuICAgICAgICB9IGVsc2UgaWYgKHRoaXMuaGFzQW5vbWFseURldGVjdGlvbihtZXRyaWMpKSB7XG4gICAgICAgICAgdGhpcy5zZXR1cEFub21hbHlEZXRlY3Rpb25BbGFybShtZXRyaWNHcm91cFdpdGhBbm5vdGF0aW9uLCBtZXRyaWMpO1xuICAgICAgICB9XG4gICAgICB9KTtcblxuICAgICAgcmV0dXJuIG1ldHJpY0dyb3VwV2l0aEFubm90YXRpb247XG4gICAgfSk7XG5cbiAgICBwcm9wcy51c2VDcmVhdGVkQWxhcm1zPy5jb25zdW1lKHRoaXMuY3JlYXRlZEFsYXJtcygpKTtcbiAgfVxuXG4gIHN1bW1hcnlXaWRnZXRzKCk6IElXaWRnZXRbXSB7XG4gICAgcmV0dXJuIHRoaXMuZ2V0QWxsV2lkZ2V0cyh0cnVlKTtcbiAgfVxuXG4gIHdpZGdldHMoKTogSVdpZGdldFtdIHtcbiAgICByZXR1cm4gdGhpcy5nZXRBbGxXaWRnZXRzKGZhbHNlKTtcbiAgfVxuXG4gIHByaXZhdGUgZ2V0QWxsV2lkZ2V0cyhzdW1tYXJ5OiBib29sZWFuKTogSVdpZGdldFtdIHtcbiAgICBjb25zdCBmaWx0ZXJlZE1ldHJpY0dyb3VwcyA9IHN1bW1hcnlcbiAgICAgID8gdGhpcy5tZXRyaWNHcm91cHMuZmlsdGVyKFxuICAgICAgICAgIChncm91cCkgPT4gZ3JvdXAubWV0cmljR3JvdXAuaW1wb3J0YW50ID8/IGZhbHNlXG4gICAgICAgIClcbiAgICAgIDogdGhpcy5tZXRyaWNHcm91cHM7XG5cbiAgICBpZiAoZmlsdGVyZWRNZXRyaWNHcm91cHMubGVuZ3RoIDwgMSkge1xuICAgICAgLy8gc2hvcnQtY2lyY3VpdCBpZiB0aGVyZSBhcmUgbm8gbWV0cmljcyBzcGVjaWZpZWRcbiAgICAgIHJldHVybiBbXTtcbiAgICB9XG5cbiAgICBjb25zdCByb3dzOiBSb3dbXSA9IFtdO1xuXG4gICAgLy8gaGVhZGVyIGFuZCBkZXNjcmlwdGlvblxuICAgIHJvd3MucHVzaChuZXcgUm93KG5ldyBNb25pdG9yaW5nSGVhZGVyV2lkZ2V0KHsgdGl0bGU6IHRoaXMudGl0bGUgfSkpKTtcbiAgICBpZiAodGhpcy5kZXNjcmlwdGlvbiAmJiAhc3VtbWFyeSkge1xuICAgICAgcm93cy5wdXNoKFxuICAgICAgICBuZXcgUm93KFxuICAgICAgICAgIHRoaXMuY3JlYXRlRGVzY3JpcHRpb25XaWRnZXQoXG4gICAgICAgICAgICB0aGlzLmRlc2NyaXB0aW9uLFxuICAgICAgICAgICAgdGhpcy5kZXNjcmlwdGlvbldpZGdldEhlaWdodFxuICAgICAgICAgIClcbiAgICAgICAgKVxuICAgICAgKTtcbiAgICB9XG5cbiAgICAvLyBncmFwaHNcbiAgICByb3dzLnB1c2goXG4gICAgICBuZXcgUm93KFxuICAgICAgICAuLi50aGlzLmNyZWF0ZUN1c3RvbU1ldHJpY0dyb3VwV2lkZ2V0cyhmaWx0ZXJlZE1ldHJpY0dyb3Vwcywgc3VtbWFyeSlcbiAgICAgIClcbiAgICApO1xuXG4gICAgcmV0dXJuIHJvd3M7XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZURlc2NyaXB0aW9uV2lkZ2V0KFxuICAgIG1hcmtkb3duOiBzdHJpbmcsXG4gICAgZGVzY3JpcHRpb25XaWRnZXRIZWlnaHQ/OiBudW1iZXJcbiAgKSB7XG4gICAgcmV0dXJuIG5ldyBUZXh0V2lkZ2V0KHtcbiAgICAgIG1hcmtkb3duLFxuICAgICAgd2lkdGg6IEZ1bGxXaWR0aCxcbiAgICAgIGhlaWdodDogZGVzY3JpcHRpb25XaWRnZXRIZWlnaHQgPz8gMSxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlQ3VzdG9tTWV0cmljR3JvdXBXaWRnZXRzKFxuICAgIGFubm90YXRlZEdyb3VwczogQ3VzdG9tTWV0cmljR3JvdXBXaXRoQW5ub3RhdGlvbnNbXSxcbiAgICBzdW1tYXJ5OiBib29sZWFuXG4gICkge1xuICAgIGNvbnN0IHdpZGdldHM6IEdyYXBoV2lkZ2V0W10gPSBbXTtcbiAgICBjb25zdCBtZXRyaWNHcm91cFdpZGdldFdpZHRoID0gcmVjb21tZW5kZWRXaWRnZXRXaWR0aChcbiAgICAgIGFubm90YXRlZEdyb3Vwcy5sZW5ndGhcbiAgICApO1xuXG4gICAgYW5ub3RhdGVkR3JvdXBzLmZvckVhY2goKGFubm90YXRlZEdyb3VwKSA9PiB7XG4gICAgICBjb25zdCBtZXRyaWNzID0gYW5ub3RhdGVkR3JvdXAubWV0cmljR3JvdXAubWV0cmljcztcbiAgICAgIGNvbnN0IGxlZnQgPSB0aGlzLnRvTWV0cmljcyhcbiAgICAgICAgbWV0cmljcy5maWx0ZXIoXG4gICAgICAgICAgKG1ldHJpYykgPT5cbiAgICAgICAgICAgICgobWV0cmljIGFzIGFueSkucG9zaXRpb24gPz8gQXhpc1Bvc2l0aW9uLkxFRlQpID09IEF4aXNQb3NpdGlvbi5MRUZUXG4gICAgICAgIClcbiAgICAgICk7XG4gICAgICBjb25zdCByaWdodCA9IHRoaXMudG9NZXRyaWNzKFxuICAgICAgICBtZXRyaWNzLmZpbHRlcihcbiAgICAgICAgICAobWV0cmljKSA9PlxuICAgICAgICAgICAgKChtZXRyaWMgYXMgYW55KS5wb3NpdGlvbiA/PyBBeGlzUG9zaXRpb24uTEVGVCkgPT1cbiAgICAgICAgICAgIEF4aXNQb3NpdGlvbi5SSUdIVFxuICAgICAgICApXG4gICAgICApO1xuICAgICAgY29uc3QgaGFzT25lTWV0cmljT25seSA9IG1ldHJpY3MubGVuZ3RoID09PSAxO1xuICAgICAgY29uc3QgaGFzQW5vbWFseURldGVjdGlvbiA9XG4gICAgICAgIG1ldHJpY3MuZmlsdGVyKChtZXRyaWMpID0+IHRoaXMuaGFzQW5vbWFseURldGVjdGlvbihtZXRyaWMpKS5sZW5ndGggPiAwO1xuICAgICAgY29uc3QgdXNlQW5vbWFseURldGVjdGlvbldpZGdldCA9IGhhc09uZU1ldHJpY09ubHkgJiYgaGFzQW5vbWFseURldGVjdGlvbjtcbiAgICAgIGxldCB0aXRsZSA9IGFubm90YXRlZEdyb3VwLm1ldHJpY0dyb3VwLnRpdGxlO1xuXG4gICAgICBpZiAoYW5ub3RhdGVkR3JvdXAudGl0bGVBZGRvbnMubGVuZ3RoID4gMCkge1xuICAgICAgICB0aXRsZSA9IGAke3RpdGxlfSAoJHthbm5vdGF0ZWRHcm91cC50aXRsZUFkZG9ucy5qb2luKFwiLCBcIil9KWA7XG4gICAgICB9XG5cbiAgICAgIGNvbnN0IGdyYXBoV2lkZ2V0UHJvcHM6IEdyYXBoV2lkZ2V0UHJvcHMgPSB7XG4gICAgICAgIHRpdGxlLFxuICAgICAgICB3aWR0aDogbWV0cmljR3JvdXBXaWRnZXRXaWR0aCxcbiAgICAgICAgaGVpZ2h0OiBzdW1tYXJ5ID8gRGVmYXVsdFN1bW1hcnlXaWRnZXRIZWlnaHQgOiBEZWZhdWx0R3JhcGhXaWRnZXRIZWlnaHQsXG4gICAgICAgIGxlZnQsXG4gICAgICAgIHJpZ2h0LFxuICAgICAgICBsZWZ0QW5ub3RhdGlvbnM6IGFubm90YXRlZEdyb3VwLmFubm90YXRpb25zLFxuICAgICAgICByaWdodEFubm90YXRpb25zOiBhbm5vdGF0ZWRHcm91cC5yaWdodEFubm90YXRpb25zLFxuICAgICAgICBsZWZ0WUF4aXM6IGFubm90YXRlZEdyb3VwLm1ldHJpY0dyb3VwLmdyYXBoV2lkZ2V0QXhpcyxcbiAgICAgICAgcmlnaHRZQXhpczogYW5ub3RhdGVkR3JvdXAubWV0cmljR3JvdXAuZ3JhcGhXaWRnZXRSaWdodEF4aXMsXG4gICAgICB9O1xuXG4gICAgICBjb25zdCB3aWRnZXQgPSB1c2VBbm9tYWx5RGV0ZWN0aW9uV2lkZ2V0XG4gICAgICAgID8gbmV3IEFub21hbHlEZXRlY3Rpb25HcmFwaFdpZGdldChncmFwaFdpZGdldFByb3BzKVxuICAgICAgICA6IGNyZWF0ZUdyYXBoV2lkZ2V0KFxuICAgICAgICAgICAgYW5ub3RhdGVkR3JvdXAubWV0cmljR3JvdXAuZ3JhcGhXaWRnZXRUeXBlID8/IEdyYXBoV2lkZ2V0VHlwZS5MSU5FLFxuICAgICAgICAgICAgZ3JhcGhXaWRnZXRQcm9wc1xuICAgICAgICAgICk7XG5cbiAgICAgIHdpZGdldHMucHVzaCh3aWRnZXQpO1xuICAgIH0pO1xuXG4gICAgcmV0dXJuIHdpZGdldHM7XG4gIH1cblxuICBwcml2YXRlIHRvTWV0cmljcyhtZXRyaWNzOiBDdXN0b21NZXRyaWNbXSk6IElNZXRyaWNbXSB7XG4gICAgY29uc3QgbWV0cmljRmFjdG9yeSA9IHRoaXMuY3JlYXRlTWV0cmljRmFjdG9yeSgpO1xuXG4gICAgcmV0dXJuIG1ldHJpY3MubWFwKChtZXRyaWMpID0+IHtcbiAgICAgIGlmICh0aGlzLmhhc0FsYXJtKG1ldHJpYykpIHtcbiAgICAgICAgLy8gbWV0cmljIHdpdGggYWxhcm1cbiAgICAgICAgcmV0dXJuIG1ldHJpY0ZhY3RvcnkuYWRhcHRNZXRyaWNQcmVzZXJ2aW5nUGVyaW9kKG1ldHJpYy5tZXRyaWMpO1xuICAgICAgfSBlbHNlIGlmICh0aGlzLmhhc0Fub21hbHlEZXRlY3Rpb24obWV0cmljKSkge1xuICAgICAgICAvLyBtZXRyaWMgd2l0aCBhbm9tYWx5IGRldGVjdGlvblxuICAgICAgICByZXR1cm4gbWV0cmljRmFjdG9yeS5jcmVhdGVNZXRyaWNBbm9tYWx5RGV0ZWN0aW9uKFxuICAgICAgICAgIG1ldHJpYy5tZXRyaWMsXG4gICAgICAgICAgbWV0cmljLmFub21hbHlEZXRlY3Rpb25TdGFuZGFyZERldmlhdGlvblRvUmVuZGVyLFxuICAgICAgICAgIGBFeHBlY3RlZCAoc3RkZXYgPSAke21ldHJpYy5hbm9tYWx5RGV0ZWN0aW9uU3RhbmRhcmREZXZpYXRpb25Ub1JlbmRlcn0pYCxcbiAgICAgICAgICB1bmRlZmluZWQsXG4gICAgICAgICAgLy8gbmVlZHMgdG8gYmUgdW5pcXVlIGluIHRoZSB3aG9sZSB3aWRnZXQgYW5kIHN0YXJ0IHdpdGggbG93ZXJjYXNlXG4gICAgICAgICAgQW5vbWFseURldGVjdGlvbk1ldHJpY0lkUHJlZml4ICtcbiAgICAgICAgICAgIGdldEhhc2hGb3JNZXRyaWNFeHByZXNzaW9uSWQobWV0cmljLmFsYXJtRnJpZW5kbHlOYW1lKSxcbiAgICAgICAgICAvLyBwcmVzZXJ2ZSB0aGUgbW9zdCBzcGVjaWZpYyBtZXRyaWMgcGVyaW9kXG4gICAgICAgICAgbWV0cmljLnBlcmlvZCA/PyBtZXRyaWMubWV0cmljLnBlcmlvZFxuICAgICAgICApO1xuICAgICAgfSBlbHNlIGlmICh0aGlzLmlzU2VhcmNoKG1ldHJpYykpIHtcbiAgICAgICAgLy8gbWV0cmljIHNlYXJjaFxuICAgICAgICByZXR1cm4gbWV0cmljRmFjdG9yeS5jcmVhdGVNZXRyaWNTZWFyY2goXG4gICAgICAgICAgbWV0cmljLnNlYXJjaFF1ZXJ5LFxuICAgICAgICAgIG1ldHJpYy5kaW1lbnNpb25zTWFwLFxuICAgICAgICAgIG1ldHJpYy5zdGF0aXN0aWMsXG4gICAgICAgICAgbWV0cmljLm5hbWVzcGFjZSxcbiAgICAgICAgICBtZXRyaWMubGFiZWwsXG4gICAgICAgICAgbWV0cmljLnBlcmlvZFxuICAgICAgICApO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgLy8gZ2VuZXJhbCBtZXRyaWNcbiAgICAgICAgcmV0dXJuIG1ldHJpY0ZhY3RvcnkuYWRhcHRNZXRyaWNQcmVzZXJ2aW5nUGVyaW9kKG1ldHJpYyk7XG4gICAgICB9XG4gICAgfSk7XG4gIH1cblxuICBwcml2YXRlIGhhc0FsYXJtKG1ldHJpYzogQ3VzdG9tTWV0cmljKTogbWV0cmljIGlzIEN1c3RvbU1ldHJpY1dpdGhBbGFybSB7XG4gICAgLy8gdHlwZSBndWFyZFxuICAgIHJldHVybiAobWV0cmljIGFzIEN1c3RvbU1ldHJpY1dpdGhBbGFybSkuYWRkQWxhcm0gIT09IHVuZGVmaW5lZDtcbiAgfVxuXG4gIHByaXZhdGUgaGFzQW5vbWFseURldGVjdGlvbihcbiAgICBtZXRyaWM6IEN1c3RvbU1ldHJpY1xuICApOiBtZXRyaWMgaXMgQ3VzdG9tTWV0cmljV2l0aEFub21hbHlEZXRlY3Rpb24ge1xuICAgIC8vIHR5cGUgZ3VhcmRcbiAgICByZXR1cm4gKFxuICAgICAgKG1ldHJpYyBhcyBDdXN0b21NZXRyaWNXaXRoQW5vbWFseURldGVjdGlvbilcbiAgICAgICAgLmFub21hbHlEZXRlY3Rpb25TdGFuZGFyZERldmlhdGlvblRvUmVuZGVyICE9PSB1bmRlZmluZWRcbiAgICApO1xuICB9XG5cbiAgcHJpdmF0ZSBpc1NlYXJjaChtZXRyaWM6IEN1c3RvbU1ldHJpYyk6IG1ldHJpYyBpcyBDdXN0b21NZXRyaWNTZWFyY2gge1xuICAgIC8vIHR5cGUgZ3VhcmRcbiAgICByZXR1cm4gKG1ldHJpYyBhcyBDdXN0b21NZXRyaWNTZWFyY2gpLnNlYXJjaFF1ZXJ5ICE9PSB1bmRlZmluZWQ7XG4gIH1cblxuICBwcml2YXRlIHNldHVwQWxhcm0oXG4gICAgbWV0cmljR3JvdXA6IEN1c3RvbU1ldHJpY0dyb3VwV2l0aEFubm90YXRpb25zLFxuICAgIG1ldHJpYzogQ3VzdG9tTWV0cmljV2l0aEFsYXJtXG4gICkge1xuICAgIGlmICh0aGlzLmlzU2VhcmNoKG1ldHJpYykpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgXCJBbGFybWluZyBvbiBzZWFyY2ggcXVlcmllcyBpcyBub3Qgc3VwcG9ydGVkIGJ5IENsb3VkV2F0Y2hcIlxuICAgICAgKTtcbiAgICB9XG5cbiAgICBmb3IgKGNvbnN0IGRpc2FtYmlndWF0b3IgaW4gbWV0cmljLmFkZEFsYXJtKSB7XG4gICAgICBjb25zdCBhbGFybVByb3BzID0gbWV0cmljLmFkZEFsYXJtW2Rpc2FtYmlndWF0b3JdO1xuICAgICAgY29uc3QgY3JlYXRlZEFsYXJtID0gdGhpcy5jdXN0b21BbGFybUZhY3RvcnkuYWRkQ3VzdG9tQWxhcm0oXG4gICAgICAgIG1ldHJpYy5tZXRyaWMsXG4gICAgICAgIG1ldHJpYy5hbGFybUZyaWVuZGx5TmFtZSxcbiAgICAgICAgZGlzYW1iaWd1YXRvcixcbiAgICAgICAgYWxhcm1Qcm9wc1xuICAgICAgKTtcbiAgICAgIGNvbnN0IHRhcmdldEFubm90YXRpb25zID1cbiAgICAgICAgKG1ldHJpYy5wb3NpdGlvbiA/PyBBeGlzUG9zaXRpb24uTEVGVCkgPT0gQXhpc1Bvc2l0aW9uLkxFRlRcbiAgICAgICAgICA/IG1ldHJpY0dyb3VwLmFubm90YXRpb25zXG4gICAgICAgICAgOiBtZXRyaWNHcm91cC5yaWdodEFubm90YXRpb25zO1xuICAgICAgdGFyZ2V0QW5ub3RhdGlvbnMucHVzaChjcmVhdGVkQWxhcm0uYW5ub3RhdGlvbik7XG4gICAgICB0aGlzLmFkZEFsYXJtKGNyZWF0ZWRBbGFybSk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBzZXR1cEFub21hbHlEZXRlY3Rpb25BbGFybShcbiAgICBtZXRyaWNHcm91cDogQ3VzdG9tTWV0cmljR3JvdXBXaXRoQW5ub3RhdGlvbnMsXG4gICAgbWV0cmljOiBDdXN0b21NZXRyaWNXaXRoQW5vbWFseURldGVjdGlvblxuICApIHtcbiAgICBpZiAodGhpcy5pc1NlYXJjaChtZXRyaWMpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIFwiQWxhcm1pbmcgb24gc2VhcmNoIHF1ZXJpZXMgaXMgbm90IHN1cHBvcnRlZCBieSBDbG91ZFdhdGNoXCJcbiAgICAgICk7XG4gICAgfVxuXG4gICAgY29uc3QgYWxhcm1TdERldnMgPSBuZXcgU2V0PG51bWJlcj4oKTtcbiAgICBjb25zdCBtZXRyaWNGYWN0b3J5ID0gdGhpcy5jcmVhdGVNZXRyaWNGYWN0b3J5KCk7XG5cbiAgICBmb3IgKGNvbnN0IGRpc2FtYmlndWF0b3IgaW4gbWV0cmljLmFkZEFsYXJtT25Bbm9tYWx5KSB7XG4gICAgICBjb25zdCBhbGFybVByb3BzID0gbWV0cmljLmFkZEFsYXJtT25Bbm9tYWx5W2Rpc2FtYmlndWF0b3JdO1xuICAgICAgaWYgKFxuICAgICAgICBhbGFybVByb3BzLmFsYXJtV2hlbkFib3ZlVGhlQmFuZCB8fFxuICAgICAgICBhbGFybVByb3BzLmFsYXJtV2hlbkJlbG93VGhlQmFuZFxuICAgICAgKSB7XG4gICAgICAgIGNvbnN0IGFub21hbHlNZXRyaWMgPSBtZXRyaWNGYWN0b3J5LmNyZWF0ZU1ldHJpY0Fub21hbHlEZXRlY3Rpb24oXG4gICAgICAgICAgLy8gQmVjYXVzZSB0aGUgbWV0cmljIHdhcyBwcm92aWRlZCB0byB1cywgd2UgdXNlIG1ldHJpY0ZhY3Rvcnkub3ZlcnJpZGVOYW1lc3BhY2UoKSB0b1xuICAgICAgICAgIC8vIGNvbmZpcm0gaXQgYWxpZ25zIHdpdGggYW55IG5hbWVzcGFjZSBvdmVycmlkZXMgcmVxdWVzdGVkIGZvciB0aGlzIE1vbml0b3JpbmdGYWNhZGVcbiAgICAgICAgICBtZXRyaWNGYWN0b3J5LmFkYXB0TWV0cmljUHJlc2VydmluZ1BlcmlvZChtZXRyaWMubWV0cmljKSxcbiAgICAgICAgICBhbGFybVByb3BzLnN0YW5kYXJkRGV2aWF0aW9uRm9yQWxhcm0sXG4gICAgICAgICAgYEJhbmQgKHN0ZGV2ICR7YWxhcm1Qcm9wcy5zdGFuZGFyZERldmlhdGlvbkZvckFsYXJtfSlgLFxuICAgICAgICAgIHVuZGVmaW5lZCxcbiAgICAgICAgICAvLyBleHByZXNzaW9uIElEIG5lZWRzIHRvIGJlIHVuaXF1ZSBhY3Jvc3MgdGhlIHdob2xlIHdpZGdldDsgbmVlZHMgdG8gc3RhcnQgd2l0aCBhIGxvd2VyY2FzZSBsZXR0ZXJcbiAgICAgICAgICBBbm9tYWx5RGV0ZWN0aW9uQWxhcm1JZFByZWZpeCArXG4gICAgICAgICAgICBnZXRIYXNoRm9yTWV0cmljRXhwcmVzc2lvbklkKFxuICAgICAgICAgICAgICBtZXRyaWMuYWxhcm1GcmllbmRseU5hbWUgKyBcIl9cIiArIGRpc2FtYmlndWF0b3JcbiAgICAgICAgICAgICksXG4gICAgICAgICAgLy8gcHJlc2VydmUgdGhlIG1vc3Qtc3BlY2lmaWMgbWV0cmljIHBlcmlvZFxuICAgICAgICAgIG1ldHJpYy5wZXJpb2QgPz8gbWV0cmljLm1ldHJpYy5wZXJpb2RcbiAgICAgICAgKTtcblxuICAgICAgICBjb25zdCBjcmVhdGVkQWxhcm0gPVxuICAgICAgICAgIHRoaXMuYW5vbWFseURldGVjdGluZ0FsYXJtRmFjdG9yeS5hZGRBbGFybVdoZW5PdXRPZkJhbmQoXG4gICAgICAgICAgICBhbm9tYWx5TWV0cmljLFxuICAgICAgICAgICAgbWV0cmljLmFsYXJtRnJpZW5kbHlOYW1lLFxuICAgICAgICAgICAgZGlzYW1iaWd1YXRvcixcbiAgICAgICAgICAgIGFsYXJtUHJvcHNcbiAgICAgICAgICApO1xuXG4gICAgICAgIC8vIG5vIG5lZWQgdG8gYWRkIGFubm90YXRpb24gc2luY2UgdGhlIGJhbmRzIGFyZSByZW5kZXJlZCBhdXRvbWF0aWNhbGx5XG4gICAgICAgIHRoaXMuYWRkQWxhcm0oY3JlYXRlZEFsYXJtKTtcbiAgICAgICAgYWxhcm1TdERldnMuYWRkKGFsYXJtUHJvcHMuc3RhbmRhcmREZXZpYXRpb25Gb3JBbGFybSk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgaWYgKGFsYXJtU3REZXZzLnNpemUgPiAwKSB7XG4gICAgICBjb25zdCBhbGFybVN0RGV2c1N0cmluZyA9IEFycmF5LmZyb20oYWxhcm1TdERldnMpLnNvcnQoKS5qb2luKFwiLCBcIik7XG4gICAgICBtZXRyaWNHcm91cC50aXRsZUFkZG9ucy5wdXNoKGBhbGFybXMgd2l0aCBzdGRldiAke2FsYXJtU3REZXZzU3RyaW5nfWApO1xuICAgIH1cbiAgfVxufVxuXG5jb25zdCBBbm9tYWx5RGV0ZWN0aW9uQWxhcm1JZFByZWZpeCA9IFwiYWxhcm1fXCI7XG5jb25zdCBBbm9tYWx5RGV0ZWN0aW9uTWV0cmljSWRQcmVmaXggPSBcImFub21hbHlfXCI7XG5jb25zdCBBbm9tYWx5QmFuZE1ldHJpY0lkU3VmZml4ID0gXCJfYmFuZFwiO1xuXG4vKipcbiAqIElOVEVSTkFMIC0gUExFQVNFIERPIE5PVCBVU0VcbiAqIFRoaXMgaXMgYSBoYWNreSBzb2x1dGlvbiB0byBtYWtlIGJhbmQgdmlzaWJsZSBpbiBHcmFwaFdpZGdldCAoZGVmYXVsdCB3aWRnZXQgb25seSByZW5kZXJzIGxpbmVzLCBub3QgdGhlIGJhbmQpLlxuICogVGhlIGNsYXNzIG1ha2VzIGFzc3VtcHRpb25zIGFib3V0IHRoZSBpbnRlcm5hbCBKU09OIHN0cnVjdHVyZSBidXQgZm91bmQgbm8gb3RoZXIgd2F5IDooLlxuICogSWRlYWxseSwgd2Ugd2FudCB0byByZW1vdmUgdGhpcyBoYWNrIG9uY2UgdGhlIGFub21hbHkgZGV0ZWN0aW9uIHJlbmRlcmluZyBpbiBDREsgZ2V0cyBpbXByb3ZlZFxuICovXG5jbGFzcyBBbm9tYWx5RGV0ZWN0aW9uR3JhcGhXaWRnZXQgZXh0ZW5kcyBHcmFwaFdpZGdldCB7XG4gIGNvbnN0cnVjdG9yKHByb3BzOiBHcmFwaFdpZGdldFByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICB9XG5cbiAgdG9Kc29uKCkge1xuICAgIGNvbnN0IGpzb24gPSBzdXBlci50b0pzb24oKTtcbiAgICBpZiAoanNvbi5sZW5ndGggIT09IDEgfHwgIWpzb24/LlswXT8ucHJvcGVydGllcz8ubWV0cmljcykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBcIlRoZSBKU09OIGlzIGV4cGVjdGVkIHRvIGhhdmUgZXhhY3RseSBvbmUgZWxlbWVudCB3aXRoIHByb3BlcnRpZXMubWV0cmljcyBwcm9wZXJ0eS5cIlxuICAgICAgKTtcbiAgICB9XG4gICAgY29uc3QgbWV0cmljczogYW55W10gPSBqc29uWzBdLnByb3BlcnRpZXMubWV0cmljcztcbiAgICBpZiAobWV0cmljcy5sZW5ndGggPCAyKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIFwiVGhlIG51bWJlciBvZiBtZXRyaWNzIG11c3QgYmUgYXQgbGVhc3QgdHdvIChtZXRyaWMgKyBhbm9tYWx5IGRldGVjdGlvbiBtYXRoKS5cIlxuICAgICAgKTtcbiAgICB9XG4gICAgY29uc3QgYW5vbWFseURldGVjdGlvbk1ldHJpY1BhcnQ6IGFueVtdID0gbWV0cmljc1swXT8udmFsdWU7XG4gICAgaWYgKFxuICAgICAgIWFub21hbHlEZXRlY3Rpb25NZXRyaWNQYXJ0IHx8XG4gICAgICBhbm9tYWx5RGV0ZWN0aW9uTWV0cmljUGFydC5sZW5ndGggIT09IDFcbiAgICApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcIkZpcnN0IG1ldHJpYyBtdXN0IGJlIGEgbWF0aCBleHByZXNzaW9uLlwiKTtcbiAgICB9XG4gICAgY29uc3QgZXZhbHVhdGVkTWV0cmljUGFydDogYW55W10gPSBtZXRyaWNzWzFdPy52YWx1ZTtcbiAgICBpZiAoXG4gICAgICAhZXZhbHVhdGVkTWV0cmljUGFydCB8fFxuICAgICAgZXZhbHVhdGVkTWV0cmljUGFydC5sZW5ndGggPCAxIHx8XG4gICAgICAhZXZhbHVhdGVkTWV0cmljUGFydFtldmFsdWF0ZWRNZXRyaWNQYXJ0Lmxlbmd0aCAtIDFdLmlkXG4gICAgKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXCJTZWNvbmQgbWV0cmljIG11c3QgaGF2ZSBhbiBJRC5cIik7XG4gICAgfVxuICAgIC8vIGJhbmQgcmVuZGVyaW5nIHJlcXVpcmVzIElEIHRvIGJlIHNldFxuICAgIGFub21hbHlEZXRlY3Rpb25NZXRyaWNQYXJ0WzBdLmlkID1cbiAgICAgIGV2YWx1YXRlZE1ldHJpY1BhcnRbZXZhbHVhdGVkTWV0cmljUGFydC5sZW5ndGggLSAxXS5pZCArXG4gICAgICBBbm9tYWx5QmFuZE1ldHJpY0lkU3VmZml4O1xuICAgIC8vIGJhbmQgcmVuZGVyaW5nIHJlcXVpcmVzIHRoZSBldmFsdWF0ZWQgbWV0cmljIHRvIGJlIHZpc2libGVcbiAgICBldmFsdWF0ZWRNZXRyaWNQYXJ0W2V2YWx1YXRlZE1ldHJpY1BhcnQubGVuZ3RoIC0gMV0udmlzaWJsZSA9IHRydWU7XG4gICAgcmV0dXJuIGpzb247XG4gIH1cbn1cbiJdfQ==