# DataMaster MCP 模块化重构方案

## 问题分析

当前 `main.py` 文件存在以下问题：
- **代码量过大**：5155行代码，包含17个工具函数
- **功能耦合**：所有功能都集中在一个文件中
- **维护困难**：代码查找、修改、测试都不方便
- **可读性差**：文件过长影响代码理解

## 模块拆分方案

### 1. 核心架构设计

```
datamaster_mcp/
├── main.py                 # MCP服务器入口 + 工具注册 (精简到200行以内)
├── core/                   # 核心功能模块
│   ├── __init__.py
│   ├── database.py         # 数据库连接和基础操作
│   ├── data_import.py      # 数据导入功能
│   ├── data_export.py      # 数据导出功能
│   ├── data_analysis.py    # 数据分析功能
│   ├── data_processing.py  # 数据处理功能
│   └── api_manager.py      # API管理功能
├── tools/                  # 工具函数模块
│   ├── __init__.py
│   ├── data_tools.py       # 数据相关工具
│   ├── db_tools.py         # 数据库相关工具
│   └── api_tools.py        # API相关工具
├── utils/                  # 工具类和辅助函数
│   ├── __init__.py
│   ├── helpers.py          # 通用辅助函数
│   ├── validators.py       # 数据验证函数
│   └── formatters.py       # 数据格式化函数
└── config/                 # 配置管理 (已存在)
    ├── __init__.py
    ├── database_config.py
    ├── api_config.py
    └── storage_config.py
```

### 2. 功能模块划分

#### 2.1 数据库模块 (`core/database.py`)
- `connect_data_source()` - 数据源连接
- `execute_sql()` - SQL执行
- `query_external_database()` - 外部数据库查询
- `list_data_sources()` - 数据源列表
- `manage_database_config()` - 数据库配置管理

#### 2.2 数据分析模块 (`core/data_analysis.py`)
- `analyze_data()` - 数据分析
- `get_data_info()` - 数据信息获取

#### 2.3 数据处理模块 (`core/data_processing.py`)
- `process_data()` - 数据处理
- `export_data()` - 数据导出

#### 2.4 API管理模块 (`core/api_manager.py`)
- `manage_api_config()` - API配置管理
- `fetch_api_data()` - API数据获取
- `api_data_preview()` - API数据预览
- `create_api_storage_session()` - 创建存储会话
- `list_api_storage_sessions()` - 列出存储会话

### 3. 重构实施步骤

#### 第一阶段：创建模块结构
1. 创建 `core/`, `tools/`, `utils/` 目录
2. 创建各模块的 `__init__.py` 文件
3. 设计模块间的接口规范

#### 第二阶段：功能迁移
1. **数据库模块**：迁移数据库相关的5个工具函数
2. **数据分析模块**：迁移数据分析相关的2个工具函数
3. **数据处理模块**：迁移数据处理相关的2个工具函数
4. **API管理模块**：迁移API相关的5个工具函数
5. **辅助函数**：迁移所有私有辅助函数到 `utils/` 模块

#### 第三阶段：主文件精简
1. 保留MCP服务器初始化代码
2. 保留工具函数注册（使用装饰器）
3. 导入各模块的工具函数
4. 移除具体实现代码

### 4. 新的 main.py 结构示例

```python
#!/usr/bin/env python3
"""
DataMaster MCP Server - 主入口文件

这是一个精简的主入口文件，只负责：
1. MCP服务器初始化
2. 工具函数注册
3. 模块导入和协调
"""

import asyncio
from mcp.server.fastmcp import FastMCP

# 导入各功能模块的工具函数
from .core.database import (
    connect_data_source,
    execute_sql,
    query_external_database,
    list_data_sources,
    manage_database_config
)
from .core.data_analysis import (
    analyze_data,
    get_data_info
)
from .core.data_processing import (
    process_data,
    export_data
)
from .core.api_manager import (
    manage_api_config,
    fetch_api_data,
    api_data_preview,
    create_api_storage_session,
    list_api_storage_sessions
)

# 初始化MCP服务器
mcp = FastMCP("DataMaster")

# 注册所有工具函数（装饰器会自动处理）
# 工具函数已在各模块中使用@mcp.tool()装饰器注册

if __name__ == "__main__":
    mcp.run()
```

### 5. 重构优势

#### 5.1 代码组织
- **模块化**：功能清晰分离，易于理解和维护
- **可读性**：每个文件专注于特定功能领域
- **可测试性**：模块独立，便于单元测试

#### 5.2 开发效率
- **并行开发**：不同开发者可以同时修改不同模块
- **快速定位**：根据功能类型快速找到相关代码
- **减少冲突**：模块分离减少代码合并冲突

#### 5.3 维护性
- **局部修改**：修改某个功能不影响其他模块
- **版本控制**：更精细的代码变更追踪
- **代码复用**：模块可以在其他项目中复用

### 6. 注意事项

#### 6.1 兼容性保证
- 保持所有工具函数的接口不变
- 确保MCP服务器的启动方式不变
- 维护现有的配置文件格式

#### 6.2 依赖管理
- 合理设计模块间依赖关系
- 避免循环依赖
- 使用依赖注入减少耦合

#### 6.3 性能考虑
- 模块导入开销最小化
- 保持数据库连接池的效率
- 避免重复初始化

### 7. 实施建议

1. **渐进式重构**：不要一次性重构所有代码，分模块逐步进行
2. **保持测试**：每个模块重构后立即测试功能完整性
3. **文档更新**：及时更新开发者文档和使用说明
4. **向后兼容**：确保重构不影响现有用户的使用

### 8. 预期效果

重构完成后：
- `main.py` 从 5155行 精简到 200行以内
- 每个功能模块控制在 500-800行
- 代码结构清晰，便于维护和扩展
- 新功能开发更加高效

这个重构方案既保持了功能的完整性，又大大提升了代码的可维护性和可扩展性。