"""Architecture service - business logic for architecture design operations."""

import logging
from typing import Any

from api.models.v1_requests import ArchitectureDesignRequest
from api.models.v1_responses import ArchitectureDesignResponse
from wistx_mcp.tools.lib.mongodb_client import MongoDBClient
from wistx_mcp.tools.lib.architecture_templates import ArchitectureTemplates
from wistx_mcp.tools.lib.template_validator import TemplateValidator
from wistx_mcp.tools.lib.vector_search import VectorSearch
from wistx_mcp.tools.lib.api_client import WISTXAPIClient
from wistx_mcp.tools import mcp_tools
from api.config import settings

logger = logging.getLogger(__name__)


class ArchitectureService:
    """Service for architecture design operations."""

    def __init__(self):
        """Initialize architecture service."""
        self.mongodb_adapter = None
        self.api_client = WISTXAPIClient()

    async def design_architecture(
        self,
        request: ArchitectureDesignRequest,
        api_key: str,
    ) -> ArchitectureDesignResponse:
        """Design architecture for DevOps/infrastructure projects.

        Args:
            request: Architecture design request
            api_key: API key for authentication

        Returns:
            Architecture design response

        Raises:
            ValueError: If invalid parameters
            RuntimeError: If operation fails
        """
        logger.info(
            "Designing architecture: action=%s, project_type=%s, project_name=%s",
            request.action,
            request.project_type,
            request.project_name,
        )

        mongodb_client = MongoDBClient()
        await mongodb_client.connect()

        try:
            if request.action == "initialize":
                if not request.project_type or not request.project_name:
                    raise ValueError("project_type and project_name are required for initialize")

                templates = ArchitectureTemplates(mongodb_client)
                template = await templates.get_template(
                    project_type=request.project_type,
                    architecture_type=request.architecture_type,
                    cloud_provider=request.cloud_provider,
                    template_id=request.template_id,
                    github_url=request.github_url,
                    user_template=request.user_template,
                )

                validator = TemplateValidator()
                validation_result = validator.validate_template(template)

                if not validation_result["valid"]:
                    if validation_result["errors"]:
                        raise ValueError(f"Invalid template: {', '.join(validation_result['errors'])}")

                compliance_context = None
                if request.include_compliance and request.compliance_standards:
                    try:
                        compliance_results = await self.api_client.get_compliance_requirements(
                            resource_types=["RDS", "EC2", "S3"],
                            standards=request.compliance_standards,
                        )
                        compliance_context = {
                            "standards": request.compliance_standards,
                            "controls": compliance_results.get("controls", []),
                        }
                    except Exception as e:
                        logger.warning("Failed to fetch compliance requirements: %s", e)

                vector_search = VectorSearch(
                    mongodb_client,
                    openai_api_key=settings.openai_api_key,
                )

                best_practices = []
                if request.include_best_practices:
                    try:
                        practices = await vector_search.search_knowledge_articles(
                            query=f"{request.project_type} best practices",
                            domains=["devops", "infrastructure"],
                            limit=10,
                        )
                        best_practices = [
                            {
                                "title": p.get("title", ""),
                                "description": p.get("summary", ""),
                            }
                            for p in practices
                        ]
                    except Exception as e:
                        logger.warning("Failed to fetch best practices: %s", e)

                return ArchitectureDesignResponse(
                    action=request.action,
                    project_name=request.project_name,
                    architecture={
                        "template": template,
                        "structure": template.get("structure", {}),
                    },
                    templates=[{"template_id": request.template_id}] if request.template_id else [],
                    compliance_context=compliance_context,
                    security_context={"enabled": request.include_security} if request.include_security else None,
                    best_practices=best_practices,
                    recommendations=["Review generated files", "Test infrastructure"],
                    output_files=[],
                )

            else:
                raise ValueError(f"Action {request.action} not yet implemented in service layer")

        finally:
            await mongodb_client.disconnect()

