"""Plan definitions and management service."""

from api.models.billing import PlanFeatures, PlanLimits, SubscriptionPlan


class PlanService:
    """Service for managing subscription plans."""

    PLANS: dict[str, SubscriptionPlan] = {
        "scout": SubscriptionPlan(
            plan_id="scout",
            name="Scout",
            description="Perfect for individuals exploring WISTX",
            monthly_price=0.0,
            annual_price=0.0,
            limits=PlanLimits(
                queries_per_month=50,
                indexes_per_month=3,
                storage_mb=1024,
                requests_per_minute=10,
                max_api_keys=1,
            ),
            features=PlanFeatures(
                compliance_queries=True,
                knowledge_queries=True,
                repository_indexing=True,
                document_indexing=True,
                custom_indexes=True,
                priority_support=False,
                sso=False,
                api_access=True,
            ),
            popular=False,
        ),
        "pro": SubscriptionPlan(
            plan_id="pro",
            name="Pro",
            description="For serious developers",
            monthly_price=7.99,
            annual_price=79.90,
            limits=PlanLimits(
                queries_per_month=500,
                indexes_per_month=5,
                storage_mb=5120,
                requests_per_minute=30,
                max_api_keys=1,
            ),
            features=PlanFeatures(
                compliance_queries=True,
                knowledge_queries=True,
                repository_indexing=True,
                document_indexing=True,
                custom_indexes=True,
                priority_support=False,
                sso=False,
                api_access=True,
            ),
            popular=False,
        ),
        "builder": SubscriptionPlan(
            plan_id="builder",
            name="Builder",
            description="For growing DevOps teams building at scale",
            monthly_price=15.99,
            annual_price=159.90,
            limits=PlanLimits(
                queries_per_month=10000,
                indexes_per_month=10,
                storage_mb=51200,
                requests_per_minute=60,
                max_api_keys=5,
            ),
            features=PlanFeatures(
                compliance_queries=True,
                knowledge_queries=True,
                repository_indexing=True,
                document_indexing=True,
                custom_indexes=True,
                priority_support=True,
                sso=False,
                api_access=True,
            ),
            popular=True,
        ),
        "architect": SubscriptionPlan(
            plan_id="architect",
            name="Architect",
            description="Enterprise-grade for large DevOps organizations",
            monthly_price=0.0,
            annual_price=0.0,
            limits=PlanLimits(
                queries_per_month=-1,
                indexes_per_month=-1,
                storage_mb=-1,
                requests_per_minute=1000,
                max_api_keys=100,
            ),
            features=PlanFeatures(
                compliance_queries=True,
                knowledge_queries=True,
                repository_indexing=True,
                document_indexing=True,
                custom_indexes=True,
                priority_support=True,
                sso=True,
                api_access=True,
            ),
            popular=False,
        ),
    }

    @classmethod
    def get_plan(cls, plan_id: str) -> SubscriptionPlan | None:
        """Get plan by ID.

        Args:
            plan_id: Plan ID

        Returns:
            SubscriptionPlan or None
        """
        return cls.PLANS.get(plan_id)

    @classmethod
    def list_plans(cls) -> list[SubscriptionPlan]:
        """List all available plans.

        Returns:
            List of subscription plans
        """
        return list(cls.PLANS.values())

    @classmethod
    def get_plan_limits(cls, plan_id: str) -> PlanLimits | None:
        """Get plan limits.

        Args:
            plan_id: Plan ID

        Returns:
            PlanLimits or None
        """
        plan = cls.get_plan(plan_id)
        return plan.limits if plan else None

    @classmethod
    def get_plan_features(cls, plan_id: str) -> PlanFeatures | None:
        """Get plan features.

        Args:
            plan_id: Plan ID

        Returns:
            PlanFeatures or None
        """
        plan = cls.get_plan(plan_id)
        return plan.features if plan else None


plan_service = PlanService()


class PlanEnforcement:
    """Centralized plan enforcement utilities."""

    @staticmethod
    def check_feature_access(plan_id: str, feature_name: str) -> bool:
        """Check if plan has access to a feature.

        Args:
            plan_id: Plan ID
            feature_name: Feature name to check

        Returns:
            True if feature is available, False otherwise
        """
        plan_features = plan_service.get_plan_features(plan_id)
        if not plan_features:
            return False

        return getattr(plan_features, feature_name, False)

    @staticmethod
    def require_feature(plan_id: str, feature_name: str) -> None:
        """Require a feature, raise exception if not available.

        Args:
            plan_id: Plan ID
            feature_name: Feature name required

        Raises:
            ValueError: If feature not available
        """
        if not PlanEnforcement.check_feature_access(plan_id, feature_name):
            raise ValueError(
                f"Feature '{feature_name}' is not available on {plan_id} plan"
            )

    @staticmethod
    def check_minimum_plan(user_plan: str, required_plan: str) -> bool:
        """Check if user plan meets minimum requirement.

        Args:
            user_plan: User's current plan
            required_plan: Minimum required plan

        Returns:
            True if user plan meets requirement
        """
        plan_hierarchy = {"scout": 1, "pro": 2, "builder": 3, "architect": 4}
        user_level = plan_hierarchy.get(user_plan, 0)
        required_level = plan_hierarchy.get(required_plan, 999)
        return user_level >= required_level

    @staticmethod
    def require_minimum_plan(user_plan: str, required_plan: str) -> None:
        """Require minimum plan, raise exception if not met.

        Args:
            user_plan: User's current plan
            required_plan: Minimum required plan

        Raises:
            ValueError: If plan requirement not met
        """
        if not PlanEnforcement.check_minimum_plan(user_plan, required_plan):
            raise ValueError(
                f"This feature requires {required_plan} plan or higher. "
                f"Your current plan: {user_plan}"
            )


plan_enforcement = PlanEnforcement()

