"""Code example data models."""

from datetime import datetime
from typing import Any, Optional

from pydantic import BaseModel, Field


class ComplianceAnalysis(BaseModel):
    """Compliance analysis for a code example."""

    applicable_standards: list[str] = Field(
        default_factory=list, description="Standards that could apply"
    )
    resource_types: list[str] = Field(
        default_factory=list, description="Resource types in code"
    )
    compliance_score: dict[str, float] = Field(
        default_factory=dict, description="Score per standard (0.0-1.0)"
    )
    missing_controls: dict[str, list[str]] = Field(
        default_factory=dict, description="Missing controls per standard"
    )
    implemented_controls: dict[str, list[str]] = Field(
        default_factory=dict, description="Implemented controls per standard"
    )
    analysis_timestamp: datetime = Field(
        default_factory=datetime.utcnow, description="Analysis timestamp"
    )


class CostAnalysis(BaseModel):
    """Cost analysis for a code example."""

    estimated_monthly: float = Field(default=0.0, description="Estimated monthly cost")
    estimated_annual: float = Field(default=0.0, description="Estimated annual cost")
    resource_costs: list[dict[str, Any]] = Field(
        default_factory=list, description="Cost breakdown by resource"
    )
    analysis_timestamp: datetime = Field(
        default_factory=datetime.utcnow, description="Analysis timestamp"
    )


class CodeExample(BaseModel):
    """Processed code example."""

    example_id: str = Field(..., description="Unique example ID")
    title: str = Field(..., description="Example title")
    description: str = Field(..., description="Example description")
    code_type: str = Field(..., description="Code type (terraform, kubernetes, docker)")
    code: str = Field(..., description="Code content")
    cloud_provider: str = Field(..., description="Cloud provider (aws, gcp, azure)")
    services: list[str] = Field(default_factory=list, description="Cloud services used")
    resources: list[str] = Field(
        default_factory=list, description="Resource types extracted from code"
    )
    github_url: str = Field(..., description="GitHub repository URL")
    file_path: str = Field(default="", description="File path in repository")
    repository_structure: Optional[dict[str, Any]] = Field(
        default=None,
        description="Repository file tree structure (nested directory/file structure)"
    )
    stars: int = Field(default=0, description="GitHub stars count")
    quality_score: int = Field(default=0, description="Quality score (0-100)")
    best_practices: list[str] = Field(
        default_factory=list, description="List of best practices identified"
    )
    metadata: dict[str, Any] = Field(
        default_factory=dict, description="Tool-specific metadata"
    )
    compliance_analysis: Optional[ComplianceAnalysis] = Field(
        default=None, description="Compliance analysis results"
    )
    cost_analysis: Optional[CostAnalysis] = Field(
        default=None, description="Cost analysis results"
    )
    relationships: dict[str, Any] = Field(
        default_factory=dict, description="Related items (code examples, articles, controls)"
    )
    contextual_description: str = Field(
        default="", description="Contextual description generated by LLM"
    )
    embedding: Optional[list[float]] = Field(
        default=None, description="Vector embedding (added in Stage 5)"
    )
    created_at: datetime = Field(
        default_factory=datetime.utcnow, description="Creation timestamp"
    )
    updated_at: datetime = Field(
        default_factory=datetime.utcnow, description="Last update timestamp"
    )
    enriched_at: Optional[datetime] = Field(
        default=None, description="Enrichment timestamp"
    )

