"""Run MongoDB migrations."""

import asyncio
import logging
import sys
from pathlib import Path

sys.path.insert(0, str(Path(__file__).parent.parent))

from api.database.migrations.migration_manager import MigrationManager
from api.database.migrations import BaseMigration

logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s - %(name)s - %(levelname)s - %(message)s",
)
logger = logging.getLogger(__name__)


def get_all_migrations() -> list[BaseMigration]:
    """Get all migration instances.

    Returns:
        List of migration instances sorted by version
    """
    from api.database.migrations import (
        Migration0001InitialSchema,
    )

    migrations = [
        Migration0001InitialSchema(),
    ]

    return sorted(migrations, key=lambda m: m.version)


async def main() -> None:
    """Run migrations."""
    logger.info("Starting migration process...")

    try:
        manager = MigrationManager()
        migrations = get_all_migrations()

        current_version = await manager.get_current_version()
        logger.info("Current database version: %d", current_version)

        await manager.migrate(migrations)

        new_version = await manager.get_current_version()
        logger.info("Migration complete. New database version: %d", new_version)

    except Exception as e:
        logger.error("Migration failed: %s", e, exc_info=True)
        sys.exit(1)


if __name__ == "__main__":
    asyncio.run(main())

