"""Setup MongoDB indexes for optimal query performance."""

import logging
import sys
from pathlib import Path

sys.path.insert(0, str(Path(__file__).parent.parent))

from api.database.mongodb import mongodb_manager

logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s - %(name)s - %(levelname)s - %(message)s",
)
logger = logging.getLogger(__name__)


def setup_indexes():
    """Create all recommended indexes for optimal query performance."""
    logger.info("Connecting to MongoDB...")
    mongodb_manager.connect()
    db = mongodb_manager.get_database()

    logger.info("Creating indexes...")

    try:
        logger.info("Creating indexes for compliance_controls collection...")
        db.compliance_controls.create_index(
            [("control_id", 1)], unique=True, name="control_id_unique", background=True
        )
        db.compliance_controls.create_index(
            [("standard", 1), ("version", 1)], name="standard_version", background=True
        )
        db.compliance_controls.create_index(
            [("source_hash", 1)], name="source_hash", background=True
        )
        db.compliance_controls.create_index(
            [("content_hash", 1)], name="content_hash", background=True
        )
        db.compliance_controls.create_index(
            [("control_id", 1), ("source_hash", 1), ("content_hash", 1)],
            name="control_id_hashes",
            background=True,
        )
        logger.info("✅ Created indexes for compliance_controls")

        logger.info("Creating indexes for pricing_data collection...")
        db.pricing_data.create_index(
            [("lookup_key", 1)], unique=True, name="lookup_key_unique", background=True
        )
        db.pricing_data.create_index(
            [("cloud", 1), ("service", 1)], name="cloud_service", background=True
        )
        db.pricing_data.create_index(
            [("region", 1)], name="region", background=True
        )
        logger.info("✅ Created indexes for pricing_data")

        logger.info("Creating indexes for code_examples collection...")
        db.code_examples.create_index(
            [("example_id", 1)], unique=True, name="example_id_unique", background=True
        )
        db.code_examples.create_index(
            [("cloud_provider", 1)], name="cloud_provider", background=True
        )
        db.code_examples.create_index(
            [("code_type", 1)], name="code_type", background=True
        )
        logger.info("✅ Created indexes for code_examples")

        logger.info("Creating indexes for best_practices collection...")
        db.best_practices.create_index(
            [("practice_id", 1)], unique=True, name="practice_id_unique", background=True
        )
        db.best_practices.create_index(
            [("category", 1)], name="category", background=True
        )
        logger.info("✅ Created indexes for best_practices")

        logger.info("Creating indexes for knowledge_articles collection...")
        db.knowledge_articles.create_index(
            [("article_id", 1)], unique=True, name="article_id_unique", background=True
        )
        db.knowledge_articles.create_index(
            [("domain", 1), ("subdomain", 1)], name="domain_subdomain", background=True
        )
        db.knowledge_articles.create_index(
            [("content_type", 1)], name="content_type", background=True
        )
        db.knowledge_articles.create_index(
            [("user_id", 1), ("organization_id", 1)], name="user_org", background=True
        )
        logger.info("✅ Created indexes for knowledge_articles")

        logger.info("Creating indexes for cost_data_focus collection...")
        db.cost_data_focus.create_index(
            [("lookup_key", 1)], unique=True, name="lookup_key_unique", background=True
        )
        db.cost_data_focus.create_index(
            [("billing_account_id", 1), ("billing_period_start", -1)],
            name="billing_account_period",
            background=True,
        )
        db.cost_data_focus.create_index(
            [("provider", 1), ("region_id", 1)], name="provider_region", background=True
        )
        db.cost_data_focus.create_index(
            [("service_category", 1), ("service_name", 1)], name="service_category_name", background=True
        )
        db.cost_data_focus.create_index(
            [("resource_type", 1), ("resource_id", 1)], name="resource_type_id", background=True
        )
        db.cost_data_focus.create_index(
            [("sku_id", 1)], name="sku_id", background=True
        )
        db.cost_data_focus.create_index(
            [("pricing_category", 1)], name="pricing_category", background=True
        )
        db.cost_data_focus.create_index(
            [("source_hash", 1)], name="source_hash", background=True
        )
        db.cost_data_focus.create_index(
            [("last_updated", -1)], name="last_updated", background=True
        )
        db.cost_data_focus.create_index(
            [("tags.BusinessUnit", 1), ("tags.Project", 1)], name="allocation_tags", background=True
        )
        logger.info("✅ Created indexes for cost_data_focus")

        logger.info("Creating indexes for infrastructure_budgets collection...")
        db.infrastructure_budgets.create_index(
            [("user_id", 1), ("status", 1)], name="user_status", background=True
        )
        db.infrastructure_budgets.create_index(
            [("scope.type", 1), ("scope.cloud_provider", 1), ("scope.environment_name", 1)],
            name="scope_lookup",
            background=True,
        )
        db.infrastructure_budgets.create_index(
            [("_id", 1), ("user_id", 1)], name="budget_user", background=True
        )
        logger.info("✅ Created indexes for infrastructure_budgets")

        logger.info("Creating indexes for infrastructure_spending collection...")
        db.infrastructure_spending.create_index(
            [("budget_id", 1), ("period", 1)], name="budget_period", background=True
        )
        db.infrastructure_spending.create_index(
            [("user_id", 1), ("period", 1)], name="user_period", background=True
        )
        db.infrastructure_spending.create_index(
            [("date", -1)], name="date_desc", background=True
        )
        db.infrastructure_spending.create_index(
            [("source_type", 1), ("source_id", 1), ("period", 1)],
            name="deduplication_key",
            background=True,
        )
        logger.info("✅ Created indexes for infrastructure_spending")

        logger.info("Creating indexes for budget_status collection...")
        db.budget_status.create_index(
            [("budget_id", 1), ("period", 1)],
            unique=True,
            name="budget_period_unique",
            background=True,
        )
        db.budget_status.create_index(
            [("user_id", 1), ("period", 1)], name="user_period_status", background=True
        )
        logger.info("✅ Created indexes for budget_status")

        logger.info("Creating compound index for cost_data_focus pricing queries...")
        db.cost_data_focus.create_index(
            [
                ("provider", 1),
                ("service_name", 1),
                ("resource_type", 1),
                ("region_id", 1),
                ("pricing_category", 1),
                ("last_updated", -1),
            ],
            name="pricing_lookup_compound",
            background=True,
        )
        logger.info("✅ Created compound index for pricing queries")

        logger.info("=" * 80)
        logger.info("✅ All indexes created successfully!")
        logger.info("=" * 80)

        logger.info("Index creation is running in background. Check status with:")
        logger.info("  db.compliance_controls.getIndexes()")

    except Exception as e:
        logger.error("Error creating indexes: %s", e, exc_info=True)
        raise
    finally:
        mongodb_manager.close()


if __name__ == "__main__":
    setup_indexes()

