"""Code examples tool - search infrastructure code examples."""

import logging
from typing import Any

from wistx_mcp.tools.lib.mongodb_client import MongoDBClient
from wistx_mcp.tools.lib.vector_search import VectorSearch
from wistx_mcp.tools.lib.auth_context import get_auth_context
from wistx_mcp.config import settings

logger = logging.getLogger(__name__)


async def get_code_examples(
    query: str,
    code_types: list[str] | None = None,
    cloud_provider: str | None = None,
    services: list[str] | None = None,
    min_quality_score: int | None = None,
    compliance_standard: str | None = None,
    limit: int = 1000,
) -> dict[str, Any]:
    """Search infrastructure code examples.
    
    Args:
        query: Search query (e.g., "RDS database with encryption")
        code_types: Filter by code types (terraform, kubernetes, docker, pulumi, etc.)
        cloud_provider: Filter by cloud provider (aws, gcp, azure)
        services: Filter by cloud services (rds, s3, ec2, etc.)
        min_quality_score: Minimum quality score (0-100)
        compliance_standard: Filter by compliance standard (PCI-DSS, HIPAA, etc.)
        limit: Maximum number of results
        
    Returns:
        Dictionary with code examples and metadata
        
    Raises:
        RuntimeError: If quota is exceeded or search fails
        ValueError: If query is invalid
    """
    if not query or not query.strip():
        raise ValueError("Query cannot be empty")
    
    if limit <= 0 or limit > 50000:
        raise ValueError("Limit must be between 1 and 50000")
    
    auth_ctx = get_auth_context()
    if auth_ctx:
        user_id = auth_ctx.get_user_id()
        if user_id:
            try:
                from api.services.quota_service import quota_service, QuotaExceededError
                
                plan = "scout"
                if auth_ctx.user_info:
                    plan = auth_ctx.user_info.get("plan", "scout")
                await quota_service.check_query_quota(user_id, plan)
            except ImportError:
                logger.debug("API quota service not available, skipping quota check")
            except QuotaExceededError as e:
                logger.warning("Quota exceeded for user %s: %s", user_id, e)
                raise RuntimeError(f"Quota exceeded: {e}") from e
            except Exception as e:
                logger.warning("Failed to check quota (continuing): %s", e)
    
    async with MongoDBClient() as client:
        vector_search = VectorSearch(client, openai_api_key=settings.openai_api_key)
        
        try:
            results = await vector_search.search_code_examples(
                query=query,
                code_types=code_types,
                cloud_provider=cloud_provider,
                services=services,
                min_quality_score=min_quality_score,
                compliance_standard=compliance_standard,
                limit=limit,
            )
            
            formatted_results = []
            for result in results:
                formatted_result = {
                    "example_id": result.get("example_id"),
                    "title": result.get("title"),
                    "description": result.get("description"),
                    "contextual_description": result.get("contextual_description"),
                    "code_type": result.get("code_type"),
                    "cloud_provider": result.get("cloud_provider"),
                    "services": result.get("services", []),
                    "resources": result.get("resources", []),
                    "code": result.get("code", ""),
                    "github_url": result.get("github_url"),
                    "file_path": result.get("file_path"),
                    "stars": result.get("stars", 0),
                    "quality_score": result.get("quality_score", 0),
                    "best_practices": result.get("best_practices", []),
                    "hybrid_score": result.get("hybrid_score", 0.0),
                    "vector_score": result.get("vector_score", 0.0),
                    "bm25_score": result.get("bm25_score", 0.0),
                }
                
                compliance_analysis = result.get("compliance_analysis")
                if compliance_analysis:
                    formatted_result["compliance_analysis"] = {
                        "applicable_standards": compliance_analysis.get("applicable_standards", []),
                        "compliance_score": compliance_analysis.get("compliance_score", {}),
                    }
                
                cost_analysis = result.get("cost_analysis")
                if cost_analysis:
                    formatted_result["cost_analysis"] = {
                        "estimated_monthly": cost_analysis.get("estimated_monthly", 0.0),
                        "estimated_annual": cost_analysis.get("estimated_annual", 0.0),
                    }
                
                formatted_results.append(formatted_result)
            
            return {
                "examples": formatted_results,
                "total": len(formatted_results),
                "query": query,
            }
        
        except ValueError as e:
            logger.warning("Invalid query parameters: %s", e)
            raise
        except Exception as e:
            logger.error("Error searching code examples: %s", e, exc_info=True)
            raise RuntimeError(f"Failed to search code examples: {e}") from e

