"""Context builder - format data for LLM context using markdown."""

import markdown as markdown_lib
from typing import Any


class MarkdownBuilder:
    """Helper class for building markdown strings consistently."""

    def __init__(self):
        """Initialize markdown builder."""
        self.lines: list[str] = []

    def add_header(self, text: str, level: int = 1) -> "MarkdownBuilder":
        """Add a header.

        Args:
            text: Header text
            level: Header level (1-6)

        Returns:
            Self for method chaining
        """
        prefix = "#" * min(max(level, 1), 6)
        self.lines.append(f"{prefix} {text}\n")
        return self

    def add_paragraph(self, text: str) -> "MarkdownBuilder":
        """Add a paragraph.

        Args:
            text: Paragraph text

        Returns:
            Self for method chaining
        """
        self.lines.append(f"{text}\n")
        return self

    def add_bold(self, text: str) -> "MarkdownBuilder":
        """Add bold text.

        Args:
            text: Text to make bold

        Returns:
            Self for method chaining
        """
        self.lines.append(f"**{text}**")
        return self

    def add_code_block(self, code: str, language: str = "") -> "MarkdownBuilder":
        """Add a code block.

        Args:
            code: Code content
            language: Language identifier

        Returns:
            Self for method chaining
        """
        lang = language if language else ""
        self.lines.append(f"```{lang}\n{code}\n```\n")
        return self

    def add_list_item(self, text: str, indent: int = 0) -> "MarkdownBuilder":
        """Add a list item.

        Args:
            text: List item text
            indent: Indentation level

        Returns:
            Self for method chaining
        """
        prefix = "  " * indent + "- "
        self.lines.append(f"{prefix}{text}\n")
        return self

    def add_separator(self) -> "MarkdownBuilder":
        """Add a horizontal separator.

        Returns:
            Self for method chaining
        """
        self.lines.append("---\n")
        return self

    def add_line_break(self) -> "MarkdownBuilder":
        """Add a line break.

        Returns:
            Self for method chaining
        """
        self.lines.append("\n")
        return self

    def build(self, validate: bool = True) -> str:
        """Build the markdown string.

        Args:
            validate: Whether to validate the markdown syntax

        Returns:
            Formatted markdown string

        Raises:
            ValueError: If markdown validation fails and validate=True
        """
        result = "".join(self.lines)
        
        if not result.strip():
            return ""
        
        if validate:
            try:
                html_output = markdown_lib.markdown(result)
                if not html_output or html_output.strip() == "":
                    import logging
                    logger = logging.getLogger(__name__)
                    logger.warning("Markdown produced empty HTML output")
            except (ValueError, TypeError, AttributeError) as e:
                import logging
                logger = logging.getLogger(__name__)
                logger.warning("Markdown validation warning: %s", e)
        
        return result

    def reset(self) -> "MarkdownBuilder":
        """Reset the builder.

        Returns:
            Self for method chaining
        """
        self.lines = []
        return self


class TOONBuilder:
    """Helper class for building TOON format strings for LLM consumption."""

    def __init__(self):
        """Initialize TOON builder."""
        self.lines: list[str] = []
        self.current_array: list[dict[str, Any]] | None = None
        self.array_fields: list[str] | None = None

    def start_array(self, fields: list[str], count: int | None = None) -> "TOONBuilder":
        """Start a tabular array with field headers.

        Args:
            fields: List of field names
            count: Optional count (unused, calculated from data)

        Returns:
            Self for method chaining
        """
        self.array_fields = fields
        self.current_array = []
        return self

    def add_array_row(self, values: dict[str, Any]) -> "TOONBuilder":
        """Add a row to the current array.

        Args:
            values: Dictionary of field values

        Returns:
            Self for method chaining
        """
        if self.current_array is None:
            raise ValueError("Must call start_array() before add_array_row()")
        self.current_array.append(values)
        return self

    def add_single_object(self, fields: list[str], values: dict[str, Any]) -> "TOONBuilder":
        """Add a single object (non-array).

        Args:
            fields: List of field names
            values: Dictionary of field values

        Returns:
            Self for method chaining
        """
        field_str = ",".join(fields)
        value_list = [self._escape_value(values.get(f, "")) for f in fields]
        value_str = ",".join(value_list)
        self.lines.append(f"{{{field_str}}}:\n{value_str}")
        return self

    def _escape_value(self, value: Any) -> str:
        """Escape TOON value (handle newlines, commas, special chars).

        Args:
            value: Value to escape

        Returns:
            Escaped string
        """
        if value is None:
            return ""
        if isinstance(value, bool):
            return "true" if value else "false"
        if isinstance(value, (int, float)):
            return str(value)
        if isinstance(value, list):
            return "|".join(str(v) for v in value)
        if isinstance(value, dict):
            return self._serialize_dict(value)

        str_value = str(value)
        if "\n" in str_value or "," in str_value or '"' in str_value:
            escaped = str_value.replace("\\", "\\\\").replace('"', '\\"').replace("\n", "\\n")
            return f'"{escaped}"'
        return str_value

    def _serialize_dict(self, d: dict[str, Any]) -> str:
        """Serialize nested dictionary to compact format.

        Args:
            d: Dictionary to serialize

        Returns:
            Serialized string
        """
        parts = []
        for k, v in d.items():
            if v is None or v == "":
                continue
            if isinstance(v, list):
                v_str = "|".join(str(item) for item in v)
            else:
                v_str = str(v)
            parts.append(f"{k}:{v_str}")
        return ";".join(parts)

    def build(self) -> str:
        """Build the final TOON string.

        Returns:
            Formatted TOON string
        """
        if self.current_array is not None and self.array_fields is not None:
            self._finalize_array()

        return "\n".join(self.lines)

    def _finalize_array(self) -> None:
        """Finalize the current array and add to lines."""
        if not self.current_array or not self.array_fields:
            return

        count = len(self.current_array)
        fields_str = ",".join(self.array_fields)

        self.lines.append(f"[{count}]{{{fields_str}}}:")

        for row in self.current_array:
            values = [self._escape_value(row.get(f, "")) for f in self.array_fields]
            self.lines.append(",".join(values))

        self.current_array = None
        self.array_fields = None

    def reset(self) -> "TOONBuilder":
        """Reset the builder.

        Returns:
            Self for method chaining
        """
        self.lines = []
        self.current_array = None
        self.array_fields = None
        return self


class ContextBuilder:
    """Build formatted context for LLM consumption using markdown."""

    @staticmethod
    def format_compliance_as_markdown(controls: list[dict[str, Any]], resource_type: str | None = None) -> str:
        """Format compliance controls as markdown for LLM consumption.

        Args:
            controls: List of compliance control dictionaries
            resource_type: Optional resource type for context

        Returns:
            Formatted markdown string
        """
        import logging
        logger = logging.getLogger(__name__)

        if not controls:
            return "No compliance controls found."

        if not isinstance(controls, list):
            logger.error("Controls must be a list, got %s", type(controls))
            return "Invalid controls format."

        builder = MarkdownBuilder()
        title = "Compliance Requirements"
        if resource_type:
            title += f" for {resource_type}"
        builder.add_header(title, level=1)

        for i, control in enumerate(controls):
            if not isinstance(control, dict):
                logger.warning("Control %d is not a dict, skipping", i)
                continue

            try:
                standard = control.get("standard", "Unknown")
                control_id = control.get("control_id", "")
                severity = control.get("severity", "MEDIUM")
                title_text = control.get("title", "")
                description = control.get("description", "")

                builder.add_header(f"{standard} {control_id}: {title_text}", level=2)
                builder.add_bold(f"Severity: {severity}")
                builder.add_line_break()
                builder.add_paragraph(str(description))

                remediation = control.get("remediation", {})
                if remediation and isinstance(remediation, dict):
                    if remediation.get("guidance"):
                        builder.add_header("Remediation Guidance", level=3)
                        builder.add_paragraph(str(remediation["guidance"]))

                    if remediation.get("code_snippet"):
                        builder.add_header("Code Example", level=3)
                        builder.add_code_block(str(remediation["code_snippet"]), language="hcl")

                verification = control.get("verification", {})
                if verification and isinstance(verification, dict):
                    if verification.get("procedure"):
                        builder.add_header("Verification", level=3)
                        builder.add_paragraph(str(verification["procedure"]))

                builder.add_separator()
            except (ValueError, TypeError, KeyError, AttributeError) as e:
                logger.warning("Error formatting control %d: %s", i, e)
                continue
            except Exception as e:  # noqa: BLE001
                logger.warning("Unexpected error formatting control %d: %s", i, e, exc_info=True)
                continue

        markdown_output = builder.build(validate=True)
        
        if not markdown_output or not markdown_output.strip():
            logger.warning("Generated markdown is empty")
            return "No compliance controls found."
        
        try:
            markdown_lib.markdown(markdown_output)
        except Exception as e:  # noqa: BLE001
            logger.error("Generated markdown failed validation: %s", e)
            return "Error formatting compliance requirements. Please try again."
        
        return markdown_output

    @staticmethod
    def format_compliance_as_toon(
        controls: list[dict[str, Any]], resource_type: str | None = None
    ) -> str:
        """Format compliance controls as TOON for LLM consumption.

        Args:
            controls: List of compliance control dictionaries
            resource_type: Optional resource type for context

        Returns:
            Formatted TOON string
        """
        import logging
        logger = logging.getLogger(__name__)

        if not controls:
            return "No compliance controls found."

        if not isinstance(controls, list):
            logger.error("Controls must be a list, got %s", type(controls))
            return "Invalid controls format."

        builder = TOONBuilder()

        if resource_type:
            builder.add_single_object(
                ["context"],
                {"context": f"Compliance Requirements for {resource_type}"}
            )
            builder.lines.append("")

        primary_fields = [
            "standard",
            "control_id",
            "title",
            "severity",
            "description"
        ]

        optional_fields = [
            "category",
            "subcategory",
            "applies_to",
            "remediation_guidance",
            "remediation_steps",
            "code_snippet",
            "verification",
            "references"
        ]

        all_fields = primary_fields + optional_fields

        builder.start_array(all_fields, len(controls))

        for i, control in enumerate(controls):
            if not isinstance(control, dict):
                logger.warning("Control %d is not a dict, skipping", i)
                continue

            try:
                row_data: dict[str, Any] = {}

                row_data["standard"] = control.get("standard", "Unknown")
                row_data["control_id"] = control.get("control_id", "")
                row_data["title"] = control.get("title", "")
                row_data["severity"] = control.get("severity", "MEDIUM")
                row_data["description"] = control.get("description", "")

                row_data["category"] = control.get("category", "")
                row_data["subcategory"] = control.get("subcategory", "")

                applies_to = control.get("applies_to", [])
                if isinstance(applies_to, list):
                    row_data["applies_to"] = ",".join(applies_to)
                else:
                    row_data["applies_to"] = str(applies_to) if applies_to else ""

                remediation = control.get("remediation", {})
                if remediation and isinstance(remediation, dict):
                    row_data["remediation_guidance"] = remediation.get("summary", "") or remediation.get("guidance", "")

                    steps = remediation.get("steps", [])
                    if isinstance(steps, list):
                        row_data["remediation_steps"] = "|".join(str(s) for s in steps)
                    else:
                        row_data["remediation_steps"] = ""

                    code_snippets = remediation.get("code_snippets", [])
                    if code_snippets and isinstance(code_snippets, list) and len(code_snippets) > 0:
                        first_snippet = code_snippets[0]
                        if isinstance(first_snippet, dict):
                            row_data["code_snippet"] = first_snippet.get("code", "")
                        elif hasattr(first_snippet, "code"):
                            row_data["code_snippet"] = first_snippet.code
                        else:
                            row_data["code_snippet"] = ""
                    elif remediation.get("code_snippet"):
                        row_data["code_snippet"] = str(remediation["code_snippet"])
                    else:
                        row_data["code_snippet"] = ""
                else:
                    row_data["remediation_guidance"] = ""
                    row_data["remediation_steps"] = ""
                    row_data["code_snippet"] = ""

                verification = control.get("verification", {})
                if verification and isinstance(verification, dict):
                    row_data["verification"] = verification.get("procedure", "")
                else:
                    row_data["verification"] = ""

                references = control.get("references", [])
                if isinstance(references, list) and references:
                    ref_urls = []
                    for ref in references:
                        if isinstance(ref, dict):
                            ref_urls.append(ref.get("url", ""))
                        elif hasattr(ref, "url"):
                            ref_urls.append(ref.url)
                    row_data["references"] = "|".join(ref_urls)
                else:
                    row_data["references"] = ""

                builder.add_array_row(row_data)

            except (ValueError, TypeError, KeyError, AttributeError) as e:
                logger.warning("Error formatting control %d: %s", i, e)
                continue
            except Exception as e:  # noqa: BLE001
                logger.warning("Unexpected error formatting control %d: %s", i, e, exc_info=True)
                continue

        toon_output = builder.build()

        if not toon_output or not toon_output.strip():
            logger.warning("Generated TOON is empty")
            return "No compliance controls found."

        return toon_output

    @staticmethod
    def format_compliance_context(controls: list[dict[str, Any]]) -> str:
        """Format compliance controls as context string (legacy method).

        Args:
            controls: List of compliance control dictionaries

        Returns:
            Formatted context string
        """
        return ContextBuilder.format_compliance_as_markdown(controls)

    @staticmethod
    def format_pricing_context(pricing: dict[str, Any]) -> str:
        """Format pricing data as context string.

        Args:
            pricing: Pricing data dictionary

        Returns:
            Formatted context string
        """
        builder = MarkdownBuilder()
        builder.add_header("Infrastructure Cost Estimate", level=2)
        builder.add_bold(f"Total Monthly: ${pricing.get('total_monthly', 0):.2f}")
        builder.add_line_break()
        builder.add_bold(f"Total Annual: ${pricing.get('total_annual', 0):.2f}")
        builder.add_line_break()

        budget_check = pricing.get("budget_check")
        if budget_check:
            builder.add_header("Budget Status", level=3)
            status = budget_check.get("status", "unknown")
            builder.add_bold(
                f"Status: {status.upper().replace('_', ' ')}"
            )
            builder.add_line_break()

            applicable_budgets = budget_check.get("applicable_budgets", [])
            if applicable_budgets:
                for budget_status in applicable_budgets:
                    builder.add_list_item(
                        f"**{budget_status['name']}**: "
                        f"${budget_status.get('current_spending', 0):.2f} spent, "
                        f"${budget_status.get('estimated_cost', 0):.2f} estimated → "
                        f"${budget_status.get('projected_spending', 0):.2f} projected "
                        f"({budget_status.get('utilization_percent', 0):.1f}% of "
                        f"${budget_status.get('budget_limit', 0):.2f} limit)"
                    )

            alerts = budget_check.get("alerts", [])
            if alerts:
                builder.add_header("Budget Alerts", level=3)
                for alert in alerts:
                    alert_type = alert.get("type", "warning").upper()
                    builder.add_list_item(
                        f"[{alert_type}] {alert.get('message', '')}"
                    )

        builder.add_header("Breakdown", level=3)

        for item in pricing.get("breakdown", []):
            resource_text = f"{item['resource']} (x{item['quantity']}): "
            resource_text += f"${item['monthly']:.2f}/month (${item['annual']:.2f}/year)"
            if item.get("region"):
                resource_text += f" [{item['region']}]"
            if item.get("pricing_category"):
                resource_text += f" ({item['pricing_category']})"
            if item.get("error"):
                resource_text += f" - [WARNING] {item['error']}"
            builder.add_list_item(resource_text)

        if pricing.get("optimizations"):
            builder.add_header("Optimization Suggestions", level=3)
            for opt in pricing["optimizations"]:
                builder.add_list_item(opt)

        return builder.build()

    @staticmethod
    def format_pricing_as_toon(pricing: dict[str, Any]) -> str:
        """Format pricing data as TOON for LLM consumption.

        Args:
            pricing: Pricing data dictionary

        Returns:
            Formatted TOON string
        """
        import logging
        logger = logging.getLogger(__name__)

        builder = TOONBuilder()

        builder.add_single_object(
            ["total_monthly", "total_annual"],
            {
                "total_monthly": f"{pricing.get('total_monthly', 0):.2f}",
                "total_annual": f"{pricing.get('total_annual', 0):.2f}"
            }
        )
        builder.lines.append("")

        budget_check = pricing.get("budget_check")
        if budget_check:
            status = budget_check.get("status", "unknown")
            builder.add_single_object(
                ["budget_status"],
                {"budget_status": status.upper().replace("_", " ")}
            )
            builder.lines.append("")

            applicable_budgets = budget_check.get("applicable_budgets", [])
            if applicable_budgets:
                budget_fields = [
                    "name",
                    "current_spending",
                    "estimated_cost",
                    "projected_spending",
                    "utilization_percent",
                    "budget_limit"
                ]
                builder.start_array(budget_fields, len(applicable_budgets))
                for budget_status in applicable_budgets:
                    builder.add_array_row({
                        "name": budget_status.get("name", ""),
                        "current_spending": f"{budget_status.get('current_spending', 0):.2f}",
                        "estimated_cost": f"{budget_status.get('estimated_cost', 0):.2f}",
                        "projected_spending": f"{budget_status.get('projected_spending', 0):.2f}",
                        "utilization_percent": f"{budget_status.get('utilization_percent', 0):.1f}",
                        "budget_limit": f"{budget_status.get('budget_limit', 0):.2f}"
                    })
                builder.lines.append("")

            alerts = budget_check.get("alerts", [])
            if alerts:
                builder.start_array(["type", "message"], len(alerts))
                for alert in alerts:
                    builder.add_array_row({
                        "type": alert.get("type", ""),
                        "message": alert.get("message", "")
                    })
                builder.lines.append("")

        breakdown = pricing.get("breakdown", [])
        if breakdown:
            breakdown_fields = [
                "resource",
                "quantity",
                "monthly",
                "annual",
                "region",
                "pricing_category",
                "error"
            ]
            builder.start_array(breakdown_fields, len(breakdown))
            for item in breakdown:
                builder.add_array_row({
                    "resource": item.get("resource", ""),
                    "quantity": str(item.get("quantity", 0)),
                    "monthly": f"{item.get('monthly', 0):.2f}",
                    "annual": f"{item.get('annual', 0):.2f}",
                    "region": item.get("region", ""),
                    "pricing_category": item.get("pricing_category", ""),
                    "error": item.get("error", "")
                })
            builder.lines.append("")

        optimizations = pricing.get("optimizations", [])
        if optimizations:
            builder.start_array(["optimization"], len(optimizations))
            for opt in optimizations:
                builder.add_array_row({"optimization": str(opt)})

        return builder.build()

    @staticmethod
    def format_code_examples_context(examples: list[dict[str, Any]]) -> str:
        """Format code examples as context string.

        Args:
            examples: List of code example dictionaries

        Returns:
            Formatted context string
        """
        if not examples:
            return "No code examples found."

        builder = MarkdownBuilder()
        builder.add_header("Code Examples", level=2)

        for example in examples:
            builder.add_header(example.get("title", "Example"), level=3)
            
            if example.get("contextual_description"):
                builder.add_paragraph(example["contextual_description"])
                builder.add_line_break()
            
            if example.get("description"):
                builder.add_bold(f"Description: {example['description']}")
                builder.add_line_break()
            
            code_type = example.get("code_type") or example.get("language") or example.get("infrastructure_type")
            cloud_provider = example.get("cloud_provider", "")
            services = example.get("services", [])
            resources = example.get("resources", [])
            
            builder.add_bold(f"Code Type: {code_type}")
            builder.add_line_break()
            
            if cloud_provider:
                builder.add_bold(f"Cloud Provider: {cloud_provider.upper()}")
                builder.add_line_break()
            
            if services:
                builder.add_bold(f"Services: {', '.join(services)}")
                builder.add_line_break()
            
            if resources:
                builder.add_bold(f"Resources: {', '.join(resources[:5])}")
                builder.add_line_break()
            
            github_url = example.get("github_url")
            if github_url:
                builder.add_bold(f"Source: {github_url}")
                builder.add_line_break()
            
            quality_score = example.get("quality_score", 0)
            stars = example.get("stars", 0)
            if quality_score > 0 or stars > 0:
                builder.add_bold(f"Quality Score: {quality_score}/100 | Stars: {stars}")
                builder.add_line_break()
            
            builder.add_code_block(example.get("code", ""), language=code_type or "")

        return builder.build()

    @staticmethod
    def format_code_examples_as_toon(examples: list[dict[str, Any]]) -> str:
        """Format code examples as TOON for LLM consumption.

        Args:
            examples: List of code example dictionaries

        Returns:
            Formatted TOON string
        """
        import logging
        logger = logging.getLogger(__name__)

        if not examples:
            return "No code examples found."

        if not isinstance(examples, list):
            logger.error("Examples must be a list, got %s", type(examples))
            return "Invalid examples format."

        builder = TOONBuilder()

        example_fields = [
            "title",
            "contextual_description",
            "description",
            "code_type",
            "cloud_provider",
            "services",
            "resources",
            "github_url",
            "quality_score",
            "stars",
            "code"
        ]

        builder.start_array(example_fields, len(examples))

        for i, example in enumerate(examples):
            if not isinstance(example, dict):
                logger.warning("Example %d is not a dict, skipping", i)
                continue

            try:
                code_type = example.get("code_type") or example.get("language") or example.get("infrastructure_type") or ""
                services = example.get("services", [])
                resources = example.get("resources", [])

                row_data: dict[str, Any] = {
                    "title": example.get("title", ""),
                    "contextual_description": example.get("contextual_description", ""),
                    "description": example.get("description", ""),
                    "code_type": code_type,
                    "cloud_provider": example.get("cloud_provider", ""),
                    "services": ",".join(services) if isinstance(services, list) else str(services),
                    "resources": ",".join(resources[:10]) if isinstance(resources, list) else str(resources),
                    "github_url": example.get("github_url", ""),
                    "quality_score": str(example.get("quality_score", 0)),
                    "stars": str(example.get("stars", 0)),
                    "code": example.get("code", "")
                }

                builder.add_array_row(row_data)

            except (ValueError, TypeError, KeyError, AttributeError) as e:
                logger.warning("Error formatting example %d: %s", i, e)
                continue
            except Exception as e:  # noqa: BLE001
                logger.warning("Unexpected error formatting example %d: %s", i, e, exc_info=True)
                continue

        toon_output = builder.build()

        if not toon_output or not toon_output.strip():
            logger.warning("Generated TOON is empty")
            return "No code examples found."

        return toon_output

    @staticmethod
    def format_web_search_results(results: dict[str, Any]) -> str:
        """Format web search results as markdown.

        Args:
            results: Web search results dictionary

        Returns:
            Formatted markdown string
        """
        builder = MarkdownBuilder()
        builder.add_header("Web Search Results", level=1)
        builder.add_bold(f"Total Results: {results.get('total', 0)}")
        builder.add_line_break()

        if results.get("web"):
            builder.add_header("Web Search Results", level=2)
            for item in results["web"][:10]:
                if item.get("type") == "answer":
                    builder.add_header(item.get("title", "AI Answer"), level=3)
                    builder.add_paragraph(item.get("content", ""))
                else:
                    builder.add_header(item.get("title", "Result"), level=3)
                    if item.get("url"):
                        builder.add_bold(f"URL: {item['url']}")
                        builder.add_line_break()
                    if item.get("content"):
                        content = item["content"][:500]
                        builder.add_paragraph(content)
                    if item.get("score"):
                        builder.add_bold(f"Relevance Score: {item['score']:.2f}")
                        builder.add_line_break()
                builder.add_separator()

        if results.get("web_results"):
            builder.add_header("Real-Time Web Research", level=2)
            web_results = results["web_results"]
            
            freshness_info = web_results.get("freshness_info", {})
            if freshness_info:
                max_age = freshness_info.get("max_age_days", 0)
                builder.add_bold(f"Data Freshness: Results from last {max_age} days")
                builder.add_line_break()
            
            if web_results.get("answer"):
                builder.add_header("AI Summary", level=3)
                builder.add_paragraph(web_results["answer"])
            if web_results.get("results"):
                builder.add_header("Web Sources", level=3)
                for item in web_results["results"][:5]:
                    builder.add_bold(item.get("title", "Source"))
                    builder.add_line_break()
                    if item.get("url"):
                        builder.add_paragraph(f"URL: {item['url']}")
                    if item.get("published_date"):
                        builder.add_paragraph(f"Published: {item['published_date']}")
                    if item.get("content"):
                        content = item["content"][:300]
                        builder.add_paragraph(content)
                    builder.add_separator()

        if results.get("security"):
            builder.add_header("Security Information", level=2)
            for item in results["security"][:10]:
                title = item.get("title") or item.get("cve_id") or "Security Item"
                builder.add_header(title, level=3)
                if item.get("description"):
                    builder.add_paragraph(item["description"])
                if item.get("severity"):
                    builder.add_bold(f"Severity: {item['severity']}")
                    builder.add_line_break()
                if item.get("source"):
                    builder.add_bold(f"Source: {item['source']}")
                    builder.add_line_break()
                if item.get("published_date"):
                    builder.add_bold(f"Published: {item['published_date']}")
                    builder.add_line_break()
                builder.add_separator()

        if results.get("compliance"):
            builder.add_header("Compliance Requirements", level=2)
            compliance_md = ContextBuilder.format_compliance_as_markdown(results["compliance"])
            builder.add_paragraph(compliance_md)

        if results.get("best_practices"):
            builder.add_header("Best Practices", level=2)
            for item in results["best_practices"][:10]:
                builder.add_header(item.get("title", "Best Practice"), level=3)
                if item.get("summary"):
                    builder.add_paragraph(item["summary"])
                if item.get("domain"):
                    builder.add_bold(f"Domain: {item['domain']}")
                    builder.add_line_break()
                builder.add_separator()

        if results.get("knowledge"):
            builder.add_header("Knowledge Base", level=2)
            for item in results["knowledge"][:10]:
                builder.add_header(item.get("title", "Knowledge Article"), level=3)
                if item.get("summary"):
                    builder.add_paragraph(item["summary"])
                if item.get("domain"):
                    builder.add_bold(f"Domain: {item['domain']}")
                    builder.add_line_break()
                builder.add_separator()

        return builder.build()

    @staticmethod
    def format_codebase_search_results(results: dict[str, Any]) -> str:
        """Format codebase search results as markdown.

        Args:
            results: Codebase search results dictionary

        Returns:
            Formatted markdown string
        """
        builder = MarkdownBuilder()
        builder.add_header("Codebase Search Results", level=1)
        builder.add_bold(f"Total Results: {results.get('total', 0)}")
        builder.add_line_break()

        if results.get("ai_analysis"):
            ai_analysis = results["ai_analysis"]
            builder.add_header("AI Analysis", level=2)
            if ai_analysis.get("analysis"):
                builder.add_paragraph(ai_analysis["analysis"])
            builder.add_separator()

        if results.get("resources"):
            builder.add_header("Resources", level=2)
            for resource in results["resources"]:
                builder.add_list_item(
                    f"**{resource.get('name', 'Unknown')}** "
                    f"({resource.get('resource_type', 'unknown')})"
                )

        if results.get("results"):
            builder.add_header("Search Results", level=2)
            for result in results["results"][:20]:
                builder.add_header(result.get("title", "Result"), level=3)
                builder.add_bold(f"Source: {result.get('source_type', 'unknown')}")
                builder.add_line_break()
                if result.get("summary"):
                    builder.add_paragraph(result["summary"])

                if result.get("content"):
                    content = result.get("content", "")[:500]
                    builder.add_code_block(content)

                if result.get("source_url"):
                    builder.add_bold(f"Source URL: {result['source_url']}")
                    builder.add_line_break()

                builder.add_separator()

        if results.get("highlights"):
            builder.add_header("Code Highlights", level=2)
            for highlight in results["highlights"][:5]:
                if highlight.get("file_path"):
                    builder.add_bold(f"File: {highlight['file_path']}")
                    builder.add_line_break()
                builder.add_code_block(highlight.get("highlight", ""))

        return builder.build()

    @staticmethod
    def format_regex_search_results(results: dict[str, Any]) -> str:
        """Format regex search results as markdown.

        Args:
            results: Regex search results dictionary

        Returns:
            Formatted markdown string
        """
        builder = MarkdownBuilder()
        builder.add_header("Regex Search Results", level=1)
        builder.add_bold(f"Total Matches: {results.get('total', 0)}")
        builder.add_line_break()

        pattern_info = results.get("pattern_info", {})
        if pattern_info:
            builder.add_header("Pattern Information", level=2)
            builder.add_bold(f"Pattern: `{pattern_info.get('pattern', '')}`")
            builder.add_line_break()

            if pattern_info.get("template"):
                builder.add_bold(f"Template: {pattern_info['template']}")
                builder.add_line_break()

            warnings = pattern_info.get("warnings", [])
            if warnings:
                builder.add_bold("Warnings:")
                builder.add_line_break()
                for warning in warnings:
                    builder.add_list_item(warning)

        performance = results.get("performance", {})
        if performance:
            builder.add_header("Performance", level=2)
            builder.add_list_item(f"Search Time: {performance.get('search_time_seconds', 0):.2f} seconds")
            builder.add_list_item(f"Files Searched: {performance.get('files_searched', 0)}")
            builder.add_list_item(f"Matches Found: {performance.get('matches_found', 0)}")

        resources = results.get("resources", [])
        if resources:
            builder.add_header("Resources", level=2)
            for resource in resources:
                builder.add_list_item(
                    f"**{resource.get('name', 'Unknown')}** "
                    f"({resource.get('resource_type', 'unknown')})"
                )

        matches = results.get("matches", [])
        if matches:
            builder.add_header("Matches", level=2)

            current_file = None
            for match in matches[:100]:
                file_path = match.get("file_path", "")
                line_number = match.get("line_number", 0)
                line_content = match.get("line_content", "")
                match_text = match.get("match_text", "")

                if file_path != current_file:
                    current_file = file_path
                    builder.add_header(f"File: {file_path}", level=3)

                builder.add_bold(f"Line {line_number}:")
                builder.add_line_break()

                highlighted_line = line_content.replace(
                    match_text,
                    f"**{match_text}**",
                    1,
                )
                builder.add_code_block(highlighted_line)

                context = match.get("context")
                if context:
                    before = context.get("before", [])
                    after = context.get("after", [])

                    if before:
                        builder.add_bold("Context (before):")
                        builder.add_line_break()
                        builder.add_code_block("\n".join(before[-3:]))

                    if after:
                        builder.add_bold("Context (after):")
                        builder.add_line_break()
                        builder.add_code_block("\n".join(after[:3]))

                builder.add_separator()
        else:
            builder.add_paragraph("No matches found.")

        return builder.build()

    @staticmethod
    def format_architecture_results(results: dict[str, Any]) -> str:
        """Format architecture design results as markdown.

        Args:
            results: Architecture results dictionary

        Returns:
            Formatted markdown string
        """
        builder = MarkdownBuilder()
        builder.add_header("Architecture Design Results", level=1)

        if "project_path" in results:
            builder.add_header("Project Initialized", level=2)
            builder.add_bold(f"Project Path: `{results.get('project_path')}`")
            builder.add_line_break()
            builder.add_bold(f"Files Created: {len(results.get('files_created', []))}")
            builder.add_line_break()

            if results.get("structure"):
                builder.add_header("Project Structure", level=3)
                for item in results["structure"][:20]:
                    builder.add_list_item(item)

            if results.get("next_steps"):
                builder.add_header("Next Steps", level=3)
                for step in results["next_steps"]:
                    builder.add_list_item(step)

        if "visualization" in results:
            builder.add_header("Architecture Visualization", level=2)
            builder.add_code_block(results.get("visualization", ""), language="mermaid")
            builder.add_paragraph(
                "**Note**: This Mermaid diagram can be rendered in Markdown viewers that support Mermaid (GitHub, GitLab, etc.)"
            )

        if "architecture_diagram" in results:
            builder.add_header("Architecture Diagram (Text)", level=2)
            builder.add_code_block(results.get("architecture_diagram", ""))

        if "components" in results:
            builder.add_header("Components", level=2)
            for component in results.get("components", []):
                builder.add_list_item(f"**{component.get('name')}** ({component.get('type')})")

        if "recommendations" in results:
            builder.add_header("Recommendations", level=2)
            for rec in results.get("recommendations", []):
                builder.add_list_item(rec)

        if "issues" in results:
            builder.add_header("Issues Found", level=2)
            for issue in results.get("issues", []):
                builder.add_list_item(f"[WARNING] {issue}")

        if "optimizations" in results:
            builder.add_header("Optimizations", level=2)
            for opt in results.get("optimizations", []):
                builder.add_list_item(f"[OK] {opt}")

        return builder.build()

    @staticmethod
    def format_troubleshooting_results(results: dict[str, Any]) -> str:
        """Format troubleshooting results as markdown.

        Args:
            results: Troubleshooting results dictionary

        Returns:
            Formatted markdown string
        """
        builder = MarkdownBuilder()
        builder.add_header("Troubleshooting Results", level=1)

        diagnosis = results.get("diagnosis", {})
        if diagnosis:
            builder.add_header("Diagnosis", level=2)
            if diagnosis.get("root_cause"):
                builder.add_bold(f"Root Cause: {diagnosis['root_cause']}")
                builder.add_line_break()
            if diagnosis.get("confidence"):
                builder.add_bold(f"Confidence: {diagnosis['confidence']}")
                builder.add_line_break()

            if diagnosis.get("issues"):
                builder.add_header("Identified Issues", level=3)
                for issue in diagnosis["issues"]:
                    builder.add_list_item(issue)

            if diagnosis.get("error_patterns"):
                builder.add_header("Error Patterns", level=3)
                for pattern in diagnosis["error_patterns"]:
                    builder.add_list_item(pattern)

            if diagnosis.get("visualization"):
                builder.add_header("Infrastructure Visualization", level=3)
                builder.add_code_block(diagnosis.get("visualization", ""), language="mermaid")
                builder.add_paragraph(
                    "**Note**: This Mermaid diagram shows the infrastructure being troubleshot. It can be rendered in Markdown viewers that support Mermaid."
                )

        if results.get("fixes"):
            builder.add_header("Recommended Fixes", level=2)
            for i, fix in enumerate(results["fixes"][:10], 1):
                builder.add_header(f"{i}. {fix.get('title', 'Fix')}", level=3)
                if fix.get("description"):
                    builder.add_paragraph(fix["description"])
                if fix.get("code_example"):
                    builder.add_code_block(fix["code_example"])
                if fix.get("url"):
                    builder.add_bold(f"Source: {fix['url']}")
                    builder.add_line_break()
                builder.add_separator()

        if results.get("prevention"):
            builder.add_header("Prevention Strategies", level=2)
            for strategy in results["prevention"]:
                builder.add_list_item(strategy)

        if results.get("related_knowledge"):
            builder.add_header("Related Knowledge", level=2)
            for article in results["related_knowledge"][:5]:
                builder.add_header(article.get("title", "Article"), level=3)
                if article.get("summary"):
                    builder.add_paragraph(article["summary"])
                builder.add_separator()

        if results.get("web_sources"):
            builder.add_header("Web Sources", level=2)
            for source in results["web_sources"][:5]:
                builder.add_header(source.get("title", "Source"), level=3)
                if source.get("url"):
                    builder.add_bold(f"URL: {source['url']}")
                    builder.add_line_break()
                if source.get("content"):
                    content = source["content"][:300]
                    builder.add_paragraph(content)
                builder.add_separator()

        return builder.build()

    @staticmethod
    def format_documentation_results(results: dict[str, Any]) -> str:
        """Format documentation generation results as markdown.

        Args:
            results: Documentation results dictionary

        Returns:
            Formatted markdown string
        """
        builder = MarkdownBuilder()
        builder.add_header("Documentation Generated", level=1)

        metadata = results.get("metadata", {})
        if metadata:
            builder.add_bold(f"Generated: {metadata.get('generated_at', '')}")
            builder.add_line_break()
            builder.add_bold(f"Document Type: {results.get('document_type', '')}")
            builder.add_line_break()
            builder.add_bold(f"Subject: {results.get('subject', '')}")
            builder.add_line_break()

        if results.get("sections"):
            builder.add_header("Document Sections", level=2)
            for section in results["sections"][:10]:
                builder.add_list_item(section)

        if results.get("content"):
            builder.add_header("Document Content", level=2)
            builder.add_paragraph(results["content"])

        return builder.build()

    @staticmethod
    def format_integration_results(results: dict[str, Any]) -> str:
        """Format integration management results as markdown.

        Args:
            results: Integration results dictionary

        Returns:
            Formatted markdown string
        """
        builder = MarkdownBuilder()
        builder.add_header(f"Integration Management: {results.get('action', 'unknown')}", level=1)

        action = results.get("action", "")

        if action == "analyze":
            builder.add_header("Analysis Results", level=2)

            if results.get("missing_connections"):
                builder.add_header("Missing Connections", level=3)
                for conn in results["missing_connections"]:
                    builder.add_list_item(f"[WARNING] {conn}")

            if results.get("dependency_issues"):
                builder.add_header("Dependency Issues", level=3)
                for issue in results["dependency_issues"]:
                    builder.add_list_item(f"[WARNING] {issue}")

            if results.get("security_gaps"):
                builder.add_header("Security Gaps", level=3)
                for gap in results["security_gaps"]:
                    builder.add_list_item(f"[SECURITY] {gap}")

            if results.get("recommendations"):
                builder.add_header("Recommendations", level=3)
                for rec in results["recommendations"]:
                    builder.add_list_item(f"[OK] {rec}")

        elif action == "generate":
            builder.add_header("Generated Integration Code", level=2)

            if results.get("description"):
                builder.add_bold(f"Pattern: {results.get('description')}")
                builder.add_line_break()

            if results.get("pattern_used"):
                builder.add_bold(f"Pattern Used: {results['pattern_used']}")
                builder.add_line_break()

            if results.get("integration_code"):
                builder.add_header("Integration Code", level=3)
                builder.add_code_block(results["integration_code"], language="hcl")

            if results.get("dependencies"):
                builder.add_header("Dependencies", level=3)
                for dep in results["dependencies"]:
                    builder.add_list_item(dep)

            if results.get("security_rules"):
                builder.add_header("Security Rules", level=3)
                for rule in results["security_rules"]:
                    builder.add_list_item(f"[SECURITY] {rule}")

            if results.get("monitoring"):
                monitoring = results["monitoring"]
                if monitoring.get("metrics"):
                    builder.add_header("Monitoring Metrics", level=3)
                    for metric in monitoring["metrics"]:
                        builder.add_list_item(metric)

        elif action == "validate":
            builder.add_header("Validation Results", level=2)

            is_valid = results.get("valid", False)
            builder.add_bold(f"Status: {'Valid' if is_valid else 'Invalid'}")
            builder.add_line_break()

            if results.get("issues"):
                builder.add_header("Issues", level=3)
                for issue in results["issues"]:
                    builder.add_list_item(f"[WARNING] {issue}")

            if results.get("fixes"):
                builder.add_header("Recommended Fixes", level=3)
                for fix in results["fixes"]:
                    builder.add_list_item(f"[OK] {fix}")

        return builder.build()

    @staticmethod
    def format_infrastructure_results(results: dict[str, Any]) -> str:
        """Format infrastructure management results as markdown.

        Args:
            results: Infrastructure results dictionary

        Returns:
            Formatted markdown string
        """
        builder = MarkdownBuilder()
        builder.add_header(f"Infrastructure Management: {results.get('action', 'unknown')}", level=1)

        action = results.get("action", "")
        resource_id = results.get("resource_id", "")
        status = results.get("status", "")

        builder.add_bold(f"Resource ID: {resource_id}")
        builder.add_line_break()
        builder.add_bold(f"Status: {status}")
        builder.add_line_break()

        if action == "create":
            if results.get("endpoints"):
                builder.add_header("Endpoints", level=2)
                endpoints = results["endpoints"]
                if isinstance(endpoints, dict):
                    for key, value in endpoints.items():
                        if isinstance(value, dict):
                            builder.add_header(key, level=3)
                            for k, v in value.items():
                                builder.add_list_item(f"**{k}**: {v}")
                        else:
                            builder.add_list_item(f"**{key}**: {value}")

            if results.get("visualization"):
                builder.add_header("Infrastructure Visualization", level=2)
                builder.add_code_block(results.get("visualization", ""), language="mermaid")
                builder.add_paragraph(
                    "**Note**: This Mermaid diagram visualizes the generated infrastructure. It can be rendered in Markdown viewers that support Mermaid (GitHub, GitLab, etc.)"
                )

            if results.get("terraform_code"):
                builder.add_header("Terraform Code", level=2)
                builder.add_code_block(results["terraform_code"], language="hcl")

            if results.get("next_steps"):
                builder.add_header("Next Steps", level=2)
                for step in results["next_steps"]:
                    builder.add_list_item(step)

        elif action == "upgrade":
            if results.get("strategy"):
                builder.add_bold(f"Upgrade Strategy: {results['strategy']}")
                builder.add_line_break()

            if results.get("steps"):
                builder.add_header("Upgrade Steps", level=2)
                for i, step in enumerate(results["steps"], 1):
                    builder.add_list_item(f"{i}. {step}")

            if results.get("rollback_plan"):
                builder.add_header("Rollback Plan", level=2)
                for step in results["rollback_plan"]:
                    builder.add_list_item(step)

            if results.get("estimated_downtime"):
                builder.add_bold(f"Estimated Downtime: {results['estimated_downtime']}")
                builder.add_line_break()

        elif action == "backup":
            if results.get("backup_commands"):
                builder.add_header("Backup Commands", level=2)
                backup_cmds = "\n".join(results["backup_commands"])
                builder.add_code_block(backup_cmds, language="bash")

            if results.get("restore_commands"):
                builder.add_header("Restore Commands", level=2)
                restore_cmds = "\n".join(results["restore_commands"])
                builder.add_code_block(restore_cmds, language="bash")

            if results.get("retention_policy"):
                builder.add_bold(f"Retention Policy: {results['retention_policy']}")
                builder.add_line_break()

        elif action == "update":
            if results.get("visualization"):
                builder.add_header("Infrastructure Visualization", level=2)
                builder.add_code_block(results.get("visualization", ""), language="mermaid")
                builder.add_paragraph(
                    "**Note**: This Mermaid diagram visualizes the updated infrastructure. It can be rendered in Markdown viewers that support Mermaid."
                )

            if results.get("terraform_code"):
                builder.add_header("Terraform Code", level=2)
                builder.add_code_block(results["terraform_code"], language="hcl")

            if results.get("update_plan"):
                builder.add_header("Update Plan", level=2)
                for step in results["update_plan"]:
                    builder.add_list_item(step)

            if results.get("rollback_plan"):
                builder.add_header("Rollback Plan", level=2)
                for step in results["rollback_plan"]:
                    builder.add_list_item(step)

            if results.get("estimated_downtime"):
                builder.add_bold(f"Estimated Downtime: {results['estimated_downtime']}")
                builder.add_line_break()

        elif action == "monitor":
            if results.get("metrics"):
                builder.add_header("Metrics", level=2)
                for metric in results["metrics"]:
                    builder.add_list_item(metric)

            if results.get("alerts"):
                builder.add_header("Alerts", level=2)
                for alert in results["alerts"]:
                    builder.add_list_item(f"[ALERT] {alert}")

            if results.get("dashboards"):
                builder.add_header("Dashboards", level=2)
                for dashboard in results["dashboards"]:
                    builder.add_list_item(dashboard)

        elif action == "optimize":
            if results.get("recommendations"):
                builder.add_header("Optimization Recommendations", level=2)
                for rec in results["recommendations"]:
                    builder.add_list_item(f"[OPTIMIZATION] {rec}")

            if results.get("estimated_savings"):
                builder.add_bold(f"Estimated Savings: {results['estimated_savings']}")
                builder.add_line_break()

            if results.get("migration_plan"):
                builder.add_header("Migration Plan", level=2)
                for step in results["migration_plan"]:
                    builder.add_list_item(step)

        return builder.build()

    @staticmethod
    def format_indexing_results(results: dict[str, Any], operation: str = "index") -> str:
        """Format indexing operation results as markdown.

        Args:
            results: Indexing results dictionary
            operation: Operation type (index, list, check, delete)

        Returns:
            Formatted markdown string
        """
        builder = MarkdownBuilder()
        builder.add_header(f"{operation.title()} Operation Results", level=1)

        if results.get("resource_id"):
            builder.add_bold(f"Resource ID: `{results['resource_id']}`")
            builder.add_line_break()

        if results.get("status"):
            status = results["status"]
            builder.add_bold(f"Status: {status}")
            builder.add_line_break()

        if results.get("message"):
            builder.add_paragraph(results["message"])

        if results.get("progress"):
            progress = results["progress"]
            builder.add_bold(f"Progress: {progress.get('percentage', 0)}%")
            builder.add_line_break()
            if progress.get("files_processed"):
                builder.add_list_item(f"Files Processed: {progress['files_processed']}")
            if progress.get("total_files"):
                builder.add_list_item(f"Total Files: {progress['total_files']}")

        if results.get("ai_analysis"):
            ai_analysis = results["ai_analysis"]
            builder.add_header("AI Analysis", level=2)
            if ai_analysis.get("analysis"):
                builder.add_paragraph(ai_analysis["analysis"])
            builder.add_separator()

        if results.get("resources"):
            builder.add_header("Resources", level=2)
            for resource in results["resources"]:
                resource_type = resource.get("resource_type", "unknown")
                name = resource.get("name", "Unknown")
                status = resource.get("status", "unknown")
                resource_id = resource.get("resource_id", "")
                
                builder.add_list_item(f"**{name}** ({resource_type}) - Status: {status}")
                if resource_id:
                    builder.add_list_item(f"Resource ID: `{resource_id}`", indent=1)
                if resource.get("created_at"):
                    builder.add_list_item(f"Created: {resource['created_at']}", indent=1)

        if results.get("total"):
            builder.add_bold(f"Total Resources: {results['total']}")
            builder.add_line_break()

        if results.get("next_steps"):
            builder.add_header("Next Steps", level=2)
            for step in results["next_steps"]:
                builder.add_list_item(step)

        return builder.build()

    @staticmethod
    def format_health_check_results(results: dict[str, Any]) -> str:
        """Format health check results as markdown.

        Args:
            results: Health check results dictionary

        Returns:
            Formatted markdown string
        """
        builder = MarkdownBuilder()
        builder.add_header("Health Check Results", level=1)

        overall_status = results.get("status", "unknown")
        builder.add_bold(f"Overall Status: {overall_status}")
        builder.add_line_break()

        if results.get("mongodb"):
            mongodb = results["mongodb"]
            mongodb_status = mongodb.get("status", "unknown")
            builder.add_header("MongoDB", level=2)
            builder.add_bold(f"Status: {mongodb_status}")
            builder.add_line_break()
            if mongodb.get("database"):
                builder.add_bold(f"Database: {mongodb['database']}")
                builder.add_line_break()
            if mongodb.get("error"):
                builder.add_bold(f"Error: {mongodb['error']}")
                builder.add_line_break()

        if results.get("pinecone"):
            pinecone = results["pinecone"]
            pinecone_status = pinecone.get("status", "unknown")
            builder.add_header("Pinecone", level=2)
            builder.add_bold(f"Status: {pinecone_status}")
            builder.add_line_break()
            if pinecone.get("index"):
                builder.add_bold(f"Index: {pinecone['index']}")
                builder.add_line_break()
            if pinecone.get("vector_count") is not None:
                builder.add_bold(f"Vector Count: {pinecone['vector_count']}")
                builder.add_line_break()
            if pinecone.get("error"):
                builder.add_bold(f"Error: {pinecone['error']}")
                builder.add_line_break()

        if results.get("api_client"):
            api_client = results["api_client"]
            api_status = api_client.get("status", "unknown")
            builder.add_header("API Client", level=2)
            builder.add_bold(f"Status: {api_status}")
            builder.add_line_break()
            if api_client.get("api_url"):
                builder.add_bold(f"API URL: {api_client['api_url']}")
                builder.add_line_break()
            if api_client.get("error"):
                builder.add_bold(f"Error: {api_client['error']}")
                builder.add_line_break()

        return builder.build()

    @staticmethod
    def format_knowledge_research_results(results: dict[str, Any]) -> str:
        """Format knowledge research results as markdown.

        Args:
            results: Knowledge research results dictionary

        Returns:
            Formatted markdown string
        """
        if results.get("markdown"):
            return results["markdown"]

        builder = MarkdownBuilder()
        builder.add_header("Knowledge Research Results", level=1)

        if results.get("research_summary"):
            summary = results["research_summary"]
            builder.add_header("Research Summary", level=2)
            if summary.get("total_found"):
                builder.add_bold(f"Total Results: {summary['total_found']}")
                builder.add_line_break()
            if summary.get("domains_covered"):
                builder.add_bold(f"Domains Covered: {', '.join(summary['domains_covered'])}")
                builder.add_line_break()
            if summary.get("key_insights"):
                builder.add_header("Key Insights", level=3)
                for insight in summary["key_insights"]:
                    builder.add_list_item(insight)

        if results.get("results"):
            builder.add_header("Knowledge Articles", level=2)
            for article in results["results"][:20]:
                builder.add_header(article.get("title", "Article"), level=3)
                if article.get("summary"):
                    builder.add_paragraph(article["summary"])
                if article.get("domain"):
                    builder.add_bold(f"Domain: {article['domain']}")
                    builder.add_line_break()
                if article.get("subdomain"):
                    builder.add_bold(f"Subdomain: {article['subdomain']}")
                    builder.add_line_break()
                if article.get("source_url"):
                    builder.add_bold(f"Source: {article['source_url']}")
                    builder.add_line_break()
                builder.add_separator()

        if results.get("web_results"):
            web_results = results["web_results"]
            builder.add_header("Web Search Results", level=2)
            
            if web_results.get("freshness_info"):
                freshness = web_results["freshness_info"]
                max_age = freshness.get("max_age_days", 0)
                builder.add_bold(f"Data Freshness: Results from last {max_age} days")
                builder.add_line_break()
            
            if web_results.get("answer"):
                builder.add_header("AI Summary", level=3)
                builder.add_paragraph(web_results["answer"])
            
            if web_results.get("results"):
                builder.add_header("Web Sources", level=3)
                for item in web_results["results"][:10]:
                    builder.add_bold(item.get("title", "Source"))
                    builder.add_line_break()
                    if item.get("url"):
                        builder.add_paragraph(f"URL: {item['url']}")
                    if item.get("published_date"):
                        builder.add_paragraph(f"Published: {item['published_date']}")
                    if item.get("content"):
                        content = item["content"][:300]
                        builder.add_paragraph(content)
                    builder.add_separator()

        return builder.build()

    @staticmethod
    def format_github_tree_results(results: dict[str, Any]) -> str:
        """Format GitHub file tree results as markdown.

        Args:
            results: GitHub tree results dictionary

        Returns:
            Formatted markdown string
        """
        builder = MarkdownBuilder()
        builder.add_header("GitHub Repository Structure", level=1)

        builder.add_bold(f"Repository: {results.get('repo_url', '')}")
        builder.add_line_break()
        builder.add_bold(f"Branch: {results.get('branch', 'main')}")
        builder.add_line_break()

        metadata = results.get("metadata", {})
        if metadata:
            builder.add_header("Repository Statistics", level=2)
            builder.add_list_item(f"Total Files: {metadata.get('total_files', 0)}")
            builder.add_list_item(f"Total Directories: {metadata.get('total_directories', 0)}")
            builder.add_list_item(f"Total Size: {metadata.get('total_size', 0)} bytes")

            if metadata.get("languages"):
                builder.add_list_item(f"Languages: {', '.join(metadata['languages'][:10])}")

        format_type = results.get("format", "json")

        if format_type == "tree":
            builder.add_header("File Tree", level=2)
            builder.add_code_block(results.get("tree", ""), language="text")
        elif format_type == "markdown":
            builder.add_header("Structure", level=2)
            builder.add_paragraph(results.get("markdown", ""))
        else:
            builder.add_header("Structure (JSON)", level=2)
            import json

            structure_json = json.dumps(results.get("structure", {}), indent=2)
            builder.add_code_block(structure_json, language="json")

        return builder.build()

    @staticmethod
    def format_visualization_results(results: dict[str, Any]) -> str:
        """Format infrastructure visualization results as markdown.

        Args:
            results: Visualization results dictionary

        Returns:
            Formatted markdown string
        """
        builder = MarkdownBuilder()
        builder.add_header(f"Infrastructure Visualization: {results.get('type', 'flow')}", level=1)

        format_type = results.get("format", "mermaid")
        builder.add_bold(f"Format: {format_type}")
        builder.add_line_break()

        metadata = results.get("metadata", {})
        if metadata:
            builder.add_header("Visualization Statistics", level=2)
            builder.add_list_item(f"Components: {metadata.get('component_count', 0)}")
            builder.add_list_item(f"Connections: {metadata.get('connection_count', 0)}")

        visualization = results.get("visualization", "")
        if visualization:
            builder.add_header("Diagram", level=2)

            if format_type == "mermaid":
                builder.add_code_block(visualization, language="mermaid")
                builder.add_paragraph(
                    "**Note**: This Mermaid diagram can be rendered in Markdown viewers that support Mermaid (GitHub, GitLab, etc.)"
                )
            else:
                builder.add_code_block(visualization, language="plantuml")

        components = results.get("components", [])
        if components:
            builder.add_header("Components", level=2)
            for comp in components[:20]:
                comp_name = comp.get("name", comp.get("id", "unknown"))
                comp_type = comp.get("type", "unknown")
                builder.add_list_item(f"**{comp_name}** ({comp_type})")

        connections = results.get("connections", [])
        if connections:
            builder.add_header("Connections", level=2)
            for conn in connections[:20]:
                from_comp = conn.get("from", "unknown")
                to_comp = conn.get("to", "unknown")
                conn_type = conn.get("type", "dependency")
                builder.add_list_item(f"{from_comp} → {to_comp} ({conn_type})")

        return builder.build()

    @staticmethod
    def format_package_search_results(results: dict[str, Any]) -> str:
        """Format package search results as markdown.

        Args:
            results: Package search results dictionary

        Returns:
            Formatted markdown string
        """
        builder = MarkdownBuilder()
        builder.add_header("Package Search Results", level=1)

        search_type = results.get("search_type", "semantic")
        total = results.get("total", 0)

        builder.add_bold(f"Search Type: {search_type.title()}")
        builder.add_line_break()
        builder.add_bold(f"Total Results: {total}")
        builder.add_line_break()

        if search_type == "hybrid":
            semantic_count = results.get("semantic_count", 0)
            regex_count = results.get("regex_count", 0)
            builder.add_bold(f"Semantic Matches: {semantic_count}")
            builder.add_line_break()
            builder.add_bold(f"Regex Matches: {regex_count}")
            builder.add_line_break()

        packages = results.get("results", [])
        matches = results.get("matches", [])

        if search_type == "regex" and matches:
            builder.add_header("Regex Matches", level=2)
            for match in matches[:50]:
                package_name = match.get("package_name", "Unknown")
                registry = match.get("registry", "unknown")
                file_path = match.get("file_path", "")
                line_number = match.get("line_number", 0)
                match_text = match.get("match_text", "")

                builder.add_header(f"{package_name} ({registry})", level=3)
                builder.add_bold(f"File: `{file_path}`")
                builder.add_line_break()
                builder.add_bold(f"Line {line_number}:")
                builder.add_line_break()
                builder.add_code_block(match_text)

                context = match.get("context")
                if context:
                    builder.add_bold("Context:")
                    builder.add_line_break()
                    builder.add_code_block(context[:500])

                builder.add_separator()

        elif packages:
            builder.add_header("Packages", level=2)
            for package in packages[:50]:
                name = package.get("name", "Unknown")
                registry = package.get("registry", "unknown")
                description = package.get("description", "")
                github_url = package.get("github_url")
                domain_tags = package.get("domain_tags", [])
                category = package.get("category")
                relevance_score = package.get("relevance_score", 0)

                builder.add_header(f"{name} ({registry})", level=3)

                if description:
                    builder.add_paragraph(description[:300])

                if domain_tags:
                    builder.add_bold(f"Domains: {', '.join(domain_tags)}")
                    builder.add_line_break()

                if category:
                    builder.add_bold(f"Category: {category}")
                    builder.add_line_break()

                if relevance_score:
                    builder.add_bold(f"Relevance Score: {relevance_score:.2f}")
                    builder.add_line_break()

                health_score = package.get("health_score", 0.0)
                health_metrics = package.get("health_metrics", {})
                if health_score > 0 or health_metrics:
                    builder.add_bold(f"Health Score: {health_score:.1f}/100")
                    builder.add_line_break()

                    if health_metrics:
                        maintenance = health_metrics.get("maintenance_status", "unknown")
                        security = health_metrics.get("security_status", "unknown")
                        builder.add_bold(f"Maintenance: {maintenance.title()} | Security: {security.title()}")
                        builder.add_line_break()

                        downloads = health_metrics.get("downloads", 0)
                        if downloads > 0:
                            builder.add_bold(f"Downloads: {downloads:,}")
                            builder.add_line_break()

                if github_url:
                    builder.add_bold(f"GitHub: {github_url}")
                    builder.add_line_break()

                regex_matches = package.get("regex_matches", [])
                if regex_matches:
                    builder.add_bold(f"Regex Matches: {len(regex_matches)}")
                    builder.add_line_break()
                    for match in regex_matches[:3]:
                        builder.add_list_item(
                            f"`{match.get('file_path', '')}`: Line {match.get('line_number', 0)}"
                        )

                builder.add_separator()

        if not packages and not matches:
            builder.add_paragraph("No packages found matching your search criteria.")

        return builder.build()

    @staticmethod
    def format_package_read_file_results(result: dict[str, Any]) -> str:
        """Format package read file results as markdown."""
        builder = MarkdownBuilder()
        builder.add_header("Package File Content", level=1)
        
        package_name = result.get("package_name", "Unknown")
        registry = result.get("registry", "Unknown")
        filename = result.get("filename", "Unknown")
        
        builder.add_bold(f"Package: {package_name}")
        builder.add_line_break()
        builder.add_bold(f"Registry: {registry}")
        builder.add_line_break()
        builder.add_bold(f"File: {filename}")
        builder.add_line_break()
        
        content = result.get("content", "")
        if content:
            builder.add_header("File Content", level=2)
            builder.add_code_block(content, language="")
        else:
            builder.add_paragraph("No content available.")
        
        return builder.build()

    @staticmethod
    def format_tool_recommendations(result: dict[str, Any]) -> str:
        """Format tool recommendations as markdown."""
        builder = MarkdownBuilder()
        builder.add_header("Recommended Tools", level=1)
        
        task = result.get("task", "")
        if task:
            builder.add_paragraph(f"**Task:** {task}")
            builder.add_line_break()
        
        category_filter = result.get("category_filter")
        if category_filter:
            builder.add_paragraph(f"**Category Filter:** {category_filter}")
            builder.add_line_break()
        
        recommendations = result.get("recommendations", [])
        count = result.get("count", len(recommendations))
        
        builder.add_paragraph(f"Found {count} recommended tool(s):")
        builder.add_line_break()
        
        for i, rec in enumerate(recommendations, 1):
            tool_name = rec.get("tool", "Unknown")
            score = rec.get("score", 0)
            reason = rec.get("reason", "")
            short_desc = rec.get("short_description", "")
            category = rec.get("category", "")
            
            builder.add_header(f"{i}. {tool_name}", level=2)
            builder.add_list_item(f"**Relevance Score:** {score:.2f}/1.0")
            builder.add_list_item(f"**Category:** {category}")
            if reason:
                builder.add_list_item(f"**Reason:** {reason}")
            if short_desc:
                builder.add_list_item(f"**Description:** {short_desc}")
            builder.add_line_break()
        
        guidance = result.get("guidance", "")
        if guidance:
            builder.add_header("Usage Guidance", level=2)
            builder.add_paragraph(guidance)
        
        return builder.build()

    @staticmethod
    def format_tools_by_category(result: dict[str, Any]) -> str:
        """Format tools by category as markdown."""
        builder = MarkdownBuilder()
        
        category = result.get("category", "Unknown")
        builder.add_header(f"Tools in Category: {category}", level=1)
        
        tools = result.get("tools", [])
        count = result.get("count", len(tools))
        
        if count == 0:
            builder.add_paragraph("No tools found in this category.")
            available_categories = result.get("available_categories", [])
            if available_categories:
                builder.add_header("Available Categories", level=2)
                for cat in available_categories:
                    builder.add_list_item(cat)
        else:
            builder.add_paragraph(f"Found {count} tool(s):")
            builder.add_line_break()
            
            for tool in tools:
                tool_name = tool.get("name", "Unknown")
                short_desc = tool.get("short_description", "")
                
                builder.add_header(tool_name, level=2)
                if short_desc:
                    builder.add_paragraph(short_desc)
                builder.add_line_break()
        
        available_categories = result.get("available_categories", [])
        if available_categories:
            builder.add_header("All Available Categories", level=2)
            for cat in available_categories:
                builder.add_list_item(cat)
        
        return builder.build()

    @staticmethod
    def format_tool_documentation(result: dict[str, Any]) -> str:
        """Format tool documentation as markdown."""
        builder = MarkdownBuilder()
        
        tool_name = result.get("tool_name", "Unknown")
        builder.add_header(f"Tool Documentation: {tool_name}", level=1)
        
        category = result.get("category", "")
        if category:
            builder.add_list_item(f"**Category:** {category}")
        
        short_desc = result.get("short_description", "")
        if short_desc:
            builder.add_header("Short Description", level=2)
            builder.add_paragraph(short_desc)
            builder.add_line_break()
        
        detailed_desc = result.get("detailed_description", "")
        if detailed_desc and detailed_desc != "Detailed description not available yet.":
            builder.add_header("Detailed Description", level=2)
            builder.add_paragraph(detailed_desc)
        else:
            builder.add_header("Detailed Description", level=2)
            builder.add_paragraph("Detailed description not available yet. This tool's documentation is being updated.")
        
        guidance = result.get("guidance", "")
        if guidance:
            builder.add_header("Usage Guidance", level=2)
            builder.add_paragraph(guidance)
        
        return builder.build()

