"""Tool description management with two-tier system (short + detailed)."""

import logging
from typing import Dict, Optional

logger = logging.getLogger(__name__)


class ToolDescriptionManager:
    """Manages tool descriptions with short and detailed versions."""

    # Short descriptions (50-150 tokens) - shown to agents by default
    SHORT_DESCRIPTIONS = {
        "wistx_get_compliance_requirements": (
            "Get compliance requirements and controls for infrastructure resources. "
            "Supports PCI-DSS, HIPAA, CIS, SOC2, NIST, ISO-27001, GDPR, FedRAMP. "
            "Returns controls, remediation guidance, and verification procedures."
        ),
        "wistx_research_knowledge_base": (
            "Research DevOps, infrastructure, compliance, FinOps, and platform engineering knowledge. "
            "Searches best practices, patterns, strategies, and guides. "
            "Returns structured knowledge with examples and recommendations."
        ),
        "wistx_calculate_infrastructure_cost": (
            "Calculate infrastructure costs for AWS, GCP, and Azure resources. "
            "Estimates compute, storage, networking, and database costs. "
            "Returns cost breakdown and optimization recommendations."
        ),
        "wistx_get_devops_infra_code_examples": (
            "Get DevOps and infrastructure code examples from curated repositories. "
            "Supports Terraform, CloudFormation, Helm, Ansible, Docker, Kubernetes. "
            "Returns production-ready code with explanations."
        ),
        "wistx_web_search": (
            "Search the web for DevOps, infrastructure, and cloud-related information. "
            "Returns recent articles, documentation, and best practices. "
            "Useful for current information and latest tools."
        ),
        "wistx_search_codebase": (
            "Search indexed code repositories for DevOps and infrastructure code. "
            "Searches Terraform, CloudFormation, Helm, Ansible, Docker, Kubernetes. "
            "Returns code snippets with file paths and line numbers."
        ),
        "wistx_regex_search": (
            "Search code repositories using regex patterns. "
            "Supports complex pattern matching across multiple files. "
            "Returns matching code sections with context."
        ),
        "wistx_design_architecture": (
            "Design and initialize DevOps/infrastructure/SRE/platform engineering projects. "
            "Creates architecture diagrams, infrastructure code, and deployment guides. "
            "Returns complete project structure and implementation plan."
        ),
        "wistx_troubleshoot_issue": (
            "Troubleshoot infrastructure and DevOps issues. "
            "Analyzes error messages and logs to identify root causes. "
            "Returns diagnosis, solutions, and prevention strategies."
        ),
        "wistx_generate_documentation": (
            "Generate comprehensive documentation and reports. "
            "Creates architecture docs, runbooks, compliance reports, and guides. "
            "Returns formatted documentation ready for use."
        ),
        "wistx_manage_integration": (
            "Manage integrations between DevOps tools and cloud services. "
            "Supports CI/CD, monitoring, logging, and infrastructure tools. "
            "Returns integration patterns and implementation code."
        ),
        "wistx_manage_infrastructure": (
            "Manage infrastructure resources and configurations. "
            "Supports AWS, GCP, Azure resource management and updates. "
            "Returns operation results and status updates."
        ),
        "wistx_search_packages": (
            "Search DevOps and infrastructure packages across registries. "
            "Searches PyPI, npm, Terraform, Crates.io, Go, Helm, Ansible, Maven, NuGet, RubyGems. "
            "Returns package information, versions, and usage examples."
        ),
        "wistx_index_repository": (
            "Index a code repository for searching and analysis. "
            "Supports Git repositories with Terraform, CloudFormation, Helm, Ansible code. "
            "Returns indexing status and search readiness."
        ),
        "wistx_get_existing_infrastructure": (
            "Get existing infrastructure configuration and state. "
            "Retrieves current AWS, GCP, Azure resource configurations. "
            "Returns infrastructure details and relationships."
        ),
        "wistx_get_recommended_tools": (
            "Get recommended tools for a specific task. "
            "Returns tool recommendations with relevance scores, categories, and usage guidance. "
            "Use this tool when you need help discovering which tools to use."
        ),
        "wistx_list_tools_by_category": (
            "List all tools in a specific category. "
            "Use this tool to discover available tools organized by category. "
            "Categories: compliance, pricing, code, infrastructure, knowledge, indexing, documentation, packages, integration."
        ),
        "wistx_get_tool_documentation": (
            "Get detailed documentation for a specific tool. "
            "Returns comprehensive information about what a tool does, when to use it, when NOT to use it, and how to use it correctly."
        ),
        "wistx_index_resource": (
            "Index content (documentation website or document file) for user-specific search. "
            "Supports website crawling and single file indexing (PDF, DOCX, Markdown). "
            "Returns indexing status and search readiness."
        ),
        "wistx_list_resources": (
            "List all indexed resources for the user. "
            "Shows repositories, documentation, or documents that have been indexed. "
            "Returns resource list with status and metadata."
        ),
        "wistx_check_resource_status": (
            "Check indexing status and progress for a specific resource. "
            "Use this tool to monitor indexing progress and see current status."
        ),
        "wistx_delete_resource": (
            "Delete an indexed resource and all associated knowledge articles. "
            "Use this tool to remove indexed content from your account."
        ),
        "wistx_search_packages_read_file": (
            "Read specific file sections from package source code using SHA256 hash. "
            "Use this tool to get complete context around code snippets found in package searches. "
            "No indexing required - fetches packages on-demand from registries."
        ),
    }

    # Detailed descriptions (original long versions) - available on-demand
    DETAILED_DESCRIPTIONS = {
        "wistx_get_compliance_requirements": (
            "Get compliance requirements for infrastructure resources.\n\n"
            "**Keywords:** compliance, PCI-DSS, HIPAA, CIS, SOC2, NIST, ISO 27001, security standards, controls, remediation\n\n"
            "**When to Use:**\n"
            "- Use when asked about compliance standards (PCI-DSS, HIPAA, CIS, SOC2, NIST-800-53, ISO-27001, GDPR, FedRAMP) for infrastructure\n"
            "- Use for AWS/GCP/Azure resources (RDS, S3, EC2, Lambda, EKS, VPC, IAM, etc.)\n"
            "- Use when need specific compliance controls, remediation guidance, or verification procedures\n"
            "- Use when building infrastructure that must meet compliance requirements\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for general compliance questions (use general knowledge)\n"
            "- Do NOT use for non-infrastructure resources (applications, databases without infrastructure context)\n"
            "- Do NOT use for compliance standards not related to infrastructure\n\n"
            "**Returns:**\n"
            "- Compliance controls with severity levels (CRITICAL, HIGH, MEDIUM, LOW)\n"
            "- Remediation guidance and code snippets\n"
            "- Verification procedures and checklists\n"
            "- Compliance report (if generate_report=true)\n\n"
            "**Common Workflows:**\n"
            "1. Design infrastructure → Get compliance requirements → Apply controls → Verify compliance\n"
            "2. Audit infrastructure → Get compliance requirements → Check violations → Remediate issues\n"
            "3. Build new resources → Get compliance requirements → Implement compliant configuration\n\n"
            "**Related Tools:**\n"
            "- `wistx_calculate_infrastructure_cost`: Use after compliance to estimate costs\n"
            "- `wistx_get_devops_infra_code_examples`: Use to find compliant code examples\n"
            "- `wistx_research_knowledge_base`: Use for compliance best practices"
        ),
        "wistx_research_knowledge_base": (
            "Deep research tool for DevOps, infrastructure, compliance, FinOps, and platform engineering knowledge.\n\n"
            "**Keywords:** research, knowledge base, best practices, patterns, strategies, guides, DevOps, infrastructure, compliance, FinOps, platform engineering\n\n"
            "**When to Use:**\n"
            "- Use for comprehensive research queries about best practices, patterns, strategies, and guides\n"
            "- Use when you need cross-domain insights and relationships\n"
            "- Use for deep research that requires both internal knowledge and real-time web search\n"
            "- Use when you need structured knowledge with examples and recommendations\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for simple factual questions (use general knowledge)\n"
            "- Do NOT use for specific compliance requirements (use `wistx_get_compliance_requirements`)\n"
            "- Do NOT use for code examples (use `wistx_get_devops_infra_code_examples`)\n\n"
            "**Returns:**\n"
            "- Structured knowledge with examples and recommendations\n"
            "- Cross-domain relationships and impacts\n"
            "- Best practices and patterns\n"
            "- Real-time web search results (if include_web_search=true)\n\n"
            "**Common Workflows:**\n"
            "1. Research topic → Get comprehensive knowledge → Apply patterns → Implement solution\n"
            "2. Learn best practices → Research strategies → Find examples → Apply to project\n\n"
            "**Related Tools:**\n"
            "- `wistx_web_search`: Use for simple web searches\n"
            "- `wistx_get_compliance_requirements`: Use for specific compliance queries"
        ),
        "wistx_calculate_infrastructure_cost": (
            "Calculate infrastructure costs for cloud resources.\n\n"
            "**Keywords:** cost, pricing, cost optimization, AWS, GCP, Azure, infrastructure cost, monthly cost, annual cost\n\n"
            "**When to Use:**\n"
            "- Use when asked about pricing, costs, or cost optimization for AWS/GCP/Azure resources\n"
            "- Use to estimate monthly/annual costs for infrastructure\n"
            "- Use to get cost breakdown and optimization suggestions\n"
            "- Use when comparing costs across cloud providers\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for application-level costs (use general knowledge)\n"
            "- Do NOT use for non-cloud infrastructure costs\n\n"
            "**Returns:**\n"
            "- Monthly/annual costs\n"
            "- Cost breakdown by service/resource\n"
            "- Optimization suggestions\n"
            "- Cost comparison across providers\n\n"
            "**Common Workflows:**\n"
            "1. Design infrastructure → Calculate costs → Optimize → Recalculate\n"
            "2. Compare providers → Calculate costs → Choose provider → Implement\n\n"
            "**Related Tools:**\n"
            "- `wistx_get_compliance_requirements`: Use to ensure compliance before cost calculation\n"
            "- `wistx_get_devops_infra_code_examples`: Use to find cost-optimized code examples"
        ),
        "wistx_get_devops_infra_code_examples": (
            "Search infrastructure code examples from curated repositories.\n\n"
            "**Keywords:** code examples, reference implementations, sample code, Terraform, Kubernetes, Docker, CloudFormation, Helm, Ansible\n\n"
            "**When to Use:**\n"
            "- Use when asked for code examples, reference implementations, or sample code for infrastructure resources\n"
            "- Use to find production-ready code for specific services or patterns\n"
            "- Use when you need compliant code examples\n"
            "- Use to find code examples with cost estimates\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for searching user's own codebase (use `wistx_search_codebase`)\n"
            "- Do NOT use for general code questions (use general knowledge)\n\n"
            "**Returns:**\n"
            "- Production-ready code examples with metadata\n"
            "- Compliance analysis\n"
            "- Cost estimates\n"
            "- Code explanations and best practices\n\n"
            "**Common Workflows:**\n"
            "1. Need code example → Search examples → Review code → Adapt to project\n"
            "2. Find compliant code → Search with compliance filter → Use example → Verify compliance\n\n"
            "**Related Tools:**\n"
            "- `wistx_search_codebase`: Use to search your own indexed codebase\n"
            "- `wistx_get_compliance_requirements`: Use to verify compliance of examples"
        ),
        "wistx_web_search": (
            "Web search for security information and general web content.\n\n"
            "**Keywords:** web search, security search, CVEs, advisories, security vulnerabilities, latest information\n\n"
            "**When to Use:**\n"
            "- Use for security searches (CVEs, advisories, security vulnerabilities)\n"
            "- Use for general web search via Tavily\n"
            "- Use when you need current information and latest tools\n"
            "- Use for recent articles, documentation, and best practices\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for compliance requirements (use `wistx_get_compliance_requirements`)\n"
            "- Do NOT use for deep research (use `wistx_research_knowledge_base`)\n"
            "- Do NOT use for searching your own codebase (use `wistx_search_codebase`)\n\n"
            "**Returns:**\n"
            "- Recent articles and documentation\n"
            "- CVE database results (if include_cves=true)\n"
            "- Security advisories (if include_advisories=true)\n"
            "- Web search results from Tavily\n\n"
            "**Common Workflows:**\n"
            "1. Need current info → Web search → Review results → Apply knowledge\n"
            "2. Check security → Search CVEs → Review advisories → Take action\n\n"
            "**Related Tools:**\n"
            "- `wistx_research_knowledge_base`: Use for deep research with web search\n"
            "- `wistx_get_compliance_requirements`: Use for compliance-specific queries"
        ),
        "wistx_search_codebase": (
            "Search user's indexed codebase including repositories, documentation, and documents.\n\n"
            "**Keywords:** codebase search, repository search, indexed code, user code, code snippets\n\n"
            "**When to Use:**\n"
            "- Use when asked to search through user's own code, documentation, or indexed resources\n"
            "- Use to find code patterns in your repositories\n"
            "- Use to search indexed documentation and documents\n"
            "- Use when you need code snippets with file paths and line numbers\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for searching curated code examples (use `wistx_get_devops_infra_code_examples`)\n"
            "- Do NOT use for regex pattern matching (use `wistx_regex_search`)\n"
            "- Do NOT use for general code questions (use general knowledge)\n\n"
            "**Returns:**\n"
            "- Code snippets with file paths and line numbers\n"
            "- AI-analyzed results with explanations\n"
            "- Pattern detection and recommendations\n"
            "- Code snippets analysis\n\n"
            "**Common Workflows:**\n"
            "1. Need to find code → Search codebase → Review results → Use code\n"
            "2. Understand patterns → Search codebase → Analyze patterns → Apply insights\n\n"
            "**Related Tools:**\n"
            "- `wistx_regex_search`: Use for exact pattern matching\n"
            "- `wistx_get_devops_infra_code_examples`: Use for curated code examples"
        ),
        "wistx_regex_search": (
            "Search user's indexed codebase using regex patterns.\n\n"
            "**Keywords:** regex search, pattern matching, security audit, compliance check, code analysis\n\n"
            "**When to Use:**\n"
            "- Use for exact pattern matching across codebase\n"
            "- Use for security audits (finding secrets, API keys, credentials)\n"
            "- Use for compliance checks (finding violations)\n"
            "- Use for code analysis with complex patterns\n"
            "- Use with pre-built templates (api_key, password, ip_address, etc.)\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for natural language queries (use `wistx_search_codebase`)\n"
            "- Do NOT use for simple text search (use `wistx_search_codebase`)\n\n"
            "**Returns:**\n"
            "- Matching code sections with context\n"
            "- File paths and line numbers\n"
            "- Surrounding code context (if include_context=true)\n\n"
            "**Common Workflows:**\n"
            "1. Security audit → Regex search for secrets → Review findings → Remediate\n"
            "2. Compliance check → Regex search for violations → Fix issues → Verify\n\n"
            "**Related Tools:**\n"
            "- `wistx_search_codebase`: Use for natural language code search\n"
            "- `wistx_get_compliance_requirements`: Use to understand compliance requirements"
        ),
        "wistx_design_architecture": (
            "Design and initialize DevOps/infrastructure/SRE/platform engineering projects.\n\n"
            "**Keywords:** architecture design, project initialization, scaffold, architecture review, architecture optimization\n\n"
            "**When to Use:**\n"
            "- Use to scaffold new projects with compliance and security built-in\n"
            "- Use to design architectures for new systems\n"
            "- Use to review existing architectures\n"
            "- Use to optimize architectures\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for simple code generation (use general knowledge)\n"
            "- Do NOT use for troubleshooting (use `wistx_troubleshoot_issue`)\n\n"
            "**Returns:**\n"
            "- Complete project structure\n"
            "- Architecture diagrams\n"
            "- Infrastructure code\n"
            "- Deployment guides\n"
            "- Implementation plan\n\n"
            "**Common Workflows:**\n"
            "1. New project → Initialize project → Design architecture → Implement\n"
            "2. Review architecture → Analyze → Optimize → Update\n\n"
            "**Related Tools:**\n"
            "- `wistx_get_compliance_requirements`: Use to ensure compliance in architecture\n"
            "- `wistx_calculate_infrastructure_cost`: Use to estimate costs for architecture"
        ),
        "wistx_troubleshoot_issue": (
            "Diagnose and fix infrastructure/code issues.\n\n"
            "**Keywords:** troubleshooting, error diagnosis, issue resolution, error analysis, fix recommendations\n\n"
            "**When to Use:**\n"
            "- Use when encountering errors or issues with infrastructure/code\n"
            "- Use to analyze error messages and logs\n"
            "- Use to identify root causes\n"
            "- Use to get fix recommendations and prevention strategies\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for architecture design (use `wistx_design_architecture`)\n"
            "- Do NOT use for general questions (use general knowledge)\n\n"
            "**Returns:**\n"
            "- Diagnosis of the issue\n"
            "- Root cause analysis\n"
            "- Fix recommendations\n"
            "- Prevention strategies\n\n"
            "**Common Workflows:**\n"
            "1. Encounter error → Troubleshoot → Get diagnosis → Apply fix\n"
            "2. Analyze logs → Troubleshoot → Identify root cause → Prevent recurrence\n\n"
            "**Related Tools:**\n"
            "- `wistx_get_compliance_requirements`: Use to check if issue is compliance-related\n"
            "- `wistx_research_knowledge_base`: Use to research similar issues"
        ),
        "wistx_generate_documentation": (
            "Generate comprehensive documentation and reports.\n\n"
            "**Keywords:** documentation generation, reports, architecture docs, runbooks, compliance reports, security reports, cost reports\n\n"
            "**When to Use:**\n"
            "- Use when asked to create documentation or reports\n"
            "- Use to generate architecture docs, runbooks, compliance reports, security reports, cost reports\n"
            "- Use to create API documentation and deployment guides\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for simple text generation (use general knowledge)\n"
            "- Do NOT use for code generation (use general knowledge)\n\n"
            "**Returns:**\n"
            "- Formatted documentation ready for use\n"
            "- Reports in requested format (markdown, PDF, HTML, JSON)\n"
            "- Comprehensive documentation with all requested sections\n\n"
            "**Common Workflows:**\n"
            "1. Need documentation → Generate docs → Review → Publish\n"
            "2. Create report → Generate report → Review → Share\n\n"
            "**Related Tools:**\n"
            "- `wistx_get_compliance_requirements`: Use to include compliance information in reports\n"
            "- `wistx_calculate_infrastructure_cost`: Use to include cost information in reports"
        ),
        "wistx_manage_integration": (
            "Analyze, generate, and validate infrastructure component integrations.\n\n"
            "**Keywords:** integration, component integration, networking, security, monitoring, service integration\n\n"
            "**When to Use:**\n"
            "- Use when working with component integrations\n"
            "- Use to analyze existing integrations\n"
            "- Use to generate integration code\n"
            "- Use to validate integrations\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for simple API calls (use general knowledge)\n"
            "- Do NOT use for application-level integrations\n\n"
            "**Returns:**\n"
            "- Integration patterns\n"
            "- Implementation code\n"
            "- Validation results\n"
            "- Integration recommendations\n\n"
            "**Common Workflows:**\n"
            "1. Need integration → Analyze requirements → Generate code → Validate\n"
            "2. Review integration → Analyze → Optimize → Update\n\n"
            "**Related Tools:**\n"
            "- `wistx_get_devops_infra_code_examples`: Use to find integration examples\n"
            "- `wistx_research_knowledge_base`: Use to research integration patterns"
        ),
        "wistx_manage_infrastructure": (
            "Manage infrastructure lifecycle for Kubernetes clusters and multi-cloud resources.\n\n"
            "**Keywords:** infrastructure management, Kubernetes, multi-cloud, cluster management, infrastructure lifecycle\n\n"
            "**When to Use:**\n"
            "- Use for Kubernetes cluster and multi-cloud management\n"
            "- Use to create, update, upgrade, backup, restore, monitor, and optimize infrastructure\n"
            "- Use for infrastructure lifecycle management\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for application deployment (use general knowledge)\n"
            "- Do NOT use for simple resource creation (use general knowledge)\n\n"
            "**Returns:**\n"
            "- Operation results\n"
            "- Status updates\n"
            "- Infrastructure state\n"
            "- Management recommendations\n\n"
            "**Common Workflows:**\n"
            "1. Manage infrastructure → Create/Update → Monitor → Optimize\n"
            "2. Backup infrastructure → Backup → Restore → Verify\n\n"
            "**Related Tools:**\n"
            "- `wistx_get_existing_infrastructure`: Use to get current infrastructure state\n"
            "- `wistx_calculate_infrastructure_cost`: Use to estimate costs for infrastructure changes"
        ),
        "wistx_search_packages": (
            "Search DevOps/infrastructure packages across multiple registries.\n\n"
            "**Keywords:** package search, PyPI, npm, Terraform Registry, Crates.io, Go Modules, Helm Charts, Ansible Galaxy, Maven Central, NuGet, RubyGems\n\n"
            "**When to Use:**\n"
            "- Use to find packages that implement compliance, cost optimization, infrastructure patterns\n"
            "- Use for semantic search (natural language), regex search (pattern matching), or hybrid search\n"
            "- Use to find packages filtered to DevOps/infrastructure domain\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for searching code examples (use `wistx_get_devops_infra_code_examples`)\n"
            "- Do NOT use for general package questions (use general knowledge)\n\n"
            "**Returns:**\n"
            "- Package information, versions, and usage examples\n"
            "- Package metadata and descriptions\n"
            "- Search results with relevance scores\n\n"
            "**Common Workflows:**\n"
            "1. Need package → Search packages → Review results → Use package\n"
            "2. Find compliant package → Search with compliance filter → Review → Integrate\n\n"
            "**Related Tools:**\n"
            "- `wistx_search_packages_read_file`: Use to read package source code\n"
            "- `wistx_get_devops_infra_code_examples`: Use to find code examples using packages"
        ),
        "wistx_index_repository": (
            "Index a GitHub repository for user-specific search.\n\n"
            "**Keywords:** index repository, GitHub, repository indexing, code indexing\n\n"
            "**When to Use:**\n"
            "- Use when asked to index or search through a GitHub repository\n"
            "- Use for both public and private repositories\n"
            "- Use to make repository code searchable\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for searching already indexed repositories (use `wistx_search_codebase`)\n"
            "- Do NOT use for indexing documentation websites (use `wistx_index_resource`)\n\n"
            "**Returns:**\n"
            "- Indexing status and search readiness\n"
            "- Resource ID for future reference\n"
            "- Indexing progress information\n\n"
            "**Common Workflows:**\n"
            "1. Need to search repo → Index repository → Wait for completion → Search codebase\n"
            "2. Add new repo → Index → Verify status → Use in searches\n\n"
            "**Related Tools:**\n"
            "- `wistx_search_codebase`: Use to search indexed repositories\n"
            "- `wistx_check_resource_status`: Use to check indexing progress"
        ),
        "wistx_get_existing_infrastructure": (
            "Get existing infrastructure context for a repository.\n\n"
            "**Keywords:** existing infrastructure, infrastructure context, infrastructure state, repository infrastructure\n\n"
            "**When to Use:**\n"
            "- Use when coding agents need to understand existing infrastructure before creating new resources\n"
            "- Use to get cost analysis, compliance status, and recommendations based on indexed repository data\n"
            "- Use to understand current infrastructure state\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for designing new infrastructure (use `wistx_design_architecture`)\n"
            "- Do NOT use for general infrastructure questions (use general knowledge)\n\n"
            "**Returns:**\n"
            "- Infrastructure details and relationships\n"
            "- Cost analysis\n"
            "- Compliance status\n"
            "- Recommendations\n\n"
            "**Common Workflows:**\n"
            "1. Understand infrastructure → Get existing infrastructure → Review → Plan changes\n"
            "2. Audit infrastructure → Get existing infrastructure → Analyze → Report\n\n"
            "**Related Tools:**\n"
            "- `wistx_get_compliance_requirements`: Use to check compliance of existing infrastructure\n"
            "- `wistx_calculate_infrastructure_cost`: Use to calculate costs for existing infrastructure"
        ),
        "wistx_get_recommended_tools": (
            "Get recommended tools for a specific task.\n\n"
            "**Keywords:** tool recommendations, tool discovery, tool suggestions, tool guidance\n\n"
            "**When to Use:**\n"
            "- Use when you need help discovering which tools to use for a given task\n"
            "- Use to get tool recommendations with relevance scores\n"
            "- Use to understand tool categories and usage guidance\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use if you already know which tool to use\n"
            "- Do NOT use for general questions (use general knowledge)\n\n"
            "**Returns:**\n"
            "- Tool recommendations with relevance scores (0-1)\n"
            "- Tool categories\n"
            "- Usage guidance\n"
            "- Explanations for recommendations\n\n"
            "**Common Workflows:**\n"
            "1. New task → Get recommended tools → Review recommendations → Use tools\n"
            "2. Explore tools → Get recommendations → Learn about tools → Use appropriate tool\n\n"
            "**Related Tools:**\n"
            "- `wistx_list_tools_by_category`: Use to list tools by category\n"
            "- `wistx_get_tool_documentation`: Use to get detailed information about recommended tools"
        ),
        "wistx_list_tools_by_category": (
            "List all tools in a specific category.\n\n"
            "**Keywords:** tool list, tool categories, tool discovery, category filter\n\n"
            "**When to Use:**\n"
            "- Use to discover available tools organized by category\n"
            "- Use when you know the category but want to see all tools in that category\n"
            "- Use to explore tools by domain (compliance, pricing, code, infrastructure, etc.)\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use if you need tool recommendations (use `wistx_get_recommended_tools`)\n"
            "- Do NOT use for general questions (use general knowledge)\n\n"
            "**Returns:**\n"
            "- List of tools in the specified category\n"
            "- Short descriptions for each tool (if include_descriptions=true)\n"
            "- Available categories list\n\n"
            "**Common Workflows:**\n"
            "1. Explore category → List tools by category → Review tools → Use appropriate tool\n"
            "2. Find tools → List by category → Get documentation → Use tool\n\n"
            "**Related Tools:**\n"
            "- `wistx_get_recommended_tools`: Use to get tool recommendations\n"
            "- `wistx_get_tool_documentation`: Use to get detailed information about tools"
        ),
        "wistx_get_tool_documentation": (
            "Get detailed documentation for a specific tool.\n\n"
            "**Keywords:** tool documentation, tool help, tool guide, tool information\n\n"
            "**When to Use:**\n"
            "- Use when you need comprehensive information about what a tool does\n"
            "- Use to understand when to use a tool and when NOT to use it\n"
            "- Use to learn how to use a tool correctly\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for general questions (use general knowledge)\n"
            "- Do NOT use if you already understand the tool\n\n"
            "**Returns:**\n"
            "- Detailed tool description\n"
            "- When to use and when NOT to use guidance\n"
            "- Usage examples\n"
            "- Related tools information\n\n"
            "**Common Workflows:**\n"
            "1. Learn about tool → Get documentation → Understand usage → Use tool\n"
            "2. Verify tool usage → Get documentation → Confirm usage → Use tool\n\n"
            "**Related Tools:**\n"
            "- `wistx_get_recommended_tools`: Use to discover tools\n"
            "- `wistx_list_tools_by_category`: Use to list tools by category"
        ),
        "wistx_index_resource": (
            "Index content (documentation website or document file) for user-specific search.\n\n"
            "**Keywords:** index resource, documentation indexing, document indexing, website crawling\n\n"
            "**When to Use:**\n"
            "- Use when asked to index documentation websites, PDFs, DOCX files, or other documents\n"
            "- Use for both website crawling and single file indexing\n"
            "- Use to make documentation searchable\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for indexing code repositories (use `wistx_index_repository`)\n"
            "- Do NOT use for searching already indexed content (use `wistx_search_codebase`)\n\n"
            "**Returns:**\n"
            "- Indexing status and search readiness\n"
            "- Resource ID for future reference\n"
            "- Indexing progress information\n\n"
            "**Common Workflows:**\n"
            "1. Need to search docs → Index resource → Wait for completion → Search codebase\n"
            "2. Add documentation → Index → Verify status → Use in searches\n\n"
            "**Related Tools:**\n"
            "- `wistx_search_codebase`: Use to search indexed resources\n"
            "- `wistx_check_resource_status`: Use to check indexing progress"
        ),
        "wistx_list_resources": (
            "List all indexed resources for the user.\n\n"
            "**Keywords:** list resources, indexed resources, resource list, resource status\n\n"
            "**When to Use:**\n"
            "- Use to see what repositories, documentation, or documents have been indexed\n"
            "- Use to check resource status and metadata\n"
            "- Use to filter resources by type or status\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for searching resources (use `wistx_search_codebase`)\n"
            "- Do NOT use for general questions (use general knowledge)\n\n"
            "**Returns:**\n"
            "- List of indexed resources\n"
            "- Resource status and metadata\n"
            "- AI-analyzed insights (if include_ai_analysis=true)\n\n"
            "**Common Workflows:**\n"
            "1. Check resources → List resources → Review status → Use in searches\n"
            "2. Manage resources → List resources → Review → Delete if needed\n\n"
            "**Related Tools:**\n"
            "- `wistx_search_codebase`: Use to search indexed resources\n"
            "- `wistx_delete_resource`: Use to remove indexed resources"
        ),
        "wistx_check_resource_status": (
            "Check indexing status and progress for a specific resource.\n\n"
            "**Keywords:** resource status, indexing status, indexing progress, resource check\n\n"
            "**When to Use:**\n"
            "- Use to monitor indexing progress for a specific resource\n"
            "- Use to check if a resource is ready for searching\n"
            "- Use to see current indexing status\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for listing all resources (use `wistx_list_resources`)\n"
            "- Do NOT use for general questions (use general knowledge)\n\n"
            "**Returns:**\n"
            "- Indexing status and progress\n"
            "- Current status (pending, indexing, completed, failed)\n"
            "- Progress information\n\n"
            "**Common Workflows:**\n"
            "1. Index resource → Check status → Wait if needed → Use resource\n"
            "2. Monitor indexing → Check status → Review progress → Continue\n\n"
            "**Related Tools:**\n"
            "- `wistx_index_repository`: Use to index a repository\n"
            "- `wistx_index_resource`: Use to index documentation or documents"
        ),
        "wistx_delete_resource": (
            "Delete an indexed resource and all associated knowledge articles.\n\n"
            "**Keywords:** delete resource, remove resource, unindex resource\n\n"
            "**When to Use:**\n"
            "- Use to remove indexed content from your account\n"
            "- Use when you no longer need a resource indexed\n"
            "- Use to clean up indexed resources\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for general questions (use general knowledge)\n"
            "- Do NOT use if you want to keep the resource indexed\n\n"
            "**Returns:**\n"
            "- Deletion confirmation\n"
            "- Status of deletion operation\n\n"
            "**Common Workflows:**\n"
            "1. Remove resource → Delete resource → Confirm deletion → Verify removal\n"
            "2. Clean up → List resources → Delete unwanted → Verify\n\n"
            "**Related Tools:**\n"
            "- `wistx_list_resources`: Use to list resources before deletion\n"
            "- `wistx_index_repository`: Use to re-index if needed"
        ),
        "wistx_search_packages_read_file": (
            "Read specific file sections from package source code using SHA256 hash.\n\n"
            "**Keywords:** read package file, package source code, file reading, code context\n\n"
            "**When to Use:**\n"
            "- Use to get complete context around code snippets found in package searches\n"
            "- Use to read specific file sections from package source code\n"
            "- Use when you need more context than search results provide\n\n"
            "**When NOT to Use:**\n"
            "- Do NOT use for searching packages (use `wistx_search_packages`)\n"
            "- Do NOT use for general code questions (use general knowledge)\n\n"
            "**Returns:**\n"
            "- File content with line numbers\n"
            "- Code context around requested section\n"
            "- Package and file metadata\n\n"
            "**Common Workflows:**\n"
            "1. Find package → Search packages → Read file → Use code\n"
            "2. Need context → Read file → Review code → Apply pattern\n\n"
            "**Related Tools:**\n"
            "- `wistx_search_packages`: Use to find packages first\n"
            "- `wistx_get_devops_infra_code_examples`: Use for curated code examples"
        ),
    }

    @classmethod
    def get_short_description(cls, tool_name: str) -> str:
        """Get short description for a tool (50-150 tokens)."""
        return cls.SHORT_DESCRIPTIONS.get(
            tool_name,
            f"Tool: {tool_name}. Use wistx_get_tool_documentation to see detailed description."
        )

    @classmethod
    def get_detailed_description(cls, tool_name: str) -> Optional[str]:
        """Get detailed description for a tool (on-demand)."""
        return cls.DETAILED_DESCRIPTIONS.get(tool_name)

    @classmethod
    def has_short_description(cls, tool_name: str) -> bool:
        """Check if short description exists for tool."""
        return tool_name in cls.SHORT_DESCRIPTIONS

    @classmethod
    def has_detailed_description(cls, tool_name: str) -> bool:
        """Check if detailed description exists for tool."""
        return tool_name in cls.DETAILED_DESCRIPTIONS

    @classmethod
    def add_short_description(cls, tool_name: str, description: str) -> None:
        """Add or update short description for a tool."""
        cls.SHORT_DESCRIPTIONS[tool_name] = description
        logger.info(f"Added short description for {tool_name}")

    @classmethod
    def add_detailed_description(cls, tool_name: str, description: str) -> None:
        """Add or update detailed description for a tool."""
        cls.DETAILED_DESCRIPTIONS[tool_name] = description
        logger.info(f"Added detailed description for {tool_name}")

