"""Tool permission matrix for authorization enforcement."""

from typing import Any

TOOL_PERMISSIONS: dict[str, dict[str, Any]] = {
    "wistx_get_compliance_requirements": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_calculate_infrastructure_cost": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_research_knowledge_base": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_get_devops_infra_code_examples": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_index_repository": {
        "required_plan": "scout",
        "required_permission": "indexing",
        "quota_required": True,
    },
    "wistx_index_resource": {
        "required_plan": "scout",
        "required_permission": "indexing",
        "quota_required": True,
    },
    "wistx_list_resources": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_check_resource_status": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_delete_resource": {
        "required_plan": "scout",
        "required_permission": "indexing",
        "quota_required": True,
    },
    "wistx_search_codebase": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_regex_search": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_web_search": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_design_architecture": {
        "required_plan": "builder",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_troubleshoot_issue": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_generate_documentation": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_manage_integration": {
        "required_plan": "builder",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_manage_infrastructure": {
        "required_plan": "builder",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_get_github_file_tree": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_visualize_infra_flow": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_get_infrastructure_context": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_search_packages": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_resolve_incident": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": True,
    },
    "wistx_get_recommended_tools": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": False,
    },
    "wistx_list_tools_by_category": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": False,
    },
    "wistx_get_tool_documentation": {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": False,
    },
}

PLAN_HIERARCHY = {"scout": 1, "pro": 2, "builder": 3, "architect": 4}


def get_tool_permissions(tool_name: str) -> dict[str, Any]:
    """Get permission requirements for a tool.
    
    Args:
        tool_name: Name of the tool
        
    Returns:
        Dictionary with required_plan, required_permission, quota_required
    """
    return TOOL_PERMISSIONS.get(tool_name, {
        "required_plan": "scout",
        "required_permission": None,
        "quota_required": True,
    })


def check_plan_access(user_plan: str, required_plan: str) -> bool:
    """Check if user plan meets minimum requirement.
    
    Args:
        user_plan: User's current plan
        required_plan: Minimum required plan
        
    Returns:
        True if user plan meets requirement
    """
    user_level = PLAN_HIERARCHY.get(user_plan, 0)
    required_level = PLAN_HIERARCHY.get(required_plan, 999)
    return user_level >= required_level

