"""Search codebase tool - search user's indexed repositories, documentation, and documents."""

import logging
from typing import Any

from bson import ObjectId

from wistx_mcp.tools.lib.mongodb_client import MongoDBClient
from wistx_mcp.tools.lib.vector_search import VectorSearch
from wistx_mcp.tools.lib.api_client import WISTXAPIClient
from wistx_mcp.tools.lib.ai_analyzer import AIAnalyzer
from wistx_mcp.tools.lib.plan_enforcement import require_query_quota
from wistx_mcp.config import settings

logger = logging.getLogger(__name__)

api_client = WISTXAPIClient()
ai_analyzer = AIAnalyzer()


@require_query_quota
async def search_codebase(
    query: str,
    api_key: str,
    repositories: list[str] | None = None,
    resource_ids: list[str] | None = None,
    resource_types: list[str] | None = None,
    file_types: list[str] | None = None,
    code_type: str | None = None,
    cloud_provider: str | None = None,
    include_sources: bool = True,
    include_ai_analysis: bool = True,
    limit: int = 1000,
) -> dict[str, Any]:
    """Search user's indexed codebase (repositories, documentation, documents).

    Args:
        query: Natural language search question
        api_key: WISTX API key for authentication
        repositories: List of repositories to search (owner/repo format, e.g., ['owner/repo', 'another/org/repo'])
        resource_ids: Filter by specific indexed resources (alternative to repositories)
        resource_types: Filter by resource type (repository, documentation, document)
        file_types: Filter by file extensions (.tf, .yaml, .py, .md, etc.)
        code_type: Filter by code type (terraform, kubernetes, docker, python)
        cloud_provider: Filter by cloud provider mentioned in code
        include_sources: Include source code snippets in results (default: True)
        include_ai_analysis: Include AI-analyzed results with explanations (default: True)
        limit: Maximum number of results

    Returns:
        Dictionary with search results:
        - results: List of matching code/documentation
        - resources: Resource information
        - total: Total results count
        - highlights: Code highlights (if include_sources=True)
        - ai_analysis: AI analysis with explanations (if include_ai_analysis=True)

    Raises:
        ValueError: If api_key is not provided or invalid parameters
        Exception: If search fails
    """
    from wistx_mcp.tools.lib.auth_context import validate_api_key_and_get_user_id

    try:
        await validate_api_key_and_get_user_id(api_key)
    except (ValueError, RuntimeError) as e:
        raise

    from wistx_mcp.tools.lib.input_sanitizer import validate_query_input

    validate_query_input(query)

    if limit < 1 or limit > 50000:
        raise ValueError("limit must be between 1 and 50000")

    logger.info(
        "Codebase search: query='%s', resources=%s, types=%s",
        query[:100],
        resource_ids,
        resource_types,
    )

    try:
        api_response = await api_client.search_codebase(
            query=query,
            repositories=repositories,
            resource_ids=resource_ids,
            resource_types=resource_types,
            file_types=file_types,
            code_type=code_type,
            cloud_provider=cloud_provider,
            include_sources=include_sources,
            include_ai_analysis=include_ai_analysis,
            limit=limit,
            api_key=api_key,
        )

        if api_response.get("data"):
            return api_response["data"]
        return api_response

    except Exception as e:
        logger.error("Error in search_codebase: %s", e, exc_info=True)
        raise

