from pydantic import BaseModel, Field
from typing import Annotated, Generic, Literal, TypeGuard
from uuid import UUID
from maleo.enums.service import OptServiceTypeT, OptServiceCategoryT
from maleo.schemas.mixins.identity import Identifier
from maleo.types.string import OptStrT
from ..enums.service import IdentifierType
from ..types.service import IdentifierValueType


class ServiceType(BaseModel, Generic[OptServiceTypeT]):
    type: OptServiceTypeT = Field(..., description="Service's type")


class Category(BaseModel, Generic[OptServiceCategoryT]):
    category: OptServiceCategoryT = Field(..., description="Service's category")


class Key(BaseModel):
    key: str = Field(..., max_length=20, description="Service's key")


class Name(BaseModel, Generic[OptStrT]):
    name: OptStrT = Field(..., max_length=20, description="Service's name")


class Secret(BaseModel):
    secret: UUID = Field(..., description="Service's secret")


class ServiceIdentifier(Identifier[IdentifierType, IdentifierValueType]):
    @property
    def column_and_value(self) -> tuple[str, IdentifierValueType]:
        return self.type.column, self.value


class IdServiceIdentifier(Identifier[Literal[IdentifierType.ID], int]):
    type: Annotated[
        Literal[IdentifierType.ID],
        Field(IdentifierType.ID, description="Identifier's type"),
    ] = IdentifierType.ID
    value: Annotated[int, Field(..., description="Identifier's value", ge=1)]


class UUIDServiceIdentifier(Identifier[Literal[IdentifierType.UUID], UUID]):
    type: Annotated[
        Literal[IdentifierType.UUID],
        Field(IdentifierType.UUID, description="Identifier's type"),
    ] = IdentifierType.UUID


class KeyServiceIdentifier(Identifier[Literal[IdentifierType.KEY], str]):
    type: Annotated[
        Literal[IdentifierType.KEY],
        Field(IdentifierType.KEY, description="Identifier's type"),
    ] = IdentifierType.KEY
    value: Annotated[str, Field(..., description="Identifier's value", max_length=20)]


class NameServiceIdentifier(Identifier[Literal[IdentifierType.NAME], str]):
    type: Annotated[
        Literal[IdentifierType.NAME],
        Field(IdentifierType.NAME, description="Identifier's type"),
    ] = IdentifierType.NAME
    value: Annotated[str, Field(..., description="Identifier's value", max_length=20)]


AnyServiceIdentifier = (
    ServiceIdentifier
    | IdServiceIdentifier
    | UUIDServiceIdentifier
    | KeyServiceIdentifier
    | NameServiceIdentifier
)


def is_id_identifier(
    identifier: AnyServiceIdentifier,
) -> TypeGuard[IdServiceIdentifier]:
    return identifier.type is IdentifierType.ID and isinstance(identifier.value, int)


def is_uuid_identifier(
    identifier: AnyServiceIdentifier,
) -> TypeGuard[UUIDServiceIdentifier]:
    return identifier.type is IdentifierType.UUID and isinstance(identifier.value, UUID)


def is_key_identifier(
    identifier: AnyServiceIdentifier,
) -> TypeGuard[KeyServiceIdentifier]:
    return identifier.type is IdentifierType.KEY and isinstance(identifier.value, str)


def is_name_identifier(
    identifier: AnyServiceIdentifier,
) -> TypeGuard[NameServiceIdentifier]:
    return identifier.type is IdentifierType.NAME and isinstance(identifier.value, str)
