from typing import overload
import abc
import datetime
import typing
import warnings

import QuantConnect
import QuantConnect.Algorithm.Framework.Portfolio
import QuantConnect.Brokerages
import QuantConnect.Data
import QuantConnect.Data.Fundamental
import QuantConnect.Data.Market
import QuantConnect.Data.UniverseSelection
import QuantConnect.Indicators
import QuantConnect.Interfaces
import QuantConnect.Orders
import QuantConnect.Orders.Fees
import QuantConnect.Orders.Fills
import QuantConnect.Orders.Slippage
import QuantConnect.Securities
import QuantConnect.Securities.CurrencyConversion
import QuantConnect.Securities.Interfaces
import QuantConnect.Securities.Positions
import QuantConnect.Securities.Volatility
import System
import System.Collections
import System.Collections.Concurrent
import System.Collections.Generic
import System.Collections.ObjectModel
import System.Collections.Specialized

IDynamicMetaObjectProvider = typing.Any
QuantConnect_Securities_SecurityDatabaseKey = typing.Any
QuantConnect_Securities_MarketHoursDatabase = typing.Any
QuantConnect_Securities_OptionFilterUniverse = typing.Any
QuantConnect_Securities_FutureFilterUniverse = typing.Any

QuantConnect_Securities_SecurityCache_GetData_T = typing.TypeVar("QuantConnect_Securities_SecurityCache_GetData_T")
QuantConnect_Securities_SecurityCache_GetAll_T = typing.TypeVar("QuantConnect_Securities_SecurityCache_GetAll_T")
QuantConnect_Securities_ContractSecurityFilterUniverse_T = typing.TypeVar("QuantConnect_Securities_ContractSecurityFilterUniverse_T")
QuantConnect_Securities_DynamicSecurityData_Get_T = typing.TypeVar("QuantConnect_Securities_DynamicSecurityData_Get_T")
QuantConnect_Securities_DynamicSecurityData_GetAll_T = typing.TypeVar("QuantConnect_Securities_DynamicSecurityData_GetAll_T")
QuantConnect_Securities_IndicatorVolatilityModel_T = typing.TypeVar("QuantConnect_Securities_IndicatorVolatilityModel_T")
QuantConnect_Securities__EventContainer_Callable = typing.TypeVar("QuantConnect_Securities__EventContainer_Callable")
QuantConnect_Securities__EventContainer_ReturnType = typing.TypeVar("QuantConnect_Securities__EventContainer_ReturnType")


class MarketHoursState(System.Enum):
    """Specifies the open/close state for a MarketHoursSegment"""

    Closed = 0
    """The market is not open (0)"""

    PreMarket = 1
    """The market is open, but before normal trading hours (1)"""

    Market = 2
    """The market is open and within normal trading hours (2)"""

    PostMarket = 3
    """The market is open, but after normal trading hours (3)"""


class MarketHoursSegment(System.Object):
    """Represents the state of an exchange during a specified time range"""

    @property
    def Start(self) -> datetime.timedelta:
        """Gets the start time for this segment"""
        ...

    @Start.setter
    def Start(self, value: datetime.timedelta):
        """Gets the start time for this segment"""
        ...

    @property
    def End(self) -> datetime.timedelta:
        """Gets the end time for this segment"""
        ...

    @End.setter
    def End(self, value: datetime.timedelta):
        """Gets the end time for this segment"""
        ...

    @property
    def State(self) -> int:
        """
        Gets the market hours state for this segment
        
        This property contains the int value of a member of the QuantConnect.Securities.MarketHoursState enum.
        """
        ...

    @State.setter
    def State(self, value: int):
        """
        Gets the market hours state for this segment
        
        This property contains the int value of a member of the QuantConnect.Securities.MarketHoursState enum.
        """
        ...

    def __init__(self, state: QuantConnect.Securities.MarketHoursState, start: datetime.timedelta, end: datetime.timedelta) -> None:
        """
        Initializes a new instance of the MarketHoursSegment class
        
        :param state: The state of the market during the specified times
        :param start: The start time of the segment
        :param end: The end time of the segment
        """
        ...

    @staticmethod
    def ClosedAllDay() -> QuantConnect.Securities.MarketHoursSegment:
        """Gets a new market hours segment representing being open all day"""
        ...

    def Contains(self, time: datetime.timedelta) -> bool:
        """
        Determines whether or not the specified time is contained within this segment
        
        :param time: The time to check
        :returns: True if this segment contains the specified time, false otherwise.
        """
        ...

    @staticmethod
    def GetMarketHoursSegments(extendedMarketOpen: datetime.timedelta, marketOpen: datetime.timedelta, marketClose: datetime.timedelta, extendedMarketClose: datetime.timedelta) -> typing.List[QuantConnect.Securities.MarketHoursSegment]:
        """
        Creates the market hours segments for the specified market open/close times
        
        :param extendedMarketOpen: The extended market open time. If no pre market, set to market open
        :param marketOpen: The regular market open time
        :param marketClose: The regular market close time
        :param extendedMarketClose: The extended market close time. If no post market, set to market close
        :returns: An array of MarketHoursSegment representing the specified market open/close times.
        """
        ...

    @staticmethod
    def OpenAllDay() -> QuantConnect.Securities.MarketHoursSegment:
        """Gets a new market hours segment representing being open all day"""
        ...

    def Overlaps(self, start: datetime.timedelta, end: datetime.timedelta) -> bool:
        """
        Determines whether or not the specified time range overlaps with this segment
        
        :param start: The start of the range
        :param end: The end of the range
        :returns: True if the specified range overlaps this time segment, false otherwise.
        """
        ...

    def ToString(self) -> str:
        """
        Returns a string that represents the current object.
        
        :returns: A string that represents the current object.
        """
        ...


class LocalMarketHours(System.Object):
    """Represents the market hours under normal conditions for an exchange and a specific day of the week in terms of local time"""

    @property
    def IsClosedAllDay(self) -> bool:
        """Gets whether or not this exchange is closed all day"""
        ...

    @property
    def IsOpenAllDay(self) -> bool:
        """Gets whether or not this exchange is closed all day"""
        ...

    @property
    def DayOfWeek(self) -> int:
        """
        Gets the day of week these hours apply to
        
        This property contains the int value of a member of the System.DayOfWeek enum.
        """
        ...

    @property
    def MarketDuration(self) -> datetime.timedelta:
        """
        Gets the tradable time during the market day.
        For a normal US equity trading day this is 6.5 hours.
        This does NOT account for extended market hours and only
        considers MarketHoursState.Market
        """
        ...

    @property
    def Segments(self) -> System.Collections.ObjectModel.ReadOnlyCollection[QuantConnect.Securities.MarketHoursSegment]:
        """Gets the individual market hours segments that define the hours of operation for this day"""
        ...

    @overload
    def __init__(self, day: System.DayOfWeek, *segments: QuantConnect.Securities.MarketHoursSegment) -> None:
        """
        Initializes a new instance of the LocalMarketHours class
        
        :param day: The day of the week these hours are applicable
        :param segments: The open/close segments defining the market hours for one day
        """
        ...

    @overload
    def __init__(self, day: System.DayOfWeek, segments: System.Collections.Generic.IEnumerable[QuantConnect.Securities.MarketHoursSegment]) -> None:
        """
        Initializes a new instance of the LocalMarketHours class
        
        :param day: The day of the week these hours are applicable
        :param segments: The open/close segments defining the market hours for one day
        """
        ...

    @overload
    def __init__(self, day: System.DayOfWeek, extendedMarketOpen: datetime.timedelta, marketOpen: datetime.timedelta, marketClose: datetime.timedelta, extendedMarketClose: datetime.timedelta) -> None:
        """
        Initializes a new instance of the LocalMarketHours class from the specified open/close times
        
        :param day: The day of week these hours apply to
        :param extendedMarketOpen: The extended market open time
        :param marketOpen: The regular market open time, must be greater than or equal to the extended market open time
        :param marketClose: The regular market close time, must be greater than the regular market open time
        :param extendedMarketClose: The extended market close time, must be greater than or equal to the regular market close time
        """
        ...

    @overload
    def __init__(self, day: System.DayOfWeek, marketOpen: datetime.timedelta, marketClose: datetime.timedelta) -> None:
        """
        Initializes a new instance of the LocalMarketHours class from the specified open/close times
        using the market open as the extended market open and the market close as the extended market close, effectively
        removing any 'extended' session from these exchange hours
        
        :param day: The day of week these hours apply to
        :param marketOpen: The regular market open time
        :param marketClose: The regular market close time, must be greater than the regular market open time
        """
        ...

    @staticmethod
    def ClosedAllDay(dayOfWeek: System.DayOfWeek) -> QuantConnect.Securities.LocalMarketHours:
        """
        Gets a LocalMarketHours instance that is always closed
        
        :param dayOfWeek: The day of week
        :returns: A LocalMarketHours instance that is always closed.
        """
        ...

    def GetMarketClose(self, time: datetime.timedelta, extendedMarket: bool, nextDaySegmentStart: typing.Optional[datetime.timedelta] = None) -> typing.Optional[datetime.timedelta]:
        """
        Gets the market closing time of day
        
        :param time: The reference time, the close returned will be the first close after the specified time if there are multiple market open segments
        :param extendedMarket: True to include extended market hours, false for regular market hours
        :param nextDaySegmentStart: Next day first segment start. This is used when the potential next market close is the last segment of the day so we need to check that segment is not continued on next day first segment. If null, it means there are no segments on the next day
        :returns: The market's closing time of day.
        """
        ...

    def GetMarketOpen(self, time: datetime.timedelta, extendedMarket: bool, previousDayLastSegment: typing.Optional[datetime.timedelta] = None) -> typing.Optional[datetime.timedelta]:
        """
        Gets the market opening time of day
        
        :param time: The reference time, the open returned will be the first open after the specified time if there are multiple market open segments
        :param extendedMarket: True to include extended market hours, false for regular market hours
        :param previousDayLastSegment: The previous days last segment. This is used when the potential next market open is the first segment of the day so we need to check that segment is not part of previous day last segment. If null, it means there were no segments on the last day
        :returns: The market's opening time of day.
        """
        ...

    @staticmethod
    def IsContinuousMarketOpen(previousSegmentEnd: typing.Optional[datetime.timedelta], nextSegmentStart: typing.Optional[datetime.timedelta], prevSegmentIsFromPrevDay: bool = True) -> bool:
        """
        Check the given segment is not part of the current previous segment
        
        :param previousSegmentEnd: Previous segment end time before the current segment
        :param nextSegmentStart: The next segment start time
        :param prevSegmentIsFromPrevDay: Indicated whether the previous segment is from the previous day or not (then it is from the same day as the next segment). Defaults to true
        :returns: True if indeed the given segment is part of the last segment. False otherwise.
        """
        ...

    @overload
    def IsOpen(self, time: datetime.timedelta, extendedMarket: bool) -> bool:
        """
        Determines if the exchange is open at the specified time
        
        :param time: The time of day to check
        :param extendedMarket: True to check exended market hours, false to check regular market hours
        :returns: True if the exchange is considered open, false otherwise.
        """
        ...

    @overload
    def IsOpen(self, start: datetime.timedelta, end: datetime.timedelta, extendedMarket: bool) -> bool:
        """
        Determines if the exchange is open during the specified interval
        
        :param start: The start time of the interval
        :param end: The end time of the interval
        :param extendedMarket: True to check exended market hours, false to check regular market hours
        :returns: True if the exchange is considered open, false otherwise.
        """
        ...

    @staticmethod
    def OpenAllDay(dayOfWeek: System.DayOfWeek) -> QuantConnect.Securities.LocalMarketHours:
        """
        Gets a LocalMarketHours instance that is always open
        
        :param dayOfWeek: The day of week
        :returns: A LocalMarketHours instance that is always open.
        """
        ...

    def ToString(self) -> str:
        """
        Returns a string that represents the current object.
        
        :returns: A string that represents the current object.
        """
        ...


class SecurityExchangeHours(System.Object):
    """
    Represents the schedule of a security exchange. This includes daily regular and extended market hours
    as well as holidays, early closes and late opens.
    """

    @property
    def TimeZone(self) -> typing.Any:
        """Gets the time zone this exchange resides in"""
        ...

    @property
    def Holidays(self) -> System.Collections.Generic.HashSet[datetime.datetime]:
        """Gets the holidays for the exchange"""
        ...

    @property
    def MarketHours(self) -> System.Collections.Generic.IReadOnlyDictionary[System.DayOfWeek, QuantConnect.Securities.LocalMarketHours]:
        """Gets the market hours for this exchange"""
        ...

    @property
    def EarlyCloses(self) -> System.Collections.Generic.IReadOnlyDictionary[datetime.datetime, datetime.timedelta]:
        """Gets the early closes for this exchange"""
        ...

    @property
    def LateOpens(self) -> System.Collections.Generic.IReadOnlyDictionary[datetime.datetime, datetime.timedelta]:
        """Gets the late opens for this exchange"""
        ...

    @property
    def RegularMarketDuration(self) -> datetime.timedelta:
        """
        Gets the most common tradable time during the market week.
        For a normal US equity trading day this is 6.5 hours.
        This does NOT account for extended market hours and only
        considers MarketHoursState.Market
        """
        ...

    @property
    def IsMarketAlwaysOpen(self) -> bool:
        """Checks whether the market is always open or not"""
        ...

    @IsMarketAlwaysOpen.setter
    def IsMarketAlwaysOpen(self, value: bool):
        """Checks whether the market is always open or not"""
        ...

    def __init__(self, timeZone: typing.Any, holidayDates: System.Collections.Generic.IEnumerable[datetime.datetime], marketHoursForEachDayOfWeek: System.Collections.Generic.IReadOnlyDictionary[System.DayOfWeek, QuantConnect.Securities.LocalMarketHours], earlyCloses: System.Collections.Generic.IReadOnlyDictionary[datetime.datetime, datetime.timedelta], lateOpens: System.Collections.Generic.IReadOnlyDictionary[datetime.datetime, datetime.timedelta]) -> None:
        """
        Initializes a new instance of the SecurityExchangeHours class
        
        :param timeZone: The time zone the dates and hours are represented in
        :param holidayDates: The dates this exchange is closed for holiday
        :param marketHoursForEachDayOfWeek: The exchange's schedule for each day of the week
        :param earlyCloses: The dates this exchange has an early close
        :param lateOpens: The dates this exchange has a late open
        """
        ...

    @staticmethod
    def AlwaysOpen(timeZone: typing.Any) -> QuantConnect.Securities.SecurityExchangeHours:
        """Gets a SecurityExchangeHours instance that is always open"""
        ...

    def GetMarketHours(self, localDateTime: typing.Union[datetime.datetime, datetime.date]) -> QuantConnect.Securities.LocalMarketHours:
        """
        Helper to access the market hours field based on the day of week
        
        :param localDateTime: The local date time to retrieve market hours for
        """
        ...

    def GetNextMarketClose(self, localDateTime: typing.Union[datetime.datetime, datetime.date], extendedMarket: bool) -> datetime.datetime:
        """
        Gets the local date time corresponding to the next market close following the specified time
        
        :param localDateTime: The time to begin searching for market close (non-inclusive)
        :param extendedMarket: True to include extended market hours in the search
        :returns: The next market closing date time following the specified local date time.
        """
        ...

    def GetNextMarketOpen(self, localDateTime: typing.Union[datetime.datetime, datetime.date], extendedMarket: bool) -> datetime.datetime:
        """
        Gets the local date time corresponding to the next market open following the specified time
        
        :param localDateTime: The time to begin searching for market open (non-inclusive)
        :param extendedMarket: True to include extended market hours in the search
        :returns: The next market opening date time following the specified local date time.
        """
        ...

    def GetNextTradingDay(self, date: typing.Union[datetime.datetime, datetime.date]) -> datetime.datetime:
        """
        Gets the next trading day
        
        :param date: The date to start searching at
        :returns: The next trading day.
        """
        ...

    def GetPreviousMarketOpen(self, localDateTime: typing.Union[datetime.datetime, datetime.date], extendedMarket: bool) -> datetime.datetime:
        """
        Gets the local date time corresponding to the previous market open to the specified time
        
        :param localDateTime: The time to begin searching for the last market open (non-inclusive)
        :param extendedMarket: True to include extended market hours in the search
        :returns: The previous market opening date time to the specified local date time.
        """
        ...

    def GetPreviousTradingDay(self, localDate: typing.Union[datetime.datetime, datetime.date]) -> datetime.datetime:
        """
        Gets the previous trading day
        
        :param localDate: The date to start searching at in this exchange's time zones
        :returns: The previous trading day.
        """
        ...

    def IsDateOpen(self, localDateTime: typing.Union[datetime.datetime, datetime.date]) -> bool:
        """
        Determines if the exchange will be open on the date specified by the local date time
        
        :param localDateTime: The date time to check if the day is open
        :returns: True if the exchange will be open on the specified date, false otherwise.
        """
        ...

    @overload
    def IsOpen(self, localDateTime: typing.Union[datetime.datetime, datetime.date], extendedMarket: bool) -> bool:
        """
        Determines if the exchange is open at the specified local date time.
        
        :param localDateTime: The time to check represented as a local time
        :param extendedMarket: True to use the extended market hours, false for just regular market hours
        :returns: True if the exchange is considered open at the specified time, false otherwise.
        """
        ...

    @overload
    def IsOpen(self, startLocalDateTime: typing.Union[datetime.datetime, datetime.date], endLocalDateTime: typing.Union[datetime.datetime, datetime.date], extendedMarket: bool) -> bool:
        """
        Determines if the exchange is open at any point in time over the specified interval.
        
        :param startLocalDateTime: The start of the interval in local time
        :param endLocalDateTime: The end of the interval in local time
        :param extendedMarket: True to use the extended market hours, false for just regular market hours
        :returns: True if the exchange is considered open at the specified time, false otherwise.
        """
        ...


class SecurityDatabaseKey(System.Object, System.IEquatable[QuantConnect_Securities_SecurityDatabaseKey]):
    """Represents the key to a single entry in the MarketHoursDatabase or the SymbolPropertiesDatabase"""

    Wildcard: str = "[*]"
    """Represents that the specified symbol or market field will match all"""

    @property
    def Market(self) -> str:
        """The market. If null, ignore market filtering"""
        ...

    @property
    def Symbol(self) -> str:
        """The symbol. If null, ignore symbol filtering"""
        ...

    @property
    def SecurityType(self) -> QuantConnect.SecurityType:
        """The security type"""
        ...

    def __init__(self, market: str, symbol: str, securityType: QuantConnect.SecurityType) -> None:
        """
        Initializes a new instance of the SecurityDatabaseKey class
        
        :param market: The market
        :param symbol: The symbol. specify null to apply to all symbols in market/security type
        :param securityType: The security type
        """
        ...

    @overload
    def Equals(self, other: QuantConnect.Securities.SecurityDatabaseKey) -> bool:
        ...

    @overload
    def Equals(self, obj: typing.Any) -> bool:
        """
        Determines whether the specified object is equal to the current object.
        
        :param obj: The object to compare with the current object.
        :returns: true if the specified object  is equal to the current object; otherwise, false.
        """
        ...

    def GetHashCode(self) -> int:
        """
        Serves as the default hash function.
        
        :returns: A hash code for the current object.
        """
        ...

    @staticmethod
    def Parse(key: str) -> QuantConnect.Securities.SecurityDatabaseKey:
        """
        Parses the specified string as a SecurityDatabaseKey
        
        :param key: The string representation of the key
        :returns: A new SecurityDatabaseKey instance.
        """
        ...

    def ToString(self) -> str:
        ...


class MarketHoursDatabase(System.Object):
    """Provides access to exchange hours and raw data times zones in various markets"""

    class Entry(System.Object):
        """Represents a single entry in the MarketHoursDatabase"""

        @property
        def DataTimeZone(self) -> typing.Any:
            """Gets the raw data time zone for this entry"""
            ...

        @property
        def ExchangeHours(self) -> QuantConnect.Securities.SecurityExchangeHours:
            """Gets the exchange hours for this entry"""
            ...

        def __init__(self, dataTimeZone: typing.Any, exchangeHours: QuantConnect.Securities.SecurityExchangeHours) -> None:
            """
            Initializes a new instance of the Entry class
            
            :param dataTimeZone: The raw data time zone
            :param exchangeHours: The security exchange hours for this entry
            """
            ...

    class AlwaysOpenMarketHoursDatabaseImpl(QuantConnect_Securities_MarketHoursDatabase):
        """This class has no documentation."""

        def __init__(self) -> None:
            ...

        def GetEntry(self, market: str, symbol: str, securityType: QuantConnect.SecurityType) -> QuantConnect.Securities.MarketHoursDatabase.Entry:
            ...

    @property
    def ExchangeHoursListing(self) -> System.Collections.Generic.List[System.Collections.Generic.KeyValuePair[QuantConnect.Securities.SecurityDatabaseKey, QuantConnect.Securities.MarketHoursDatabase.Entry]]:
        """Gets all the exchange hours held by this provider"""
        ...

    AlwaysOpen: QuantConnect.Securities.MarketHoursDatabase
    """Gets a MarketHoursDatabase that always returns SecurityExchangeHours.AlwaysOpen"""

    def __init__(self, exchangeHours: System.Collections.Generic.IReadOnlyDictionary[QuantConnect.Securities.SecurityDatabaseKey, QuantConnect.Securities.MarketHoursDatabase.Entry]) -> None:
        """
        Initializes a new instance of the MarketHoursDatabase class
        
        :param exchangeHours: The full listing of exchange hours by key
        """
        ...

    def ContainsKey(self, key: QuantConnect.Securities.SecurityDatabaseKey) -> bool:
        """
        Determines if the database contains the specified key
        
        This method is protected.
        
        :param key: The key to search for
        :returns: True if an entry is found, otherwise false.
        """
        ...

    @staticmethod
    @overload
    def FromDataFolder() -> QuantConnect.Securities.MarketHoursDatabase:
        """
        Gets the instance of the MarketHoursDatabase class produced by reading in the market hours
        data found in /Data/market-hours/
        
        :returns: A MarketHoursDatabase class that represents the data in the market-hours folder.
        """
        ...

    @staticmethod
    @overload
    def FromDataFolder(dataFolder: str) -> QuantConnect.Securities.MarketHoursDatabase:
        """
        Gets the instance of the MarketHoursDatabase class produced by reading in the market hours
        data found in /Data/market-hours/
        
        :param dataFolder: Path to the data folder
        :returns: A MarketHoursDatabase class that represents the data in the market-hours folder.
        """
        ...

    @staticmethod
    def FromFile(path: str) -> QuantConnect.Securities.MarketHoursDatabase:
        """
        Reads the specified file as a market hours database instance
        
        :param path: The market hours database file path
        :returns: A new instance of the MarketHoursDatabase class.
        """
        ...

    @staticmethod
    def GetDatabaseSymbolKey(symbol: typing.Union[QuantConnect.Symbol, str]) -> str:
        """
        Gets the correct string symbol to use as a database key
        
        :param symbol: The symbol
        :returns: The symbol string used in the database ke.
        """
        ...

    def GetDataTimeZone(self, market: str, symbol: typing.Union[QuantConnect.Symbol, str], securityType: QuantConnect.SecurityType) -> typing.Any:
        """
        Performs a lookup using the specified information and returns the data's time zone if found,
        if an entry is not found, an exception is thrown
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded
        :param securityType: The security type of the symbol
        :returns: The raw data time zone for the specified security.
        """
        ...

    @overload
    def GetEntry(self, market: str, symbol: str, securityType: QuantConnect.SecurityType) -> QuantConnect.Securities.MarketHoursDatabase.Entry:
        """
        Gets the entry for the specified market/symbol/security-type
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded
        :param securityType: The security type of the symbol
        :returns: The entry matching the specified market/symbol/security-type.
        """
        ...

    @overload
    def GetEntry(self, market: str, symbol: typing.Union[QuantConnect.Symbol, str], securityType: QuantConnect.SecurityType) -> QuantConnect.Securities.MarketHoursDatabase.Entry:
        """
        Gets the entry for the specified market/symbol/security-type
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded (Symbol class)
        :param securityType: The security type of the symbol
        :returns: The entry matching the specified market/symbol/security-type.
        """
        ...

    @overload
    def GetExchangeHours(self, configuration: QuantConnect.Data.SubscriptionDataConfig) -> QuantConnect.Securities.SecurityExchangeHours:
        """
        Convenience method for retrieving exchange hours from market hours database using a subscription config
        
        :param configuration: The subscription data config to get exchange hours for
        :returns: The configure exchange hours for the specified configuration.
        """
        ...

    @overload
    def GetExchangeHours(self, market: str, symbol: typing.Union[QuantConnect.Symbol, str], securityType: QuantConnect.SecurityType) -> QuantConnect.Securities.SecurityExchangeHours:
        """
        Convenience method for retrieving exchange hours from market hours database using a subscription config
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded
        :param securityType: The security type of the symbol
        :returns: The exchange hours for the specified security.
        """
        ...

    @staticmethod
    def Reset() -> None:
        """
        Resets the market hours database, forcing a reload when reused.
        Called in tests where multiple algorithms are run sequentially,
        and we need to guarantee that every test starts with the same environment.
        """
        ...

    def SetEntry(self, market: str, symbol: str, securityType: QuantConnect.SecurityType, exchangeHours: QuantConnect.Securities.SecurityExchangeHours, dataTimeZone: typing.Any = None) -> QuantConnect.Securities.MarketHoursDatabase.Entry:
        """
        Sets the entry for the specified market/symbol/security-type.
        This is intended to be used by custom data and other data sources that don't have explicit
        entries in market-hours-database.csv. At run time, the algorithm can update the market hours
        database via calls to AddData.
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded
        :param securityType: The security type of the symbol
        :param exchangeHours: The exchange hours for the specified symbol
        :param dataTimeZone: The time zone of the symbol's raw data. Optional, defaults to the exchange time zone
        :returns: The entry matching the specified market/symbol/security-type.
        """
        ...

    def SetEntryAlwaysOpen(self, market: str, symbol: str, securityType: QuantConnect.SecurityType, timeZone: typing.Any) -> QuantConnect.Securities.MarketHoursDatabase.Entry:
        """
        Convenience method for the common custom data case.
        Sets the entry for the specified symbol using SecurityExchangeHours.AlwaysOpen(timeZone)
        This sets the data time zone equal to the exchange time zone as well.
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded
        :param securityType: The security type of the symbol
        :param timeZone: The time zone of the symbol's exchange and raw data
        :returns: The entry matching the specified market/symbol/security-type.
        """
        ...

    @overload
    def TryGetEntry(self, market: str, symbol: typing.Union[QuantConnect.Symbol, str], securityType: QuantConnect.SecurityType, entry: typing.Optional[QuantConnect.Securities.MarketHoursDatabase.Entry]) -> typing.Union[bool, QuantConnect.Securities.MarketHoursDatabase.Entry]:
        """
        Tries to get the entry for the specified market/symbol/security-type
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded
        :param securityType: The security type of the symbol
        :param entry: The entry found if any
        :returns: True if the entry was present, else false.
        """
        ...

    @overload
    def TryGetEntry(self, market: str, symbol: str, securityType: QuantConnect.SecurityType, entry: typing.Optional[QuantConnect.Securities.MarketHoursDatabase.Entry]) -> typing.Union[bool, QuantConnect.Securities.MarketHoursDatabase.Entry]:
        """
        Tries to get the entry for the specified market/symbol/security-type
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded
        :param securityType: The security type of the symbol
        :param entry: The entry found if any
        :returns: True if the entry was present, else false.
        """
        ...


class SymbolProperties(System.Object):
    """Represents common properties for a specific security, uniquely identified by market, symbol and security type"""

    @property
    def Description(self) -> str:
        """The description of the security"""
        ...

    @property
    def QuoteCurrency(self) -> str:
        """The quote currency of the security"""
        ...

    @property
    def ContractMultiplier(self) -> float:
        """The contract multiplier for the security"""
        ...

    @ContractMultiplier.setter
    def ContractMultiplier(self, value: float):
        """The contract multiplier for the security"""
        ...

    @property
    def MinimumPriceVariation(self) -> float:
        """The minimum price variation (tick size) for the security"""
        ...

    @MinimumPriceVariation.setter
    def MinimumPriceVariation(self, value: float):
        """The minimum price variation (tick size) for the security"""
        ...

    @property
    def LotSize(self) -> float:
        """The lot size (lot size of the order) for the security"""
        ...

    @property
    def MarketTicker(self) -> str:
        """The market ticker"""
        ...

    @property
    def MinimumOrderSize(self) -> typing.Optional[float]:
        """
        The minimum order size allowed
        For crypto/forex pairs it's expected to be expressed in base or quote currency
        i.e For BTC/USD the minimum order size allowed with GDAX is 0.0001 BTC
        while on Binance the minimum order size allowed is 10 USD
        """
        ...

    @property
    def PriceMagnifier(self) -> float:
        """
        Allows normalizing live asset prices to US Dollars for Lean consumption. In some exchanges,
        for some securities, data is expressed in cents like for example for corn futures ('ZC').
        """
        ...

    def __init__(self, description: str, quoteCurrency: str, contractMultiplier: float, minimumPriceVariation: float, lotSize: float, marketTicker: str, minimumOrderSize: typing.Optional[float] = None, priceMagnifier: float = 1) -> None:
        """Creates an instance of the SymbolProperties class"""
        ...

    @staticmethod
    def GetDefault(quoteCurrency: str) -> QuantConnect.Securities.SymbolProperties:
        """
        Gets a default instance of the SymbolProperties class for the specified
        
        :param quoteCurrency: The quote currency of the symbol
        :returns: A default instance of theSymbolProperties class.
        """
        ...


class SymbolPropertiesDatabase(System.Object):
    """Provides access to specific properties for various symbols"""

    def __init__(self, file: str) -> None:
        """
        Initialize a new instance of SymbolPropertiesDatabase using the given file
        
        This method is protected.
        
        :param file: File to read from
        """
        ...

    @overload
    def ContainsKey(self, market: str, symbol: str, securityType: QuantConnect.SecurityType) -> bool:
        """
        Check whether symbol properties exists for the specified market/symbol/security-type
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded
        :param securityType: The security type of the symbol
        """
        ...

    @overload
    def ContainsKey(self, market: str, symbol: typing.Union[QuantConnect.Symbol, str], securityType: QuantConnect.SecurityType) -> bool:
        """
        Check whether symbol properties exists for the specified market/symbol/security-type
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded (Symbol class)
        :param securityType: The security type of the symbol
        """
        ...

    @staticmethod
    def FromCsvLine(line: str, key: typing.Optional[QuantConnect.Securities.SecurityDatabaseKey]) -> typing.Union[QuantConnect.Securities.SymbolProperties, QuantConnect.Securities.SecurityDatabaseKey]:
        """
        Creates a new instance of SymbolProperties from the specified csv line
        
        This method is protected.
        
        :param line: The csv line to be parsed
        :param key: The key used to uniquely identify this security
        :returns: A new SymbolProperties for the specified csv line.
        """
        ...

    @staticmethod
    def FromDataFolder() -> QuantConnect.Securities.SymbolPropertiesDatabase:
        """
        Gets the instance of the SymbolPropertiesDatabase class produced by reading in the symbol properties
        data found in /Data/symbol-properties/
        
        :returns: A SymbolPropertiesDatabase class that represents the data in the symbol-properties folder.
        """
        ...

    def GetSymbolProperties(self, market: str, symbol: typing.Union[QuantConnect.Symbol, str], securityType: QuantConnect.SecurityType, defaultQuoteCurrency: str) -> QuantConnect.Securities.SymbolProperties:
        """
        Gets the symbol properties for the specified market/symbol/security-type
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param symbol: The particular symbol being traded (Symbol class)
        :param securityType: The security type of the symbol
        :param defaultQuoteCurrency: Specifies the quote currency to be used when returning a default instance of an entry is not found in the database
        :returns: The symbol properties matching the specified market/symbol/security-type or null if not found.
        """
        ...

    @overload
    def GetSymbolPropertiesList(self, market: str, securityType: QuantConnect.SecurityType) -> System.Collections.Generic.IEnumerable[System.Collections.Generic.KeyValuePair[QuantConnect.Securities.SecurityDatabaseKey, QuantConnect.Securities.SymbolProperties]]:
        """
        Gets a list of symbol properties for the specified market/security-type
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :param securityType: The security type of the symbol
        :returns: An IEnumerable of symbol properties matching the specified market/security-type.
        """
        ...

    @overload
    def GetSymbolPropertiesList(self, market: str) -> System.Collections.Generic.IEnumerable[System.Collections.Generic.KeyValuePair[QuantConnect.Securities.SecurityDatabaseKey, QuantConnect.Securities.SymbolProperties]]:
        """
        Gets a list of symbol properties for the specified market
        
        :param market: The market the exchange resides in, i.e, 'usa', 'fxcm', ect...
        :returns: An IEnumerable of symbol properties matching the specified market.
        """
        ...

    def SetEntry(self, market: str, symbol: str, securityType: QuantConnect.SecurityType, properties: QuantConnect.Securities.SymbolProperties) -> bool:
        """
        Set SymbolProperties entry for a particular market, symbol and security type.
        
        :param market: Market of the entry
        :param symbol: Symbol of the entry
        :param securityType: Type of security for the entry
        :param properties: The new symbol properties to store
        :returns: True if successful.
        """
        ...

    def TryGetMarket(self, symbol: str, securityType: QuantConnect.SecurityType, market: typing.Optional[str]) -> typing.Union[bool, str]:
        """
        Tries to get the market for the provided symbol/security type
        
        :param symbol: The particular symbol being traded
        :param securityType: The security type of the symbol
        :param market: The market the exchange resides in Market
        :returns: True if market was retrieved, false otherwise.
        """
        ...


class IRegisteredSecurityDataTypesProvider(metaclass=abc.ABCMeta):
    """Provides the set of base data types registered in the algorithm"""

    def RegisterType(self, type: typing.Type) -> bool:
        """
        Registers the specified type w/ the provider
        
        :returns: True if the type was previously not registered.
        """
        ...

    def TryGetType(self, name: str, type: typing.Optional[typing.Type]) -> typing.Union[bool, typing.Type]:
        """
        Determines if the specified type is registered or not and returns it
        
        :returns: True if the type was previously registered.
        """
        ...

    def UnregisterType(self, type: typing.Type) -> bool:
        """
        Removes the registration for the specified type
        
        :returns: True if the type was previously registered.
        """
        ...


class SecurityService(System.Object, QuantConnect.Interfaces.ISecurityService):
    """This class implements interface ISecurityService providing methods for creating new Security"""

    def __init__(self, cashBook: QuantConnect.Securities.CashBook, marketHoursDatabase: QuantConnect.Securities.MarketHoursDatabase, symbolPropertiesDatabase: QuantConnect.Securities.SymbolPropertiesDatabase, securityInitializerProvider: QuantConnect.Interfaces.ISecurityInitializerProvider, registeredTypes: QuantConnect.Securities.IRegisteredSecurityDataTypesProvider, cacheProvider: QuantConnect.Securities.SecurityCacheProvider, primaryExchangeProvider: QuantConnect.Interfaces.IPrimaryExchangeProvider = None) -> None:
        """Creates a new instance of the SecurityService class"""
        ...

    @overload
    def CreateSecurity(self, symbol: typing.Union[QuantConnect.Symbol, str], subscriptionDataConfigList: System.Collections.Generic.List[QuantConnect.Data.SubscriptionDataConfig], leverage: float = 0, addToSymbolCache: bool = True, underlying: QuantConnect.Securities.Security = None) -> QuantConnect.Securities.Security:
        """Creates a new security"""
        ...

    @overload
    def CreateSecurity(self, symbol: typing.Union[QuantConnect.Symbol, str], subscriptionDataConfig: QuantConnect.Data.SubscriptionDataConfig, leverage: float = 0, addToSymbolCache: bool = True, underlying: QuantConnect.Securities.Security = None) -> QuantConnect.Securities.Security:
        """Creates a new security"""
        ...

    def SetLiveMode(self, isLiveMode: bool) -> None:
        """
        Set live mode state of the algorithm
        
        :param isLiveMode: True, live mode is enabled
        """
        ...


class SecurityManager(QuantConnect.ExtendedDictionary[QuantConnect.Securities.Security], System.Collections.Generic.IDictionary[QuantConnect.Symbol, QuantConnect.Securities.Security], System.Collections.Specialized.INotifyCollectionChanged, typing.Iterable[System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.Security]]):
    """Enumerable security management class for grouping security objects into an array and providing any common properties."""

    @property
    def CollectionChanged(self) -> _EventContainer[typing.Callable[[System.Object, System.Collections.Specialized.NotifyCollectionChangedEventArgs], None], None]:
        """Event fired when a security is added or removed from this collection"""
        ...

    @CollectionChanged.setter
    def CollectionChanged(self, value: _EventContainer[typing.Callable[[System.Object, System.Collections.Specialized.NotifyCollectionChangedEventArgs], None], None]):
        """Event fired when a security is added or removed from this collection"""
        ...

    @property
    def UtcTime(self) -> datetime.datetime:
        """Gets the most recent time this manager was updated"""
        ...

    @property
    def Count(self) -> int:
        """Count of the number of securities in the collection."""
        ...

    @property
    def IsReadOnly(self) -> bool:
        """Flag indicating if the internal array is read only."""
        ...

    @property
    def Keys(self) -> System.Collections.Generic.ICollection[QuantConnect.Symbol]:
        """List of the symbol-keys in the collection of securities."""
        ...

    @property
    def GetKeys(self) -> System.Collections.Generic.IEnumerable[QuantConnect.Symbol]:
        """
        Gets an System.Collections.Generic.ICollection`1 containing the Symbol objects of the System.Collections.Generic.IDictionary`2.
        
        This property is protected.
        """
        ...

    @property
    def GetValues(self) -> System.Collections.Generic.IEnumerable[QuantConnect.Securities.Security]:
        """
        Gets an System.Collections.Generic.ICollection`1 containing the values in the System.Collections.Generic.IDictionary`2.
        
        This property is protected.
        """
        ...

    @property
    def Values(self) -> System.Collections.Generic.ICollection[QuantConnect.Securities.Security]:
        """Get a list of the security objects for this collection."""
        ...

    def __getitem__(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> QuantConnect.Securities.Security:
        """
        Indexer method for the security manager to access the securities objects by their symbol.
        
        :param symbol: Symbol object indexer
        :returns: Security.
        """
        ...

    def __init__(self, timeKeeper: QuantConnect.Interfaces.ITimeKeeper) -> None:
        """Initialise the algorithm security manager with two empty dictionaries"""
        ...

    def __setitem__(self, symbol: typing.Union[QuantConnect.Symbol, str], value: QuantConnect.Securities.Security) -> None:
        """
        Indexer method for the security manager to access the securities objects by their symbol.
        
        :param symbol: Symbol object indexer
        :returns: Security.
        """
        ...

    @overload
    def Add(self, symbol: typing.Union[QuantConnect.Symbol, str], security: QuantConnect.Securities.Security) -> None:
        """
        Add a new security with this symbol to the collection.
        
        :param symbol: symbol for security we're trading
        :param security: security object
        """
        ...

    @overload
    def Add(self, security: QuantConnect.Securities.Security) -> None:
        """
        Add a new security with this symbol to the collection.
        
        :param security: security object
        """
        ...

    @overload
    def Add(self, pair: System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.Security]) -> None:
        """Add a symbol-security by its key value pair."""
        ...

    def Clear(self) -> None:
        """Clear the securities array to delete all the portfolio and asset information."""
        ...

    def Contains(self, pair: System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.Security]) -> bool:
        """
        Check if this collection contains this key value pair.
        
        :param pair: Search key-value pair
        :returns: Bool true if contains this key-value pair.
        """
        ...

    def ContainsKey(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> bool:
        """
        Check if this collection contains this symbol.
        
        :param symbol: Symbol we're checking for.
        :returns: Bool true if contains this symbol pair.
        """
        ...

    def CopyTo(self, array: typing.List[System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.Security]], number: int) -> None:
        """
        Copy from the internal array to an external array.
        
        :param array: Array we're outputting to
        :param number: Starting index of array
        """
        ...

    @overload
    def CreateSecurity(self, symbol: typing.Union[QuantConnect.Symbol, str], subscriptionDataConfigList: System.Collections.Generic.List[QuantConnect.Data.SubscriptionDataConfig], leverage: float = 0, addToSymbolCache: bool = True, underlying: QuantConnect.Securities.Security = None) -> QuantConnect.Securities.Security:
        """Creates a new security"""
        ...

    @overload
    def CreateSecurity(self, symbol: typing.Union[QuantConnect.Symbol, str], subscriptionDataConfig: QuantConnect.Data.SubscriptionDataConfig, leverage: float = 0, addToSymbolCache: bool = True, underlying: QuantConnect.Securities.Security = None) -> QuantConnect.Securities.Security:
        """Creates a new security"""
        ...

    @overload
    def GetEnumerator(self) -> System.Collections.Generic.IEnumerator[System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.Security]]:
        """
        Get the enumerator for this security collection.
        
        :returns: Enumerable key value pair.
        """
        ...

    @overload
    def GetEnumerator(self) -> System.Collections.IEnumerator:
        """
        Get the enumerator for this securities collection.
        
        :returns: Enumerator.
        """
        ...

    def OnCollectionChanged(self, changedEventArgs: System.Collections.Specialized.NotifyCollectionChangedEventArgs) -> None:
        """
        Event invocator for the CollectionChanged event
        
        This method is protected.
        
        :param changedEventArgs: Event arguments for the CollectionChanged event
        """
        ...

    @overload
    def Remove(self, pair: System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.Security]) -> bool:
        """
        Remove a key value of of symbol-securities from the collections.
        
        :param pair: Key Value pair of symbol-security to remove
        :returns: Boolean true on success.
        """
        ...

    @overload
    def Remove(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> bool:
        """
        Remove this symbol security: Dictionary interface implementation.
        
        :param symbol: Symbol we're searching for
        :returns: true success.
        """
        ...

    def SetLiveMode(self, isLiveMode: bool) -> None:
        """
        Set live mode state of the algorithm
        
        :param isLiveMode: True, live mode is enabled
        """
        ...

    def SetSecurityService(self, securityService: QuantConnect.Securities.SecurityService) -> None:
        """Sets the Security Service to be used"""
        ...

    def TryGetValue(self, symbol: typing.Union[QuantConnect.Symbol, str], security: typing.Optional[QuantConnect.Securities.Security]) -> typing.Union[bool, QuantConnect.Securities.Security]:
        """
        Try and get this security object with matching symbol and return true on success.
        
        :param symbol: String search symbol
        :param security: Output Security object
        :returns: True on successfully locating the security object.
        """
        ...


class Cash(System.Object):
    """Represents a holding of a currency in cash."""

    @property
    def Updated(self) -> _EventContainer[typing.Callable[[System.Object, System.EventArgs], None], None]:
        """
        Event fired when this instance is updated
        AddAmount, SetAmount, Update
        """
        ...

    @Updated.setter
    def Updated(self, value: _EventContainer[typing.Callable[[System.Object, System.EventArgs], None], None]):
        """
        Event fired when this instance is updated
        AddAmount, SetAmount, Update
        """
        ...

    @property
    def SecuritySymbols(self) -> System.Collections.Generic.IEnumerable[QuantConnect.Symbol]:
        """
        Gets the symbols of the securities required to provide conversion rates.
        If this cash represents the account currency, then an empty enumerable is returned.
        """
        ...

    @property
    def CurrencyConversion(self) -> QuantConnect.Securities.CurrencyConversion.ICurrencyConversion:
        """Gets the object that calculates the conversion rate to account currency"""
        ...

    @CurrencyConversion.setter
    def CurrencyConversion(self, value: QuantConnect.Securities.CurrencyConversion.ICurrencyConversion):
        """Gets the object that calculates the conversion rate to account currency"""
        ...

    @property
    def Symbol(self) -> str:
        """Gets the symbol used to represent this cash"""
        ...

    @property
    def Amount(self) -> float:
        """Gets or sets the amount of cash held"""
        ...

    @Amount.setter
    def Amount(self, value: float):
        """Gets or sets the amount of cash held"""
        ...

    @property
    def ConversionRate(self) -> float:
        """Gets the conversion rate into account currency"""
        ...

    @ConversionRate.setter
    def ConversionRate(self, value: float):
        """Gets the conversion rate into account currency"""
        ...

    @property
    def CurrencySymbol(self) -> str:
        """The symbol of the currency, such as $"""
        ...

    @property
    def ValueInAccountCurrency(self) -> float:
        """Gets the value of this cash in the account currency"""
        ...

    def __init__(self, symbol: str, amount: float, conversionRate: float) -> None:
        """
        Initializes a new instance of the Cash class
        
        :param symbol: The symbol used to represent this cash
        :param amount: The amount of this currency held
        :param conversionRate: The initial conversion rate of this currency into the CashBook.AccountCurrency
        """
        ...

    def AddAmount(self, amount: float) -> float:
        """
        Adds the specified amount of currency to this Cash instance and returns the new total.
        This operation is thread-safe
        
        :param amount: The amount of currency to be added
        :returns: The amount of currency directly after the addition.
        """
        ...

    def EnsureCurrencyDataFeed(self, securities: QuantConnect.Securities.SecurityManager, subscriptions: QuantConnect.Data.SubscriptionManager, marketMap: System.Collections.Generic.IReadOnlyDictionary[QuantConnect.SecurityType, str], changes: QuantConnect.Data.UniverseSelection.SecurityChanges, securityService: QuantConnect.Interfaces.ISecurityService, accountCurrency: str, defaultResolution: QuantConnect.Resolution = ...) -> System.Collections.Generic.List[QuantConnect.Data.SubscriptionDataConfig]:
        """
        Ensures that we have a data feed to convert this currency into the base currency.
        This will add a SubscriptionDataConfig and create a Security at the lowest resolution if one is not found.
        
        :param securities: The security manager
        :param subscriptions: The subscription manager used for searching and adding subscriptions
        :param marketMap: The market map that decides which market the new security should be in
        :param changes: Will be used to consume SecurityChanges.AddedSecurities
        :param securityService: Will be used to create required new Security
        :param accountCurrency: The account currency
        :param defaultResolution: The default resolution to use for the internal subscriptions
        :returns: Returns the added SubscriptionDataConfig, otherwise null.
        """
        ...

    def SetAmount(self, amount: float) -> None:
        """
        Sets the Quantity to the specified amount
        
        :param amount: The amount to set the quantity to
        """
        ...

    @overload
    def ToString(self) -> str:
        """
        Returns a string that represents the current Cash.
        
        :returns: A string that represents the current Cash.
        """
        ...

    @overload
    def ToString(self, accountCurrency: str) -> str:
        """
        Returns a string that represents the current Cash.
        
        :returns: A string that represents the current Cash.
        """
        ...

    def Update(self) -> None:
        """Marks this cash object's conversion rate as being potentially outdated"""
        ...


class SecurityCache(System.Object):
    """Base class caching caching spot for security data and any other temporary properties."""

    @property
    def Price(self) -> float:
        """Gets the most recent price submitted to this cache"""
        ...

    @Price.setter
    def Price(self, value: float):
        """Gets the most recent price submitted to this cache"""
        ...

    @property
    def Open(self) -> float:
        """Gets the most recent open submitted to this cache"""
        ...

    @Open.setter
    def Open(self, value: float):
        """Gets the most recent open submitted to this cache"""
        ...

    @property
    def High(self) -> float:
        """Gets the most recent high submitted to this cache"""
        ...

    @High.setter
    def High(self, value: float):
        """Gets the most recent high submitted to this cache"""
        ...

    @property
    def Low(self) -> float:
        """Gets the most recent low submitted to this cache"""
        ...

    @Low.setter
    def Low(self, value: float):
        """Gets the most recent low submitted to this cache"""
        ...

    @property
    def Close(self) -> float:
        """Gets the most recent close submitted to this cache"""
        ...

    @Close.setter
    def Close(self, value: float):
        """Gets the most recent close submitted to this cache"""
        ...

    @property
    def BidPrice(self) -> float:
        """Gets the most recent bid submitted to this cache"""
        ...

    @BidPrice.setter
    def BidPrice(self, value: float):
        """Gets the most recent bid submitted to this cache"""
        ...

    @property
    def AskPrice(self) -> float:
        """Gets the most recent ask submitted to this cache"""
        ...

    @AskPrice.setter
    def AskPrice(self, value: float):
        """Gets the most recent ask submitted to this cache"""
        ...

    @property
    def BidSize(self) -> float:
        """Gets the most recent bid size submitted to this cache"""
        ...

    @BidSize.setter
    def BidSize(self, value: float):
        """Gets the most recent bid size submitted to this cache"""
        ...

    @property
    def AskSize(self) -> float:
        """Gets the most recent ask size submitted to this cache"""
        ...

    @AskSize.setter
    def AskSize(self, value: float):
        """Gets the most recent ask size submitted to this cache"""
        ...

    @property
    def Volume(self) -> float:
        """Gets the most recent volume submitted to this cache"""
        ...

    @Volume.setter
    def Volume(self, value: float):
        """Gets the most recent volume submitted to this cache"""
        ...

    @property
    def OpenInterest(self) -> int:
        """Gets the most recent open interest submitted to this cache"""
        ...

    @OpenInterest.setter
    def OpenInterest(self, value: int):
        """Gets the most recent open interest submitted to this cache"""
        ...

    def AddData(self, data: QuantConnect.Data.BaseData) -> None:
        """
        Add a new market data point to the local security cache for the current market price.
        Rules:
            Don't cache fill forward data.
            Always return the last observation.
            If two consecutive data has the same time stamp and one is Quotebars and the other Tradebar, prioritize the Quotebar.
        """
        ...

    def AddDataList(self, data: System.Collections.Generic.IReadOnlyList[QuantConnect.Data.BaseData], dataType: typing.Type, containsFillForwardData: typing.Optional[bool] = None) -> None:
        """Add a list of market data points to the local security cache for the current market price."""
        ...

    def GetAll(self) -> System.Collections.Generic.IEnumerable[QuantConnect_Securities_SecurityCache_GetAll_T]:
        """
        Gets all data points of the specified type from the most recent time step
        that produced data for that type
        """
        ...

    @overload
    def GetData(self) -> QuantConnect.Data.BaseData:
        """
        Get last data packet received for this security if any else null
        
        :returns: BaseData type of the security.
        """
        ...

    @overload
    def GetData(self) -> QuantConnect_Securities_SecurityCache_GetData_T:
        """
        Get last data packet received for this security of the specified type
        
        :returns: The last data packet, null if none received of type.
        """
        ...

    def HasData(self, type: typing.Type) -> bool:
        """Gets whether or not this dynamic data instance has data stored for the specified type"""
        ...

    def Reset(self) -> None:
        """Reset cache storage and free memory"""
        ...

    @staticmethod
    def ShareTypeCacheInstance(sourceToShare: QuantConnect.Securities.SecurityCache, targetToModify: QuantConnect.Securities.SecurityCache) -> None:
        """
        Helper method that modifies the target security cache instance to use the
        type cache of the source
        
        :param sourceToShare: The source cache to use
        :param targetToModify: The target security cache that will be modified
        """
        ...

    def StoreData(self, data: System.Collections.Generic.IReadOnlyList[QuantConnect.Data.BaseData], dataType: typing.Type) -> None:
        """
        Stores the specified data list in the cache WITHOUT updating any of the cache properties, such as Price
        
        :param data: The collection of data to store in this cache
        :param dataType: The data type
        """
        ...

    def TryGetValue(self, type: typing.Type, data: typing.Optional[System.Collections.Generic.IReadOnlyList[QuantConnect.Data.BaseData]]) -> typing.Union[bool, System.Collections.Generic.IReadOnlyList[QuantConnect.Data.BaseData]]:
        """Gets whether or not this dynamic data instance has data stored for the specified type"""
        ...


class CashAmount:
    """Represents a cash amount which can be converted to account currency using a currency converter"""

    @property
    def Amount(self) -> float:
        """The amount of cash"""
        ...

    @property
    def Currency(self) -> str:
        """The currency in which the cash amount is denominated"""
        ...

    def __init__(self, amount: float, currency: str) -> None:
        """
        Initializes a new instance of the CashAmount class
        
        :param amount: The amount
        :param currency: The currency
        """
        ...

    def Equals(self, obj: typing.Any) -> bool:
        """
        Used to compare two CashAmount instances.
        Useful to compare against the default instance
        
        :param obj: The other object to compare with
        :returns: True if Currency and Amount are equal.
        """
        ...

    def GetHashCode(self) -> int:
        """
        Get Hash Code for this Object
        
        :returns: Integer Hash Code.
        """
        ...


class ICurrencyConverter(metaclass=abc.ABCMeta):
    """Provides the ability to convert cash amounts to the account currency"""

    @property
    @abc.abstractmethod
    def AccountCurrency(self) -> str:
        """Gets account currency"""
        ...

    def ConvertToAccountCurrency(self, cashAmount: QuantConnect.Securities.CashAmount) -> QuantConnect.Securities.CashAmount:
        """
        Converts a cash amount to the account currency
        
        :param cashAmount: The CashAmount instance to convert
        :returns: A new CashAmount instance denominated in the account currency.
        """
        ...


class SecurityHolding(System.Object):
    """SecurityHolding is a base class for purchasing and holding a market item which manages the asset portfolio"""

    @property
    def QuantityChanged(self) -> _EventContainer[typing.Callable[[System.Object, QuantConnect.Securities.SecurityHoldingQuantityChangedEventArgs], None], None]:
        """Event raised each time the holdings quantity is changed."""
        ...

    @QuantityChanged.setter
    def QuantityChanged(self, value: _EventContainer[typing.Callable[[System.Object, QuantConnect.Securities.SecurityHoldingQuantityChangedEventArgs], None], None]):
        """Event raised each time the holdings quantity is changed."""
        ...

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """
        The security being held
        
        This property is protected.
        """
        ...

    @property
    def Target(self) -> QuantConnect.Algorithm.Framework.Portfolio.IPortfolioTarget:
        """Gets the current target holdings for this security"""
        ...

    @Target.setter
    def Target(self, value: QuantConnect.Algorithm.Framework.Portfolio.IPortfolioTarget):
        """Gets the current target holdings for this security"""
        ...

    @property
    def AveragePrice(self) -> float:
        """Average price of the security holdings."""
        ...

    @AveragePrice.setter
    def AveragePrice(self, value: float):
        """Average price of the security holdings."""
        ...

    @property
    def Quantity(self) -> float:
        """Quantity of the security held."""
        ...

    @Quantity.setter
    def Quantity(self, value: float):
        """Quantity of the security held."""
        ...

    @property
    def Symbol(self) -> QuantConnect.Symbol:
        """Symbol identifier of the underlying security."""
        ...

    @property
    def Type(self) -> int:
        """
        The security type of the symbol
        
        This property contains the int value of a member of the QuantConnect.SecurityType enum.
        """
        ...

    @property
    def Leverage(self) -> float:
        """Leverage of the underlying security."""
        ...

    @property
    def HoldingsCost(self) -> float:
        """Acquisition cost of the security total holdings in units of the account's currency."""
        ...

    @property
    def UnleveredHoldingsCost(self) -> float:
        """Unlevered Acquisition cost of the security total holdings in units of the account's currency."""
        ...

    @property
    def Price(self) -> float:
        """Current market price of the security."""
        ...

    @Price.setter
    def Price(self, value: float):
        """Current market price of the security."""
        ...

    @property
    def AbsoluteHoldingsCost(self) -> float:
        """Absolute holdings cost for current holdings in units of the account's currency."""
        ...

    @property
    def UnleveredAbsoluteHoldingsCost(self) -> float:
        """Unlevered absolute acquisition cost of the security total holdings in units of the account's currency."""
        ...

    @property
    def HoldingsValue(self) -> float:
        """Market value of our holdings in units of the account's currency."""
        ...

    @property
    def AbsoluteHoldingsValue(self) -> float:
        """Absolute of the market value of our holdings in units of the account's currency."""
        ...

    @property
    def HoldStock(self) -> bool:
        """Boolean flat indicating if we hold any of the security"""
        ...

    @property
    def Invested(self) -> bool:
        """Boolean flat indicating if we hold any of the security"""
        ...

    @property
    def TotalSaleVolume(self) -> float:
        """The total transaction volume for this security since the algorithm started in units of the account's currency."""
        ...

    @property
    def TotalFees(self) -> float:
        """Total fees for this company since the algorithm started in units of the account's currency."""
        ...

    @property
    def IsLong(self) -> bool:
        """Boolean flag indicating we have a net positive holding of the security."""
        ...

    @property
    def IsShort(self) -> bool:
        """BBoolean flag indicating we have a net negative holding of the security."""
        ...

    @property
    def AbsoluteQuantity(self) -> float:
        """Absolute quantity of holdings of this security"""
        ...

    @property
    def LastTradeProfit(self) -> float:
        """Record of the closing profit from the last trade conducted in units of the account's currency."""
        ...

    @property
    def Profit(self) -> float:
        """Calculate the total profit for this security in units of the account's currency."""
        ...

    @property
    def NetProfit(self) -> float:
        """Return the net for this company measured by the profit less fees in units of the account's currency."""
        ...

    @property
    def UnrealizedProfitPercent(self) -> float:
        """Gets the unrealized profit as a percenage of holdings cost"""
        ...

    @property
    def UnrealizedProfit(self) -> float:
        """Unrealized profit of this security when absolute quantity held is more than zero in units of the account's currency."""
        ...

    @overload
    def __init__(self, security: QuantConnect.Securities.Security, currencyConverter: QuantConnect.Securities.ICurrencyConverter) -> None:
        """
        Create a new holding class instance setting the initial properties to $0.
        
        :param security: The security being held
        :param currencyConverter: A currency converter instance
        """
        ...

    @overload
    def __init__(self, holding: QuantConnect.Securities.SecurityHolding) -> None:
        """
        Create a new holding class instance copying the initial properties
        
        This method is protected.
        
        :param holding: The security being held
        """
        ...

    def AddNewFee(self, newFee: float) -> None:
        """Adds a fee to the running total of total fees in units of the account's currency."""
        ...

    def AddNewProfit(self, profitLoss: float) -> None:
        """
        Adds a profit record to the running total of profit in units of the account's currency.
        
        :param profitLoss: The cash change in portfolio from closing a position
        """
        ...

    def AddNewSale(self, saleValue: float) -> None:
        """Adds a new sale value to the running total trading volume in units of the account's currency."""
        ...

    @overload
    def GetQuantityValue(self, quantity: float) -> float:
        """
        Gets the total value of the specified  of shares of this security
        in the account currency
        
        :param quantity: The quantity of shares
        :returns: The value of the quantity of shares in the account currency.
        """
        ...

    @overload
    def GetQuantityValue(self, quantity: float, price: float) -> float:
        """
        Gets the total value of the specified  of shares of this security
        in the account currency
        
        :param quantity: The quantity of shares
        :param price: The current price
        :returns: The value of the quantity of shares in the account currency.
        """
        ...

    def OnQuantityChanged(self, previousAveragePrice: float, previousQuantity: float) -> None:
        """
        Event invocator for the QuantityChanged event
        
        This method is protected.
        """
        ...

    @overload
    def SetHoldings(self, averagePrice: float, quantity: int) -> None:
        """Set the quantity of holdings and their average price after processing a portfolio fill."""
        ...

    @overload
    def SetHoldings(self, averagePrice: float, quantity: float) -> None:
        """Set the quantity of holdings and their average price after processing a portfolio fill."""
        ...

    def SetLastTradeProfit(self, lastTradeProfit: float) -> None:
        """
        Set the last trade profit for this security from a Portfolio.ProcessFill call in units of the account's currency.
        
        :param lastTradeProfit: Value of the last trade profit
        """
        ...

    def ToString(self) -> str:
        """Writes out the properties of this instance to string"""
        ...

    def TotalCloseProfit(self) -> float:
        """Profit if we closed the holdings right now including the approximate fees in units of the account's currency."""
        ...

    def UpdateMarketPrice(self, closingPrice: float) -> None:
        """
        Update local copy of closing price value.
        
        :param closingPrice: Price of the underlying asset to be used for calculating market price / portfolio value
        """
        ...


class SecurityExchange(System.Object):
    """Base exchange class providing information and helper tools for reading the current exchange situation"""

    @property
    def Hours(self) -> QuantConnect.Securities.SecurityExchangeHours:
        """Gets the SecurityExchangeHours for this exchange"""
        ...

    @Hours.setter
    def Hours(self, value: QuantConnect.Securities.SecurityExchangeHours):
        """Gets the SecurityExchangeHours for this exchange"""
        ...

    @property
    def TimeZone(self) -> typing.Any:
        """Gets the time zone for this exchange"""
        ...

    @property
    def TradingDaysPerYear(self) -> int:
        """Number of trading days per year for this security. By default the market is open 365 days per year."""
        ...

    @property
    def LocalTime(self) -> datetime.datetime:
        """Time from the most recent data"""
        ...

    @LocalTime.setter
    def LocalTime(self, value: datetime.datetime):
        """Time from the most recent data"""
        ...

    @property
    def ExchangeOpen(self) -> bool:
        """Boolean property for quickly testing if the exchange is open."""
        ...

    @property
    def ClosingSoon(self) -> bool:
        """Boolean property for quickly testing if the exchange is 10 minutes away from closing."""
        ...

    def __init__(self, exchangeHours: QuantConnect.Securities.SecurityExchangeHours) -> None:
        """
        Initializes a new instance of the SecurityExchange class using the specified
        exchange hours to determine open/close times
        
        :param exchangeHours: Contains the weekly exchange schedule plus holidays
        """
        ...

    def DateIsOpen(self, dateToCheck: typing.Union[datetime.datetime, datetime.date]) -> bool:
        """
        Check if the *date* is open.
        
        :param dateToCheck: Date to check
        :returns: Return true if the exchange is open for this date.
        """
        ...

    def DateTimeIsOpen(self, dateTime: typing.Union[datetime.datetime, datetime.date]) -> bool:
        """
        Check if this DateTime is open.
        
        :param dateTime: DateTime to check
        :returns: Boolean true if the market is open.
        """
        ...

    def IsClosingSoon(self, minutesToClose: int) -> bool:
        """
        Determines if the exchange is going to close in the next provided minutes
        
        :param minutesToClose: Minutes to close to check
        :returns: Returns true if the exchange is going to close in the next provided minutes.
        """
        ...

    def IsOpenDuringBar(self, barStartTime: typing.Union[datetime.datetime, datetime.date], barEndTime: typing.Union[datetime.datetime, datetime.date], isExtendedMarketHours: bool) -> bool:
        """Determines if the exchange was open at any time between start and stop"""
        ...

    def SetLocalDateTimeFrontier(self, newLocalTime: typing.Union[datetime.datetime, datetime.date]) -> None:
        """
        Set the current datetime in terms of the exchange's local time zone
        
        :param newLocalTime: Most recent data tick
        """
        ...

    def SetMarketHours(self, marketHoursSegments: System.Collections.Generic.IEnumerable[QuantConnect.Securities.MarketHoursSegment], *days: System.DayOfWeek) -> None:
        """
        Sets the regular market hours for the specified days If no days are specified then
        all days will be updated.
        
        :param marketHoursSegments: Specifies each segment of the market hours, such as premarket/market/postmark
        :param days: The days of the week to set these times for
        """
        ...


class IOrderProvider(metaclass=abc.ABCMeta):
    """Represents a type capable of fetching Order instances by its QC order id or by a brokerage id"""

    @property
    @abc.abstractmethod
    def OrdersCount(self) -> int:
        """Gets the current number of orders that have been processed"""
        ...

    def GetOpenOrders(self, filter: typing.Callable[[QuantConnect.Orders.Order], bool] = None) -> System.Collections.Generic.List[QuantConnect.Orders.Order]:
        """
        Gets open orders matching the specified filter. Specifying null will return an enumerable
        of all open orders.
        
        :param filter: Delegate used to filter the orders
        :returns: All filtered open orders this order provider currently holds.
        """
        ...

    def GetOpenOrderTickets(self, filter: typing.Callable[[QuantConnect.Orders.OrderTicket], bool] = None) -> System.Collections.Generic.IEnumerable[QuantConnect.Orders.OrderTicket]:
        """
        Gets and enumerable of opened OrderTicket matching the specified
        
        :param filter: The filter predicate used to find the required order tickets. If null is specified then all tickets are returned
        :returns: An enumerable of opened OrderTicket matching the specified.
        """
        ...

    def GetOrderByBrokerageId(self, brokerageId: str) -> QuantConnect.Orders.Order:
        """
        Gets the order by its brokerage id
        
        :param brokerageId: The brokerage id to fetch
        :returns: The first order matching the brokerage id, or null if no match is found.
        """
        ...

    def GetOrderById(self, orderId: int) -> QuantConnect.Orders.Order:
        """
        Get the order by its id
        
        :param orderId: Order id to fetch
        :returns: A clone of the order with the specified id, or null if no match is found.
        """
        ...

    def GetOrders(self, filter: typing.Callable[[QuantConnect.Orders.Order], bool] = None) -> System.Collections.Generic.IEnumerable[QuantConnect.Orders.Order]:
        """
        Gets all orders matching the specified filter. Specifying null will return an enumerable
        of all orders.
        
        :param filter: Delegate used to filter the orders
        :returns: All orders this order provider currently holds by the specified filter.
        """
        ...

    def GetOrderTicket(self, orderId: int) -> QuantConnect.Orders.OrderTicket:
        """
        Gets the order ticket for the specified order id. Returns null if not found
        
        :param orderId: The order's id
        :returns: The order ticket with the specified id, or null if not found.
        """
        ...

    def GetOrderTickets(self, filter: typing.Callable[[QuantConnect.Orders.OrderTicket], bool] = None) -> System.Collections.Generic.IEnumerable[QuantConnect.Orders.OrderTicket]:
        """
        Gets and enumerable of OrderTicket matching the specified
        
        :param filter: The filter predicate used to find the required order tickets. If null is specified then all tickets are returned
        :returns: An enumerable of OrderTicket matching the specified.
        """
        ...


class IOrderProcessor(QuantConnect.Securities.IOrderProvider, metaclass=abc.ABCMeta):
    """Represents a type capable of processing orders"""

    def Process(self, request: QuantConnect.Orders.OrderRequest) -> QuantConnect.Orders.OrderTicket:
        """
        Adds the specified order to be processed
        
        :param request: The OrderRequest to be processed
        :returns: The OrderTicket for the corresponding OrderRequest.OrderId.
        """
        ...


class SecurityTransactionManager(System.Object, QuantConnect.Securities.IOrderProvider):
    """Algorithm Transactions Manager - Recording Transactions"""

    @property
    def UtcTime(self) -> datetime.datetime:
        """Gets the time the security information was last updated"""
        ...

    @property
    def TransactionRecord(self) -> System.Collections.Generic.Dictionary[datetime.datetime, float]:
        """Trade record of profits and losses for each trade statistics calculations"""
        ...

    @property
    def MinimumOrderSize(self) -> float:
        """
        Configurable minimum order value to ignore bad orders, or orders with unrealistic sizes
        
        MinimumOrderSize is obsolete and will not be used, please use Settings.MinimumOrderMarginPortfolioPercentage instead
        """
        warnings.warn("MinimumOrderSize is obsolete and will not be used, please use Settings.MinimumOrderMarginPortfolioPercentage instead", DeprecationWarning)

    @property
    def MinimumOrderQuantity(self) -> int:
        """
        Configurable minimum order size to ignore bad orders, or orders with unrealistic sizes
        
        MinimumOrderQuantity is obsolete and will not be used, please use Settings.MinimumOrderMarginPortfolioPercentage instead
        """
        warnings.warn("MinimumOrderQuantity is obsolete and will not be used, please use Settings.MinimumOrderMarginPortfolioPercentage instead", DeprecationWarning)

    @property
    def LastOrderId(self) -> int:
        """Get the last order id."""
        ...

    @property
    def MarketOrderFillTimeout(self) -> datetime.timedelta:
        """Configurable timeout for market order fills"""
        ...

    @MarketOrderFillTimeout.setter
    def MarketOrderFillTimeout(self, value: datetime.timedelta):
        """Configurable timeout for market order fills"""
        ...

    @property
    def OrdersCount(self) -> int:
        """Gets the current number of orders that have been processed"""
        ...

    def __init__(self, algorithm: QuantConnect.Interfaces.IAlgorithm, security: QuantConnect.Securities.SecurityManager) -> None:
        """Initialise the transaction manager for holding and processing orders."""
        ...

    def AddOrder(self, request: QuantConnect.Orders.SubmitOrderRequest) -> QuantConnect.Orders.OrderTicket:
        """
        Add an order to collection and return the unique order id or negative if an error.
        
        :param request: A request detailing the order to be submitted
        :returns: New unique, increasing orderid.
        """
        ...

    def AddTransactionRecord(self, time: typing.Union[datetime.datetime, datetime.date], transactionProfitLoss: float) -> None:
        """
        Record the transaction value and time in a list to later be processed for statistics creation.
        
        :param time: Time of order processed
        :param transactionProfitLoss: Profit Loss.
        """
        ...

    @overload
    def CancelOpenOrders(self) -> System.Collections.Generic.List[QuantConnect.Orders.OrderTicket]:
        """
        Cancels all open orders for all symbols
        
        :returns: List containing the cancelled order tickets.
        """
        ...

    @overload
    def CancelOpenOrders(self, symbol: typing.Union[QuantConnect.Symbol, str], tag: str = None) -> System.Collections.Generic.List[QuantConnect.Orders.OrderTicket]:
        """
        Cancels all open orders for the specified symbol
        
        :param symbol: The symbol whose orders are to be cancelled
        :param tag: Custom order tag
        :returns: List containing the cancelled order tickets.
        """
        ...

    def CancelOrder(self, orderId: int, orderTag: str = None) -> QuantConnect.Orders.OrderTicket:
        """
        Added alias for RemoveOrder -
        
        :param orderId: Order id we wish to cancel
        :param orderTag: Tag to indicate from where this method was called
        """
        ...

    def GetIncrementOrderId(self) -> int:
        """
        Get a new order id, and increment the internal counter.
        
        :returns: New unique int order id.
        """
        ...

    @overload
    def GetOpenOrders(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> System.Collections.Generic.List[QuantConnect.Orders.Order]:
        """
        Get a list of all open orders for a symbol.
        
        :param symbol: The symbol for which to return the orders
        :returns: List of open orders.
        """
        ...

    @overload
    def GetOpenOrders(self, filter: typing.Callable[[QuantConnect.Orders.Order], bool] = None) -> System.Collections.Generic.List[QuantConnect.Orders.Order]:
        """
        Gets open orders matching the specified filter. Specifying null will return an enumerable
        of all open orders.
        
        :param filter: Delegate used to filter the orders
        :returns: All filtered open orders this order provider currently holds.
        """
        ...

    @overload
    def GetOpenOrders(self, filter: typing.Any) -> System.Collections.Generic.List[QuantConnect.Orders.Order]:
        """
        Gets open orders matching the specified filter. However, this method can be confused with the
        override that takes a Symbol as parameter. For this reason it first checks if it can convert
        the parameter into a symbol. If that conversion cannot be aplied it assumes the parameter is
        a Python function object and not a Python representation of a Symbol.
        
        :param filter: Python function object used to filter the orders
        :returns: All filtered open orders this order provider currently holds.
        """
        ...

    @overload
    def GetOpenOrdersRemainingQuantity(self, filter: typing.Callable[[QuantConnect.Orders.OrderTicket], bool] = None) -> float:
        """
        Gets the remaining quantity to be filled from open orders, i.e. order size minus quantity filled
        
        :param filter: Filters the order tickets to be included in the aggregate quantity remaining to be filled
        :returns: Total quantity that hasn't been filled yet for all orders that were not filtered.
        """
        ...

    @overload
    def GetOpenOrdersRemainingQuantity(self, filter: typing.Any) -> float:
        """
        Gets the remaining quantity to be filled from open orders, i.e. order size minus quantity filled
        However, this method can be confused with the override that takes a Symbol as parameter. For this reason
        it first checks if it can convert the parameter into a symbol. If that conversion cannot be aplied it
        assumes the parameter is a Python function object and not a Python representation of a Symbol.
        
        :param filter: Filters the order tickets to be included in the aggregate quantity remaining to be filled
        :returns: Total quantity that hasn't been filled yet for all orders that were not filtered.
        """
        ...

    @overload
    def GetOpenOrdersRemainingQuantity(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> float:
        """
        Gets the remaining quantity to be filled from open orders for a Symbol, i.e. order size minus quantity filled
        
        :param symbol: Symbol to get the remaining quantity of currently open orders
        :returns: Total quantity that hasn't been filled yet for orders matching the Symbol.
        """
        ...

    @overload
    def GetOpenOrderTickets(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> System.Collections.Generic.IEnumerable[QuantConnect.Orders.OrderTicket]:
        """
        Get an enumerable of open OrderTicket for the specified symbol
        
        :param symbol: The symbol for which to return the order tickets
        :returns: An enumerable of open OrderTicket.
        """
        ...

    @overload
    def GetOpenOrderTickets(self, filter: typing.Callable[[QuantConnect.Orders.OrderTicket], bool] = None) -> System.Collections.Generic.IEnumerable[QuantConnect.Orders.OrderTicket]:
        """
        Gets an enumerable of opened OrderTicket matching the specified
        
        :param filter: The filter predicate used to find the required order tickets
        :returns: An enumerable of opened OrderTicket matching the specified.
        """
        ...

    @overload
    def GetOpenOrderTickets(self, filter: typing.Any) -> System.Collections.Generic.IEnumerable[QuantConnect.Orders.OrderTicket]:
        """
        Gets an enumerable of opened OrderTicket matching the specified 
        However, this method can be confused with the override that takes a Symbol as parameter. For this reason
        it first checks if it can convert the parameter into a symbol. If that conversion cannot be aplied it
        assumes the parameter is a Python function object and not a Python representation of a Symbol.
        
        :param filter: The Python function filter used to find the required order tickets
        :returns: An enumerable of opened OrderTicket matching the specified.
        """
        ...

    def GetOrderByBrokerageId(self, brokerageId: str) -> QuantConnect.Orders.Order:
        """
        Gets the order by its brokerage id
        
        :param brokerageId: The brokerage id to fetch
        :returns: The first order matching the brokerage id, or null if no match is found.
        """
        ...

    def GetOrderById(self, orderId: int) -> QuantConnect.Orders.Order:
        """
        Get the order by its id
        
        :param orderId: Order id to fetch
        :returns: A clone of the order with the specified id, or null if no match is found.
        """
        ...

    @overload
    def GetOrders(self, filter: typing.Callable[[QuantConnect.Orders.Order], bool] = None) -> System.Collections.Generic.IEnumerable[QuantConnect.Orders.Order]:
        """
        Gets all orders matching the specified filter. Specifying null will return an enumerable
        of all orders.
        
        :param filter: Delegate used to filter the orders
        :returns: All orders this order provider currently holds by the specified filter.
        """
        ...

    @overload
    def GetOrders(self, filter: typing.Any) -> System.Collections.Generic.IEnumerable[QuantConnect.Orders.Order]:
        """
        Gets all orders matching the specified filter.
        
        :param filter: Python function object used to filter the orders
        :returns: All orders this order provider currently holds by the specified filter.
        """
        ...

    def GetOrderTicket(self, orderId: int) -> QuantConnect.Orders.OrderTicket:
        """
        Gets the order ticket for the specified order id. Returns null if not found
        
        :param orderId: The order's id
        :returns: The order ticket with the specified id, or null if not found.
        """
        ...

    @overload
    def GetOrderTickets(self, filter: typing.Callable[[QuantConnect.Orders.OrderTicket], bool] = None) -> System.Collections.Generic.IEnumerable[QuantConnect.Orders.OrderTicket]:
        """
        Gets an enumerable of OrderTicket matching the specified
        
        :param filter: The filter predicate used to find the required order tickets
        :returns: An enumerable of OrderTicket matching the specified.
        """
        ...

    @overload
    def GetOrderTickets(self, filter: typing.Any) -> System.Collections.Generic.IEnumerable[QuantConnect.Orders.OrderTicket]:
        """
        Gets an enumerable of OrderTicket matching the specified
        
        :param filter: The Python function filter used to find the required order tickets
        :returns: An enumerable of OrderTicket matching the specified.
        """
        ...

    def ProcessRequest(self, request: QuantConnect.Orders.OrderRequest) -> QuantConnect.Orders.OrderTicket:
        """
        Processes the order request
        
        :param request: The request to be processed
        :returns: The order ticket for the request.
        """
        ...

    def RemoveOrder(self, orderId: int, tag: str = None) -> QuantConnect.Orders.OrderTicket:
        """
        Remove this order from outstanding queue: user is requesting a cancel.
        
        :param orderId: Specific order id to remove
        :param tag: Tag request
        """
        ...

    def SetOrderProcessor(self, orderProvider: QuantConnect.Securities.IOrderProcessor) -> None:
        """
        Sets the IOrderProvider used for fetching orders for the algorithm
        
        :param orderProvider: The IOrderProvider to be used to manage fetching orders
        """
        ...

    def UpdateOrder(self, request: QuantConnect.Orders.UpdateOrderRequest) -> QuantConnect.Orders.OrderTicket:
        """
        Update an order yet to be filled such as stop or limit orders.
        
        :param request: Request detailing how the order should be updated
        """
        ...

    def WaitForOrder(self, orderId: int) -> bool:
        """
        Wait for a specific order to be either Filled, Invalid or Canceled
        
        :param orderId: The id of the order to wait for
        :returns: True if we successfully wait for the fill, false if we were unable to wait. This may be because it is not a market order or because the timeout was reached.
        """
        ...


class CashBook(System.Object, System.Collections.Generic.IDictionary[str, QuantConnect.Securities.Cash], QuantConnect.Securities.ICurrencyConverter, typing.Iterable[System.Collections.Generic.KeyValuePair[str, QuantConnect.Securities.Cash]]):
    """Provides a means of keeping track of the different cash holdings of an algorithm"""

    class UpdateType(System.Enum):
        """The different types of Updated events"""

        Added = 0
        """A new Cash.Symbol was added (0)"""

        Removed = 1
        """One or more Cash instances were removed (1)"""

        Updated = 2
        """An existing Cash.Symbol was updated (2)"""

    @property
    def Updated(self) -> _EventContainer[typing.Callable[[System.Object, QuantConnect.Securities.CashBook.UpdateType], None], None]:
        """
        Event fired when a Cash instance is added or removed, and when
        the Cash.Updated is triggered for the currently hold instances
        """
        ...

    @Updated.setter
    def Updated(self, value: _EventContainer[typing.Callable[[System.Object, QuantConnect.Securities.CashBook.UpdateType], None], None]):
        """
        Event fired when a Cash instance is added or removed, and when
        the Cash.Updated is triggered for the currently hold instances
        """
        ...

    @property
    def AccountCurrency(self) -> str:
        """Gets the base currency used"""
        ...

    @AccountCurrency.setter
    def AccountCurrency(self, value: str):
        """Gets the base currency used"""
        ...

    @property
    def TotalValueInAccountCurrency(self) -> float:
        """Gets the total value of the cash book in units of the base currency"""
        ...

    @property
    def Count(self) -> int:
        ...

    @property
    def IsReadOnly(self) -> bool:
        """Gets a value indicating whether this instance is read only."""
        ...

    @property
    def Keys(self) -> System.Collections.Generic.ICollection[str]:
        """Gets the keys."""
        ...

    @property
    def Values(self) -> System.Collections.Generic.ICollection[QuantConnect.Securities.Cash]:
        """Gets the values."""
        ...

    def __getitem__(self, symbol: str) -> QuantConnect.Securities.Cash:
        """
        Gets or sets the QuantConnect.Securities.Cash with the specified symbol.
        
        :param symbol: Symbol.
        """
        ...

    def __init__(self) -> None:
        """Initializes a new instance of the CashBook class."""
        ...

    def __setitem__(self, symbol: str, value: QuantConnect.Securities.Cash) -> None:
        """
        Gets or sets the QuantConnect.Securities.Cash with the specified symbol.
        
        :param symbol: Symbol.
        """
        ...

    @overload
    def Add(self, symbol: str, quantity: float, conversionRate: float) -> None:
        """
        Adds a new cash of the specified symbol and quantity
        
        :param symbol: The symbol used to reference the new cash
        :param quantity: The amount of new cash to start
        :param conversionRate: The conversion rate used to determine the initial portfolio value/starting capital impact caused by this currency position.
        """
        ...

    @overload
    def Add(self, item: System.Collections.Generic.KeyValuePair[str, QuantConnect.Securities.Cash]) -> None:
        """
        Add the specified item to this CashBook.
        
        :param item: KeyValuePair of symbol -> Cash item
        """
        ...

    @overload
    def Add(self, symbol: str, value: QuantConnect.Securities.Cash) -> None:
        """
        Add the specified key and value.
        
        :param symbol: The symbol of the Cash value.
        :param value: Value.
        """
        ...

    def Clear(self) -> None:
        """Clear this instance of all Cash entries."""
        ...

    def Contains(self, item: System.Collections.Generic.KeyValuePair[str, QuantConnect.Securities.Cash]) -> bool:
        """
        Determines whether the current collection contains the specified value.
        
        :param item: Item.
        """
        ...

    def ContainsKey(self, symbol: str) -> bool:
        """
        Determines whether the current instance contains an entry with the specified symbol.
        
        :param symbol: Key.
        :returns: true, if key was contained, false otherwise.
        """
        ...

    def Convert(self, sourceQuantity: float, sourceCurrency: str, destinationCurrency: str) -> float:
        """
        Converts a quantity of source currency units into the specified destination currency
        
        :param sourceQuantity: The quantity of source currency to be converted
        :param sourceCurrency: The source currency symbol
        :param destinationCurrency: The destination currency symbol
        :returns: The converted value.
        """
        ...

    @overload
    def ConvertToAccountCurrency(self, sourceQuantity: float, sourceCurrency: str) -> float:
        """
        Converts a quantity of source currency units into the account currency
        
        :param sourceQuantity: The quantity of source currency to be converted
        :param sourceCurrency: The source currency symbol
        :returns: The converted value.
        """
        ...

    @overload
    def ConvertToAccountCurrency(self, cashAmount: QuantConnect.Securities.CashAmount) -> QuantConnect.Securities.CashAmount:
        ...

    def CopyTo(self, array: typing.List[System.Collections.Generic.KeyValuePair[str, QuantConnect.Securities.Cash]], arrayIndex: int) -> None:
        """
        Copies to the specified array.
        
        :param array: Array.
        :param arrayIndex: Array index.
        """
        ...

    def EnsureCurrencyDataFeeds(self, securities: QuantConnect.Securities.SecurityManager, subscriptions: QuantConnect.Data.SubscriptionManager, marketMap: System.Collections.Generic.IReadOnlyDictionary[QuantConnect.SecurityType, str], changes: QuantConnect.Data.UniverseSelection.SecurityChanges, securityService: QuantConnect.Interfaces.ISecurityService, defaultResolution: QuantConnect.Resolution = ...) -> System.Collections.Generic.List[QuantConnect.Data.SubscriptionDataConfig]:
        """
        Checks the current subscriptions and adds necessary currency pair feeds to provide real time conversion data
        
        :param securities: The SecurityManager for the algorithm
        :param subscriptions: The SubscriptionManager for the algorithm
        :param marketMap: The market map that decides which market the new security should be in
        :param changes: Will be used to consume SecurityChanges.AddedSecurities
        :param securityService: Will be used to create required new Security
        :param defaultResolution: The default resolution to use for the internal subscriptions
        :returns: Returns a list of added currency SubscriptionDataConfig.
        """
        ...

    @overload
    def GetEnumerator(self) -> System.Collections.Generic.IEnumerator[System.Collections.Generic.KeyValuePair[str, QuantConnect.Securities.Cash]]:
        """
        Gets the enumerator.
        
        :returns: The enumerator.
        """
        ...

    @overload
    def GetEnumerator(self) -> System.Collections.IEnumerator:
        ...

    @overload
    def Remove(self, symbol: str) -> bool:
        """
        Remove the Cash item corresponding to the specified symbol
        
        :param symbol: The symbolto be removed
        """
        ...

    @overload
    def Remove(self, item: System.Collections.Generic.KeyValuePair[str, QuantConnect.Securities.Cash]) -> bool:
        """
        Remove the specified item.
        
        :param item: Item.
        """
        ...

    def ToString(self) -> str:
        """
        Returns a string that represents the current object.
        
        :returns: A string that represents the current object.
        """
        ...

    def TryGetValue(self, symbol: str, value: typing.Optional[QuantConnect.Securities.Cash]) -> typing.Union[bool, QuantConnect.Securities.Cash]:
        """
        Try to get the value.
        
        :param symbol: The symbol.
        :param value: Value.
        :returns: true, if get value was tryed, false otherwise.
        """
        ...


class IMarginCallModel(metaclass=abc.ABCMeta):
    """Represents the model responsible for picking which orders should be executed during a margin call"""

    def ExecuteMarginCall(self, generatedMarginCallOrders: System.Collections.Generic.IEnumerable[QuantConnect.Orders.SubmitOrderRequest]) -> System.Collections.Generic.List[QuantConnect.Orders.OrderTicket]:
        """
        Executes synchronous orders to bring the account within margin requirements.
        
        :param generatedMarginCallOrders: These are the margin call orders that were generated by individual security margin models.
        :returns: The list of orders that were actually executed.
        """
        ...

    def GetMarginCallOrders(self, issueMarginCallWarning: typing.Optional[bool]) -> typing.Union[System.Collections.Generic.List[QuantConnect.Orders.SubmitOrderRequest], bool]:
        """
        Scan the portfolio and the updated data for a potential margin call situation which may get the holdings below zero!
        If there is a margin call, liquidate the portfolio immediately before the portfolio gets sub zero.
        
        :param issueMarginCallWarning: Set to true if a warning should be issued to the algorithm
        :returns: True for a margin call on the holdings.
        """
        ...


class UnsettledCashAmount(System.Object):
    """Represents a pending cash amount waiting for settlement time"""

    @property
    def SettlementTimeUtc(self) -> datetime.datetime:
        """The settlement time (in UTC)"""
        ...

    @SettlementTimeUtc.setter
    def SettlementTimeUtc(self, value: datetime.datetime):
        """The settlement time (in UTC)"""
        ...

    @property
    def Currency(self) -> str:
        """The currency symbol"""
        ...

    @Currency.setter
    def Currency(self, value: str):
        """The currency symbol"""
        ...

    @property
    def Amount(self) -> float:
        """The amount of cash"""
        ...

    @Amount.setter
    def Amount(self, value: float):
        """The amount of cash"""
        ...

    def __init__(self, settlementTimeUtc: typing.Union[datetime.datetime, datetime.date], currency: str, amount: float) -> None:
        """Creates a new instance of the UnsettledCashAmount class"""
        ...


class SecurityPortfolioManager(QuantConnect.ExtendedDictionary[QuantConnect.Securities.SecurityHolding], System.Collections.Generic.IDictionary[QuantConnect.Symbol, QuantConnect.Securities.SecurityHolding], QuantConnect.Securities.ISecurityProvider, typing.Iterable[System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.SecurityHolding]]):
    """
    Portfolio manager class groups popular properties and makes them accessible through one interface.
    It also provide indexing by the vehicle symbol to get the Security.Holding objects.
    """

    @property
    def Securities(self) -> QuantConnect.Securities.SecurityManager:
        """Local access to the securities collection for the portfolio summation."""
        ...

    @Securities.setter
    def Securities(self, value: QuantConnect.Securities.SecurityManager):
        """Local access to the securities collection for the portfolio summation."""
        ...

    @property
    def Transactions(self) -> QuantConnect.Securities.SecurityTransactionManager:
        """Local access to the transactions collection for the portfolio summation and updates."""
        ...

    @Transactions.setter
    def Transactions(self, value: QuantConnect.Securities.SecurityTransactionManager):
        """Local access to the transactions collection for the portfolio summation and updates."""
        ...

    @property
    def Positions(self) -> QuantConnect.Securities.Positions.PositionManager:
        """Local access to the position manager"""
        ...

    @Positions.setter
    def Positions(self, value: QuantConnect.Securities.Positions.PositionManager):
        """Local access to the position manager"""
        ...

    @property
    def PositionGroups(self) -> QuantConnect.Securities.Positions.PositionGroupCollection:
        """Current read only position groups collection"""
        ...

    @property
    def CashBook(self) -> QuantConnect.Securities.CashBook:
        """Gets the cash book that keeps track of all currency holdings (only settled cash)"""
        ...

    @property
    def UnsettledCashBook(self) -> QuantConnect.Securities.CashBook:
        """Gets the cash book that keeps track of all currency holdings (only unsettled cash)"""
        ...

    @property
    def Count(self) -> int:
        """Count the securities objects in the portfolio."""
        ...

    @property
    def IsReadOnly(self) -> bool:
        """Check if the underlying securities array is read only."""
        ...

    @property
    def GetKeys(self) -> System.Collections.Generic.IEnumerable[QuantConnect.Symbol]:
        """
        Gets an System.Collections.Generic.ICollection`1 containing the Symbol objects of the System.Collections.Generic.IDictionary`2.
        
        This property is protected.
        """
        ...

    @property
    def GetValues(self) -> System.Collections.Generic.IEnumerable[QuantConnect.Securities.SecurityHolding]:
        """
        Gets an System.Collections.Generic.ICollection`1 containing the values in the System.Collections.Generic.IDictionary`2.
        
        This property is protected.
        """
        ...

    @property
    def Keys(self) -> System.Collections.Generic.ICollection[QuantConnect.Symbol]:
        """Symbol keys collection of the underlying assets in the portfolio."""
        ...

    @property
    def Values(self) -> System.Collections.Generic.ICollection[QuantConnect.Securities.SecurityHolding]:
        """Collection of securities objects in the portfolio."""
        ...

    @property
    def Cash(self) -> float:
        ...

    @property
    def UnsettledCash(self) -> float:
        """Sum of all currencies in account in US dollars (only unsettled cash)"""
        ...

    @property
    def TotalUnleveredAbsoluteHoldingsCost(self) -> float:
        """Absolute value of cash discounted from our total cash by the holdings we own."""
        ...

    @property
    def TotalAbsoluteHoldingsCost(self) -> float:
        """
        Gets the total absolute holdings cost of the portfolio. This sums up the individual
        absolute cost of each holding
        """
        ...

    @property
    def TotalHoldingsValue(self) -> float:
        """Absolute sum the individual items in portfolio."""
        ...

    @property
    def HoldStock(self) -> bool:
        """Boolean flag indicating we have any holdings in the portfolio."""
        ...

    @property
    def Invested(self) -> bool:
        """Alias for HoldStock. Check if we have and holdings."""
        ...

    @property
    def TotalUnrealisedProfit(self) -> float:
        """Get the total unrealised profit in our portfolio from the individual security unrealized profits."""
        ...

    @property
    def TotalUnrealizedProfit(self) -> float:
        """Get the total unrealised profit in our portfolio from the individual security unrealized profits."""
        ...

    @property
    def TotalPortfolioValue(self) -> float:
        """Total portfolio value if we sold all holdings at current market rates."""
        ...

    @property
    def TotalFees(self) -> float:
        """Total fees paid during the algorithm operation across all securities in portfolio."""
        ...

    @property
    def TotalProfit(self) -> float:
        """Sum of all gross profit across all securities in portfolio."""
        ...

    @property
    def TotalSaleVolume(self) -> float:
        """Total sale volume since the start of algorithm operations."""
        ...

    @property
    def TotalMarginUsed(self) -> float:
        """Gets the total margin used across all securities in the account's currency"""
        ...

    @property
    def MarginRemaining(self) -> float:
        """Gets the remaining margin on the account in the account's currency"""
        ...

    @property
    def MarginCallModel(self) -> QuantConnect.Securities.IMarginCallModel:
        """
        Gets or sets the MarginCallModel for the portfolio. This
        is used to executed margin call orders.
        """
        ...

    @MarginCallModel.setter
    def MarginCallModel(self, value: QuantConnect.Securities.IMarginCallModel):
        """
        Gets or sets the MarginCallModel for the portfolio. This
        is used to executed margin call orders.
        """
        ...

    def __getitem__(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> QuantConnect.Securities.SecurityHolding:
        """
        Indexer for the PortfolioManager class to access the underlying security holdings objects.
        
        :param symbol: Symbol object indexer
        :returns: SecurityHolding class from the algorithm securities.
        """
        ...

    def __init__(self, securityManager: QuantConnect.Securities.SecurityManager, transactions: QuantConnect.Securities.SecurityTransactionManager, defaultOrderProperties: QuantConnect.Interfaces.IOrderProperties = None) -> None:
        """Initialise security portfolio manager."""
        ...

    def __setitem__(self, symbol: typing.Union[QuantConnect.Symbol, str], value: QuantConnect.Securities.SecurityHolding) -> None:
        """
        Indexer for the PortfolioManager class to access the underlying security holdings objects.
        
        :param symbol: Symbol object indexer
        :returns: SecurityHolding class from the algorithm securities.
        """
        ...

    @overload
    def Add(self, symbol: typing.Union[QuantConnect.Symbol, str], holding: QuantConnect.Securities.SecurityHolding) -> None:
        ...

    @overload
    def Add(self, pair: System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.SecurityHolding]) -> None:
        """
        Add a new securities key value pair to the portfolio.
        
        :param pair: Key value pair of dictionary
        """
        ...

    def AddTransactionRecord(self, time: typing.Union[datetime.datetime, datetime.date], transactionProfitLoss: float) -> None:
        """
        Record the transaction value and time in a list to later be processed for statistics creation.
        
        :param time: Time of order processed
        :param transactionProfitLoss: Profit Loss.
        """
        ...

    def AddUnsettledCashAmount(self, item: QuantConnect.Securities.UnsettledCashAmount) -> None:
        """
        Adds an item to the list of unsettled cash amounts
        
        :param item: The item to add
        """
        ...

    def ApplyDividend(self, dividend: QuantConnect.Data.Market.Dividend, liveMode: bool, mode: QuantConnect.DataNormalizationMode) -> None:
        """
        Applies a dividend to the portfolio
        
        :param dividend: The dividend to be applied
        :param liveMode: True if live mode, false for backtest
        :param mode: The DataNormalizationMode for this security
        """
        ...

    def ApplySplit(self, split: QuantConnect.Data.Market.Split, liveMode: bool, mode: QuantConnect.DataNormalizationMode) -> None:
        """
        Applies a split to the portfolio
        
        :param split: The split to be applied
        :param liveMode: True if live mode, false for backtest
        :param mode: The DataNormalizationMode for this security
        """
        ...

    def Clear(self) -> None:
        """Clear the portfolio of securities objects."""
        ...

    def Contains(self, pair: System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.SecurityHolding]) -> bool:
        """
        Check if the key-value pair is in the portfolio.
        
        :param pair: Pair we're searching for
        :returns: True if we have this object.
        """
        ...

    def ContainsKey(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> bool:
        """
        Check if the portfolio contains this symbol string.
        
        :param symbol: String search symbol for the security
        :returns: Boolean true if portfolio contains this symbol.
        """
        ...

    def CopyTo(self, array: typing.List[System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.SecurityHolding]], index: int) -> None:
        """
        Copy contents of the portfolio collection to a new destination.
        
        :param array: Destination array
        :param index: Position in array to start copying
        """
        ...

    def GetBuyingPower(self, symbol: typing.Union[QuantConnect.Symbol, str], direction: QuantConnect.Orders.OrderDirection = ...) -> float:
        """
        Gets the margin available for trading a specific symbol in a specific direction.
        Alias for GetMarginRemaining(Symbol, OrderDirection)
        
        :param symbol: The symbol to compute margin remaining for
        :param direction: The order/trading direction
        :returns: The maximum order size that is currently executable in the specified direction.
        """
        ...

    @overload
    def GetEnumerator(self) -> System.Collections.Generic.IEnumerator[System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.SecurityHolding]]:
        """
        Get the enumerator for the underlying securities collection.
        
        :returns: Enumerable key value pair.
        """
        ...

    @overload
    def GetEnumerator(self) -> System.Collections.IEnumerator:
        """
        Get the enumerator for the underlying securities collection.
        
        :returns: Enumerator.
        """
        ...

    @overload
    def GetMarginRemaining(self, totalPortfolioValue: float) -> float:
        """
        Gets the remaining margin on the account in the account's currency
        for the given total portfolio value
        
        :param totalPortfolioValue: The total portfolio value TotalPortfolioValue
        """
        ...

    @overload
    def GetMarginRemaining(self, symbol: typing.Union[QuantConnect.Symbol, str], direction: QuantConnect.Orders.OrderDirection = ...) -> float:
        """
        Gets the margin available for trading a specific symbol in a specific direction.
        
        :param symbol: The symbol to compute margin remaining for
        :param direction: The order/trading direction
        :returns: The maximum order size that is currently executable in the specified direction.
        """
        ...

    def GetSecurity(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> QuantConnect.Securities.Security:
        """
        Retrieves a summary of the holdings for the specified symbol
        
        :param symbol: The symbol to get holdings for
        :returns: The holdings for the symbol or null if the symbol is invalid and/or not in the portfolio.
        """
        ...

    def InvalidateTotalPortfolioValue(self) -> None:
        """
        Will flag the current TotalPortfolioValue as invalid
        so it is recalculated when gotten
        """
        ...

    def LogMarginInformation(self, orderRequest: QuantConnect.Orders.OrderRequest = None) -> None:
        """Logs margin information for debugging"""
        ...

    def ProcessFill(self, fill: QuantConnect.Orders.OrderEvent) -> None:
        """Calculate the new average price after processing a partial/complete order fill event."""
        ...

    @overload
    def Remove(self, pair: System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Securities.SecurityHolding]) -> bool:
        """
        Remove this keyvalue pair from the portfolio.
        
        :param pair: Key value pair of dictionary
        """
        ...

    @overload
    def Remove(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> bool:
        """
        Remove this symbol from the portfolio.
        
        :param symbol: Symbol of dictionary
        """
        ...

    def ScanForCashSettlement(self, timeUtc: typing.Union[datetime.datetime, datetime.date]) -> None:
        """Scan the portfolio to check if unsettled funds should be settled"""
        ...

    def SetAccountCurrency(self, accountCurrency: str) -> None:
        """
        Sets the account currency cash symbol this algorithm is to manage.
        
        :param accountCurrency: The account currency cash symbol to set
        """
        ...

    @overload
    def SetCash(self, cash: float) -> None:
        """
        Set the account currency cash this algorithm is to manage.
        
        :param cash: Decimal cash value of portfolio
        """
        ...

    @overload
    def SetCash(self, symbol: str, cash: float, conversionRate: float) -> None:
        """
        Set the cash for the specified symbol
        
        :param symbol: The cash symbol to set
        :param cash: Decimal cash value of portfolio
        :param conversionRate: The current conversion rate for the
        """
        ...

    @overload
    def SetMarginCallModel(self, marginCallModel: QuantConnect.Securities.IMarginCallModel) -> None:
        """
        Sets the margin call model
        
        :param marginCallModel: Model that represents a portfolio's model to executed margin call orders.
        """
        ...

    @overload
    def SetMarginCallModel(self, pyObject: typing.Any) -> None:
        """
        Sets the margin call model
        
        :param pyObject: Model that represents a portfolio's model to executed margin call orders.
        """
        ...

    def TryGetValue(self, symbol: typing.Union[QuantConnect.Symbol, str], holding: typing.Optional[QuantConnect.Securities.SecurityHolding]) -> typing.Union[bool, QuantConnect.Securities.SecurityHolding]:
        """
        Attempt to get the value of the securities holding class if this symbol exists.
        
        :param symbol: String search symbol
        :param holding: Holdings object of this security
        :returns: Boolean true if successful locating and setting the holdings object.
        """
        ...


class ISecurityPortfolioModel(metaclass=abc.ABCMeta):
    """Performs order fill application to portfolio"""

    def ProcessFill(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, fill: QuantConnect.Orders.OrderEvent) -> None:
        """
        Performs application of an OrderEvent to the portfolio
        
        :param portfolio: The algorithm's portfolio
        :param security: The fill's security
        :param fill: The order event fill object to be applied
        """
        ...


class MaintenanceMargin(System.Object):
    """Result type for IBuyingPowerModel.GetMaintenanceMargin"""

    Zero: QuantConnect.Securities.MaintenanceMargin
    """Gets an instance of MaintenanceMargin with zero values."""

    @property
    def Value(self) -> float:
        """The maintenance margin value in account currency"""
        ...

    def __init__(self, value: float) -> None:
        """
        Initializes a new instance of the MaintenanceMargin class
        
        :param value: The maintenance margin
        """
        ...


class InitialMargin(System.Object):
    """
    Result type for IBuyingPowerModel.GetInitialMarginRequirement
    and IBuyingPowerModel.GetInitialMarginRequiredForOrder
    """

    Zero: QuantConnect.Securities.InitialMargin
    """Gets an instance of InitialMargin with zero values"""

    @property
    def Value(self) -> float:
        """The initial margin value in account currency"""
        ...

    def __init__(self, value: float) -> None:
        """
        Initializes a new instance of the InitialMargin class
        
        :param value: The initial margin
        """
        ...


class HasSufficientBuyingPowerForOrderResult(System.Object):
    """Contains the information returned by IBuyingPowerModel.HasSufficientBuyingPowerForOrder"""

    @property
    def IsSufficient(self) -> bool:
        """Gets true if there is sufficient buying power to execute an order"""
        ...

    @property
    def Reason(self) -> str:
        """Gets the reason for insufficient buying power to execute an order"""
        ...

    def __init__(self, isSufficient: bool, reason: str = None) -> None:
        """
        Initializes a new instance of the HasSufficientBuyingPowerForOrderResult class
        
        :param isSufficient: True if the order can be executed
        :param reason: The reason for insufficient buying power
        """
        ...


class GetMaximumOrderQuantityResult(System.Object):
    """
    Contains the information returned by IBuyingPowerModel.GetMaximumOrderQuantityForTargetBuyingPower
    and  IBuyingPowerModel.GetMaximumOrderQuantityForDeltaBuyingPower
    """

    @property
    def Quantity(self) -> float:
        """Returns the maximum quantity for the order"""
        ...

    @property
    def Reason(self) -> str:
        """Returns the reason for which the maximum order quantity is zero"""
        ...

    @property
    def IsError(self) -> bool:
        """Returns true if the zero order quantity is an error condition and will be shown to the user."""
        ...

    @overload
    def __init__(self, quantity: float, reason: str = None) -> None:
        """
        Initializes a new instance of the GetMaximumOrderQuantityResult class
        
        :param quantity: Returns the maximum quantity for the order
        :param reason: The reason for which the maximum order quantity is zero
        """
        ...

    @overload
    def __init__(self, quantity: float, reason: str, isError: bool = True) -> None:
        """
        Initializes a new instance of the GetMaximumOrderQuantityResult class
        
        :param quantity: Returns the maximum quantity for the order
        :param reason: The reason for which the maximum order quantity is zero
        :param isError: True if the zero order quantity is an error condition
        """
        ...


class ReservedBuyingPowerForPosition(System.Object):
    """Defines the result for IBuyingPowerModel.GetReservedBuyingPowerForPosition"""

    @property
    def AbsoluteUsedBuyingPower(self) -> float:
        """Gets the reserved buying power"""
        ...

    def __init__(self, reservedBuyingPowerForPosition: float) -> None:
        """
        Initializes a new instance of the ReservedBuyingPowerForPosition class
        
        :param reservedBuyingPowerForPosition: The reserved buying power for the security's holdings
        """
        ...


class BuyingPower(System.Object):
    """Defines the result for IBuyingPowerModel.GetBuyingPower"""

    @property
    def Value(self) -> float:
        """Gets the buying power"""
        ...

    def __init__(self, buyingPower: float) -> None:
        """
        Initializes a new instance of the BuyingPower class
        
        :param buyingPower: The buying power
        """
        ...


class IBuyingPowerModel(metaclass=abc.ABCMeta):
    """Represents a security's model of buying power"""

    def GetBuyingPower(self, parameters: QuantConnect.Securities.BuyingPowerParameters) -> QuantConnect.Securities.BuyingPower:
        """
        Gets the buying power available for a trade
        
        :param parameters: A parameters object containing the algorithm's portfolio, security, and order direction
        :returns: The buying power available for the trade.
        """
        ...

    def GetInitialMarginRequiredForOrder(self, parameters: QuantConnect.Securities.InitialMarginRequiredForOrderParameters) -> QuantConnect.Securities.InitialMargin:
        """
        Gets the total margin required to execute the specified order in units of the account currency including fees
        
        :param parameters: An object containing the portfolio, the security and the order
        :returns: The total margin in terms of the currency quoted in the order.
        """
        ...

    def GetInitialMarginRequirement(self, parameters: QuantConnect.Securities.InitialMarginParameters) -> QuantConnect.Securities.InitialMargin:
        """
        The margin that must be held in order to increase the position by the provided quantity
        
        :param parameters: An object containing the security and quantity
        :returns: The initial margin required for the provided security and quantity.
        """
        ...

    def GetLeverage(self, security: QuantConnect.Securities.Security) -> float:
        """
        Gets the current leverage of the security
        
        :param security: The security to get leverage for
        :returns: The current leverage in the security.
        """
        ...

    def GetMaintenanceMargin(self, parameters: QuantConnect.Securities.MaintenanceMarginParameters) -> QuantConnect.Securities.MaintenanceMargin:
        """
        Gets the margin currently allocated to the specified holding
        
        :param parameters: An object containing the security and holdings quantity/cost/value
        :returns: The maintenance margin required for the provided holdings quantity/cost/value.
        """
        ...

    def GetMaximumOrderQuantityForDeltaBuyingPower(self, parameters: QuantConnect.Securities.GetMaximumOrderQuantityForDeltaBuyingPowerParameters) -> QuantConnect.Securities.GetMaximumOrderQuantityResult:
        """
        Get the maximum market order quantity to obtain a delta in the buying power used by a security.
        The deltas sign defines the position side to apply it to, positive long, negative short.
        
        :param parameters: An object containing the portfolio, the security and the delta buying power
        :returns: Returns the maximum allowed market order quantity and if zero, also the reason.
        """
        ...

    def GetMaximumOrderQuantityForTargetBuyingPower(self, parameters: QuantConnect.Securities.GetMaximumOrderQuantityForTargetBuyingPowerParameters) -> QuantConnect.Securities.GetMaximumOrderQuantityResult:
        """
        Get the maximum market order quantity to obtain a position with a given buying power percentage.
        Will not take into account free buying power.
        
        :param parameters: An object containing the portfolio, the security and the target signed buying power percentage
        :returns: Returns the maximum allowed market order quantity and if zero, also the reason.
        """
        ...

    def GetReservedBuyingPowerForPosition(self, parameters: QuantConnect.Securities.ReservedBuyingPowerForPositionParameters) -> QuantConnect.Securities.ReservedBuyingPowerForPosition:
        """
        Gets the amount of buying power reserved to maintain the specified position
        
        :param parameters: A parameters object containing the security
        :returns: The reserved buying power in account currency.
        """
        ...

    def HasSufficientBuyingPowerForOrder(self, parameters: QuantConnect.Securities.HasSufficientBuyingPowerForOrderParameters) -> QuantConnect.Securities.HasSufficientBuyingPowerForOrderResult:
        """
        Check if there is sufficient buying power to execute this order.
        
        :param parameters: An object containing the portfolio, the security and the order
        :returns: Returns buying power information for an order.
        """
        ...

    def SetLeverage(self, security: QuantConnect.Securities.Security, leverage: float) -> None:
        """
        Sets the leverage for the applicable securities, i.e, equities
        
        :param security: The security to set leverage for
        :param leverage: The new leverage
        """
        ...


class ISettlementModel(metaclass=abc.ABCMeta):
    """Represents the model responsible for applying cash settlement rules"""

    def ApplyFunds(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, applicationTimeUtc: typing.Union[datetime.datetime, datetime.date], currency: str, amount: float) -> None:
        """
        Applies cash settlement rules
        
        :param portfolio: The algorithm's portfolio
        :param security: The fill's security
        :param applicationTimeUtc: The fill time (in UTC)
        :param currency: The currency symbol
        :param amount: The amount of cash to apply
        """
        ...


class IVolatilityModel(metaclass=abc.ABCMeta):
    """Represents a model that computes the volatility of a security"""

    @property
    @abc.abstractmethod
    def Volatility(self) -> float:
        """Gets the volatility of the security as a percentage"""
        ...

    def GetHistoryRequirements(self, security: QuantConnect.Securities.Security, utcTime: typing.Union[datetime.datetime, datetime.date]) -> System.Collections.Generic.IEnumerable[QuantConnect.Data.HistoryRequest]:
        """
        Returns history requirements for the volatility model expressed in the form of history request
        
        :param security: The security of the request
        :param utcTime: The date/time of the request
        :returns: History request object list, or empty if no requirements.
        """
        ...

    def Update(self, security: QuantConnect.Securities.Security, data: QuantConnect.Data.BaseData) -> None:
        """
        Updates this model using the new price information in
        the specified security instance
        
        :param security: The security to calculate volatility for
        :param data: The new data used to update the model
        """
        ...


class IPriceVariationModel(metaclass=abc.ABCMeta):
    """Gets the minimum price variation of a given security"""

    def GetMinimumPriceVariation(self, parameters: QuantConnect.Securities.GetMinimumPriceVariationParameters) -> float:
        """
        Get the minimum price variation from a security
        
        :param parameters: An object containing the method parameters
        :returns: Decimal minimum price variation of a given security.
        """
        ...


class DynamicSecurityData(IDynamicMetaObjectProvider):
    """
    Provides access to a security's data via it's type. This implementation supports dynamic access
    by type name.
    """

    def __init__(self, registeredTypes: QuantConnect.Securities.IRegisteredSecurityDataTypesProvider, cache: QuantConnect.Securities.SecurityCache) -> None:
        """
        Initializes a new instance of the DynamicSecurityData class
        
        :param registeredTypes: Provides all the registered data types for the algorithm
        :param cache: The security cache
        """
        ...

    def Get(self, type: QuantConnect_Securities_DynamicSecurityData_Get_T) -> QuantConnect_Securities_DynamicSecurityData_Get_T:
        """
        Get the matching cached object in a python friendly accessor
        
        :param type: Type to search for
        :returns: Matching object.
        """
        ...

    def GetAll(self, type: QuantConnect_Securities_DynamicSecurityData_GetAll_T) -> System.Collections.Generic.IReadOnlyList[QuantConnect_Securities_DynamicSecurityData_GetAll_T]:
        """
        Get all the matching types with a python friendly overload.
        
        :param type: Search type
        :returns: List of matching objects cached.
        """
        ...

    def GetMetaObject(self, parameter: typing.Any) -> typing.Any:
        """
        Returns the System.Dynamic.DynamicMetaObject responsible for binding operations performed on this object.
        
        :param parameter: The expression tree representation of the runtime value.
        :returns: The System.Dynamic.DynamicMetaObject to bind this object.
        """
        ...

    def GetProperty(self, name: str) -> System.Object:
        """
        Gets the property's value with the specified name. This is a case-insensitve search.
        
        :param name: The property name to access
        :returns: object value of BaseData.
        """
        ...

    def HasData(self) -> bool:
        """Gets whether or not this dynamic data instance has data stored for the specified type"""
        ...

    def HasProperty(self, name: str) -> bool:
        """
        Gets whether or not this dynamic data instance has a property with the specified name.
        This is a case-insensitive search.
        
        :param name: The property name to check for
        :returns: True if the property exists, false otherwise.
        """
        ...

    def SetProperty(self, name: str, value: typing.Any) -> System.Object:
        """
        Sets the property with the specified name to the value. This is a case-insensitve search.
        
        DynamicSecurityData is a view of the SecurityCache. It is readonly, properties can not be set
        
        :param name: The property name to set
        :param value: The new property value
        :returns: Returns the input value back to the caller.
        """
        warnings.warn("DynamicSecurityData is a view of the SecurityCache. It is readonly, properties can not be set", DeprecationWarning)


class Security(System.Object, QuantConnect.Interfaces.ISecurityPrice):
    """A base vehicle properties class for providing a common interface to all assets in QuantConnect."""

    @property
    def SubscriptionsBag(self) -> System.Collections.Concurrent.ConcurrentBag[QuantConnect.Data.SubscriptionDataConfig]:
        """
        Collection of SubscriptionDataConfigs for this security.
        Uses concurrent bag to avoid list enumeration threading issues
        
        This field is protected.
        """
        ...

    @property
    def ShortableProvider(self) -> QuantConnect.Interfaces.IShortableProvider:
        """
        This securities IShortableProvider
        
        This property is protected.
        """
        ...

    @ShortableProvider.setter
    def ShortableProvider(self, value: QuantConnect.Interfaces.IShortableProvider):
        """
        This securities IShortableProvider
        
        This property is protected.
        """
        ...

    NullLeverage: float = 0
    """A null security leverage value"""

    @property
    def Subscriptions(self) -> System.Collections.Generic.IEnumerable[QuantConnect.Data.SubscriptionDataConfig]:
        """Gets all the subscriptions for this security"""
        ...

    @property
    def Symbol(self) -> QuantConnect.Symbol:
        """Symbol for the asset."""
        ...

    @property
    def QuoteCurrency(self) -> QuantConnect.Securities.Cash:
        """Gets the Cash object used for converting the quote currency to the account currency"""
        ...

    @property
    def SymbolProperties(self) -> QuantConnect.Securities.SymbolProperties:
        """Gets the symbol properties for this security"""
        ...

    @property
    def Type(self) -> int:
        """
        Type of the security.
        
        This property contains the int value of a member of the QuantConnect.SecurityType enum.
        """
        ...

    @property
    def Resolution(self) -> int:
        """
        Resolution of data requested for this security.
        
        This property contains the int value of a member of the QuantConnect.Resolution enum.
        
        This property is obsolete. Use the 'SubscriptionDataConfig' exposed by 'SubscriptionManager'
        """
        warnings.warn("This property is obsolete. Use the 'SubscriptionDataConfig' exposed by 'SubscriptionManager'", DeprecationWarning)

    @Resolution.setter
    def Resolution(self, value: int):
        """
        Resolution of data requested for this security.
        
        This property contains the int value of a member of the QuantConnect.Resolution enum.
        
        This property is obsolete. Use the 'SubscriptionDataConfig' exposed by 'SubscriptionManager'
        """
        warnings.warn("This property is obsolete. Use the 'SubscriptionDataConfig' exposed by 'SubscriptionManager'", DeprecationWarning)

    @property
    def IsFillDataForward(self) -> bool:
        """
        Indicates the data will use previous bars when there was no trading in this time period. This was a configurable datastream setting set in initialization.
        
        This property is obsolete. Use the 'SubscriptionDataConfig' exposed by 'SubscriptionManager'
        """
        warnings.warn("This property is obsolete. Use the 'SubscriptionDataConfig' exposed by 'SubscriptionManager'", DeprecationWarning)

    @IsFillDataForward.setter
    def IsFillDataForward(self, value: bool):
        """
        Indicates the data will use previous bars when there was no trading in this time period. This was a configurable datastream setting set in initialization.
        
        This property is obsolete. Use the 'SubscriptionDataConfig' exposed by 'SubscriptionManager'
        """
        warnings.warn("This property is obsolete. Use the 'SubscriptionDataConfig' exposed by 'SubscriptionManager'", DeprecationWarning)

    @property
    def IsExtendedMarketHours(self) -> bool:
        """
        Indicates the security will continue feeding data after the primary market hours have closed. This was a configurable setting set in initialization.
        
        This property is obsolete. Use the 'SubscriptionDataConfig' exposed by 'SubscriptionManager'
        """
        warnings.warn("This property is obsolete. Use the 'SubscriptionDataConfig' exposed by 'SubscriptionManager'", DeprecationWarning)

    @IsExtendedMarketHours.setter
    def IsExtendedMarketHours(self, value: bool):
        """
        Indicates the security will continue feeding data after the primary market hours have closed. This was a configurable setting set in initialization.
        
        This property is obsolete. Use the 'SubscriptionDataConfig' exposed by 'SubscriptionManager'
        """
        warnings.warn("This property is obsolete. Use the 'SubscriptionDataConfig' exposed by 'SubscriptionManager'", DeprecationWarning)

    @property
    def DataNormalizationMode(self) -> int:
        """
        Gets the data normalization mode used for this security
        
        This property contains the int value of a member of the QuantConnect.DataNormalizationMode enum.
        
        This property is obsolete. Use the 'SubscriptionDataConfig' exposed by 'SubscriptionManager'
        """
        warnings.warn("This property is obsolete. Use the 'SubscriptionDataConfig' exposed by 'SubscriptionManager'", DeprecationWarning)

    @DataNormalizationMode.setter
    def DataNormalizationMode(self, value: int):
        """
        Gets the data normalization mode used for this security
        
        This property contains the int value of a member of the QuantConnect.DataNormalizationMode enum.
        
        This property is obsolete. Use the 'SubscriptionDataConfig' exposed by 'SubscriptionManager'
        """
        warnings.warn("This property is obsolete. Use the 'SubscriptionDataConfig' exposed by 'SubscriptionManager'", DeprecationWarning)

    @property
    def SubscriptionDataConfig(self) -> QuantConnect.Data.SubscriptionDataConfig:
        """
        Gets the subscription configuration for this security
        
        This property returns only the first subscription. Use the 'Subscriptions' property for all of this security's subscriptions.
        """
        warnings.warn("This property returns only the first subscription. Use the 'Subscriptions' property for all of this security's subscriptions.", DeprecationWarning)

    @property
    def HasData(self) -> bool:
        """There has been at least one datapoint since our algorithm started running for us to determine price."""
        ...

    @property
    def IsTradable(self) -> bool:
        """Gets or sets whether or not this security should be considered tradable"""
        ...

    @IsTradable.setter
    def IsTradable(self, value: bool):
        """Gets or sets whether or not this security should be considered tradable"""
        ...

    @property
    def IsDelisted(self) -> bool:
        """True if the security has been delisted from exchanges and is no longer tradable"""
        ...

    @IsDelisted.setter
    def IsDelisted(self, value: bool):
        """True if the security has been delisted from exchanges and is no longer tradable"""
        ...

    @property
    def Cache(self) -> QuantConnect.Securities.SecurityCache:
        """Data cache for the security to store previous price information."""
        ...

    @Cache.setter
    def Cache(self, value: QuantConnect.Securities.SecurityCache):
        """Data cache for the security to store previous price information."""
        ...

    @property
    def Holdings(self) -> QuantConnect.Securities.SecurityHolding:
        """Holdings class contains the portfolio, cash and processes order fills."""
        ...

    @Holdings.setter
    def Holdings(self, value: QuantConnect.Securities.SecurityHolding):
        """Holdings class contains the portfolio, cash and processes order fills."""
        ...

    @property
    def Exchange(self) -> QuantConnect.Securities.SecurityExchange:
        """Exchange class contains the market opening hours, along with pre-post market hours."""
        ...

    @Exchange.setter
    def Exchange(self, value: QuantConnect.Securities.SecurityExchange):
        """Exchange class contains the market opening hours, along with pre-post market hours."""
        ...

    @property
    def FeeModel(self) -> QuantConnect.Orders.Fees.IFeeModel:
        """Fee model used to compute order fees for this security"""
        ...

    @FeeModel.setter
    def FeeModel(self, value: QuantConnect.Orders.Fees.IFeeModel):
        """Fee model used to compute order fees for this security"""
        ...

    @property
    def FillModel(self) -> QuantConnect.Orders.Fills.IFillModel:
        """Fill model used to produce fill events for this security"""
        ...

    @FillModel.setter
    def FillModel(self, value: QuantConnect.Orders.Fills.IFillModel):
        """Fill model used to produce fill events for this security"""
        ...

    @property
    def SlippageModel(self) -> QuantConnect.Orders.Slippage.ISlippageModel:
        """Slippage model use to compute slippage of market orders"""
        ...

    @SlippageModel.setter
    def SlippageModel(self, value: QuantConnect.Orders.Slippage.ISlippageModel):
        """Slippage model use to compute slippage of market orders"""
        ...

    @property
    def PortfolioModel(self) -> QuantConnect.Securities.ISecurityPortfolioModel:
        """Gets the portfolio model used by this security"""
        ...

    @PortfolioModel.setter
    def PortfolioModel(self, value: QuantConnect.Securities.ISecurityPortfolioModel):
        """Gets the portfolio model used by this security"""
        ...

    @property
    def BuyingPowerModel(self) -> QuantConnect.Securities.IBuyingPowerModel:
        """Gets the buying power model used for this security"""
        ...

    @BuyingPowerModel.setter
    def BuyingPowerModel(self, value: QuantConnect.Securities.IBuyingPowerModel):
        """Gets the buying power model used for this security"""
        ...

    @property
    def MarginModel(self) -> QuantConnect.Securities.IBuyingPowerModel:
        """Gets the buying power model used for this security, an alias for BuyingPowerModel"""
        ...

    @MarginModel.setter
    def MarginModel(self, value: QuantConnect.Securities.IBuyingPowerModel):
        """Gets the buying power model used for this security, an alias for BuyingPowerModel"""
        ...

    @property
    def SettlementModel(self) -> QuantConnect.Securities.ISettlementModel:
        """Gets the settlement model used for this security"""
        ...

    @SettlementModel.setter
    def SettlementModel(self, value: QuantConnect.Securities.ISettlementModel):
        """Gets the settlement model used for this security"""
        ...

    @property
    def VolatilityModel(self) -> QuantConnect.Securities.IVolatilityModel:
        """Gets the volatility model used for this security"""
        ...

    @VolatilityModel.setter
    def VolatilityModel(self, value: QuantConnect.Securities.IVolatilityModel):
        """Gets the volatility model used for this security"""
        ...

    @property
    def DataFilter(self) -> QuantConnect.Securities.Interfaces.ISecurityDataFilter:
        """
        Customizable data filter to filter outlier ticks before they are passed into user event handlers.
        By default all ticks are passed into the user algorithms.
        """
        ...

    @DataFilter.setter
    def DataFilter(self, value: QuantConnect.Securities.Interfaces.ISecurityDataFilter):
        """
        Customizable data filter to filter outlier ticks before they are passed into user event handlers.
        By default all ticks are passed into the user algorithms.
        """
        ...

    @property
    def PriceVariationModel(self) -> QuantConnect.Securities.IPriceVariationModel:
        """
        Customizable price variation model used to define the minimum price variation of this security.
        By default minimum price variation is a constant find in the symbol-properties-database.
        """
        ...

    @PriceVariationModel.setter
    def PriceVariationModel(self, value: QuantConnect.Securities.IPriceVariationModel):
        """
        Customizable price variation model used to define the minimum price variation of this security.
        By default minimum price variation is a constant find in the symbol-properties-database.
        """
        ...

    @property
    def Data(self) -> QuantConnect.Securities.DynamicSecurityData:
        """Provides dynamic access to data in the cache"""
        ...

    @property
    def HoldStock(self) -> bool:
        """Read only property that checks if we currently own stock in the company."""
        ...

    @property
    def Invested(self) -> bool:
        """Alias for HoldStock - Do we have any of this security"""
        ...

    @property
    def LocalTime(self) -> datetime.datetime:
        """Local time for this market"""
        ...

    @property
    def Price(self) -> float:
        """Get the current value of the security."""
        ...

    @property
    def Leverage(self) -> float:
        """Leverage for this Security."""
        ...

    @property
    def High(self) -> float:
        """If this uses tradebar data, return the most recent high."""
        ...

    @property
    def Low(self) -> float:
        """If this uses tradebar data, return the most recent low."""
        ...

    @property
    def Close(self) -> float:
        """If this uses tradebar data, return the most recent close."""
        ...

    @property
    def Open(self) -> float:
        """If this uses tradebar data, return the most recent open."""
        ...

    @property
    def Volume(self) -> float:
        """Access to the volume of the equity today"""
        ...

    @property
    def BidPrice(self) -> float:
        """Gets the most recent bid price if available"""
        ...

    @property
    def BidSize(self) -> float:
        """Gets the most recent bid size if available"""
        ...

    @property
    def AskPrice(self) -> float:
        """Gets the most recent ask price if available"""
        ...

    @property
    def AskSize(self) -> float:
        """Gets the most recent ask size if available"""
        ...

    @property
    def OpenInterest(self) -> int:
        """Access to the open interest of the security today"""
        ...

    @property
    def Fundamentals(self) -> QuantConnect.Data.Fundamental.Fundamentals:
        """Gets the fundamental data associated with the security if there is any, otherwise null."""
        ...

    @overload
    def __init__(self, exchangeHours: QuantConnect.Securities.SecurityExchangeHours, config: QuantConnect.Data.SubscriptionDataConfig, quoteCurrency: QuantConnect.Securities.Cash, symbolProperties: QuantConnect.Securities.SymbolProperties, currencyConverter: QuantConnect.Securities.ICurrencyConverter, registeredTypesProvider: QuantConnect.Securities.IRegisteredSecurityDataTypesProvider, cache: QuantConnect.Securities.SecurityCache) -> None:
        """Construct a new security vehicle based on the user options."""
        ...

    @overload
    def __init__(self, symbol: typing.Union[QuantConnect.Symbol, str], exchangeHours: QuantConnect.Securities.SecurityExchangeHours, quoteCurrency: QuantConnect.Securities.Cash, symbolProperties: QuantConnect.Securities.SymbolProperties, currencyConverter: QuantConnect.Securities.ICurrencyConverter, registeredTypesProvider: QuantConnect.Securities.IRegisteredSecurityDataTypesProvider, cache: QuantConnect.Securities.SecurityCache) -> None:
        """Construct a new security vehicle based on the user options."""
        ...

    @overload
    def __init__(self, symbol: typing.Union[QuantConnect.Symbol, str], quoteCurrency: QuantConnect.Securities.Cash, symbolProperties: QuantConnect.Securities.SymbolProperties, exchange: QuantConnect.Securities.SecurityExchange, cache: QuantConnect.Securities.SecurityCache, portfolioModel: QuantConnect.Securities.ISecurityPortfolioModel, fillModel: QuantConnect.Orders.Fills.IFillModel, feeModel: QuantConnect.Orders.Fees.IFeeModel, slippageModel: QuantConnect.Orders.Slippage.ISlippageModel, settlementModel: QuantConnect.Securities.ISettlementModel, volatilityModel: QuantConnect.Securities.IVolatilityModel, buyingPowerModel: QuantConnect.Securities.IBuyingPowerModel, dataFilter: QuantConnect.Securities.Interfaces.ISecurityDataFilter, priceVariationModel: QuantConnect.Securities.IPriceVariationModel, currencyConverter: QuantConnect.Securities.ICurrencyConverter, registeredTypesProvider: QuantConnect.Securities.IRegisteredSecurityDataTypesProvider) -> None:
        """
        Construct a new security vehicle based on the user options.
        
        This method is protected.
        """
        ...

    @overload
    def __init__(self, config: QuantConnect.Data.SubscriptionDataConfig, quoteCurrency: QuantConnect.Securities.Cash, symbolProperties: QuantConnect.Securities.SymbolProperties, exchange: QuantConnect.Securities.SecurityExchange, cache: QuantConnect.Securities.SecurityCache, portfolioModel: QuantConnect.Securities.ISecurityPortfolioModel, fillModel: QuantConnect.Orders.Fills.IFillModel, feeModel: QuantConnect.Orders.Fees.IFeeModel, slippageModel: QuantConnect.Orders.Slippage.ISlippageModel, settlementModel: QuantConnect.Securities.ISettlementModel, volatilityModel: QuantConnect.Securities.IVolatilityModel, buyingPowerModel: QuantConnect.Securities.IBuyingPowerModel, dataFilter: QuantConnect.Securities.Interfaces.ISecurityDataFilter, priceVariationModel: QuantConnect.Securities.IPriceVariationModel, currencyConverter: QuantConnect.Securities.ICurrencyConverter, registeredTypesProvider: QuantConnect.Securities.IRegisteredSecurityDataTypesProvider) -> None:
        """
        Temporary convenience constructor
        
        This method is protected.
        """
        ...

    def GetLastData(self) -> QuantConnect.Data.BaseData:
        """
        Get the last price update set to the security if any else null
        
        :returns: BaseData object for this security.
        """
        ...

    def IsCustomData(self) -> bool:
        """
        Returns true if the security contains at least one subscription that represents custom data
        
        This method is obsolete. Use the 'SubscriptionDataConfig' exposed by" +
                    " 'SubscriptionManager' and the 'IsCustomData()' extension method
        """
        warnings.warn("This method is obsolete. Use the 'SubscriptionDataConfig' exposed by" +
                    " 'SubscriptionManager' and the 'IsCustomData()' extension method", DeprecationWarning)

    def RefreshDataNormalizationModeProperty(self) -> None:
        """
        This method will refresh the value of the DataNormalizationMode property.
        This is required for backward-compatibility.
        TODO: to be deleted with the DataNormalizationMode property
        """
        ...

    @overload
    def SetBuyingPowerModel(self, buyingPowerModel: QuantConnect.Securities.IBuyingPowerModel) -> None:
        """
        Sets the buying power model
        
        :param buyingPowerModel: Model that represents a security's model of buying power
        """
        ...

    @overload
    def SetBuyingPowerModel(self, pyObject: typing.Any) -> None:
        """
        Sets the buying power model
        
        :param pyObject: Model that represents a security's model of buying power
        """
        ...

    def SetDataNormalizationMode(self, mode: QuantConnect.DataNormalizationMode) -> None:
        """
        Sets the data normalization mode to be used by this security
        
        This method is obsolete. Use the 'SubscriptionDataConfig' exposed by" +
                    " 'SubscriptionManager' and the 'SetDataNormalizationMode()' extension method
        """
        warnings.warn("This method is obsolete. Use the 'SubscriptionDataConfig' exposed by" +
                    " 'SubscriptionManager' and the 'SetDataNormalizationMode()' extension method", DeprecationWarning)

    @overload
    def SetFeeModel(self, feelModel: QuantConnect.Orders.Fees.IFeeModel) -> None:
        """
        Sets the fee model
        
        :param feelModel: Model that represents a fee model
        """
        ...

    @overload
    def SetFeeModel(self, feelModel: typing.Any) -> None:
        """
        Sets the fee model
        
        :param feelModel: Model that represents a fee model
        """
        ...

    @overload
    def SetFillModel(self, fillModel: QuantConnect.Orders.Fills.IFillModel) -> None:
        """
        Sets the fill model
        
        :param fillModel: Model that represents a fill model
        """
        ...

    @overload
    def SetFillModel(self, fillModel: typing.Any) -> None:
        """
        Sets the fill model
        
        :param fillModel: Model that represents a fill model
        """
        ...

    def SetLeverage(self, leverage: float) -> None:
        """
        Set the leverage parameter for this security
        
        :param leverage: Leverage for this asset
        """
        ...

    def SetLocalTimeKeeper(self, localTimeKeeper: QuantConnect.LocalTimeKeeper) -> None:
        """
        Sets the LocalTimeKeeper to be used for this Security.
        This is the source of this instance's time.
        
        :param localTimeKeeper: The source of this Security's time.
        """
        ...

    @overload
    def SetMarginModel(self, marginModel: QuantConnect.Securities.IBuyingPowerModel) -> None:
        """
        Sets the margin model
        
        :param marginModel: Model that represents a security's model of buying power
        """
        ...

    @overload
    def SetMarginModel(self, pyObject: typing.Any) -> None:
        """
        Sets the margin model
        
        :param pyObject: Model that represents a security's model of buying power
        """
        ...

    def SetMarketPrice(self, data: QuantConnect.Data.BaseData) -> None:
        """
        Update any security properties based on the latest market data and time
        
        :param data: New data packet from LEAN
        """
        ...

    def SetShortableProvider(self, shortableProvider: QuantConnect.Interfaces.IShortableProvider) -> None:
        """
        Set Shortable Provider for this Security
        
        :param shortableProvider: Provider to use
        """
        ...

    @overload
    def SetSlippageModel(self, slippageModel: QuantConnect.Orders.Slippage.ISlippageModel) -> None:
        """
        Sets the slippage model
        
        :param slippageModel: Model that represents a slippage model
        """
        ...

    @overload
    def SetSlippageModel(self, slippageModel: typing.Any) -> None:
        """
        Sets the slippage model
        
        :param slippageModel: Model that represents a slippage model
        """
        ...

    @overload
    def SetVolatilityModel(self, volatilityModel: QuantConnect.Securities.IVolatilityModel) -> None:
        """
        Sets the volatility model
        
        :param volatilityModel: Model that represents a volatility model
        """
        ...

    @overload
    def SetVolatilityModel(self, volatilityModel: typing.Any) -> None:
        """
        Sets the volatility model
        
        :param volatilityModel: Model that represents a volatility model
        """
        ...

    def ToString(self) -> str:
        """
        Returns a string that represents the current object.
        
        :returns: A string that represents the current object.
        """
        ...

    def Update(self, data: System.Collections.Generic.IReadOnlyList[QuantConnect.Data.BaseData], dataType: typing.Type, containsFillForwardData: typing.Optional[bool] = None) -> None:
        """
        Updates all of the security properties, such as price/OHLCV/bid/ask based
        on the data provided. Data is also stored into the security's data cache
        
        :param data: The security update data
        :param dataType: The data type
        :param containsFillForwardData: Flag indicating whether  contains any fill forward bar or not
        """
        ...

    def UpdateConsumersMarketPrice(self, data: QuantConnect.Data.BaseData) -> None:
        """
        Update market price of this Security
        
        This method is protected.
        
        :param data: Data to pull price from
        """
        ...


class BuyingPowerParameters(System.Object):
    """Defines the parameters for IBuyingPowerModel.GetBuyingPower"""

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """Gets the security"""
        ...

    @property
    def Portfolio(self) -> QuantConnect.Securities.SecurityPortfolioManager:
        """Gets the algorithm's portfolio"""
        ...

    @property
    def Direction(self) -> int:
        """
        Gets the direction in which buying power is to be computed
        
        This property contains the int value of a member of the QuantConnect.Orders.OrderDirection enum.
        """
        ...

    def __init__(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, direction: QuantConnect.Orders.OrderDirection) -> None:
        """
        Initializes a new instance of the BuyingPowerParameters class
        
        :param portfolio: The algorithm's portfolio
        :param security: The security
        :param direction: The direction to compute buying power in
        """
        ...

    def Result(self, buyingPower: float, currency: str) -> QuantConnect.Securities.BuyingPower:
        """
        Creates the result using the specified buying power
        
        :param buyingPower: The buying power
        :param currency: The units the buying power is denominated in
        :returns: The buying power.
        """
        ...

    def ResultInAccountCurrency(self, buyingPower: float) -> QuantConnect.Securities.BuyingPower:
        """
        Creates the result using the specified buying power in units of the account currency
        
        :param buyingPower: The buying power
        :returns: The buying power.
        """
        ...


class GetMinimumPriceVariationParameters(System.Object):
    """Defines the parameters for IPriceVariationModel.GetMinimumPriceVariation"""

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """Gets the security"""
        ...

    @property
    def ReferencePrice(self) -> float:
        """Gets the reference price to be used for the calculation"""
        ...

    def __init__(self, security: QuantConnect.Securities.Security, referencePrice: float) -> None:
        """
        Initializes a new instance of the GetMinimumPriceVariationParameters class
        
        :param security: The security
        :param referencePrice: The reference price to be used for the calculation
        """
        ...


class SecurityPriceVariationModel(System.Object, QuantConnect.Securities.IPriceVariationModel):
    """
    Provides default implementation of IPriceVariationModel
    for use in defining the minimum price variation.
    """

    def GetMinimumPriceVariation(self, parameters: QuantConnect.Securities.GetMinimumPriceVariationParameters) -> float:
        """
        Get the minimum price variation from a security
        
        :param parameters: An object containing the method parameters
        :returns: Decimal minimum price variation of a given security.
        """
        ...


class EquityPriceVariationModel(QuantConnect.Securities.SecurityPriceVariationModel):
    """
    Provides an implementation of IPriceVariationModel
    for use in defining the minimum price variation for a given equity
    under Regulation NMS – Rule 612 (a.k.a – the “sub-penny rule”)
    """

    def GetMinimumPriceVariation(self, parameters: QuantConnect.Securities.GetMinimumPriceVariationParameters) -> float:
        """
        Get the minimum price variation from a security
        
        :param parameters: An object containing the method parameters
        :returns: Decimal minimum price variation of a given security.
        """
        ...


class IDerivativeSecurityFilterUniverse(metaclass=abc.ABCMeta):
    """Represents derivative symbols universe used in filtering."""

    @property
    @abc.abstractmethod
    def Underlying(self) -> QuantConnect.Data.BaseData:
        """The underlying price data"""
        ...

    @property
    @abc.abstractmethod
    def IsDynamic(self) -> bool:
        """True if the universe is dynamic and filter needs to be reapplied during trading day"""
        ...


class IDerivativeSecurityFilter(metaclass=abc.ABCMeta):
    """Filters a set of derivative symbols using the underlying price data."""

    def Filter(self, universe: QuantConnect.Securities.IDerivativeSecurityFilterUniverse) -> QuantConnect.Securities.IDerivativeSecurityFilterUniverse:
        """
        Filters the input set of symbols represented by the universe
        
        :param universe: derivative symbols universe used in filtering
        :returns: The filtered set of symbols.
        """
        ...


class FuncSecurityDerivativeFilter(System.Object, QuantConnect.Securities.IDerivativeSecurityFilter):
    """Provides a functional implementation of IDerivativeSecurityFilter"""

    def __init__(self, filter: typing.Callable[[QuantConnect.Securities.IDerivativeSecurityFilterUniverse], QuantConnect.Securities.IDerivativeSecurityFilterUniverse]) -> None:
        """
        Initializes a new instance of the FuncSecurityDerivativeFilter class
        
        :param filter: The functional implementation of the Filter method
        """
        ...

    def Filter(self, universe: QuantConnect.Securities.IDerivativeSecurityFilterUniverse) -> QuantConnect.Securities.IDerivativeSecurityFilterUniverse:
        """
        Filters the input set of symbols represented by the universe
        
        :param universe: Derivative symbols universe used in filtering
        :returns: The filtered set of symbols.
        """
        ...


class SecurityDataFilter(System.Object, QuantConnect.Securities.Interfaces.ISecurityDataFilter):
    """Base class implementation for packet by packet data filtering mechanism to dynamically detect bad ticks."""

    def __init__(self) -> None:
        """Initialize data filter class"""
        ...

    def Filter(self, vehicle: QuantConnect.Securities.Security, data: QuantConnect.Data.BaseData) -> bool:
        """
        Filter the data packet passing through this method by returning true to accept, or false to fail/reject the data point.
        
        :param vehicle: Security vehicle for filter
        :param data: BasData data object we're filtering
        """
        ...


class GetMaximumOrderQuantityForDeltaBuyingPowerParameters(System.Object):
    """Defines the parameters for IBuyingPowerModel.GetMaximumOrderQuantityForDeltaBuyingPower"""

    @property
    def Portfolio(self) -> QuantConnect.Securities.SecurityPortfolioManager:
        """Gets the algorithm's portfolio"""
        ...

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """Gets the security"""
        ...

    @property
    def DeltaBuyingPower(self) -> float:
        """The delta buying power."""
        ...

    @property
    def SilenceNonErrorReasons(self) -> bool:
        """
        True enables the IBuyingPowerModel to skip setting GetMaximumOrderQuantityResult.Reason
        for non error situations, for performance
        """
        ...

    @property
    def MinimumOrderMarginPortfolioPercentage(self) -> float:
        """Configurable minimum order margin portfolio percentage to ignore bad orders, orders with unrealistic small sizes"""
        ...

    def __init__(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, deltaBuyingPower: float, minimumOrderMarginPortfolioPercentage: float, silenceNonErrorReasons: bool = False) -> None:
        """
        Initializes a new instance of the GetMaximumOrderQuantityForDeltaBuyingPowerParameters class
        
        :param portfolio: The algorithm's portfolio
        :param security: The security
        :param deltaBuyingPower: The delta buying power to apply. Sign defines the position side to apply the delta
        :param minimumOrderMarginPortfolioPercentage: Configurable minimum order margin portfolio percentage to ignore orders with unrealistic small sizes
        :param silenceNonErrorReasons: True will not return GetMaximumOrderQuantityResult.Reason set for non error situation, this is for performance
        """
        ...


class ISecuritySeeder(metaclass=abc.ABCMeta):
    """Used to seed the security with the correct price"""

    def SeedSecurity(self, security: QuantConnect.Securities.Security) -> bool:
        """
        Seed the security
        
        :param security: Security being seeded
        :returns: true if the security was seeded, false otherwise.
        """
        ...


class SecuritySeeder(System.Object):
    """Provides access to a null implementation for ISecuritySeeder"""

    Null: QuantConnect.Securities.ISecuritySeeder = ...
    """Gets an instance of ISecuritySeeder that is a no-op"""


class ISecurityInitializer(metaclass=abc.ABCMeta):
    """Represents a type capable of initializing a new security"""

    def Initialize(self, security: QuantConnect.Securities.Security) -> None:
        """
        Initializes the specified security
        
        :param security: The security to be initialized
        """
        ...


class FuncSecurityInitializer(System.Object, QuantConnect.Securities.ISecurityInitializer):
    """Provides a functional implementation of ISecurityInitializer"""

    def __init__(self, initializer: typing.Callable[[QuantConnect.Securities.Security], None]) -> None:
        """
        Initializes a new instance of the FuncSecurityInitializer class
        
        :param initializer: The functional implementation of ISecurityInitializer.Initialize
        """
        ...

    def Initialize(self, security: QuantConnect.Securities.Security) -> None:
        """
        Initializes the specified security
        
        :param security: The security to be initialized
        """
        ...


class ReservedBuyingPowerForPositionParameters(System.Object):
    """Defines the parameters for IBuyingPowerModel.GetReservedBuyingPowerForPosition"""

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """Gets the security"""
        ...

    def __init__(self, security: QuantConnect.Securities.Security) -> None:
        """
        Initializes a new instance of the ReservedBuyingPowerForPositionParameters class
        
        :param security: The security
        """
        ...

    def ResultInAccountCurrency(self, reservedBuyingPower: float) -> QuantConnect.Securities.ReservedBuyingPowerForPosition:
        """
        Creates the result using the specified reserved buying power in units of the account currency
        
        :param reservedBuyingPower: The reserved buying power in units of the account currency
        :returns: The reserved buying power.
        """
        ...


class AdjustedPriceVariationModel(System.Object, QuantConnect.Securities.IPriceVariationModel):
    """
    Provides an implementation of IPriceVariationModel
    for use when data is DataNormalizationMode.Adjusted.
    """

    def GetMinimumPriceVariation(self, parameters: QuantConnect.Securities.GetMinimumPriceVariationParameters) -> float:
        """
        Get the minimum price variation from a security
        
        :param parameters: An object containing the method parameters
        :returns: Zero.
        """
        ...


class InitialMarginRequiredForOrderParameters(System.Object):
    """Defines the parameters for BuyingPowerModel.GetInitialMarginRequiredForOrder"""

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """Gets the security"""
        ...

    @property
    def Order(self) -> QuantConnect.Orders.Order:
        """Gets the order"""
        ...

    @property
    def CurrencyConverter(self) -> QuantConnect.Securities.ICurrencyConverter:
        """Gets the currency converter"""
        ...

    def __init__(self, currencyConverter: QuantConnect.Securities.ICurrencyConverter, security: QuantConnect.Securities.Security, order: QuantConnect.Orders.Order) -> None:
        """
        Initializes a new instance of the InitialMarginRequiredForOrderParameters class
        
        :param currencyConverter: The currency converter
        :param security: The security
        :param order: The order
        """
        ...


class MaintenanceMarginParameters(System.Object):
    """Parameters for IBuyingPowerModel.GetMaintenanceMargin"""

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """Gets the security"""
        ...

    @property
    def Quantity(self) -> float:
        """Gets the quantity of the security"""
        ...

    @property
    def AbsoluteQuantity(self) -> float:
        """Gets the absolute quantity of the security"""
        ...

    @property
    def HoldingsCost(self) -> float:
        """Gets the holdings cost of the security"""
        ...

    @property
    def AbsoluteHoldingsCost(self) -> float:
        """Gets the absolute holdings cost of the security"""
        ...

    @property
    def HoldingsValue(self) -> float:
        """Gets the holdings value of the security"""
        ...

    @property
    def AbsoluteHoldingsValue(self) -> float:
        """Gets the absolute holdings value of the security"""
        ...

    def __init__(self, security: QuantConnect.Securities.Security, quantity: float, holdingsCost: float, holdingsValue: float) -> None:
        """
        Initializes a new instance of the MaintenanceMarginParameters class
        
        :param security: The security
        :param quantity: The quantity
        :param holdingsCost: The holdings cost
        :param holdingsValue: The holdings value
        """
        ...

    @staticmethod
    def ForCurrentHoldings(security: QuantConnect.Securities.Security) -> QuantConnect.Securities.MaintenanceMarginParameters:
        """
        Creates a new instance of the MaintenanceMarginParameters class to compute the maintenance margin
        required to support the algorithm's current holdings
        """
        ...

    @staticmethod
    def ForQuantityAtCurrentPrice(security: QuantConnect.Securities.Security, quantity: float) -> QuantConnect.Securities.MaintenanceMarginParameters:
        """
        Creates a new instance of the MaintenanceMarginParameters class to compute the maintenance margin
        required to support the specified quantity of holdings at current market prices
        """
        ...

    def ForUnderlying(self, quantity: float) -> QuantConnect.Securities.MaintenanceMarginParameters:
        """Creates a new instance of MaintenanceMarginParameters for the security's underlying"""
        ...


class InitialMarginParameters(System.Object):
    """Parameters for IBuyingPowerModel.GetInitialMarginRequirement"""

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """Gets the security"""
        ...

    @property
    def Quantity(self) -> float:
        """Gets the quantity"""
        ...

    def __init__(self, security: QuantConnect.Securities.Security, quantity: float) -> None:
        """
        Initializes a new instance of the InitialMarginParameters class
        
        :param security: The security
        :param quantity: The quantity
        """
        ...

    def ForUnderlying(self) -> QuantConnect.Securities.InitialMarginParameters:
        """Creates a new instance of InitialMarginParameters for the security's underlying"""
        ...


class HasSufficientBuyingPowerForOrderParameters(System.Object):
    """Defines the parameters for IBuyingPowerModel.HasSufficientBuyingPowerForOrder"""

    @property
    def Portfolio(self) -> QuantConnect.Securities.SecurityPortfolioManager:
        """Gets the algorithm's portfolio"""
        ...

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """Gets the security"""
        ...

    @property
    def Order(self) -> QuantConnect.Orders.Order:
        """Gets the order"""
        ...

    def __init__(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, order: QuantConnect.Orders.Order) -> None:
        """
        Initializes a new instance of the HasSufficientBuyingPowerForOrderParameters class
        
        :param portfolio: The algorithm's portfolio
        :param security: The security
        :param order: The order
        """
        ...

    def ForUnderlying(self, order: QuantConnect.Orders.Order) -> QuantConnect.Securities.HasSufficientBuyingPowerForOrderParameters:
        """
        Creates a new HasSufficientBuyingPowerForOrderParameters targeting the security's underlying.
        If the security does not implement IDerivativeSecurity then an InvalidCastException
        will be thrown. If the order's symbol does not match the underlying then an ArgumentException will
        be thrown.
        
        :param order: The new order targeting the underlying
        :returns: New parameters instance suitable for invoking the sufficient capital method for the underlying security.
        """
        ...

    @overload
    def Insufficient(self, reason: str) -> QuantConnect.Securities.HasSufficientBuyingPowerForOrderResult:
        """Creates a new result indicating that there is insufficient buying power for the contemplated order"""
        ...

    @overload
    def Insufficient(self, reason: System.FormattableString) -> QuantConnect.Securities.HasSufficientBuyingPowerForOrderResult:
        """Creates a new result indicating that there is insufficient buying power for the contemplated order"""
        ...

    def Sufficient(self) -> QuantConnect.Securities.HasSufficientBuyingPowerForOrderResult:
        """Creates a new result indicating that there is sufficient buying power for the contemplated order"""
        ...


class GetMaximumOrderQuantityForTargetBuyingPowerParameters(System.Object):
    """Defines the parameters for IBuyingPowerModel.GetMaximumOrderQuantityForTargetBuyingPower"""

    @property
    def Portfolio(self) -> QuantConnect.Securities.SecurityPortfolioManager:
        """Gets the algorithm's portfolio"""
        ...

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """Gets the security"""
        ...

    @property
    def TargetBuyingPower(self) -> float:
        """Gets the target signed percentage buying power"""
        ...

    @property
    def SilenceNonErrorReasons(self) -> bool:
        """
        True enables the IBuyingPowerModel to skip setting GetMaximumOrderQuantityResult.Reason
        for non error situations, for performance
        """
        ...

    @property
    def MinimumOrderMarginPortfolioPercentage(self) -> float:
        """Configurable minimum order margin portfolio percentage to ignore bad orders, orders with unrealistic small sizes"""
        ...

    def __init__(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, targetBuyingPower: float, minimumOrderMarginPortfolioPercentage: float, silenceNonErrorReasons: bool = False) -> None:
        """
        Initializes a new instance of the GetMaximumOrderQuantityForTargetBuyingPowerParameters class
        
        :param portfolio: The algorithm's portfolio
        :param security: The security
        :param targetBuyingPower: The target percentage buying power
        :param minimumOrderMarginPortfolioPercentage: Configurable minimum order margin portfolio percentage to ignore orders with unrealistic small sizes
        :param silenceNonErrorReasons: True will not return GetMaximumOrderQuantityResult.Reason set for non error situation, this is for performance
        """
        ...


class BuyingPowerModel(System.Object, QuantConnect.Securities.IBuyingPowerModel):
    """Provides a base class for all buying power models"""

    @property
    def RequiredFreeBuyingPowerPercent(self) -> float:
        """
        The percentage used to determine the required unused buying power for the account.
        
        This field is protected.
        """
        ...

    @RequiredFreeBuyingPowerPercent.setter
    def RequiredFreeBuyingPowerPercent(self, value: float):
        """
        The percentage used to determine the required unused buying power for the account.
        
        This field is protected.
        """
        ...

    @overload
    def __init__(self) -> None:
        """Initializes a new instance of the BuyingPowerModel with no leverage (1x)"""
        ...

    @overload
    def __init__(self, initialMarginRequirement: float, maintenanceMarginRequirement: float, requiredFreeBuyingPowerPercent: float) -> None:
        """
        Initializes a new instance of the BuyingPowerModel
        
        :param initialMarginRequirement: The percentage of an order's absolute cost that must be held in free cash in order to place the order
        :param maintenanceMarginRequirement: The percentage of the holding's absolute cost that must be held in free cash in order to avoid a margin call
        :param requiredFreeBuyingPowerPercent: The percentage used to determine the required unused buying power for the account.
        """
        ...

    @overload
    def __init__(self, leverage: float, requiredFreeBuyingPowerPercent: float = 0) -> None:
        """
        Initializes a new instance of the BuyingPowerModel
        
        :param leverage: The leverage
        :param requiredFreeBuyingPowerPercent: The percentage used to determine the required unused buying power for the account.
        """
        ...

    def GetAmountToOrder(self, security: QuantConnect.Securities.Security, targetMargin: float, marginForOneUnit: float, finalMargin: typing.Optional[float]) -> typing.Union[float, float]:
        """
        Helper function that determines the amount to order to get to a given target safely.
        Meaning it will either be at or just below target always.
        
        :param security: Security we are to determine order size for
        :param targetMargin: Target margin allocated
        :param marginForOneUnit: Margin requirement for one unit; used in our initial order guess
        :param finalMargin: Output the final margin allocated to this security
        :returns: The size of the order to get safely to our target.
        """
        ...

    def GetBuyingPower(self, parameters: QuantConnect.Securities.BuyingPowerParameters) -> QuantConnect.Securities.BuyingPower:
        """
        Gets the buying power available for a trade
        
        :param parameters: A parameters object containing the algorithm's portfolio, security, and order direction
        :returns: The buying power available for the trade.
        """
        ...

    def GetInitialMarginRequiredForOrder(self, parameters: QuantConnect.Securities.InitialMarginRequiredForOrderParameters) -> QuantConnect.Securities.InitialMargin:
        """
        Gets the total margin required to execute the specified order in units of the account currency including fees
        
        :param parameters: An object containing the portfolio, the security and the order
        :returns: The total margin in terms of the currency quoted in the order.
        """
        ...

    def GetInitialMarginRequirement(self, parameters: QuantConnect.Securities.InitialMarginParameters) -> QuantConnect.Securities.InitialMargin:
        """
        The margin that must be held in order to increase the position by the provided quantity
        
        :param parameters: An object containing the security and quantity of shares
        :returns: The initial margin required for the provided security and quantity.
        """
        ...

    def GetLeverage(self, security: QuantConnect.Securities.Security) -> float:
        """
        Gets the current leverage of the security
        
        :param security: The security to get leverage for
        :returns: The current leverage in the security.
        """
        ...

    def GetMaintenanceMargin(self, parameters: QuantConnect.Securities.MaintenanceMarginParameters) -> QuantConnect.Securities.MaintenanceMargin:
        """
        Gets the margin currently allocated to the specified holding
        
        :param parameters: An object containing the security and holdings quantity/cost/value
        :returns: The maintenance margin required for the provided holdings quantity/cost/value.
        """
        ...

    def GetMarginRemaining(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, direction: QuantConnect.Orders.OrderDirection) -> float:
        """
        Gets the margin cash available for a trade
        
        This method is protected.
        
        :param portfolio: The algorithm's portfolio
        :param security: The security to be traded
        :param direction: The direction of the trade
        :returns: The margin available for the trade.
        """
        ...

    def GetMaximumOrderQuantityForDeltaBuyingPower(self, parameters: QuantConnect.Securities.GetMaximumOrderQuantityForDeltaBuyingPowerParameters) -> QuantConnect.Securities.GetMaximumOrderQuantityResult:
        """
        Get the maximum market order quantity to obtain a delta in the buying power used by a security.
        The deltas sign defines the position side to apply it to, positive long, negative short.
        
        :param parameters: An object containing the portfolio, the security and the delta buying power
        :returns: Returns the maximum allowed market order quantity and if zero, also the reason.
        """
        ...

    def GetMaximumOrderQuantityForTargetBuyingPower(self, parameters: QuantConnect.Securities.GetMaximumOrderQuantityForTargetBuyingPowerParameters) -> QuantConnect.Securities.GetMaximumOrderQuantityResult:
        """
        Get the maximum market order quantity to obtain a position with a given buying power percentage.
        Will not take into account free buying power.
        
        :param parameters: An object containing the portfolio, the security and the target signed buying power percentage
        :returns: Returns the maximum allowed market order quantity and if zero, also the reason.
        """
        ...

    def GetReservedBuyingPowerForPosition(self, parameters: QuantConnect.Securities.ReservedBuyingPowerForPositionParameters) -> QuantConnect.Securities.ReservedBuyingPowerForPosition:
        """
        Gets the amount of buying power reserved to maintain the specified position
        
        :param parameters: A parameters object containing the security
        :returns: The reserved buying power in account currency.
        """
        ...

    def HasSufficientBuyingPowerForOrder(self, parameters: QuantConnect.Securities.HasSufficientBuyingPowerForOrderParameters) -> QuantConnect.Securities.HasSufficientBuyingPowerForOrderResult:
        """
        Check if there is sufficient buying power to execute this order.
        
        :param parameters: An object containing the portfolio, the security and the order
        :returns: Returns buying power information for an order.
        """
        ...

    def SetLeverage(self, security: QuantConnect.Securities.Security, leverage: float) -> None:
        """
        Sets the leverage for the applicable securities, i.e, equities
        
        :param leverage: The new leverage
        """
        ...


class SecurityMarginModel(QuantConnect.Securities.BuyingPowerModel):
    """Represents a simple, constant margin model by specifying the percentages of required margin."""

    @overload
    def __init__(self) -> None:
        """Initializes a new instance of the SecurityMarginModel with no leverage (1x)"""
        ...

    @overload
    def __init__(self, initialMarginRequirement: float, maintenanceMarginRequirement: float, requiredFreeBuyingPowerPercent: float) -> None:
        """
        Initializes a new instance of the SecurityMarginModel
        
        :param initialMarginRequirement: The percentage of an order's absolute cost that must be held in free cash in order to place the order
        :param maintenanceMarginRequirement: The percentage of the holding's absolute cost that must be held in free cash in order to avoid a margin call
        :param requiredFreeBuyingPowerPercent: The percentage used to determine the required unused buying power for the account.
        """
        ...

    @overload
    def __init__(self, leverage: float, requiredFreeBuyingPowerPercent: float = 0) -> None:
        """
        Initializes a new instance of the SecurityMarginModel
        
        :param leverage: The leverage
        :param requiredFreeBuyingPowerPercent: The percentage used to determine the required unused buying power for the account.
        """
        ...


class SecurityPortfolioModel(System.Object, QuantConnect.Securities.ISecurityPortfolioModel):
    """
    Provides a default implementation of ISecurityPortfolioModel that simply
    applies the fills to the algorithm's portfolio. This implementation is intended to
    handle all security types.
    """

    def ProcessFill(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, fill: QuantConnect.Orders.OrderEvent) -> None:
        """
        Performs application of an OrderEvent to the portfolio
        
        :param portfolio: The algorithm's portfolio
        :param security: The fill's security
        :param fill: The order event fill object to be applied
        """
        ...


class FuncSecuritySeeder(System.Object, QuantConnect.Securities.ISecuritySeeder):
    """Seed a security price from a history function"""

    @overload
    def __init__(self, seedFunction: typing.Any) -> None:
        """
        Constructor that takes as a parameter the security used to seed the price
        
        :param seedFunction: The seed function to use
        """
        ...

    @overload
    def __init__(self, seedFunction: typing.Callable[[QuantConnect.Securities.Security], QuantConnect.Data.BaseData]) -> None:
        """
        Constructor that takes as a parameter the security used to seed the price
        
        :param seedFunction: The seed function to use
        """
        ...

    @overload
    def __init__(self, seedFunction: typing.Callable[[QuantConnect.Securities.Security], System.Collections.Generic.IEnumerable[QuantConnect.Data.BaseData]]) -> None:
        """
        Constructor that takes as a parameter the security used to seed the price
        
        :param seedFunction: The seed function to use
        """
        ...

    def SeedSecurity(self, security: QuantConnect.Securities.Security) -> bool:
        """
        Seed the security
        
        :param security: Security being seeded
        :returns: true if the security was seeded, false otherwise.
        """
        ...


class BuyingPowerModelExtensions(System.Object):
    """Provides extension methods as backwards compatibility shims"""

    @staticmethod
    @overload
    def AboveMinimumOrderMarginPortfolioPercentage(model: QuantConnect.Securities.IBuyingPowerModel, security: QuantConnect.Securities.Security, quantity: float, portfolioManager: QuantConnect.Securities.SecurityPortfolioManager, minimumOrderMarginPortfolioPercentage: float) -> bool:
        """
        Helper method to determine if the requested quantity is above the algorithm minimum order margin portfolio percentage
        
        :param model: The buying power model
        :param security: The security
        :param quantity: The quantity of shares
        :param portfolioManager: The algorithm's portfolio
        :param minimumOrderMarginPortfolioPercentage: Minimum order margin portfolio percentage to ignore bad orders, orders with unrealistic small sizes
        :returns: True if this order quantity is above the minimum requested.
        """
        ...

    @staticmethod
    @overload
    def AboveMinimumOrderMarginPortfolioPercentage(portfolioManager: QuantConnect.Securities.SecurityPortfolioManager, minimumOrderMarginPortfolioPercentage: float, absFinalOrderMargin: float) -> bool:
        """
        Helper method to determine if the requested quantity is above the algorithm minimum order margin portfolio percentage
        
        :param portfolioManager: The algorithm's portfolio
        :param minimumOrderMarginPortfolioPercentage: Minimum order margin portfolio percentage to ignore bad orders, orders with unrealistic small sizes
        :param absFinalOrderMargin: The calculated order margin value
        :returns: True if this order quantity is above the minimum requested.
        """
        ...

    @staticmethod
    def GetBuyingPower(model: QuantConnect.Securities.IBuyingPowerModel, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, direction: QuantConnect.Orders.OrderDirection) -> float:
        """
        Gets the buying power available for a trade
        
        :param model: The IBuyingPowerModel
        :param portfolio: The algorithm's portfolio
        :param security: The security to be traded
        :param direction: The direction of the trade
        :returns: The buying power available for the trade.
        """
        ...

    @staticmethod
    def GetInitialMarginRequirement(model: QuantConnect.Securities.IBuyingPowerModel, security: QuantConnect.Securities.Security, quantity: float) -> float:
        """
        Gets the margin currently allocated to the specified holding
        
        :param model: The buying power model
        :param security: The security
        :param quantity: The quantity of shares
        :returns: The initial margin required for the provided security and quantity.
        """
        ...

    @staticmethod
    def GetMaintenanceMargin(model: QuantConnect.Securities.IBuyingPowerModel, security: QuantConnect.Securities.Security) -> float:
        """
        Gets the margin currently allocated to the specified holding
        
        :param model: The buying power model
        :param security: The security
        :returns: The maintenance margin required for the provided holdings quantity/cost/value.
        """
        ...

    @staticmethod
    def GetMaximumOrderQuantityForTargetBuyingPower(model: QuantConnect.Securities.IBuyingPowerModel, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, target: float, minimumOrderMarginPortfolioPercentage: float) -> QuantConnect.Securities.GetMaximumOrderQuantityResult:
        """
        Get the maximum market order quantity to obtain a position with a given value in account currency
        
        :param model: The IBuyingPowerModel
        :param portfolio: The algorithm's portfolio
        :param security: The security to be traded
        :param target: The target percent holdings
        :param minimumOrderMarginPortfolioPercentage: Configurable minimum order margin portfolio percentage to ignore orders with unrealistic small sizes
        :returns: Returns the maximum allowed market order quantity and if zero, also the reason.
        """
        ...

    @staticmethod
    def GetReservedBuyingPowerForPosition(model: QuantConnect.Securities.IBuyingPowerModel, security: QuantConnect.Securities.Security) -> float:
        """
        Gets the amount of buying power reserved to maintain the specified position
        
        :param model: The IBuyingPowerModel
        :param security: The security
        :returns: The reserved buying power in account currency.
        """
        ...

    @staticmethod
    def HasSufficientBuyingPowerForOrder(model: QuantConnect.Securities.IBuyingPowerModel, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, order: QuantConnect.Orders.Order) -> QuantConnect.Securities.HasSufficientBuyingPowerForOrderResult:
        """
        Check if there is sufficient buying power to execute this order.
        
        :param model: The IBuyingPowerModel
        :param portfolio: The algorithm's portfolio
        :param security: The security to be traded
        :param order: The order
        :returns: Returns buying power information for an order.
        """
        ...


class IOrderEventProvider(metaclass=abc.ABCMeta):
    """Represents a type with a new OrderEvent event EventHandler."""

    @property
    @abc.abstractmethod
    def NewOrderEvent(self) -> _EventContainer[typing.Callable[[System.Object, QuantConnect.Orders.OrderEvent], None], None]:
        """Event fired when there is a new OrderEvent"""
        ...

    @NewOrderEvent.setter
    @abc.abstractmethod
    def NewOrderEvent(self, value: _EventContainer[typing.Callable[[System.Object, QuantConnect.Orders.OrderEvent], None], None]):
        """Event fired when there is a new OrderEvent"""
        ...


class AccountEvent(System.Object):
    """Messaging class signifying a change in a user's account"""

    @property
    def CashBalance(self) -> float:
        """Gets the total cash balance of the account in units of CurrencySymbol"""
        ...

    @CashBalance.setter
    def CashBalance(self, value: float):
        """Gets the total cash balance of the account in units of CurrencySymbol"""
        ...

    @property
    def CurrencySymbol(self) -> str:
        """Gets the currency symbol"""
        ...

    @CurrencySymbol.setter
    def CurrencySymbol(self, value: str):
        """Gets the currency symbol"""
        ...

    def __init__(self, currencySymbol: str, cashBalance: float) -> None:
        """
        Creates an AccountEvent
        
        :param currencySymbol: The currency's symbol
        :param cashBalance: The total cash balance of the account
        """
        ...

    def ToString(self) -> str:
        """
        Returns a string that represents the current object.
        
        :returns: A string that represents the current object.
        """
        ...


class SecurityCacheDataStoredEventArgs(System.EventArgs):
    """Event args for SecurityCache's DataStored event"""

    @property
    def DataType(self) -> typing.Type:
        """The type of data that was stored, such as TradeBar"""
        ...

    @property
    def Data(self) -> System.Collections.Generic.IReadOnlyList[QuantConnect.Data.BaseData]:
        """The list of data points stored"""
        ...

    def __init__(self, dataType: typing.Type, data: System.Collections.Generic.IReadOnlyList[QuantConnect.Data.BaseData]) -> None:
        """
        Initializes a new instance of the SecurityCacheDataStoredEventArgs class
        
        :param dataType: The type of data
        :param data: The list of data points
        """
        ...


class OrderProviderExtensions(System.Object):
    """Provides extension methods for the IOrderProvider interface"""

    @staticmethod
    @overload
    def GetOrderByBrokerageId(orderProvider: QuantConnect.Securities.IOrderProvider, brokerageId: int) -> QuantConnect.Orders.Order:
        """
        Gets the order by its brokerage id
        
        :param orderProvider: The order provider to search
        :param brokerageId: The brokerage id to fetch
        :returns: The first order matching the brokerage id, or null if no match is found.
        """
        ...

    @staticmethod
    @overload
    def GetOrderByBrokerageId(orderProvider: QuantConnect.Securities.IOrderProvider, brokerageId: int) -> QuantConnect.Orders.Order:
        """
        Gets the order by its brokerage id
        
        :param orderProvider: The order provider to search
        :param brokerageId: The brokerage id to fetch
        :returns: The first order matching the brokerage id, or null if no match is found.
        """
        ...


class SecurityInitializer(System.Object):
    """Provides static access to the Null security initializer"""

    Null: QuantConnect.Securities.ISecurityInitializer = ...
    """Gets an implementation of ISecurityInitializer that is a no-op"""


class ErrorCurrencyConverter(System.Object, QuantConnect.Securities.ICurrencyConverter):
    """
    Provides an implementation of ICurrencyConverter for use in
    tests that don't depend on this behavior.
    """

    @property
    def AccountCurrency(self) -> str:
        """Gets account currency"""
        ...

    Instance: QuantConnect.Securities.ICurrencyConverter = ...
    """
    Provides access to the single instance of ErrorCurrencyConverter.
    This is done this way to ensure usage is explicit.
    """

    def ConvertToAccountCurrency(self, cashAmount: QuantConnect.Securities.CashAmount) -> QuantConnect.Securities.CashAmount:
        """
        Converts a cash amount to the account currency
        
        :param cashAmount: The CashAmount instance to convert
        :returns: A new CashAmount instance denominated in the account currency.
        """
        ...


class CompositeSecurityInitializer(System.Object, QuantConnect.Securities.ISecurityInitializer):
    """
    Provides an implementation of ISecurityInitializer that executes
    each initializer in order
    """

    def __init__(self, *initializers: QuantConnect.Securities.ISecurityInitializer) -> None:
        """
        Initializes a new instance of the CompositeSecurityInitializer class
        
        :param initializers: The initializers to execute in order
        """
        ...

    def Initialize(self, security: QuantConnect.Securities.Security) -> None:
        """
        Execute each of the internally held initializers in sequence
        
        :param security: The security to be initialized
        """
        ...


class ISecurityProvider(metaclass=abc.ABCMeta):
    """Represents a type capable of fetching the holdings for the specified symbol"""

    def GetSecurity(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> QuantConnect.Securities.Security:
        """
        Retrieves a summary of the holdings for the specified symbol
        
        :param symbol: The symbol to get holdings for
        :returns: The holdings for the symbol or null if the symbol is invalid and/or not in the portfolio.
        """
        ...


class SecurityCacheProvider(System.Object):
    """A helper class that will provide SecurityCache instances"""

    def __init__(self, securityProvider: QuantConnect.Securities.ISecurityProvider) -> None:
        """
        Creates a new instance
        
        :param securityProvider: The security provider to use
        """
        ...

    def GetSecurityCache(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> QuantConnect.Securities.SecurityCache:
        """
        Will return the SecurityCache instance to use for a give Symbol.
        If the provided Symbol is a custom type which has an underlying we will try to use the
        underlying SecurityCache type cache, if the underlying is not present we will keep track
        of the custom Symbol in case it is added later.
        
        :returns: The cache instance to use.
        """
        ...


class SecurityDefinitionSymbolResolver(System.Object):
    """
    Resolves standardized security definitions such as FIGI, CUSIP, ISIN, SEDOL into
    a properly mapped Lean Symbol.
    """

    def __init__(self, dataProvider: QuantConnect.Interfaces.IDataProvider = None, securitiesDefinitionKey: str = None) -> None:
        """
        Creates an instance of the symbol resolver
        
        :param dataProvider: Data provider used to obtain symbol mappings data
        :param securitiesDefinitionKey: Location to read the securities definition data from
        """
        ...

    def CompositeFIGI(self, compositeFigi: str, tradingDate: typing.Union[datetime.datetime, datetime.date]) -> QuantConnect.Symbol:
        """
        Converts an asset's composite FIGI into a Lean Symbol
        
        :param compositeFigi: The composite Financial Instrument Global Identifier (FIGI) of a security
        :param tradingDate: The date that the stock was trading at with the composite FIGI provided. This is used to get the ticker of the symbol on this date.
        :returns: The Lean Symbol corresponding to the composite FIGI on the trading date provided.
        """
        ...

    def CUSIP(self, cusip: str, tradingDate: typing.Union[datetime.datetime, datetime.date]) -> QuantConnect.Symbol:
        """
        Converts CUSIP into a Lean Symbol
        
        :param cusip: The Committee on Uniform Securities Identification Procedures (CUSIP) number of a security
        :param tradingDate: The date that the stock was trading at with the CUSIP provided. This is used to get the ticker of the symbol on this date.
        :returns: The Lean Symbol corresponding to the CUSIP number on the trading date provided.
        """
        ...

    def ISIN(self, isin: str, tradingDate: typing.Union[datetime.datetime, datetime.date]) -> QuantConnect.Symbol:
        """
        Converts ISIN into a Lean Symbol
        
        :param isin: The International Securities Identification Number (ISIN) of a security
        :param tradingDate: The date that the stock was trading at with the ISIN provided. This is used to get the ticker of the symbol on this date.
        :returns: The Lean Symbol corresponding to the ISIN on the trading date provided.
        """
        ...

    def SEDOL(self, sedol: str, tradingDate: typing.Union[datetime.datetime, datetime.date]) -> QuantConnect.Symbol:
        """
        Converts SEDOL into a Lean Symbol
        
        :param sedol: The Stock Exchange Daily Official List (SEDOL) security identifier of a security
        :param tradingDate: The date that the stock was trading at with the SEDOL provided. This is used to get the ticker of the symbol on this date.
        :returns: The Lean Symbol corresponding to the SEDOL on the trading date provided.
        """
        ...


class SecurityProviderExtensions(System.Object):
    """Provides extension methods for the ISecurityProvider interface."""

    @staticmethod
    def GetHoldingsQuantity(provider: QuantConnect.Securities.ISecurityProvider, symbol: typing.Union[QuantConnect.Symbol, str]) -> float:
        """
        Extension method to return the quantity of holdings, if no holdings are present, then zero is returned.
        
        :param provider: The ISecurityProvider
        :param symbol: The symbol we want holdings quantity for
        :returns: The quantity of holdings for the specified symbol.
        """
        ...


class IDerivativeSecurity(metaclass=abc.ABCMeta):
    """Defines a security as a derivative of another security"""

    @property
    @abc.abstractmethod
    def Underlying(self) -> QuantConnect.Securities.Security:
        """Gets or sets the underlying security for the derivative"""
        ...

    @Underlying.setter
    @abc.abstractmethod
    def Underlying(self, value: QuantConnect.Securities.Security):
        """Gets or sets the underlying security for the derivative"""
        ...


class AccountCurrencyImmediateSettlementModel(System.Object, QuantConnect.Securities.ISettlementModel):
    """Represents the model responsible for applying cash settlement rules"""

    def ApplyFunds(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, applicationTimeUtc: typing.Union[datetime.datetime, datetime.date], currency: str, amount: float) -> None:
        """
        Applies cash settlement rules
        
        :param portfolio: The algorithm's portfolio
        :param security: The fill's security
        :param applicationTimeUtc: The fill time (in UTC)
        :param currency: The currency symbol
        :param amount: The amount of cash to apply
        """
        ...


class DelayedSettlementModel(System.Object, QuantConnect.Securities.ISettlementModel):
    """Represents the model responsible for applying cash settlement rules"""

    def __init__(self, numberOfDays: int, timeOfDay: datetime.timedelta) -> None:
        """
        Creates an instance of the DelayedSettlementModel class
        
        :param numberOfDays: The number of days required for settlement
        :param timeOfDay: The time of day used for settlement
        """
        ...

    def ApplyFunds(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, applicationTimeUtc: typing.Union[datetime.datetime, datetime.date], currency: str, amount: float) -> None:
        """
        Applies cash settlement rules
        
        :param portfolio: The algorithm's portfolio
        :param security: The fill's security
        :param applicationTimeUtc: The fill time (in UTC)
        :param currency: The currency symbol
        :param amount: The amount of cash to apply
        """
        ...


class SecurityEventArgs(System.Object, metaclass=abc.ABCMeta):
    """Defines a base class for Security related events"""

    @property
    def Security(self) -> QuantConnect.Securities.Security:
        """Gets the security related to this event"""
        ...

    def __init__(self, security: QuantConnect.Securities.Security) -> None:
        """
        Initializes a new instance of the SecurityEventArgs class
        
        This method is protected.
        
        :param security: The security
        """
        ...


class SecurityHoldingQuantityChangedEventArgs(QuantConnect.Securities.SecurityEventArgs):
    """
    Event arguments for the SecurityHolding.QuantityChanged event.
    The event data contains the previous quantity/price. The current quantity/price
    can be accessed via the SecurityEventArgs.Security property
    """

    @property
    def PreviousQuantity(self) -> float:
        """Gets the holdings quantity before this change"""
        ...

    @property
    def PreviousAveragePrice(self) -> float:
        """Gets the average holdings price before this change"""
        ...

    def __init__(self, security: QuantConnect.Securities.Security, previousAveragePrice: float, previousQuantity: float) -> None:
        """
        Initializes a new instance of the SecurityHoldingQuantityChangedEventArgs class
        
        :param security: The security
        :param previousAveragePrice: The security's previous average holdings price
        :param previousQuantity: The security's previous holdings quantity
        """
        ...


class PatternDayTradingMarginModel(QuantConnect.Securities.SecurityMarginModel):
    """
    Represents a simple margining model where margin/leverage depends on market state (open or close).
    During regular market hours, leverage is 4x, otherwise 2x
    """

    @overload
    def __init__(self) -> None:
        """Initializes a new instance of the PatternDayTradingMarginModel"""
        ...

    @overload
    def __init__(self, closedMarketLeverage: float, openMarketLeverage: float) -> None:
        """
        Initializes a new instance of the PatternDayTradingMarginModel
        
        :param closedMarketLeverage: Leverage used outside regular market hours
        :param openMarketLeverage: Leverage used during regular market hours
        """
        ...

    def GetInitialMarginRequirement(self, parameters: QuantConnect.Securities.InitialMarginParameters) -> QuantConnect.Securities.InitialMargin:
        """The percentage of an order's absolute cost that must be held in free cash in order to place the order"""
        ...

    def GetLeverage(self, security: QuantConnect.Securities.Security) -> float:
        """
        Gets the current leverage of the security
        
        :param security: The security to get leverage for
        :returns: The current leverage in the security.
        """
        ...

    def GetMaintenanceMargin(self, parameters: QuantConnect.Securities.MaintenanceMarginParameters) -> QuantConnect.Securities.MaintenanceMargin:
        """The percentage of the holding's absolute cost that must be held in free cash in order to avoid a margin call"""
        ...

    def SetLeverage(self, security: QuantConnect.Securities.Security, leverage: float) -> None:
        """
        Sets the leverage for the applicable securities, i.e, equities
        
        :param security: The security to set leverage to
        :param leverage: The new leverage
        """
        ...


class MarginCallModel(System.Object):
    """Provides access to a null implementation for IMarginCallModel"""

    Null: QuantConnect.Securities.IMarginCallModel = ...
    """
    Gets an instance of IMarginCallModel that will always
    return an empty list of executed orders.
    """


class IContinuousSecurity(metaclass=abc.ABCMeta):
    """A continuous security that get's mapped during his life"""

    @property
    @abc.abstractmethod
    def Mapped(self) -> QuantConnect.Symbol:
        """Gets or sets the currently mapped symbol for the security"""
        ...

    @Mapped.setter
    @abc.abstractmethod
    def Mapped(self, value: QuantConnect.Symbol):
        """Gets or sets the currently mapped symbol for the security"""
        ...


class BrokerageModelSecurityInitializer(System.Object, QuantConnect.Securities.ISecurityInitializer):
    """
    Provides an implementation of ISecurityInitializer that initializes a security
    by settings the Security.FillModel, Security.FeeModel,
    Security.SlippageModel, and the Security.SettlementModel properties
    """

    @overload
    def __init__(self) -> None:
        """
        Initializes a new instance of the BrokerageModelSecurityInitializer class
        for the specified algorithm
        """
        ...

    @overload
    def __init__(self, brokerageModel: QuantConnect.Brokerages.IBrokerageModel, securitySeeder: QuantConnect.Securities.ISecuritySeeder) -> None:
        """
        Initializes a new instance of the BrokerageModelSecurityInitializer class
        for the specified algorithm
        
        :param brokerageModel: The brokerage model used to initialize the security models
        :param securitySeeder: An ISecuritySeeder used to seed the initial price of the security
        """
        ...

    def Initialize(self, security: QuantConnect.Securities.Security) -> None:
        """
        Initializes the specified security by setting up the models
        
        :param security: The security to be initialized
        """
        ...


class ContractSecurityFilterUniverse(typing.Generic[QuantConnect_Securities_ContractSecurityFilterUniverse_T], System.Object, QuantConnect.Securities.IDerivativeSecurityFilterUniverse, typing.Iterable[QuantConnect.Symbol], metaclass=abc.ABCMeta):
    """
    Base class for contract symbols filtering universes.
    Used by OptionFilterUniverse and FutureFilterUniverse
    """

    class ContractExpirationType(System.Enum):
        """
        Defines listed contract types with Flags attribute
        
        This class is protected.
        """

        Standard = 1
        """Standard contracts"""

        Weekly = 2
        """Non standard weekly contracts"""

    @property
    def Type(self) -> QuantConnect.Securities.ContractSecurityFilterUniverse.ContractExpirationType:
        """
        Expiration Types allowed through the filter
        Standards only by default
        
        This field is protected.
        """
        ...

    @Type.setter
    def Type(self, value: QuantConnect.Securities.ContractSecurityFilterUniverse.ContractExpirationType):
        """
        Expiration Types allowed through the filter
        Standards only by default
        
        This field is protected.
        """
        ...

    @property
    def UnderlyingInternal(self) -> QuantConnect.Data.BaseData:
        """
        The underlying price data
        
        This field is protected.
        """
        ...

    @UnderlyingInternal.setter
    def UnderlyingInternal(self, value: QuantConnect.Data.BaseData):
        """
        The underlying price data
        
        This field is protected.
        """
        ...

    @property
    def AllSymbols(self) -> System.Collections.Generic.IEnumerable[QuantConnect.Symbol]:
        """
        All Symbols in this filter
        Marked internal for use by extensions
        """
        ...

    @AllSymbols.setter
    def AllSymbols(self, value: System.Collections.Generic.IEnumerable[QuantConnect.Symbol]):
        """
        All Symbols in this filter
        Marked internal for use by extensions
        """
        ...

    @property
    def IsDynamicInternal(self) -> bool:
        """
        Mark this filter dynamic for regular reapplying
        Marked internal for use by extensions
        """
        ...

    @IsDynamicInternal.setter
    def IsDynamicInternal(self, value: bool):
        """
        Mark this filter dynamic for regular reapplying
        Marked internal for use by extensions
        """
        ...

    @property
    def IsDynamic(self) -> bool:
        """True if the universe is dynamic and filter needs to be reapplied"""
        ...

    @property
    def Underlying(self) -> QuantConnect.Data.BaseData:
        """The underlying price data"""
        ...

    @overload
    def __init__(self) -> None:
        """
        Constructs ContractSecurityFilterUniverse
        
        This method is protected.
        """
        ...

    @overload
    def __init__(self, allSymbols: System.Collections.Generic.IEnumerable[QuantConnect.Symbol], underlying: QuantConnect.Data.BaseData) -> None:
        """
        Constructs ContractSecurityFilterUniverse
        
        This method is protected.
        """
        ...

    def BackMonth(self) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Returns first of back month contracts
        
        :returns: Universe with filter applied.
        """
        ...

    def BackMonths(self) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Returns a list of back month contracts
        
        :returns: Universe with filter applied.
        """
        ...

    @overload
    def Contracts(self, contracts: typing.Any) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Explicitly sets the selected contract symbols for this universe.
        This overrides and and all other methods of selecting symbols assuming it is called last.
        
        :param contracts: The option contract symbol objects to select
        :returns: Universe with filter applied.
        """
        ...

    @overload
    def Contracts(self, contracts: System.Collections.Generic.IEnumerable[QuantConnect.Symbol]) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Explicitly sets the selected contract symbols for this universe.
        This overrides and and all other methods of selecting symbols assuming it is called last.
        
        :param contracts: The option contract symbol objects to select
        :returns: Universe with filter applied.
        """
        ...

    @overload
    def Contracts(self, contractSelector: typing.Callable[[System.Collections.Generic.IEnumerable[QuantConnect.Symbol]], System.Collections.Generic.IEnumerable[QuantConnect.Symbol]]) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Sets a function used to filter the set of available contract filters. The input to the 'contractSelector'
        function will be the already filtered list if any other filters have already been applied.
        
        :param contractSelector: The option contract symbol objects to select
        :returns: Universe with filter applied.
        """
        ...

    @overload
    def Expiration(self, minExpiry: datetime.timedelta, maxExpiry: datetime.timedelta) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Applies filter selecting options contracts based on a range of expiration dates relative to the current day
        
        :param minExpiry: The minimum time until expiry to include, for example, TimeSpan.FromDays(10) would exclude contracts expiring in more than 10 days
        :param maxExpiry: The maximum time until expiry to include, for example, TimeSpan.FromDays(10) would exclude contracts expiring in less than 10 days
        :returns: Universe with filter applied.
        """
        ...

    @overload
    def Expiration(self, minExpiryDays: int, maxExpiryDays: int) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Applies filter selecting contracts based on a range of expiration dates relative to the current day
        
        :param minExpiryDays: The minimum time, expressed in days, until expiry to include, for example, 10 would exclude contracts expiring in more than 10 days
        :param maxExpiryDays: The maximum time, expressed in days, until expiry to include, for example, 10 would exclude contracts expiring in less than 10 days
        :returns: Universe with filter applied.
        """
        ...

    def FrontMonth(self) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Returns front month contract
        
        :returns: Universe with filter applied.
        """
        ...

    @overload
    def GetEnumerator(self) -> System.Collections.Generic.IEnumerator[QuantConnect.Symbol]:
        """
        IEnumerable interface method implementation
        
        :returns: IEnumerator of Symbols in Universe.
        """
        ...

    @overload
    def GetEnumerator(self) -> System.Collections.IEnumerator:
        """IEnumerable interface method implementation"""
        ...

    def IncludeWeeklys(self) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Includes universe of non-standard weeklys contracts (if any) into selection
        
        :returns: Universe with filter applied.
        """
        ...

    def IsStandard(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> bool:
        """
        Function to determine if the given symbol is a standard contract
        
        This method is protected.
        
        :returns: True if standard type.
        """
        ...

    def OnlyApplyFilterAtMarketOpen(self) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Instructs the engine to only filter contracts on the first time step of each market day.
        
        :returns: Universe with filter applied.
        """
        ...

    def Refresh(self, allSymbols: System.Collections.Generic.IEnumerable[QuantConnect.Symbol], underlying: QuantConnect.Data.BaseData) -> None:
        """
        Refreshes this filter universe
        
        :param allSymbols: All the contract symbols for the Universe
        :param underlying: The current underlying last data point
        """
        ...

    def StandardsOnly(self) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Sets universe of standard contracts (if any) as selection
        Contracts by default are standards; only needed to switch back if changed
        
        :returns: Universe with filter applied.
        """
        ...

    def WeeklysOnly(self) -> QuantConnect_Securities_ContractSecurityFilterUniverse_T:
        """
        Sets universe of weeklys contracts (if any) as selection
        
        :returns: Universe with filter applied.
        """
        ...


class DefaultMarginCallModel(System.Object, QuantConnect.Securities.IMarginCallModel):
    """Represents the model responsible for picking which orders should be executed during a margin call"""

    @property
    def Portfolio(self) -> QuantConnect.Securities.SecurityPortfolioManager:
        """
        Gets the portfolio that margin calls will be transacted against
        
        This property is protected.
        """
        ...

    @property
    def DefaultOrderProperties(self) -> QuantConnect.Interfaces.IOrderProperties:
        """
        Gets the default order properties to be used in margin call orders
        
        This property is protected.
        """
        ...

    def __init__(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, defaultOrderProperties: QuantConnect.Interfaces.IOrderProperties) -> None:
        """
        Initializes a new instance of the DefaultMarginCallModel class
        
        :param portfolio: The portfolio object to receive margin calls
        :param defaultOrderProperties: The default order properties to be used in margin call orders
        """
        ...

    def ExecuteMarginCall(self, generatedMarginCallOrders: System.Collections.Generic.IEnumerable[QuantConnect.Orders.SubmitOrderRequest]) -> System.Collections.Generic.List[QuantConnect.Orders.OrderTicket]:
        """
        Executes synchronous orders to bring the account within margin requirements.
        
        :param generatedMarginCallOrders: These are the margin call orders that were generated by individual security margin models.
        :returns: The list of orders that were actually executed.
        """
        ...

    def GenerateMarginCallOrder(self, security: QuantConnect.Securities.Security, totalPortfolioValue: float, totalUsedMargin: float) -> QuantConnect.Orders.SubmitOrderRequest:
        """
        Generates a new order for the specified security taking into account the total margin
        used by the account. Returns null when no margin call is to be issued.
        
        This method is protected.
        
        :param security: The security to generate a margin call order for
        :param totalPortfolioValue: The net liquidation value for the entire account
        :param totalUsedMargin: The total margin used by the account in units of base currency
        :returns: An order object representing a liquidation order to be executed to bring the account within margin requirements.
        """
        ...

    def GetMarginCallOrders(self, issueMarginCallWarning: typing.Optional[bool]) -> typing.Union[System.Collections.Generic.List[QuantConnect.Orders.SubmitOrderRequest], bool]:
        """
        Scan the portfolio and the updated data for a potential margin call situation which may get the holdings below zero!
        If there is a margin call, liquidate the portfolio immediately before the portfolio gets sub zero.
        
        :param issueMarginCallWarning: Set to true if a warning should be issued to the algorithm
        :returns: True for a margin call on the holdings.
        """
        ...


class IdentityCurrencyConverter(System.Object, QuantConnect.Securities.ICurrencyConverter):
    """
    Provides an implementation of ICurrencyConverter that does NOT perform conversions.
    This implementation will throw if the specified cashAmount is not in units of account currency.
    """

    @property
    def AccountCurrency(self) -> str:
        """Gets account currency"""
        ...

    def __init__(self, accountCurrency: str) -> None:
        """
        Initializes a new instance of the ICurrencyConverter class
        
        :param accountCurrency: The algorithm's account currency
        """
        ...

    def ConvertToAccountCurrency(self, cashAmount: QuantConnect.Securities.CashAmount) -> QuantConnect.Securities.CashAmount:
        """
        Converts a cash amount to the account currency.
        This implementation can only handle cash amounts in units of the account currency.
        
        :param cashAmount: The CashAmount instance to convert
        :returns: A new CashAmount instance denominated in the account currency.
        """
        ...


class ImmediateSettlementModel(System.Object, QuantConnect.Securities.ISettlementModel):
    """Represents the model responsible for applying cash settlement rules"""

    def ApplyFunds(self, portfolio: QuantConnect.Securities.SecurityPortfolioManager, security: QuantConnect.Securities.Security, applicationTimeUtc: typing.Union[datetime.datetime, datetime.date], currency: str, amount: float) -> None:
        """
        Applies cash settlement rules
        
        :param portfolio: The algorithm's portfolio
        :param security: The fill's security
        :param applicationTimeUtc: The fill time (in UTC)
        :param currency: The currency symbol
        :param amount: The amount of cash to apply
        """
        ...


class IBaseCurrencySymbol(metaclass=abc.ABCMeta):
    """Interface for various currency symbols"""

    @property
    @abc.abstractmethod
    def BaseCurrencySymbol(self) -> str:
        """Gets the currency acquired by going long this currency pair"""
        ...


class UniverseManager(System.Object, System.Collections.Generic.IDictionary[QuantConnect.Symbol, QuantConnect.Data.UniverseSelection.Universe], System.Collections.Specialized.INotifyCollectionChanged, typing.Iterable[System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Data.UniverseSelection.Universe]]):
    """Manages the algorithm's collection of universes"""

    @property
    def CollectionChanged(self) -> _EventContainer[typing.Callable[[System.Object, System.Collections.Specialized.NotifyCollectionChangedEventArgs], None], None]:
        """Event fired when a universe is added or removed"""
        ...

    @CollectionChanged.setter
    def CollectionChanged(self, value: _EventContainer[typing.Callable[[System.Object, System.Collections.Specialized.NotifyCollectionChangedEventArgs], None], None]):
        """Event fired when a universe is added or removed"""
        ...

    @property
    def ActiveSecurities(self) -> System.Collections.Generic.IReadOnlyDictionary[QuantConnect.Symbol, QuantConnect.Securities.Security]:
        """
        Read-only dictionary containing all active securities. An active security is
        a security that is currently selected by the universe or has holdings or open orders.
        """
        ...

    @property
    def Count(self) -> int:
        """Gets the number of elements contained in the System.Collections.Generic.ICollection`1."""
        ...

    @property
    def IsReadOnly(self) -> bool:
        """Gets a value indicating whether the System.Collections.Generic.ICollection`1 is read-only."""
        ...

    @property
    def Keys(self) -> System.Collections.Generic.ICollection[QuantConnect.Symbol]:
        """Gets an System.Collections.Generic.ICollection`1 containing the keys of the System.Collections.Generic.IDictionary`2."""
        ...

    @property
    def Values(self) -> System.Collections.Generic.ICollection[QuantConnect.Data.UniverseSelection.Universe]:
        """Gets an System.Collections.Generic.ICollection`1 containing the values in the System.Collections.Generic.IDictionary`2."""
        ...

    def __getitem__(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> QuantConnect.Data.UniverseSelection.Universe:
        """
        Gets or sets the element with the specified key.
        
        :param symbol: The key of the element to get or set.
        :returns: The element with the specified key.
        """
        ...

    def __init__(self) -> None:
        """Initializes a new instance of the UniverseManager class"""
        ...

    def __setitem__(self, symbol: typing.Union[QuantConnect.Symbol, str], value: QuantConnect.Data.UniverseSelection.Universe) -> None:
        """
        Gets or sets the element with the specified key.
        
        :param symbol: The key of the element to get or set.
        :returns: The element with the specified key.
        """
        ...

    @overload
    def Add(self, item: System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Data.UniverseSelection.Universe]) -> None:
        """
        Adds an item to the System.Collections.Generic.ICollection`1.
        
        :param item: The object to add to the System.Collections.Generic.ICollection`1.
        """
        ...

    @overload
    def Add(self, key: typing.Union[QuantConnect.Symbol, str], universe: QuantConnect.Data.UniverseSelection.Universe) -> None:
        """
        Adds an element with the provided key and value to the System.Collections.Generic.IDictionary`2.
        
        :param key: The object to use as the key of the element to add.
        :param universe: The object to use as the value of the element to add.
        """
        ...

    def Clear(self) -> None:
        """Removes all items from the System.Collections.Generic.ICollection`1."""
        ...

    def Contains(self, item: System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Data.UniverseSelection.Universe]) -> bool:
        """
        Determines whether the System.Collections.Generic.ICollection`1 contains a specific value.
        
        :param item: The object to locate in the System.Collections.Generic.ICollection`1.
        :returns: true if  is found in the System.Collections.Generic.ICollection`1; otherwise, false.
        """
        ...

    def ContainsKey(self, key: typing.Union[QuantConnect.Symbol, str]) -> bool:
        """
        Determines whether the System.Collections.Generic.IDictionary`2 contains an element with the specified key.
        
        :param key: The key to locate in the System.Collections.Generic.IDictionary`2.
        :returns: true if the System.Collections.Generic.IDictionary`2 contains an element with the key; otherwise, false.
        """
        ...

    def CopyTo(self, array: typing.List[System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Data.UniverseSelection.Universe]], arrayIndex: int) -> None:
        """
        Copies the elements of the System.Collections.Generic.ICollection`1 to an System.Array, starting at a particular System.Array index.
        
        :param array: The one-dimensional System.Array that is the destination of the elements copied from System.Collections.Generic.ICollection`1. The System.Array must have zero-based indexing.
        :param arrayIndex: The zero-based index in  at which copying begins.
        """
        ...

    @overload
    def GetEnumerator(self) -> System.Collections.Generic.IEnumerator[System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Data.UniverseSelection.Universe]]:
        ...

    @overload
    def GetEnumerator(self) -> System.Collections.IEnumerator:
        """
        Returns an enumerator that iterates through a collection.
        
        :returns: An System.Collections.IEnumerator object that can be used to iterate through the collection.
        """
        ...

    def OnCollectionChanged(self, e: System.Collections.Specialized.NotifyCollectionChangedEventArgs) -> None:
        """This method is protected."""
        ...

    @overload
    def Remove(self, item: System.Collections.Generic.KeyValuePair[QuantConnect.Symbol, QuantConnect.Data.UniverseSelection.Universe]) -> bool:
        """
        Removes the first occurrence of a specific object from the System.Collections.Generic.ICollection`1.
        
        :param item: The object to remove from the System.Collections.Generic.ICollection`1.
        :returns: true if  was successfully removed from the System.Collections.Generic.ICollection`1; otherwise, false. This method also returns false if  is not found in the original System.Collections.Generic.ICollection`1.
        """
        ...

    @overload
    def Remove(self, key: typing.Union[QuantConnect.Symbol, str]) -> bool:
        """
        Removes the element with the specified key from the System.Collections.Generic.IDictionary`2.
        
        :param key: The key of the element to remove.
        :returns: true if the element is successfully removed; otherwise, false.  This method also returns false if  was not found in the original System.Collections.Generic.IDictionary`2.
        """
        ...

    def TryGetValue(self, key: typing.Union[QuantConnect.Symbol, str], value: typing.Optional[QuantConnect.Data.UniverseSelection.Universe]) -> typing.Union[bool, QuantConnect.Data.UniverseSelection.Universe]:
        """
        Gets the value associated with the specified key.
        
        :param key: The key whose value to get.
        :param value: When this method returns, the value associated with the specified key, if the key is found; otherwise, the default value for the type of the  parameter. This parameter is passed uninitialized.
        :returns: true if the object that implements System.Collections.Generic.IDictionary`2 contains an element with the specified key; otherwise, false.
        """
        ...


class RegisteredSecurityDataTypesProvider(System.Object, QuantConnect.Securities.IRegisteredSecurityDataTypesProvider):
    """
    Provides an implementation of IRegisteredSecurityDataTypesProvider that permits the
    consumer to modify the expected types
    """

    Null: QuantConnect.Securities.IRegisteredSecurityDataTypesProvider = ...
    """Provides a reference to an instance of IRegisteredSecurityDataTypesProvider that contains no registered types"""

    def RegisterType(self, type: typing.Type) -> bool:
        """
        Registers the specified type w/ the provider
        
        :returns: True if the type was previously not registered.
        """
        ...

    def TryGetType(self, name: str, type: typing.Optional[typing.Type]) -> typing.Union[bool, typing.Type]:
        """Gets an enumerable of data types expected to be contained in a DynamicSecurityData instance"""
        ...

    def UnregisterType(self, type: typing.Type) -> bool:
        """
        Removes the registration for the specified type
        
        :returns: True if the type was previously registered.
        """
        ...


class SecurityDefinition(System.Object):
    """
    Helper class containing various unique identifiers for a given
    SecurityIdentifier, such as FIGI, ISIN, CUSIP, SEDOL.
    """

    @property
    def SecurityIdentifier(self) -> QuantConnect.SecurityIdentifier:
        """
        The unique SecurityIdentifier identified by
        the industry-standard security identifiers contained within this class.
        """
        ...

    @SecurityIdentifier.setter
    def SecurityIdentifier(self, value: QuantConnect.SecurityIdentifier):
        """
        The unique SecurityIdentifier identified by
        the industry-standard security identifiers contained within this class.
        """
        ...

    @property
    def CUSIP(self) -> str:
        """The Committee on Uniform Securities Identification Procedures (CUSIP) number of a security"""
        ...

    @CUSIP.setter
    def CUSIP(self, value: str):
        """The Committee on Uniform Securities Identification Procedures (CUSIP) number of a security"""
        ...

    @property
    def CompositeFIGI(self) -> str:
        """The composite Financial Instrument Global Identifier (FIGI) of a security"""
        ...

    @CompositeFIGI.setter
    def CompositeFIGI(self, value: str):
        """The composite Financial Instrument Global Identifier (FIGI) of a security"""
        ...

    @property
    def SEDOL(self) -> str:
        """The Stock Exchange Daily Official List (SEDOL) security identifier of a security"""
        ...

    @SEDOL.setter
    def SEDOL(self, value: str):
        """The Stock Exchange Daily Official List (SEDOL) security identifier of a security"""
        ...

    @property
    def ISIN(self) -> str:
        """The International Securities Identification Number (ISIN) of a security"""
        ...

    @ISIN.setter
    def ISIN(self, value: str):
        """The International Securities Identification Number (ISIN) of a security"""
        ...

    @staticmethod
    def FromCsvLine(line: str) -> QuantConnect.Securities.SecurityDefinition:
        """
        Parses a single line of CSV and converts it into an instance
        
        :param line: Line of CSV
        :returns: SecurityDefinition instance.
        """
        ...

    @staticmethod
    def Read(dataProvider: QuantConnect.Interfaces.IDataProvider, securitiesDefinitionKey: str) -> System.Collections.Generic.List[QuantConnect.Securities.SecurityDefinition]:
        """
        Reads data from the specified file and converts it to a list of SecurityDefinition
        
        :param dataProvider: Data provider used to obtain symbol mappings data
        :param securitiesDefinitionKey: Location to read the securities definition data from
        :returns: List of security definitions.
        """
        ...

    @staticmethod
    def TryRead(dataProvider: QuantConnect.Interfaces.IDataProvider, securitiesDatabaseKey: str, securityDefinitions: typing.Optional[System.Collections.Generic.List[QuantConnect.Securities.SecurityDefinition]]) -> typing.Union[bool, System.Collections.Generic.List[QuantConnect.Securities.SecurityDefinition]]:
        """
        Attempts to read data from the specified file and convert it into a list of SecurityDefinition
        
        :param dataProvider: Data provider used to obtain symbol mappings data
        :param securitiesDatabaseKey: Location of the file to read from
        :param securityDefinitions: Security definitions read
        :returns: true if data was read successfully, false otherwise.
        """
        ...


class ConstantBuyingPowerModel(QuantConnect.Securities.BuyingPowerModel):
    """
    Provides an implementation of IBuyingPowerModel that uses an absurdly low margin
    requirement to ensure all orders have sufficient margin provided the portfolio is not underwater.
    """

    def __init__(self, marginRequiredPerUnitInAccountCurrency: float) -> None:
        """
        Initializes a new instance of the ConstantBuyingPowerModel class
        
        :param marginRequiredPerUnitInAccountCurrency: The constant amount of margin required per single unit of an asset. Each unit is defined as a quantity of 1 and NOT based on the lot size.
        """
        ...

    def GetInitialMarginRequirement(self, parameters: QuantConnect.Securities.InitialMarginParameters) -> QuantConnect.Securities.InitialMargin:
        """
        The margin that must be held in order to increase the position by the provided quantity
        
        :param parameters: An object containing the security and quantity of shares
        :returns: The initial margin required for the provided security and quantity.
        """
        ...

    def GetMaintenanceMargin(self, parameters: QuantConnect.Securities.MaintenanceMarginParameters) -> QuantConnect.Securities.MaintenanceMargin:
        """
        Gets the margin currently allocated to the specified holding
        
        :param parameters: An object containing the security
        :returns: The maintenance margin required for the provided holdings quantity/cost/value.
        """
        ...

    def SetLeverage(self, security: QuantConnect.Securities.Security, leverage: float) -> None:
        """
        Sets the leverage for the applicable securities, i.e, equities
        
        :param leverage: The new leverage
        """
        ...


class CashBuyingPowerModel(QuantConnect.Securities.BuyingPowerModel):
    """Represents a buying power model for cash accounts"""

    def __init__(self) -> None:
        """Initializes a new instance of the CashBuyingPowerModel class"""
        ...

    def GetBuyingPower(self, parameters: QuantConnect.Securities.BuyingPowerParameters) -> QuantConnect.Securities.BuyingPower:
        """
        Gets the buying power available for a trade
        
        :param parameters: A parameters object containing the algorithm's portfolio, security, and order direction
        :returns: The buying power available for the trade.
        """
        ...

    def GetInitialMarginRequirement(self, parameters: QuantConnect.Securities.InitialMarginParameters) -> QuantConnect.Securities.InitialMargin:
        """
        The margin that must be held in order to increase the position by the provided quantity
        
        :param parameters: An object containing the security and quantity of shares
        """
        ...

    def GetLeverage(self, security: QuantConnect.Securities.Security) -> float:
        """
        Gets the current leverage of the security
        
        :param security: The security to get leverage for
        :returns: The current leverage in the security.
        """
        ...

    def GetMaximumOrderQuantityForDeltaBuyingPower(self, parameters: QuantConnect.Securities.GetMaximumOrderQuantityForDeltaBuyingPowerParameters) -> QuantConnect.Securities.GetMaximumOrderQuantityResult:
        """
        Get the maximum market order quantity to obtain a delta in the buying power used by a security.
        The deltas sign defines the position side to apply it to, positive long, negative short.
        
        :param parameters: An object containing the portfolio, the security and the delta buying power
        :returns: Returns the maximum allowed market order quantity and if zero, also the reason.
        """
        ...

    def GetMaximumOrderQuantityForTargetBuyingPower(self, parameters: QuantConnect.Securities.GetMaximumOrderQuantityForTargetBuyingPowerParameters) -> QuantConnect.Securities.GetMaximumOrderQuantityResult:
        """
        Get the maximum market order quantity to obtain a position with a given buying power percentage.
        Will not take into account free buying power.
        
        :param parameters: An object containing the portfolio, the security and the target signed buying power percentage
        :returns: Returns the maximum allowed market order quantity and if zero, also the reason.
        """
        ...

    def GetReservedBuyingPowerForPosition(self, parameters: QuantConnect.Securities.ReservedBuyingPowerForPositionParameters) -> QuantConnect.Securities.ReservedBuyingPowerForPosition:
        """
        Gets the amount of buying power reserved to maintain the specified position
        
        :param parameters: A parameters object containing the security
        :returns: The reserved buying power in account currency.
        """
        ...

    def HasSufficientBuyingPowerForOrder(self, parameters: QuantConnect.Securities.HasSufficientBuyingPowerForOrderParameters) -> QuantConnect.Securities.HasSufficientBuyingPowerForOrderResult:
        """
        Check if there is sufficient buying power to execute this order.
        
        :param parameters: An object containing the portfolio, the security and the order
        :returns: Returns buying power information for an order.
        """
        ...

    def SetLeverage(self, security: QuantConnect.Securities.Security, leverage: float) -> None:
        """
        Sets the leverage for the applicable securities, i.e, equities
        
        :param security: The security to set leverage for
        :param leverage: The new leverage
        """
        ...


class OptionFilterUniverse(QuantConnect.Securities.ContractSecurityFilterUniverse[QuantConnect_Securities_OptionFilterUniverse]):
    """Represents options symbols universe used in filtering."""

    @overload
    def __init__(self) -> None:
        """Constructs OptionFilterUniverse"""
        ...

    @overload
    def __init__(self, allSymbols: System.Collections.Generic.IEnumerable[QuantConnect.Symbol], underlying: QuantConnect.Data.BaseData) -> None:
        """Constructs OptionFilterUniverse"""
        ...

    def CallsOnly(self) -> QuantConnect.Securities.OptionFilterUniverse:
        """
        Sets universe of call options (if any) as a selection
        
        :returns: Universe with filter applied.
        """
        ...

    def IsStandard(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> bool:
        """
        Determine if the given Option contract symbol is standard
        
        This method is protected.
        
        :returns: True if standard.
        """
        ...

    def PutsOnly(self) -> QuantConnect.Securities.OptionFilterUniverse:
        """
        Sets universe of put options (if any) as a selection
        
        :returns: Universe with filter applied.
        """
        ...

    def Refresh(self, allSymbols: System.Collections.Generic.IEnumerable[QuantConnect.Symbol], underlying: QuantConnect.Data.BaseData, exchangeDateChange: bool = True) -> None:
        """
        Refreshes this option filter universe and allows specifying if the exchange date changed from last call
        
        :param allSymbols: All the options contract symbols
        :param underlying: The current underlying last data point
        :param exchangeDateChange: True if the exchange data has chanced since the last call or construction
        """
        ...

    def Strikes(self, minStrike: int, maxStrike: int) -> QuantConnect.Securities.OptionFilterUniverse:
        """
        Applies filter selecting options contracts based on a range of strikes in relative terms
        
        :param minStrike: The minimum strike relative to the underlying price, for example, -1 would filter out contracts further than 1 strike below market price
        :param maxStrike: The maximum strike relative to the underlying price, for example, +1 would filter out contracts further than 1 strike above market price
        :returns: Universe with filter applied.
        """
        ...


class OptionFilterUniverseEx(System.Object):
    """Extensions for Linq support"""

    @staticmethod
    def Select(universe: QuantConnect.Securities.OptionFilterUniverse, mapFunc: typing.Callable[[QuantConnect.Symbol], QuantConnect.Symbol]) -> QuantConnect.Securities.OptionFilterUniverse:
        """
        Maps universe
        
        :param universe: Universe to apply the filter too
        :param mapFunc: Symbol function to determine which Symbols are filtered
        :returns: Universe with filter applied.
        """
        ...

    @staticmethod
    def SelectMany(universe: QuantConnect.Securities.OptionFilterUniverse, mapFunc: typing.Callable[[QuantConnect.Symbol], System.Collections.Generic.IEnumerable[QuantConnect.Symbol]]) -> QuantConnect.Securities.OptionFilterUniverse:
        """
        Binds universe
        
        :param universe: Universe to apply the filter too
        :param mapFunc: Symbol function to determine which Symbols are filtered
        :returns: Universe with filter applied.
        """
        ...

    @staticmethod
    def Where(universe: QuantConnect.Securities.OptionFilterUniverse, predicate: typing.Callable[[QuantConnect.Symbol], bool]) -> QuantConnect.Securities.OptionFilterUniverse:
        """
        Filters universe
        
        :param universe: Universe to apply the filter too
        :param predicate: Bool function to determine which Symbol are filtered
        :returns: Universe with filter applied.
        """
        ...

    @staticmethod
    def WhereContains(universe: QuantConnect.Securities.OptionFilterUniverse, filterList: System.Collections.Generic.List[QuantConnect.Symbol]) -> QuantConnect.Securities.OptionFilterUniverse:
        """
        Updates universe to only contain the symbols in the list
        
        :param universe: Universe to apply the filter too
        :param filterList: List of Symbols to keep in the Universe
        :returns: Universe with filter applied.
        """
        ...


class FutureFilterUniverse(QuantConnect.Securities.ContractSecurityFilterUniverse[QuantConnect_Securities_FutureFilterUniverse]):
    """Represents futures symbols universe used in filtering."""

    def __init__(self, allSymbols: System.Collections.Generic.IEnumerable[QuantConnect.Symbol], underlying: QuantConnect.Data.BaseData) -> None:
        """Constructs FutureFilterUniverse"""
        ...

    def ExpirationCycle(self, months: typing.List[int]) -> QuantConnect.Securities.FutureFilterUniverse:
        """
        Applies filter selecting futures contracts based on expiration cycles. See FutureExpirationCycles for details
        
        :param months: Months to select contracts from
        :returns: Universe with filter applied.
        """
        ...

    def IsStandard(self, symbol: typing.Union[QuantConnect.Symbol, str]) -> bool:
        """
        Determine if the given Future contract symbol is standard
        
        This method is protected.
        
        :returns: True if contract is standard.
        """
        ...


class FutureFilterUniverseEx(System.Object):
    """Extensions for Linq support"""

    @staticmethod
    def Select(universe: QuantConnect.Securities.FutureFilterUniverse, mapFunc: typing.Callable[[QuantConnect.Symbol], QuantConnect.Symbol]) -> QuantConnect.Securities.FutureFilterUniverse:
        """
        Maps universe
        
        :param universe: Universe to apply the filter too
        :param mapFunc: Symbol function to determine which Symbols are filtered
        :returns: FutureFilterUniverse with filter applied.
        """
        ...

    @staticmethod
    def SelectMany(universe: QuantConnect.Securities.FutureFilterUniverse, mapFunc: typing.Callable[[QuantConnect.Symbol], System.Collections.Generic.IEnumerable[QuantConnect.Symbol]]) -> QuantConnect.Securities.FutureFilterUniverse:
        """
        Binds universe
        
        :param universe: Universe to apply the filter too
        :param mapFunc: Symbols function to determine which Symbols are filtered
        :returns: FutureFilterUniverse with filter applied.
        """
        ...

    @staticmethod
    def Where(universe: QuantConnect.Securities.FutureFilterUniverse, predicate: typing.Callable[[QuantConnect.Symbol], bool]) -> QuantConnect.Securities.FutureFilterUniverse:
        """
        Filters universe
        
        :param universe: Universe to apply the filter too
        :param predicate: Bool function to determine which Symbol are filtered
        :returns: FutureFilterUniverse with filter applied.
        """
        ...


class Futures(System.Object):
    """Futures static class contains shortcut definitions of major futures contracts available for trading"""

    class Grains(System.Object):
        """Grains and Oilseeds group"""

        BlackSeaCornFinanciallySettledPlatts: str = "BCF"
        """Black Sea Corn Financially Settled (Platts) Futures"""

        BlackSeaWheatFinanciallySettledPlatts: str = "BWF"
        """Black Sea Wheat Financially Settled (Platts) Futures"""

        SRWWheat: str = "ZW"
        """Chicago SRW Wheat Futures"""

        Wheat: str = ...
        """Default wheat contract is SRWWheat"""

        HRWWheat: str = "KE"
        """KC HRW Wheat Futures"""

        Corn: str = "ZC"
        """Corn Futures"""

        Soybeans: str = "ZS"
        """Soybeans Futures"""

        SoybeanMeal: str = "ZM"
        """Soybean Meal Futures"""

        SoybeanOil: str = "ZL"
        """Soybean Oil Futures"""

        Oats: str = "ZO"
        """Oats Futures"""

    class Currencies(System.Object):
        """Currencies group"""

        USD: str = "DX"
        """U.S. Dollar Index Futures"""

        GBP: str = "6B"
        """British Pound Futures"""

        CAD: str = "6C"
        """Canadian Dollar Futures"""

        JPY: str = "6J"
        """Japanese Yen Futures"""

        CHF: str = "6S"
        """Swiss Franc Futures"""

        EUR: str = "6E"
        """Euro FX Futures"""

        AUD: str = "6A"
        """Australian Dollar Futures"""

        NZD: str = "6N"
        """New Zealand Dollar Futures"""

        RUB: str = "6R"
        """Russian Ruble Futures"""

        BRL: str = "6L"
        """Brazillian Real Futures"""

        MXN: str = "6M"
        """Mexican Peso Futures"""

        ZAR: str = "6Z"
        """South African Rand Futures"""

        AUDCAD: str = "ACD"
        """Australian Dollar/Canadian Dollar Futures"""

        AUDJPY: str = "AJY"
        """Australian Dollar/Japanese Yen Futures"""

        AUDNZD: str = "ANE"
        """Australian Dollar/New Zealand Dollar Futures"""

        BTC: str = "BTC"
        """Bitcoin Futures"""

        CADJPY: str = "CJY"
        """Canadian Dollar/Japanese Yen Futures"""

        StandardSizeUSDOffshoreRMBCNH: str = "CNH"
        """Standard-Size USD/Offshore RMB (CNH) Futures"""

        EuroFXEmini: str = "E7"
        """E-mini Euro FX Futures"""

        EURAUD: str = "EAD"
        """Euro/Australian Dollar Futures"""

        EURCAD: str = "ECD"
        """Euro/Canadian Dollar Futures"""

        EURSEK: str = "ESK"
        """Euro/Swedish Krona Futures"""

        JapaneseYenEmini: str = "J7"
        """E-mini Japanese Yen Futures"""

        MicroEUR: str = "M6E"
        """Micro EUR/USD Futures"""

        MicroAUD: str = "M6A"
        """Micro AUD/USD Futures"""

        MicroGBP: str = "M6B"
        """Micro GBP/USD Futures"""

        MicroCADUSD: str = "MCD"
        """Micro CAD/USD Futures"""

        MicroJPY: str = "MJY"
        """Micro JPY/USD Futures"""

        MicroCHF: str = "MSF"
        """Micro CHF/USD Futures"""

        MicroUSDJPY: str = "M6J"
        """Micro USD/JPY Futures"""

        MicroINRUSD: str = "MIR"
        """Micro INR/USD Futures"""

        MicroCAD: str = "M6C"
        """Micro USD/CAD Futures"""

        MicroUSDCHF: str = "M6S"
        """Micro USD/CHF Futures"""

        MicroUSDCNH: str = "MNH"
        """Micro USD/CNH Futures"""

        MicroEther: str = "MET"
        """Micro Ether Futures"""

        MicroBTC: str = "MBT"
        """Micro Bitcoin Futures"""

        BTICMicroEther: str = "MRB"
        """BTIC on Micro Ether Futures"""

        BTICMicroBTC: str = "MIB"
        """BTIC on Micro Bitcoin Futures"""

    class Energies(System.Object):
        """Energies group"""

        PropaneNonLDHMontBelvieu: str = "1S"
        """Propane Non LDH Mont Belvieu (OPIS) BALMO Futures"""

        ArgusPropaneFarEastIndexBALMO: str = "22"
        """Argus Propane Far East Index BALMO Futures"""

        MiniEuropeanThreePointPercentFiveFuelOilBargesPlatts: str = "A0D"
        """Mini European 3.5% Fuel Oil Barges FOB Rdam (Platts) Futures"""

        MiniSingaporeFuelOil180CstPlatts: str = "A0F"
        """Mini Singapore Fuel Oil 180 cst (Platts) Futures"""

        GulfCoastULSDPlattsUpDownBALMO: str = "A1L"
        """Gulf Coast ULSD (Platts) Up-Down BALMO Futures"""

        GulfCoastJetPlattsUpDownBALMO: str = "A1M"
        """Gulf Coast Jet (Platts) Up-Down BALMO Futures"""

        PropaneNonLDHMontBelvieuOPIS: str = "A1R"
        """Propane Non-LDH Mont Belvieu (OPIS) Futures"""

        EuropeanPropaneCIFARAArgusBALMO: str = "A32"
        """European Propane CIF ARA (Argus) BALMO Futures"""

        PremiumUnleadedGasoline10ppmFOBMEDPlatts: str = "A3G"
        """Premium Unleaded Gasoline 10 ppm FOB MED (Platts) Futures"""

        ArgusPropaneFarEastIndex: str = "A7E"
        """Argus Propane Far East Index Futures"""

        GasolineEurobobOxyNWEBargesArgusCrackSpreadBALMO: str = "A7I"
        """Gasoline Euro-bob Oxy NWE Barges (Argus) Crack Spread BALMO Futures"""

        MontBelvieuNaturalGasolineOPIS: str = "A7Q"
        """Mont Belvieu Natural Gasoline (OPIS) Futures"""

        MontBelvieuNormalButaneOPISBALMO: str = "A8J"
        """Mont Belvieu Normal Butane (OPIS) BALMO Futures"""

        ConwayPropaneOPIS: str = "A8K"
        """Conway Propane (OPIS) Futures"""

        MontBelvieuLDHPropaneOPISBALMO: str = "A8O"
        """Mont Belvieu LDH Propane (OPIS) BALMO Futures"""

        ArgusPropaneFarEastIndexVsEuropeanPropaneCIFARAArgus: str = "A91"
        """Argus Propane Far East Index vs. European Propane CIF ARA (Argus) Futures"""

        ArgusPropaneSaudiAramco: str = "A9N"
        """Argus Propane (Saudi Aramco) Futures"""

        GroupThreeULSDPlattsVsNYHarborULSD: str = "AA6"
        """Group Three ULSD (Platts) vs. NY Harbor ULSD Futures"""

        GroupThreeSuboctaneGasolinePlattsVsRBOB: str = "AA8"
        """Group Three Sub-octane Gasoliine (Platts) vs. RBOB Futures"""

        SingaporeFuelOil180cstPlattsBALMO: str = "ABS"
        """Singapore Fuel Oil 180 cst (Platts) BALMO Futures"""

        SingaporeFuelOil380cstPlattsBALMO: str = "ABT"
        """Singapore Fuel Oil 380 cst (Platts) BALMO Futures"""

        MontBelvieuEthaneOPIS: str = "AC0"
        """Mont Belvieu Ethane (OPIS) Futures"""

        MontBelvieuNormalButaneOPIS: str = "AD0"
        """Mont Belvieu Normal Butane (OPIS) Futures"""

        BrentCrudeOilVsDubaiCrudeOilPlatts: str = "ADB"
        """Brent Crude Oil vs. Dubai Crude Oil (Platts) Futures"""

        ArgusLLSvsWTIArgusTradeMonth: str = "AE5"
        """Argus LLS vs. WTI (Argus) Trade Month Futures"""

        SingaporeGasoilPlattsVsLowSulphurGasoilFutures: str = "AGA"
        """Singapore Gasoil (Platts) vs. Low Sulphur Gasoil Futures"""

        LosAngelesCARBOBGasolineOPISvsRBOBGasoline: str = "AJL"
        """Los Angeles CARBOB Gasoline (OPIS) vs. RBOB Gasoline Futures"""

        LosAngelesJetOPISvsNYHarborULSD: str = "AJS"
        """Los Angeles Jet (OPIS) vs. NY Harbor ULSD Futures"""

        LosAngelesCARBDieselOPISvsNYHarborULSD: str = "AKL"
        """Los Angeles CARB Diesel (OPIS) vs. NY Harbor ULSD Futures"""

        EuropeanNaphthaPlattsBALMO: str = "AKZ"
        """European Naphtha (Platts) BALMO Futures"""

        EuropeanPropaneCIFARAArgus: str = "APS"
        """European Propane CIF ARA (Argus) Futures"""

        MontBelvieuNaturalGasolineOPISBALMO: str = "AR0"
        """Mont Belvieu Natural Gasoline (OPIS) BALMO Futures"""

        RBOBGasolineCrackSpread: str = "ARE"
        """RBOB Gasoline Crack Spread Futures"""

        GulfCoastHSFOPlattsBALMO: str = "AVZ"
        """Gulf Coast HSFO (Platts) BALMO Futures"""

        MarsArgusVsWTITradeMonth: str = "AYV"
        """Mars (Argus) vs. WTI Trade Month Futures"""

        MarsArgusVsWTIFinancial: str = "AYX"
        """Mars (Argus) vs. WTI Financial Futures"""

        EthanolT2FOBRdamIncludingDutyPlatts: str = "AZ1"
        """Ethanol T2 FOB Rdam Including Duty (Platts) Futures"""

        MontBelvieuLDHPropaneOPIS: str = "B0"
        """Mont Belvieu LDH Propane (OPIS) Futures"""

        GasolineEurobobOxyNWEBargesArgus: str = "B7H"
        """Gasoline Euro-bob Oxy NWE Barges (Argus) Futures"""

        WTIBrentFinancial: str = "BK"
        """WTI-Brent Financial Futures"""

        ThreePointFivePercentFuelOilBargesFOBRdamPlattsCrackSpread1000mt: str = "BOO"
        """3.5% Fuel Oil Barges FOB Rdam (Platts) Crack Spread (1000mt) Futures"""

        GasolineEurobobOxyNWEBargesArgusBALMO: str = "BR7"
        """Gasoline Euro-bob Oxy NWE Barges (Argus) BALMO Futures"""

        BrentLastDayFinancial: str = "BZ"
        """Brent Last Day Financial Futures"""

        CrudeOilWTI: str = "CL"
        """Crude Oil WTI Futures"""

        GulfCoastCBOBGasolineA2PlattsVsRBOBGasoline: str = "CRB"
        """Gulf Coast CBOB Gasoline A2 (Platts) vs. RBOB Gasoline Futures"""

        ClearbrookBakkenSweetCrudeOilMonthlyIndexNetEnergy: str = "CSW"
        """Clearbrook Bakken Sweet Crude Oil Monthly Index (Net Energy) Futures"""

        WTIFinancial: str = "CSX"
        """WTI Financial Futures"""

        ChicagoEthanolPlatts: str = "CU"
        """Chicago Ethaanol (Platts) Futures"""

        SingaporeMogas92UnleadedPlattsBrentCrackSpread: str = "D1N"
        """Singapore Mogas 92 Unleaded (Platts) Brent Crack Spread Futures"""

        DubaiCrudeOilPlattsFinancial: str = "DCB"
        """Dubai Crude Oil (Platts) Financial Futures"""

        JapanCnFNaphthaPlattsBALMO: str = "E6"
        """Japan C&F Naphtha (Platts) BALMO Futures"""

        Ethanol: str = "EH"
        """Ethanol Futures"""

        EuropeanNaphthaPlattsCrackSpread: str = "EN"
        """European Naphtha (Platts) Crack Spread Futures"""

        EuropeanPropaneCIFARAArgusVsNaphthaCargoesCIFNWEPlatts: str = "EPN"
        """European Propane CIF ARA (Argus) vs. Naphtha Cargoes CIF NWE (Platts) Futures"""

        SingaporeFuelOil380cstPlattsVsEuropeanThreePointFivePercentFuelOilBargesFOBRdamPlatts: str = "EVC"
        """Singapore Fuel Oil 380 cst (Platts) vs. European 3.5% Fuel Oil Barges FOB Rdam (Platts) Futures"""

        EastWestGasolineSpreadPlattsArgus: str = "EWG"
        """East-West Gasoline Spread (Platts-Argus) Futures"""

        EastWestNaphthaJapanCFvsCargoesCIFNWESpreadPlatts: str = "EWN"
        """East-West Naphtha: Japan C&F vs. Cargoes CIF NWE Spread (Platts) Futures"""

        RBOBGasolineVsEurobobOxyNWEBargesArgusThreeHundredFiftyThousandGallons: str = "EXR"
        """RBOB Gasoline vs. Euro-bob Oxy NWE Barges (Argus) (350,000 gallons) Futures"""

        ThreePointFivePercentFuelOilBargesFOBRdamPlattsCrackSpread: str = "FO"
        """3.5% Fuel Oil Barges FOB Rdam (Platts) Crack Spread Futures"""

        FreightRouteTC14Baltic: str = "FRC"
        """Freight Route TC14 (Baltic) Futures"""

        OnePercentFuelOilCargoesFOBNWEPlattsVsThreePointFivePercentFuelOilBargesFOBRdamPlatts: str = "FSS"
        """1% Fuel Oil Cargoes FOB NWE (Platts) vs. 3.5% Fuel Oil Barges FOB Rdam (Platts) Futures"""

        GulfCoastHSFOPlattsVsEuropeanThreePointFivePercentFuelOilBargesFOBRdamPlatts: str = "GCU"
        """Gulf Coast HSFO (Platts) vs. European 3.5% Fuel Oil Barges FOB Rdam (Platts) Futures"""

        WTIHoustonCrudeOil: str = "HCL"
        """WTI Houston Crude Oil Futures"""

        NaturalGasHenryHubLastDayFinancial: str = "HH"
        """Natural Gas (Henry Hub) Last-day Financial Futures"""

        HeatingOil: str = "HO"
        """Heating Oil Futures"""

        NaturalGasHenryHubPenultimateFinancial: str = "HP"
        """Natural Gas (Henry Hub) Penultimate Financial Futures"""

        WTIHoustonArgusVsWTITradeMonth: str = "HTT"
        """WTI Houston (Argus) vs. WTI Trade Month Futures"""

        Gasoline: str = "RB"
        """Gasoline RBOB Futures"""

        NaturalGas: str = "NG"
        """Natural Gas Futures"""

        BrentCrude: str = "B"
        """Brent Crude Futures"""

        LowSulfurGasoil: str = "G"
        """Low Sulfur Gasoil"""

        MicroCrudeOilWTI: str = "MCL"
        """Micro WTI Crude Oil Futures"""

        MicroSingaporeFOBMarineFuelZeroPointFivePercetPlatts: str = "S5O"
        """Micro Singapore FOB Marine Fuel 0.5% (Platts) Futures"""

        MicroGasoilZeroPointOnePercentBargesFOBARAPlatts: str = "M1B"
        """Micro Gasoil 0.1% Barges FOB ARA (Platts) Futures"""

        MicroEuropeanFOBRdamMarineFuelZeroPointFivePercentBargesPlatts: str = "R5O"
        """Micro European FOB Rdam Marine Fuel 0.5% Barges (Platts) Futures"""

        MicroEuropeanThreePointFivePercentOilBargesFOBRdamPlatts: str = "MEF"
        """Micro European 3.5% Fuel Oil Barges FOB Rdam (Platts) Futures"""

        MicroSingaporeFuelOil380CSTPlatts: str = "MAF"
        """Micro Singapore Fuel Oil 380CST (Platts) Futures"""

        MicroCoalAPIFivefobNewcastleArgusMcCloskey: str = "M5F"
        """Micro Coal (API 5) fob Newcastle (Argus/McCloskey) Futures"""

        MicroEuropeanThreePointFivePercentFuelOilCargoesFOBMedPlatts: str = "M35"
        """Micro European 3.5% Fuel Oil Cargoes FOB Med (Platts) Futures"""

    class Financials(System.Object):
        """Financials group"""

        Y30TreasuryBond: str = "ZB"
        """30Y U.S. Treasury Bond Futures"""

        Y10TreasuryNote: str = "ZN"
        """10Y U.S. Treasury Note Futures"""

        Y5TreasuryNote: str = "ZF"
        """5Y U.S. Treasury Note Futures"""

        Y2TreasuryNote: str = "ZT"
        """2Y U.S. Treasury Note Futures"""

        EuroDollar: str = "GE"
        """EuroDollar Futures"""

        FiveYearUSDMACSwap: str = "F1U"
        """5-Year USD MAC Swap Futures"""

        UltraUSTreasuryBond: str = "UB"
        """Ultra U.S. Treasury Bond Futures"""

        UltraTenYearUSTreasuryNote: str = "TN"
        """Ultra 10-Year U.S. Treasury Note Futures"""

        MicroY10TreasuryNote: str = "10Y"
        """Micro 10-Year Yield Futures"""

        MicroY30TreasuryBond: str = "30Y"
        """Micro 30-Year Yield Futures"""

        MicroY2TreasuryBond: str = "2YY"
        """Micro 2-Year Yield Futures"""

        MicroY5TreasuryBond: str = "5YY"
        """Micro 5-Year Yield Futures"""

    class Indices(System.Object):
        """Indices group"""

        SP500EMini: str = "ES"
        """E-mini S&P 500 Futures"""

        NASDAQ100EMini: str = "NQ"
        """E-mini NASDAQ 100 Futures"""

        Dow30EMini: str = "YM"
        """E-mini Dow Indu 30 Futures"""

        VIX: str = "VX"
        """CBOE Volatility Index Futures"""

        Russell2000EMini: str = "RTY"
        """E-mini Russell 2000 Futures"""

        Nikkei225Dollar: str = "NKD"
        """Nikkei-225 Dollar Futures"""

        BloombergCommodityIndex: str = "AW"
        """Bloomberg Commodity Index Futures"""

        NASDAQ100BiotechnologyEMini: str = "BIO"
        """E-mini Nasdaq-100 Biotechnology Index Futures"""

        FTSEEmergingEmini: str = "EI"
        """E-mini FTSE Emerging Index Futures"""

        SP400MidCapEmini: str = "EMD"
        """E-mini S&P MidCap 400 Futures"""

        SPGSCICommodity: str = "GD"
        """S&P-GSCI Commodity Index Futures"""

        USDDenominatedIbovespa: str = "IBV"
        """USD-Denominated Ibovespa Index Futures"""

        MSCITaiwanIndex: str = "TW"
        """USD-Denominated MSCI Taiwan Index Futures"""

        Nikkei225Yen: str = "NK"
        """Nikkei-225 Yen denominated Index Futures"""

        Nifty50: str = "NIFTY"
        """NSE Nifty50 Index Futures"""

        BankNifty: str = "BANKNIFTY"
        """NSE BankNifty Futures"""

        BseSensex: str = "SENSEX"
        """S&P BSE Sensex Index Futures"""

        HangSeng: str = "HSI"
        """Hang Seng Index"""

        MicroSP500EMini: str = "MES"
        """Micro E-mini S&P 500 Index Futures"""

        MicroNASDAQ100EMini: str = "MNQ"
        """Micro E-mini Nasdaq-100 Index Futures"""

        MicroRussell2000EMini: str = "M2K"
        """Micro E-mini Russell 2000 Index Futures"""

        MicroDow30EMini: str = "MYM"
        """Micro E-mini Dow Jones Industrial Average Index Futures"""

        MSCIEuropeNTR: str = "M1EU"
        """MSCI Europe Net Total Return (USD) Futures"""

        MSCIJapanNTR: str = "M1JP"
        """MSCI Japan Net Total Return Futures"""

        MSCIEmergingMarketsAsiaNTR: str = "M1MSA"
        """MSCI Emerging Markets Asia Net Total Return Futures"""

        MSCIEafeIndex: str = "MXEA"
        """MSCI EAFE Index Futures"""

        MSCIEmergingMarketsIndex: str = "MXEF"
        """MSCI Emerging Markets Index Futures"""

        MSCIUsaIndex: str = "MXUS"
        """MSCI USA Index Futures"""

    class Forestry(System.Object):
        """Forestry group"""

        RandomLengthLumber: str = "LBS"
        """Random Length Lumber Futures"""

    class Meats(System.Object):
        """Meats group"""

        LiveCattle: str = "LE"
        """Live Cattle Futures"""

        FeederCattle: str = "GF"
        """Feeder Cattle Futures"""

        LeanHogs: str = "HE"
        """Lean Hogs Futures"""

    class Metals(System.Object):
        """Metals group"""

        Gold: str = "GC"
        """Gold Futures"""

        Silver: str = "SI"
        """Silver Futures"""

        Platinum: str = "PL"
        """Platinum Futures"""

        Palladium: str = "PA"
        """Palladium Futures"""

        AluminumMWUSTransactionPremiumPlatts25MT: str = "AUP"
        """Aluminum MW U.S. Transaction Premium Platts (25MT) Futures"""

        AluminiumEuropeanPremiumDutyPaidMetalBulletin: str = "EDP"
        """Aluminium European Premium Duty-Paid (Metal Bulletin) Futures"""

        Copper: str = "HG"
        """Copper Futures"""

        USMidwestDomesticHotRolledCoilSteelCRUIndex: str = "HRC"
        """U.S. Midwest Domestic Hot-Rolled Coil Steel (CRU) Index Futures"""

        MicroGold: str = "MGC"
        """Micro Gold Futures"""

        MicroSilver: str = "SIL"
        """Micro Silver Futures"""

        MicroGoldTAS: str = "MGT"
        """Micro Gold TAS Futures"""

        MicroPalladium: str = "PAM"
        """Micro Palladium Futures"""

        MiniNYGold: str = "YG"
        """Mini Sized NY Gold Futures"""

        MiniNYSilver: str = "YI"
        """Mini Sized NY Silver Futures"""

        Gold100Oz: str = "ZG"
        """Gold 100 Oz Futures"""

        Silver5000Oz: str = "ZI"
        """Silver 5000 Oz Futures"""

    class Softs(System.Object):
        """Softs group"""

        Cotton2: str = "CT"
        """Cotton #2 Futures"""

        OrangeJuice: str = "OJ"
        """Orange Juice Futures"""

        Coffee: str = "KC"
        """Coffee C Arabica Futures"""

        Sugar11: str = "SB"
        """Sugar #11 Futures ICE"""

        Sugar11CME: str = "YO"
        """Sugar #11 Futures CME"""

        Cocoa: str = "CC"
        """Cocoa Futures"""

    class Dairy(System.Object):
        """Dairy group"""

        CashSettledButter: str = "CB"
        """Cash-settled Butter Futures"""

        CashSettledCheese: str = "CSC"
        """Cash-settled Cheese Futures"""

        ClassIIIMilk: str = "DC"
        """Class III Milk Futures"""

        DryWhey: str = "DY"
        """Dry Whey Futures"""

        ClassIVMilk: str = "GDK"
        """Class IV Milk Futures"""

        NonfatDryMilk: str = "GNF"
        """Non-fat Dry Milk Futures"""


class FutureExpirationCycles(System.Object):
    """Static class contains definitions of popular futures expiration cycles"""

    January: typing.List[int] = ...
    """January Cycle: Expirations in January, April, July, October (the first month of each quarter)"""

    February: typing.List[int] = ...
    """February Cycle: Expirations in February, May, August, November (second month)"""

    March: typing.List[int] = ...
    """March Cycle: Expirations in March, June, September, December (third month)"""

    AllYear: typing.List[int] = ...
    """All Year Cycle: Expirations in every month of the year"""

    GJMQVZ: typing.List[int] = ...
    """GJMQVZ Cycle"""

    GJKMNQVZ: typing.List[int] = ...
    """GJKMNQVZ Cycle"""

    HMUZ: typing.List[int] = ...
    """HMUZ Cycle"""

    HKNUZ: typing.List[int] = ...
    """HKNUZ Cycle"""

    HKNV: typing.List[int] = ...
    """HKNV Cycle"""

    HKNVZ: typing.List[int] = ...
    """HKNVZ Cycle"""

    FHKNUX: typing.List[int] = ...
    """FHKNUX Cycle"""

    FHJKQUVX: typing.List[int] = ...
    """FHJKQUVX Cycle"""

    HKNUVZ: typing.List[int] = ...
    """HKNUVZ Cycle"""

    FHKNUVZ: typing.List[int] = ...
    """FHKNQUVZ Cycle"""

    FHKNQUVZ: typing.List[int] = ...
    """FHKMQUVZ Cycle"""

    FHKNQUX: typing.List[int] = ...
    """FHKNQUX Cycle"""

    FGHJKMNQUVXZ: typing.List[int] = ...
    """FGHJKMNQUVXZ Cycle"""


class IndicatorVolatilityModel(typing.Generic[QuantConnect_Securities_IndicatorVolatilityModel_T], System.Object, QuantConnect.Securities.IVolatilityModel):
    """
    Provides an implementation of IVolatilityModel that uses an indicator
    to compute its value
    """

    @property
    def Volatility(self) -> float:
        """Gets the volatility of the security as a percentage"""
        ...

    @overload
    def __init__(self, indicator: QuantConnect.Indicators.IIndicator[QuantConnect_Securities_IndicatorVolatilityModel_T]) -> None:
        """
        Initializes a new instance of the IVolatilityModel using
        the specified . The 
        is assumed to but updated externally from this model, such as being registered
        into the consolidator system.
        
        :param indicator: The auto-updating indicator
        """
        ...

    @overload
    def __init__(self, indicator: QuantConnect.Indicators.IIndicator[QuantConnect_Securities_IndicatorVolatilityModel_T], indicatorUpdate: typing.Callable[[QuantConnect.Securities.Security, QuantConnect.Data.BaseData, QuantConnect.Indicators.IIndicator[QuantConnect_Securities_IndicatorVolatilityModel_T]], None]) -> None:
        """
        Initializes a new instance of the IVolatilityModel using
        the specified . The 
        is assumed to but updated externally from this model, such as being registered
        into the consolidator system.
        
        :param indicator: The auto-updating indicator
        :param indicatorUpdate: Function delegate used to update the indicator on each call to Update
        """
        ...

    def GetHistoryRequirements(self, security: QuantConnect.Securities.Security, utcTime: typing.Union[datetime.datetime, datetime.date]) -> System.Collections.Generic.IEnumerable[QuantConnect.Data.HistoryRequest]:
        """
        Returns history requirements for the volatility model expressed in the form of history request
        
        :param security: The security of the request
        :param utcTime: The date/time of the request
        :returns: History request object list, or empty if no requirements.
        """
        ...

    def Update(self, security: QuantConnect.Securities.Security, data: QuantConnect.Data.BaseData) -> None:
        """
        Updates this model using the new price information in
        the specified security instance
        
        :param security: The security to calculate volatility for
        :param data: The new piece of data for the security
        """
        ...


class StandardDeviationOfReturnsVolatilityModel(QuantConnect.Securities.Volatility.BaseVolatilityModel):
    """
    Provides an implementation of IVolatilityModel that computes the
    annualized sample standard deviation of daily returns as the volatility of the security
    """

    @property
    def Volatility(self) -> float:
        """Gets the volatility of the security as a percentage"""
        ...

    @overload
    def __init__(self, periods: int, resolution: typing.Optional[QuantConnect.Resolution] = None, updateFrequency: typing.Optional[datetime.timedelta] = None) -> None:
        """
        Initializes a new instance of the StandardDeviationOfReturnsVolatilityModel class
        
        :param periods: The max number of samples in the rolling window to be considered for calculating the standard deviation of returns
        :param resolution: Resolution of the price data inserted into the rolling window series to calculate standard deviation. Will be used as the default value for update frequency if a value is not provided for . This only has a material effect in live mode. For backtesting, this value does not cause any behavioral changes.
        :param updateFrequency: Frequency at which we insert new values into the rolling window for the standard deviation calculation
        """
        ...

    @overload
    def __init__(self, resolution: QuantConnect.Resolution, updateFrequency: typing.Optional[datetime.timedelta] = None) -> None:
        """
        Initializes a new instance of the StandardDeviationOfReturnsVolatilityModel class
        
        :param resolution: Resolution of the price data inserted into the rolling window series to calculate standard deviation. Will be used as the default value for update frequency if a value is not provided for . This only has a material effect in live mode. For backtesting, this value does not cause any behavioral changes.
        :param updateFrequency: Frequency at which we insert new values into the rolling window for the standard deviation calculation
        """
        ...

    def GetHistoryRequirements(self, security: QuantConnect.Securities.Security, utcTime: typing.Union[datetime.datetime, datetime.date]) -> System.Collections.Generic.IEnumerable[QuantConnect.Data.HistoryRequest]:
        """
        Returns history requirements for the volatility model expressed in the form of history request
        
        :param security: The security of the request
        :param utcTime: The date of the request
        :returns: History request object list, or empty if no requirements.
        """
        ...

    def Update(self, security: QuantConnect.Securities.Security, data: QuantConnect.Data.BaseData) -> None:
        """
        Updates this model using the new price information in
        the specified security instance
        
        :param security: The security to calculate volatility for
        :param data: Data to update the volatility model with
        """
        ...


class VolatilityModel(System.Object):
    """Provides access to a null implementation for IVolatilityModel"""

    Null: QuantConnect.Securities.IVolatilityModel = ...
    """
    Gets an instance of IVolatilityModel that will always
    return 0 for its volatility and does nothing during Update.
    """


class RelativeStandardDeviationVolatilityModel(QuantConnect.Securities.Volatility.BaseVolatilityModel):
    """
    Provides an implementation of IVolatilityModel that computes the
    relative standard deviation as the volatility of the security
    """

    @property
    def Volatility(self) -> float:
        """Gets the volatility of the security as a percentage"""
        ...

    def __init__(self, periodSpan: datetime.timedelta, periods: int) -> None:
        """
        Initializes a new instance of the RelativeStandardDeviationVolatilityModel class
        
        :param periodSpan: The time span representing one 'period' length
        :param periods: The nuber of 'period' lengths to wait until updating the value
        """
        ...

    def GetHistoryRequirements(self, security: QuantConnect.Securities.Security, utcTime: typing.Union[datetime.datetime, datetime.date]) -> System.Collections.Generic.IEnumerable[QuantConnect.Data.HistoryRequest]:
        """
        Returns history requirements for the volatility model expressed in the form of history request
        
        :param security: The security of the request
        :param utcTime: The date/time of the request
        :returns: History request object list, or empty if no requirements.
        """
        ...

    def Update(self, security: QuantConnect.Securities.Security, data: QuantConnect.Data.BaseData) -> None:
        """
        Updates this model using the new price information in
        the specified security instance
        
        :param security: The security to calculate volatility for
        """
        ...


class _EventContainer(typing.Generic[QuantConnect_Securities__EventContainer_Callable, QuantConnect_Securities__EventContainer_ReturnType]):
    """This class is used to provide accurate autocomplete on events and cannot be imported."""

    def __call__(self, *args: typing.Any, **kwargs: typing.Any) -> QuantConnect_Securities__EventContainer_ReturnType:
        """Fires the event."""
        ...

    def __iadd__(self, item: QuantConnect_Securities__EventContainer_Callable) -> None:
        """Registers an event handler."""
        ...

    def __isub__(self, item: QuantConnect_Securities__EventContainer_Callable) -> None:
        """Unregisters an event handler."""
        ...


