import click
import json
from convisoappsec.sast.sastbox import SASTBox
from convisoappsec.flowcli.context import pass_flow_context
from convisoappsec.common.box import ContainerWrapper
from convisoappsec.logger import LOGGER
from convisoappsec.flowcli.companies.ls import Companies
from convisoappsec.flowcli.findings import import_sarif


class VulnCloser:
    @staticmethod
    @pass_flow_context
    @click.pass_context
    def verify_issues_to_close(context, flow_context, conviso_api, company_id, asset_id, scan_type):
        """ Method that checks if any vulnerability can be closed on the conviso platform """

        # verify if vulnerability auto close flag is enable to continue with auto close vulnerabilities on cp
        if not context.params['vulnerability_auto_close']:
            return

        if company_id is None:
            companies = Companies()
            company = companies.ls(flow_context, company_id=company_id)
            company_id = company[0]['id']

        statuses = ['CREATED', 'IDENTIFIED', 'IN_PROGRESS', 'AWAITING_VALIDATION', 'FIX_ACCEPTED']

        # at this point a full scan is performed to get all issues hashes from all vulnerabilities on the code base
        if scan_type == 'sca':
            return

            issues_from_cp = conviso_api.issues.auto_close_vulnerabilities(company_id, asset_id, statuses, 'SCA_FINDING')
            items_with_fix_accepted = [item for item in issues_from_cp if item['status'] == 'FIX_ACCEPTED']
            issues_from_cp = [item for item in issues_from_cp if item['status'] != 'FIX_ACCEPTED']

            if len(issues_from_cp) == 0:
                VulnCloser.colored_output("No vulnerabilities was founded on conviso platform.", bold=True)
                return

            hash_issues = VulnCloser.execute_full_sca_scan()
        else:
            VulnCloser.colored_output("[*] Verifying if any vulnerability was fixed ...", color='blue', bold=True)
            issues_from_cp = conviso_api.issues.auto_close_vulnerabilities(company_id, asset_id, statuses, 'SAST_FINDING')
            items_with_fix_accepted = [item for item in issues_from_cp if item['status'] == 'FIX_ACCEPTED']
            issues_from_cp = [item for item in issues_from_cp if item['status'] != 'FIX_ACCEPTED']

            if len(issues_from_cp) == 0:
                VulnCloser.colored_output("No vulnerabilities was founded on conviso platform!", bold=True)
                return

            hash_issues = VulnCloser.execute_full_sast_scan()

        if len(hash_issues) == 0:
            VulnCloser.colored_output("No vulnerabilities was founded congrats !!!", color='green', bold=True)
            return

        # set is a collection which is unordered, unchangeable*, and unindexed. No duplicate members.
        set_of_hash_issues = set(hash_issues)

        # this is where the magic happens, comparing hash data from cp and the full scan above.
        # if the full scan has fewer vulnerabilities than has on the cp, the extras vulnerabilities are solved on cp.
        differences = [
            {'id': item['id'], 'originalIssueIdFromTool': item['originalIssueIdFromTool']}
            for item in issues_from_cp if item['originalIssueIdFromTool'] not in set_of_hash_issues
        ]

        # this is where we filter the issues to validate if the vulnerability was adding back for some reason
        # in this case we see if the issue with status == fix_accepted is on most recent full scan, if this happens
        # we will reopen that issue without creating a new one. =)
        issues_to_reopen = [
            {'id': item['id'], 'originalIssueIdFromTool': item['originalIssueIdFromTool']}
            for item in items_with_fix_accepted if item['originalIssueIdFromTool'] in set_of_hash_issues
        ]

        if issues_to_reopen:
            for issue in issues_to_reopen:
                issue_id = issue['id']
                status = 'CREATED'

                conviso_api.issues.update_issue_status(issue_id=issue_id, status=status)

        if len(differences) == 0:
            VulnCloser.colored_output("No vulnerabilities was fixed ...", bold=True)
            return

        VulnCloser.colored_output(
            "Fixing {issues} vulnerabilities on conviso platform ...".format(issues=len(differences)),
            bold=True
        )

        for issue in differences:
            issue_id = issue['id']
            status = 'FIX_ACCEPTED'

            conviso_api.issues.update_issue_status(issue_id=issue_id, status=status)

    @staticmethod
    @click.pass_context
    @pass_flow_context
    def execute_full_sast_scan(flow_context, context):
        """ This performs a full sast scan execution to verify hash issues on conviso platform """

        sastbox_registry = context.params['sastbox_registry']
        sastbox_repository_name = context.params['sastbox_repository_name']
        sastbox_tag = context.params['sastbox_tag']
        repository_dir = context.params['repository_dir']

        sastbox = SASTBox(
            registry=sastbox_registry,
            repository_name=sastbox_repository_name,
            tag=sastbox_tag,
        )

        reports = sastbox.run_scan_diff(
            repository_dir, None, None
        )

        results_filepaths = [str(r) for r in reports]
        file_path = results_filepaths[0]
        converted_results = [import_sarif.entrypoint.perform_command(flow_context, file_path, internal=True)]

        hash_issues = []

        for report in converted_results:
            try:
                with open(report, 'r') as report_file:
                    report_content = json.load(report_file)
                    issues = report_content.get("issues", [])
                    hash_issues.extend(issue.get("hash_issue") for issue in issues)

            except (FileNotFoundError, json.JSONDecodeError) as e:
                print(f"Error processing {file_path}: {e}")

        return hash_issues

    @staticmethod
    @click.pass_context
    @pass_flow_context
    def execute_full_sca_scan(flow_context, context):
        """ This performs a full sca scan execution to verify hash issues on conviso platform """
        conviso_rest_api = flow_context.create_conviso_rest_api_client()
        token = conviso_rest_api.docker_registry.get_sast_token()
        scanner_timeout = context.params['scanner_timeout']
        sca_scanners = VulnCloser.sca_scanners()

        scabox = ContainerWrapper(token=token, containers_map=sca_scanners, logger=LOGGER, timeout=scanner_timeout)
        scabox.run()

        results_filepaths = []
        for unit in scabox.scanners:
            file_path = unit.results
            if file_path:
                results_filepaths.append(file_path)

        hash_issues = []

        for report_path in results_filepaths:
            try:
                with open(report_path, 'r') as report_file:
                    report_content = json.load(report_file)
                    issues = report_content.get("issues", [])
                    hash_issues.extend(issue.get("hash_issue") for issue in issues)

            except (FileNotFoundError, json.JSONDecodeError) as e:
                print(f"Error processing {report_path}: {e}")

        return hash_issues

    @staticmethod
    @click.pass_context
    def sca_scanners(context):
        required_codebase_path = '/code'
        dandelion_image_name = 'scabox-dandelion'
        ramphastos_image_name = 'scabox-ramphastos'

        repository_dir = context.params['repository_dir']

        scanners = {
            dandelion_image_name: {
                'repository_name': dandelion_image_name,
                'tag': 'latest',
                'command': [
                    '-c', required_codebase_path,
                    '-f', 'json',
                    '-o', '/{}.json'.format(dandelion_image_name)
                ],
                'repository_dir': repository_dir
            },
            ramphastos_image_name: {
                'repository_name': ramphastos_image_name,
                'tag': 'latest',
                'command': [
                    '-c', required_codebase_path,
                    '-f', 'json',
                    '-o', '/{}.json'.format(ramphastos_image_name)
                ],
                'repository_dir': repository_dir
            },
        }

        return scanners

    @staticmethod
    def perform_full_scan():
        """ Method to perform a full scan to verify if any vulnerability could be close. """
        return NotImplemented

    @staticmethod
    def colored_output(msg, color=None, bold=False, new_line=True, to_stderr=False):
        """
        Outputs colored text using click.echo().

        Args:
            msg (str): The message to be printed.
            color (str): The color of the text. Options: 'red', 'green', 'yellow', 'blue', 'magenta', 'cyan', 'white'.
            bold (bool): Whether the text should be bold.
            new_line (bool): Whether to add a newline after the message.
            to_stderr (bool): Whether to print to stderr.

        Returns:
            None
        """

        COLORS = {
            'red': "\033[31m",
            'green': "\033[32m",
            'yellow': "\033[33m",
            'blue': "\033[34m",
            'magenta': "\033[35m",
            'cyan': "\033[36m",
            'white': "\033[37m"
        }
        RESET = "\033[0m"
        BOLD = "\033[1m" if bold else ""

        # Construct the colored message
        colored_msg = ''
        if color in COLORS:
            colored_msg += COLORS[color]
        if bold:
            colored_msg += BOLD
        colored_msg += msg
        colored_msg += RESET

        # Output the message using click.echo()
        click.echo(colored_msg, nl=new_line, err=to_stderr)
