import click
import json
from convisoappsec.flowcli import help_option
from convisoappsec.flowcli.context import pass_flow_context
from convisoappsec.sast.sastbox import SASTBox
from convisoappsec.flowcli.common import (asset_id_option, project_code_option)
from convisoappsec.flowcli.findings import import_sarif
from convisoappsec.flowcli.requirements_verifier import RequirementsVerifier
from convisoappsec.flow import GitAdapter


@click.command()
@project_code_option(required=False)
@asset_id_option(required=False)
@click.option(
    "-s",
    "--start-commit",
    required=False,
    help="If no value is set so the empty tree hash commit is used.",
)
@click.option(
    "-e",
    "--end-commit",
    required=False,
    help="""If no value is set so the HEAD commit
    from the current branch is used""",
)
@click.option(
    "-r",
    "--repository-dir",
    default=".",
    show_default=True,
    type=click.Path(
        exists=True,
        resolve_path=True,
    ),
    required=False,
    help="The source code repository directory.",
)
@click.option(
    "--fail-on-severity-threshold",
    required=False,
    help="If the threshold of the informed severity and higher has reach, then the command will fail after send the results to AppSec Flow.\n \
    The severity levels are: UNDEFINED, INFO, LOW, MEDIUM, HIGH, CRITICAL.",
    type=click.Tuple([str, int]),
    default=(None, None),
)
@click.option(
    "--fail-on-threshold",
    required=False,
    help="If the threshold has reach then the command will fail after send the result to AppSec Flow",
    type=int,
    default=False,
)
@click.option(
    "--send-to-flow/--no-send-to-flow",
    default=True,
    show_default=True,
    required=False,
    hidden=True,
    help="""Enable or disable the ability of send analysis result
    reports to flow. When --send-to-flow option is set the --project-code
    option is required""",
)
@click.option(
    "--deploy-id",
    default=None,
    required=False,
    hidden=True,
    envvar=("CONVISO_DEPLOY_ID", "FLOW_DEPLOY_ID")
)
@click.option(
    "--sastbox-registry",
    default="",
    required=False,
    hidden=True,
    envvar=("CONVISO_SASTBOX_REGISTRY", "FLOW_SASTBOX_REGISTRY"),
)
@click.option(
    "--sastbox-repository-name",
    default="",
    required=False,
    hidden=True,
    envvar=("CONVISO_SASTBOX_REPOSITORY_NAME", "FLOW_SASTBOX_REPOSITORY_NAME"),
)
@click.option(
    "--sastbox-tag",
    default=SASTBox.DEFAULT_TAG,
    required=False,
    hidden=True,
    envvar=("CONVISO_SASTBOX_TAG", "FLOW_SASTBOX_TAG"),
)
@click.option(
    "--sastbox-skip-login/--sastbox-no-skip-login",
    default=False,
    required=False,
    hidden=True,
    envvar=("CONVISO_SASTBOX_SKIP_LOGIN", "FLOW_SASTBOX_SKIP_LOGIN"),
)
@click.option(
    '--experimental',
    default=False,
    is_flag=True,
    hidden=True,
    help="Enable experimental features.",
)
@click.option(
    "--company-id",
    required=False,
    envvar=("CONVISO_COMPANY_ID", "FLOW_COMPANY_ID"),
    help="Company ID on Conviso Platform",
)
@click.option(
    '--asset-name',
    required=False,
    envvar=("CONVISO_ASSET_NAME", "FLOW_ASSET_NAME"),
    help="Provides a asset name.",
)
@click.option(
    '--vulnerability-auto-close',
    default=False,
    is_flag=True,
    hidden=True,
    help="Enable auto fixing vulnerabilities on cp.",
)
@click.option(
    '--from-ast',
    default=False,
    is_flag=True,
    hidden=True,
    help="Internal use only.",
)
@help_option
@pass_flow_context
@click.pass_context
def run(context,
        flow_context,
        project_code,
        asset_id,
        company_id,
        end_commit,
        start_commit,
        repository_dir,
        send_to_flow,
        deploy_id,
        sastbox_registry,
        sastbox_repository_name,
        sastbox_tag,
        sastbox_skip_login,
        fail_on_threshold,
        fail_on_severity_threshold,
        experimental,
        asset_name,
        vulnerability_auto_close,
        from_ast):

    context.params['company_id'] = company_id if company_id is not None else None

    if not from_ast:
        prepared_context = RequirementsVerifier.prepare_context(context)

        params_to_copy = [
            'project_code', 'asset_id', 'start_commit', 'end_commit',
            'repository_dir', 'send_to_flow', 'deploy_id', 'sastbox_registry',
            'sastbox_repository_name', 'sastbox_tag', 'sastbox_skip_login',
            'experimental', 'asset_name', 'vulnerability_auto_close', 'company_id'
        ]

        for param_name in params_to_copy:
            context.params[param_name] = (
                    locals()[param_name] or prepared_context.params[param_name]
            )

    conviso_api = flow_context.create_conviso_api_client_beta()
    company_id = context.params['company_id']
    asset_id = context.params['asset_id']
    statuses = ['CREATED', 'IDENTIFIED', 'IN_PROGRESS', 'AWAITING_VALIDATION', 'FIX_ACCEPTED']

    page = 1
    merged_issues = []

    while True:
        sast_issues_from_cp = conviso_api.issues.auto_close_vulnerabilities(
            company_id, asset_id, statuses, 'SAST_FINDING', page
        )

        total_pages = sast_issues_from_cp['metadata']['totalPages']
        sast_issues_collection = sast_issues_from_cp['collection']
        sast_issues_collection = [item for item in sast_issues_collection if item['scanSource'] == 'conviso_scanner']

        merged_issues.extend(sast_issues_collection)

        if total_pages == page:
            break
        else:
            page += 1

    items_with_fix_accepted = [item for item in merged_issues if item['status'] == 'FIX_ACCEPTED']
    issues_without_fix_accepted = [item for item in merged_issues if item['status'] != 'FIX_ACCEPTED']

    if len(sast_issues_from_cp) == 0:
        log_func("No vulnerabilities was founded on conviso platform!")
        return

    hash_issues = perform_sastbox_scan(
        sastbox_registry,
        sastbox_repository_name,
        sastbox_tag,
        sastbox_skip_login,
        repository_dir,
        end_commit,
        start_commit,
        log_func,
    )

    if len(hash_issues) == 0:
        log_func("No vulnerabilities was founded congrats !!!")
        return

    set_of_hash_issues = set(hash_issues)

    # this is where the magic happens, comparing hash data from cp and the full scan above.
    # if the full scan has fewer vulnerabilities than has on the cp, the extras vulnerabilities are solved on cp.
    differences = [
        {'id': item['id'], 'originalIssueIdFromTool': item['originalIssueIdFromTool']}
        for item in issues_without_fix_accepted if item['originalIssueIdFromTool'] not in set_of_hash_issues
    ]

    # this is where we filter the issues to validate if the vulnerability was adding back for some reason
    # in this case we see if the issue with status == fix_accepted is on most recent full scan, if this happens
    # we will reopen that issue without creating a new one. =)
    issues_to_reopen = [
        {'id': item['id'], 'originalIssueIdFromTool': item['originalIssueIdFromTool']}
        for item in items_with_fix_accepted if item['originalIssueIdFromTool'] in set_of_hash_issues
    ]

    if issues_to_reopen:
        for issue in issues_to_reopen:
            issue_id = issue['id']
            status = 'CREATED'

            conviso_api.issues.update_issue_status(issue_id=issue_id, status=status)

    if len(differences) == 0:
        log_func("No vulnerabilities was fixed yet ...")
        return

    log_func("Fixing {issues} vulnerabilities on conviso platform ...".format(issues=len(differences)))

    for issue in differences:
        issue_id = issue['id']
        status = 'FIX_ACCEPTED'

        conviso_api.issues.update_issue_status(issue_id=issue_id, status=status)


@pass_flow_context
def perform_sastbox_scan(
        flow_context,
        sastbox_registry,
        sastbox_repository_name,
        sastbox_tag,
        sastbox_skip_login,
        repository_dir,
        end_commit,
        start_commit,
        logger,
):
    sastbox = SASTBox(
        registry=sastbox_registry,
        repository_name=sastbox_repository_name,
        tag=sastbox_tag,
    )

    conviso_rest_api = flow_context.create_conviso_rest_api_client()
    token = conviso_rest_api.docker_registry.get_sast_token()
    git_adapter = GitAdapter(repository_dir)
    end_commit = git_adapter.head_commit
    start_commit = git_adapter.empty_repository_tree_commit

    reports = sastbox.run_scan_diff(
        repository_dir, end_commit, start_commit, token=token
    )

    results_filepaths = [str(r) for r in reports]
    file_path = results_filepaths[0]
    converted_results = [import_sarif.entrypoint.perform_command(flow_context, file_path, internal = True)]

    hash_issues = []

    for report in converted_results:
        try:
            with open(report, 'r') as report_file:
                report_content = json.load(report_file)
                issues = report_content.get("issues", [])
                hash_issues.extend(issue.get("hash_issue") for issue in issues)

        except (FileNotFoundError, json.JSONDecodeError) as e:
            print(f"Error processing {file_path}: {e}")

    return hash_issues


def log_func(msg, new_line=True):
    click.echo(click.style(msg, bold=True, fg='blue'), nl=new_line, err=True)
