"""Templates for generating proxy project files."""


def get_readme_template(port: int, master_key: str) -> str:
    """Get README.md template content.

    Args:
        port: Port number for the proxy
        master_key: Master key for authentication

    Returns:
        README content
    """
    return f"""# LiteLLM Proxy Project

This project was generated by llm-anygate-cli to set up a LiteLLM proxy server.

## Configuration

- **Port**: {port}
- **Master Key**: {master_key} (change for production!)
- **Config File**: config.yaml

## Setup

1. Install LiteLLM with proxy support:
   ```bash
   pip install 'litellm[proxy]'
   ```

2. Set up environment variables:
   - Copy `env.example` to `.env`
   - Add your API keys to `.env`

3. Start the proxy server:
   ```bash
   llm-anygate-cli start
   ```

## Usage

The proxy server provides an OpenAI-compatible API at:
```
http://localhost:{port}
```

### Example with curl:
```bash
curl http://localhost:{port}/v1/models \\
  -H "Authorization: Bearer {master_key}"
```

### Example with OpenAI Python client:
```python
from openai import OpenAI

client = OpenAI(
    base_url="http://localhost:{port}/v1",
    api_key="{master_key}"
)

response = client.chat.completions.create(
    model="gpt-5-2025-08-07",  # or any model from config.yaml
    messages=[{{"role": "user", "content": "Hello!"}}]
)
```

## Models

Check `config.yaml` for available models and their configurations.

## Security

**Important**: The default master key `{master_key}` is not secure. 
For production use, generate a secure key and update:
1. The `master_key` in `config.yaml`
2. The `LITELLM_MASTER_KEY` in your `.env` file

## Troubleshooting

- **Port already in use**: Use `--port` option with `llm-anygate-cli start` or update anygate.yaml
- **Authentication errors**: Check your API keys in .env file
- **Model not found**: Verify model names in config.yaml

## Documentation

- [LiteLLM Documentation](https://docs.litellm.ai/)
- [LiteLLM Proxy Guide](https://docs.litellm.ai/docs/simple_proxy)
"""


def get_env_template(master_key: str) -> str:
    """Get env.example file content.

    Args:
        master_key: Default master key

    Returns:
        env.example template content
    """
    return f"""# LiteLLM Proxy Environment Variables
# Copy this file to .env and set your actual API keys

# Primary OpenAI API key (required for yunwu.ai and other OpenAI-compatible endpoints)
OPENAI_API_KEY=your-actual-openai-api-key-here

# OpenAI API base URL (for custom endpoints or proxies)
# Common values: https://api.openai.com/v1, https://yunwu.ai/v1
OPENAI_BASE_URL=https://api.openai.com/v1

# Master key for LiteLLM proxy authentication
# This overrides the master_key setting in config.yaml
LITELLM_MASTER_KEY={master_key}

# Optional: Preferred port (can be overridden by command line)
LITELLM_PREFER_PORT=4567

# Other provider API keys (uncomment if needed)
# ANTHROPIC_API_KEY=your-anthropic-api-key-here
# GEMINI_API_KEY=your-gemini-api-key-here

# Azure OpenAI (if using Azure instead of standard OpenAI)
# AZURE_API_KEY=your-azure-api-key-here
# AZURE_API_BASE=https://your-resource.openai.azure.com
# AZURE_API_VERSION=2024-02-01

# Optional LiteLLM settings
# LITELLM_LOG_LEVEL=INFO
# LITELLM_TELEMETRY=false
"""


def get_shell_script_template(port: int) -> str:
    """Get shell script template for Unix/macOS.

    Args:
        port: Port number for the proxy

    Returns:
        Shell script content
    """
    return f"""#!/bin/bash

# Start LiteLLM Proxy Server
# This script starts the LiteLLM proxy with the configuration in config.yaml

echo "Starting LiteLLM Proxy Server..."
echo "Port: {port}"
echo "Config: config.yaml"
echo ""

# Check if .env file exists and load it
if [ -f .env ]; then
    echo "Loading environment variables from .env..."
    export $(cat .env | grep -v '^#' | xargs)
fi

# Check if litellm is installed
if ! command -v litellm &> /dev/null; then
    echo "Error: litellm is not installed."
    echo "Please run: pip install 'litellm[proxy]'"
    exit 1
fi

# Start the proxy server
echo "Starting proxy server..."
litellm --config config.yaml --port {port}
"""


def get_powershell_script_template(port: int) -> str:
    """Get PowerShell script template for Windows.

    Args:
        port: Port number for the proxy

    Returns:
        PowerShell script content
    """
    return f"""# Start LiteLLM Proxy Server (Windows PowerShell)
# This script starts the LiteLLM proxy with the configuration in config.yaml

Write-Host "Starting LiteLLM Proxy Server..." -ForegroundColor Green
Write-Host "Port: {port}"
Write-Host "Config: config.yaml"
Write-Host ""

# Check if .env file exists and load it
if (Test-Path .env) {{
    Write-Host "Loading environment variables from .env..." -ForegroundColor Yellow
    Get-Content .env | ForEach-Object {{
        if ($_ -notmatch '^#' -and $_ -match '=') {{
            $parts = $_ -split '=', 2
            $name = $parts[0].Trim()
            $value = $parts[1].Trim()
            [System.Environment]::SetEnvironmentVariable($name, $value, [System.EnvironmentVariableTarget]::Process)
        }}
    }}
}}

# Check if litellm is installed
try {{
    $null = litellm --version
}} catch {{
    Write-Host "Error: litellm is not installed." -ForegroundColor Red
    Write-Host "Please run: pip install 'litellm[proxy]'" -ForegroundColor Yellow
    exit 1
}}

# Start the proxy server
Write-Host "Starting proxy server..." -ForegroundColor Green
litellm --config config.yaml --port {port}
"""
