import importlib
import pkgutil

import pytest

import examples


def discover_modules(package):
    """Discover all submodules in a package."""
    package_path = package.__path__[0]
    for _, name, is_pkg in pkgutil.iter_modules([package_path]):
        if not is_pkg:
            yield name


@pytest.mark.parametrize("module_name", discover_modules(examples))
def test_dynamic_import(module_name):
    """Dynamically imports modules from 'examples' to catch static/syntax errors."""
    try:
        # Dynamically import the module
        module = importlib.import_module(f"examples.{module_name}")
        assert module  # Simple check to see if the module is not None
    except Exception as e:
        pytest.fail(f"Failed to import {module_name} due to: {e}")


@pytest.mark.parametrize(
    "module",
    [name for _, name, _ in pkgutil.iter_modules(["..examples"])],
)
def test_import_all_from_examples(module):
    """
    Test to dynamically import all modules in 'examples'
    package to catch static/syntax errors.
    """
    package_name = "examples"
    full_module_name = f"{package_name}.{module}"
    try:
        importlib.import_module(full_module_name)
        assert True  # Import succeeded
    except Exception as e:
        pytest.fail(f"Failed to import {full_module_name}: {e}")
