import os
import pandas as pd
import subprocess
import shutil

from bigecyhmm.visualisation import compute_relative_abundance_per_tax_id, read_esmecata_proteome_file, compute_bigecyhmm_functions_abundance, \
                                    compute_bigecyhmm_functions_occurrence, create_visualisation, compute_abundance_per_tax_rank
from bigecyhmm.utils import read_measures_file


def test_compute_relative_abundance_per_tax_id():
    sample_abundance = {'sample_1': {'org_1': 100, 'org_2': 100, 'org_3': 0},
                        'sample_2': {'org_1': 0, 'org_2': 200, 'org_3': 600},
                        'sample_3': {'org_1': 0, 'org_2': 120, 'org_3': 400}}

    sample_tot_abundance = {'sample_1': 200,
                  'sample_2': 800,
                  'sample_3': 520}

    observation_names_tax_id_names = {'org_1': 'tax_id_name_1', 'org_2': 'tax_id_name_2', 'org_3': 'tax_id_name_2'}

    abundance_data = compute_relative_abundance_per_tax_id(sample_abundance, sample_tot_abundance, observation_names_tax_id_names)
    expected_abundance_data = {'sample_1': {'tax_id_name_1': 0.5, 'tax_id_name_2': 0.5},
                               'sample_2': {'tax_id_name_1': 0, 'tax_id_name_2': 1},
                               'sample_3': {'tax_id_name_1': 0, 'tax_id_name_2': 1}}

    for sample in expected_abundance_data:
        for tax_id_name in expected_abundance_data[sample]:
            assert expected_abundance_data[sample][tax_id_name] == abundance_data[sample][tax_id_name]


def test_compute_relative_abundance_per_tax_id_file():
    proteome_tax_id_file = os.path.join('input_data', 'esmecata_output_folder', '0_proteomes',  'proteome_tax_id.tsv')
    abundance_file_path = os.path.join('input_data', 'proteome_tax_id_abundance.tsv')

    sample_abundance, sample_tot_abundance = read_measures_file(abundance_file_path)
    observation_names_tax_id_names, observation_names_tax_ranks = read_esmecata_proteome_file(proteome_tax_id_file)
    data_abundance_taxon_sample, sample_abundance_tax_rank, data_abundance_organism_sample = compute_abundance_per_tax_rank(sample_abundance, observation_names_tax_ranks, sample_tot_abundance)

    expected_abundance_data = {'sample_1': {'genus': 100, 'family': 100, 'order': 0}, 'sample_2': {'genus': 0, 'family': 200, 'order': 600}, 'sample_3': {'genus': 0, 'family': 120, 'order': 400}}

    for sample in expected_abundance_data:
        for tax_rank in expected_abundance_data[sample]:
            assert expected_abundance_data[sample][tax_rank] == sample_abundance_tax_rank[sample][tax_rank]


def test_tax_rank_relative_abundance_file():
    proteome_tax_id_file = os.path.join('input_data', 'esmecata_output_folder', '0_proteomes',  'proteome_tax_id.tsv')
    abundance_file_path = os.path.join('input_data', 'proteome_tax_id_abundance.tsv')

    sample_abundance, sample_tot_abundance = read_measures_file(abundance_file_path)
    observation_names_tax_id_names, observation_names_tax_ranks = read_esmecata_proteome_file(proteome_tax_id_file)
    abundance_data = compute_relative_abundance_per_tax_id(sample_abundance, sample_tot_abundance, observation_names_tax_id_names)

    expected_abundance_data = {'sample_1': {'tax_id_name_1': 0.5, 'tax_id_name_2': 0.5},
                               'sample_2': {'tax_id_name_1': 0, 'tax_id_name_2': 1},
                               'sample_3': {'tax_id_name_1': 0, 'tax_id_name_2': 1}}

    for sample in expected_abundance_data:
        for tax_id_name in expected_abundance_data[sample]:
            assert expected_abundance_data[sample][tax_id_name] == abundance_data[sample][tax_id_name]


def test_compute_bigecyhmm_functions_occurrence_functions():
    bigecyhmm_cycle_file = os.path.join('input_data', 'bigecyhmm_output_folder', 'function_presence.tsv')

    function_occurrence_organisms, all_studied_organisms = compute_bigecyhmm_functions_occurrence(bigecyhmm_cycle_file)

    expected_function_occurrence_organisms = {'4-aminobutyrate aminotransferase and related aminotransferases 4-aminobutyrate aminotransferase and related aminotransferases': {'tax_id_name_2': 1.0}, 'Aminotransferase class I and II aminotransferase class I and II': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Aspartate racemase aspartate racemase ': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Histidinol-phosphate/aromatic aminotransferase histidinol-phosphate/aromatic aminotransferase': {'tax_id_name_2': 1.0}, 'Serine-pyruvate aminotransferase/archaeal aspartate aminotransferase serine-pyruvate aminotransferase/archaeal aspartate aminotransferase': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Fatty acid degradation acyl-CoA dehydrogenase': {'tax_id_name_1': 1.0}, 'Phenol => Benzoyl-CoA ubiX': {'tax_id_name_2': 1.0}, 'Alcohol utilization adh': {'tax_id_name_2': 1.0}, 'Acetogenesis ack': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Acetogenesis pta': {'tax_id_name_2': 1.0}, 'Formate <=> CO2 + H2 hycE': {'tax_id_name_2': 1.0}, 'Formate oxidation fdwB': {'tax_id_name_1': 1.0}, 'Formate oxidation fdoH': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Formate oxidation fdhB': {'tax_id_name_2': 1.0}, 'Aerobic CO oxidation coxS': {'tax_id_name_2': 1.0}, 'Aerobic CO oxidation coxM': {'tax_id_name_2': 1.0}, 'DC/4-HB 4hbl': {'tax_id_name_2': 1.0}, 'Wood Ljungdahl pathway (methyl branch) fthfs': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Wood Ljungdahl pathway (methyl branch) fhs': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'N2 fixation nifD': {'tax_id_name_2': 1.0}, 'N2 fixation nifK': {'tax_id_name_2': 1.0}, 'N2 fixation nifH': {'tax_id_name_2': 1.0}, 'Nitrite reduction to ammonia nrfH': {'tax_id_name_2': 1.0}, 'Nitrite reduction to ammonia nrfA': {'tax_id_name_2': 1.0}, 'Sulfite reduction dsrD': {'tax_id_name_2': 1.0}, 'Sulfur oxidation dsrA': {'tax_id_name_2': 1.0}, 'Sulfur oxidation dsrB': {'tax_id_name_2': 1.0}, 'Sulfur oxidation sdo': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Sulfite reduction asrA': {'tax_id_name_2': 1.0}, 'Sulfite reduction asrB': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Sulfite reduction asrC': {'tax_id_name_2': 1.0}, 'Sulfate reduction aprA': {'tax_id_name_2': 1.0}, 'Sulfate reduction sat': {'tax_id_name_2': 1.0}, 'Sulfate reduction apsA': {'tax_id_name_2': 1.0}, 'Sulfate reduction apsK': {'tax_id_name_2': 1.0}, 'Thiosulfate disproportionation phsA': {'tax_id_name_2': 1.0}, 'FeFe hydrogenase fefe-group-a2': {'tax_id_name_1': 1.0}, 'FeFe hydrogenase fefe-group-b': {'tax_id_name_1': 1.0}, 'Ni-Fe Hydrogenase nife-group-1': {'tax_id_name_2': 1.0}, 'Ni-Fe Hydrogenase nife-group-4a-g': {'tax_id_name_2': 1.0}, 'Complex I (NADH-quinone oxidoreductase) nuoB': {'tax_id_name_2': 1.0}, 'Complex I (NADH-quinone oxidoreductase) nuoC': {'tax_id_name_2': 1.0}, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpA (V/A-type)': {'tax_id_name_1': 1.0}, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpB (V/A-type)': {'tax_id_name_1': 1.0}, 'Complex V (F-type H+-transporting ATPase) atpA (F-type)': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Complex V (F-type H+-transporting ATPase) atpD (F-type)': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydA': {'tax_id_name_2': 1.0}, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydB': {'tax_id_name_2': 1.0}, 'Halogenated compounds breakdown E3.8.1.2': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Iron reduction FmnA': {'tax_id_name_1': 1.0}, 'Iron reduction FmnB': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Iron reduction EetA': {'tax_id_name_1': 1.0}, 'Iron reduction EetB': {'tax_id_name_1': 1.0}, 'Iron reduction DmkB': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Iron reduction DFE_0449': {'tax_id_name_2': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrA': {'tax_id_name_2': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrB': {'tax_id_name_2': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrC': {'tax_id_name_2': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrD': {'tax_id_name_2': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrJ': {'tax_id_name_2': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrK': {'tax_id_name_2': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrM': {'tax_id_name_2': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrO': {'tax_id_name_2': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrP': {'tax_id_name_2': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrT': {'tax_id_name_2': 1.0}, 'Dissimilatory sulfur metabolism | Sulfite reduction asrA': {'tax_id_name_2': 1.0}, 'Dissimilatory sulfur metabolism | Sulfite reduction asrB': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Dissimilatory sulfur metabolism | Sulfite reduction asrC': {'tax_id_name_2': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation sdo': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Dissimilatory sulfur metabolism | Sulfate reduction aprA': {'tax_id_name_2': 1.0}, 'Dissimilatory sulfur metabolism | Sulfate reduction aprB': {'tax_id_name_2': 1.0}, 'Dissimilatory sulfur metabolism | Sulfate reduction sat': {'tax_id_name_2': 1.0}, 'Dissimilatory sulfur metabolism | Thiosulfate disproportionation phsA': {'tax_id_name_2': 1.0}, "Assimilatory sulfur metabolism| Sulfate <-> APS (Adenosine 5'-phosphosulfate) cysN": {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, "Assimilatory sulfur metabolism| APS (Adenosine 5'-phosphosulfate) <-> PAPS (3'-Phosphoadenosine 5'-phosphosulfate) cysC": {'tax_id_name_2': 1.0}, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysK': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysM': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) ddhA': {'tax_id_name_2': 1.0}, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) dmsA': {'tax_id_name_2': 1.0}, 'DMSO metabolism | DMS (Dimethyl sulfide) <-> Methanethiol dmoB': {'tax_id_name_2': 1.0}, 'Metabolism of organic sulfur | Cysteine <-> Cystathionine metB': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Metabolism of organic sulfur | Cysteine <-> 3-mercaptopyruvate aspB': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Metabolism of organic sulfur | OAHS (O-acetylhomoserine) + Sulfide -> Homocysteine metY': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine malY': {'tax_id_name_1': 1.0}, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine metC': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Metabolism of organic sulfur | MHO (L-methionine (R)-S-oxide) -> Methionine) msrC': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Metabolism of organic sulfur | Homocysteine -> Methionine metH': {'tax_id_name_2': 1.0}, 'Metabolism of organic sulfur | Homocysteine <-> Methionine metE': {'tax_id_name_2': 1.0}, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> Homocysteine ahcY': {'tax_id_name_2': 1.0}, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> SRH (S-ribosylhomocysteine) mtnN': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysK': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysM': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide malY': {'tax_id_name_1': 1.0}, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide metC': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Sulfur-related amino acid metabolism | Methionine -> Methanethiol megL': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}, 'Sulfur-related amino acid metabolism | Methionine -> SAMe (S-adenosyl-L-methionine) metK': {'tax_id_name_1': 1.0, 'tax_id_name_2': 1.0}}

    for function in expected_function_occurrence_organisms:
        for organism in expected_function_occurrence_organisms[function]:
            assert function_occurrence_organisms[function][organism] == expected_function_occurrence_organisms[function][organism]


def test_compute_bigecyhmm_functions_occurrence_functions_from_esmecata():
    bigecyhmm_cycle_file = os.path.join('input_data', 'bigecyhmm_output_folder', 'function_presence.tsv')
    proteome_tax_id_file = os.path.join('input_data', 'esmecata_output_folder', '0_proteomes', 'proteome_tax_id.tsv')

    observation_names_tax_id_names, observation_names_tax_ranks = read_esmecata_proteome_file(proteome_tax_id_file)

    tax_id_names_observation_names = {}
    for observation_name in observation_names_tax_id_names:
        tax_id_name = observation_names_tax_id_names[observation_name]
        if tax_id_name not in tax_id_names_observation_names:
            tax_id_names_observation_names[tax_id_name] = [observation_name]
        else:
            tax_id_names_observation_names[tax_id_name].append(observation_name)
    function_occurrence_organisms, all_studied_organisms = compute_bigecyhmm_functions_occurrence(bigecyhmm_cycle_file, tax_id_names_observation_names)

    expected_function_occurrence_organisms = {'4-aminobutyrate aminotransferase and related aminotransferases 4-aminobutyrate aminotransferase and related aminotransferases': {'org_2': 1.0, 'org_3': 1.0}, 'Aminotransferase class I and II aminotransferase class I and II': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Aspartate racemase aspartate racemase ': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Histidinol-phosphate/aromatic aminotransferase histidinol-phosphate/aromatic aminotransferase': {'org_2': 1.0, 'org_3': 1.0}, 'Serine-pyruvate aminotransferase/archaeal aspartate aminotransferase serine-pyruvate aminotransferase/archaeal aspartate aminotransferase': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Fatty acid degradation acyl-CoA dehydrogenase': {'org_1': 1.0}, 'Phenol => Benzoyl-CoA ubiX': {'org_2': 1.0, 'org_3': 1.0}, 'Alcohol utilization adh': {'org_2': 1.0, 'org_3': 1.0}, 'Acetogenesis ack': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Acetogenesis pta': {'org_2': 1.0, 'org_3': 1.0}, 'Formate <=> CO2 + H2 hycE': {'org_2': 1.0, 'org_3': 1.0}, 'Formate oxidation fdwB': {'org_1': 1.0}, 'Formate oxidation fdoH': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Formate oxidation fdhB': {'org_2': 1.0, 'org_3': 1.0}, 'Aerobic CO oxidation coxS': {'org_2': 1.0, 'org_3': 1.0}, 'Aerobic CO oxidation coxM': {'org_2': 1.0, 'org_3': 1.0}, 'DC/4-HB 4hbl': {'org_2': 1.0, 'org_3': 1.0}, 'Wood Ljungdahl pathway (methyl branch) fthfs': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Wood Ljungdahl pathway (methyl branch) fhs': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'N2 fixation nifD': {'org_2': 1.0, 'org_3': 1.0}, 'N2 fixation nifK': {'org_2': 1.0, 'org_3': 1.0}, 'N2 fixation nifH': {'org_2': 1.0, 'org_3': 1.0}, 'Nitrite reduction to ammonia nrfH': {'org_2': 1.0, 'org_3': 1.0}, 'Nitrite reduction to ammonia nrfA': {'org_2': 1.0, 'org_3': 1.0}, 'Sulfite reduction dsrD': {'org_2': 1.0, 'org_3': 1.0}, 'Sulfur oxidation dsrA': {'org_2': 1.0, 'org_3': 1.0}, 'Sulfur oxidation dsrB': {'org_2': 1.0, 'org_3': 1.0}, 'Sulfur oxidation sdo': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Sulfite reduction asrA': {'org_2': 1.0, 'org_3': 1.0}, 'Sulfite reduction asrB': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Sulfite reduction asrC': {'org_2': 1.0, 'org_3': 1.0}, 'Sulfate reduction aprA': {'org_2': 1.0, 'org_3': 1.0}, 'Sulfate reduction sat': {'org_2': 1.0, 'org_3': 1.0}, 'Sulfate reduction apsA': {'org_2': 1.0, 'org_3': 1.0}, 'Sulfate reduction apsK': {'org_2': 1.0, 'org_3': 1.0}, 'Thiosulfate disproportionation phsA': {'org_2': 1.0, 'org_3': 1.0}, 'FeFe hydrogenase fefe-group-a2': {'org_1': 1.0}, 'FeFe hydrogenase fefe-group-b': {'org_1': 1.0}, 'Ni-Fe Hydrogenase nife-group-1': {'org_2': 1.0, 'org_3': 1.0}, 'Ni-Fe Hydrogenase nife-group-4a-g': {'org_2': 1.0, 'org_3': 1.0}, 'Complex I (NADH-quinone oxidoreductase) nuoB': {'org_2': 1.0, 'org_3': 1.0}, 'Complex I (NADH-quinone oxidoreductase) nuoC': {'org_2': 1.0, 'org_3': 1.0}, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpA (V/A-type)': {'org_1': 1.0}, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpB (V/A-type)': {'org_1': 1.0}, 'Complex V (F-type H+-transporting ATPase) atpA (F-type)': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Complex V (F-type H+-transporting ATPase) atpD (F-type)': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydA': {'org_2': 1.0, 'org_3': 1.0}, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydB': {'org_2': 1.0, 'org_3': 1.0}, 'Halogenated compounds breakdown E3.8.1.2': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Iron reduction FmnA': {'org_1': 1.0}, 'Iron reduction FmnB': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Iron reduction EetA': {'org_1': 1.0}, 'Iron reduction EetB': {'org_1': 1.0}, 'Iron reduction DmkB': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Iron reduction DFE_0449': {'org_2': 1.0, 'org_3': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrA': {'org_2': 1.0, 'org_3': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrB': {'org_2': 1.0, 'org_3': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrC': {'org_2': 1.0, 'org_3': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrD': {'org_2': 1.0, 'org_3': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrJ': {'org_2': 1.0, 'org_3': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrK': {'org_2': 1.0, 'org_3': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrM': {'org_2': 1.0, 'org_3': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrO': {'org_2': 1.0, 'org_3': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrP': {'org_2': 1.0, 'org_3': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrT': {'org_2': 1.0, 'org_3': 1.0}, 'Dissimilatory sulfur metabolism | Sulfite reduction asrA': {'org_2': 1.0, 'org_3': 1.0}, 'Dissimilatory sulfur metabolism | Sulfite reduction asrB': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Dissimilatory sulfur metabolism | Sulfite reduction asrC': {'org_2': 1.0, 'org_3': 1.0}, 'Dissimilatory sulfur metabolism | Sulfur oxidation sdo': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Dissimilatory sulfur metabolism | Sulfate reduction aprA': {'org_2': 1.0, 'org_3': 1.0}, 'Dissimilatory sulfur metabolism | Sulfate reduction aprB': {'org_2': 1.0, 'org_3': 1.0}, 'Dissimilatory sulfur metabolism | Sulfate reduction sat': {'org_2': 1.0, 'org_3': 1.0}, 'Dissimilatory sulfur metabolism | Thiosulfate disproportionation phsA': {'org_2': 1.0, 'org_3': 1.0}, "Assimilatory sulfur metabolism| Sulfate <-> APS (Adenosine 5'-phosphosulfate) cysN": {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, "Assimilatory sulfur metabolism| APS (Adenosine 5'-phosphosulfate) <-> PAPS (3'-Phosphoadenosine 5'-phosphosulfate) cysC": {'org_2': 1.0, 'org_3': 1.0}, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysK': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysM': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) ddhA': {'org_2': 1.0, 'org_3': 1.0}, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) dmsA': {'org_2': 1.0, 'org_3': 1.0}, 'DMSO metabolism | DMS (Dimethyl sulfide) <-> Methanethiol dmoB': {'org_2': 1.0, 'org_3': 1.0}, 'Metabolism of organic sulfur | Cysteine <-> Cystathionine metB': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Metabolism of organic sulfur | Cysteine <-> 3-mercaptopyruvate aspB': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Metabolism of organic sulfur | OAHS (O-acetylhomoserine) + Sulfide -> Homocysteine metY': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine malY': {'org_1': 1.0}, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine metC': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Metabolism of organic sulfur | MHO (L-methionine (R)-S-oxide) -> Methionine) msrC': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Metabolism of organic sulfur | Homocysteine -> Methionine metH': {'org_2': 1.0, 'org_3': 1.0}, 'Metabolism of organic sulfur | Homocysteine <-> Methionine metE': {'org_2': 1.0, 'org_3': 1.0}, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> Homocysteine ahcY': {'org_2': 1.0, 'org_3': 1.0}, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> SRH (S-ribosylhomocysteine) mtnN': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysK': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysM': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide malY': {'org_1': 1.0}, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide metC': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Sulfur-related amino acid metabolism | Methionine -> Methanethiol megL': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}, 'Sulfur-related amino acid metabolism | Methionine -> SAMe (S-adenosyl-L-methionine) metK': {'org_1': 1.0, 'org_2': 1.0, 'org_3': 1.0}}

    for function in expected_function_occurrence_organisms:
        for organism in expected_function_occurrence_organisms[function]:
            assert function_occurrence_organisms[function][organism] == expected_function_occurrence_organisms[function][organism]



def test_compute_bigecyhmm_functions_abundance_functions():
    abundance_file_path = os.path.join('input_data', 'abundance_file_from_genomes.tsv')
    bigecyhmm_cycle_file = os.path.join('input_data', 'bigecyhmm_output_folder', 'function_presence.tsv')

    sample_abundance, sample_tot_abundance = read_measures_file(abundance_file_path)

    function_abundance_samples, function_relative_abundance_samples, function_participation_samples = compute_bigecyhmm_functions_abundance(bigecyhmm_cycle_file, sample_abundance, sample_tot_abundance)

    expected_function_abundance_samples = {'sample_1': {'Aminotransferase class I and II aminotransferase class I and II': 200.0, 'Aspartate racemase aspartate racemase ': 200.0, 'Serine-pyruvate aminotransferase/archaeal aspartate aminotransferase serine-pyruvate aminotransferase/archaeal aspartate aminotransferase': 200.0, 'Fatty acid degradation acyl-CoA dehydrogenase': 100.0, 'Acetogenesis ack': 200.0, 'Formate oxidation fdwB': 100.0, 'Formate oxidation fdoH': 200.0, 'Wood Ljungdahl pathway (methyl branch) fthfs': 200.0, 'Wood Ljungdahl pathway (methyl branch) fhs': 200.0, 'Sulfur oxidation sdo': 200.0, 'Sulfite reduction asrB': 200.0, 'FeFe hydrogenase fefe-group-a2': 100.0, 'FeFe hydrogenase fefe-group-b': 100.0, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpA (V/A-type)': 100.0, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpB (V/A-type)': 100.0, 'Complex V (F-type H+-transporting ATPase) atpA (F-type)': 200.0, 'Complex V (F-type H+-transporting ATPase) atpD (F-type)': 200.0, 'Halogenated compounds breakdown E3.8.1.2': 200.0, 'Iron reduction FmnA': 100.0, 'Iron reduction FmnB': 200.0, 'Iron reduction EetA': 100.0, 'Iron reduction EetB': 100.0, 'Iron reduction DmkB': 200.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrB': 200.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation sdo': 200.0, "Assimilatory sulfur metabolism| Sulfate <-> APS (Adenosine 5'-phosphosulfate) cysN": 200.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysK': 200.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysM': 200.0, 'Metabolism of organic sulfur | Cysteine <-> Cystathionine metB': 200.0, 'Metabolism of organic sulfur | Cysteine <-> 3-mercaptopyruvate aspB': 200.0, 'Metabolism of organic sulfur | OAHS (O-acetylhomoserine) + Sulfide -> Homocysteine metY': 200.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine malY': 100.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine metC': 200.0, 'Metabolism of organic sulfur | MHO (L-methionine (R)-S-oxide) -> Methionine) msrC': 200.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> SRH (S-ribosylhomocysteine) mtnN': 200.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysK': 200.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysM': 200.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide malY': 100.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide metC': 200.0, 'Sulfur-related amino acid metabolism | Methionine -> Methanethiol megL': 200.0, 'Sulfur-related amino acid metabolism | Methionine -> SAMe (S-adenosyl-L-methionine) metK': 200.0, '4-aminobutyrate aminotransferase and related aminotransferases 4-aminobutyrate aminotransferase and related aminotransferases': 100.0, 'Histidinol-phosphate/aromatic aminotransferase histidinol-phosphate/aromatic aminotransferase': 100.0, 'Phenol => Benzoyl-CoA ubiX': 100.0, 'Alcohol utilization adh': 100.0, 'Acetogenesis pta': 100.0, 'Formate <=> CO2 + H2 hycE': 100.0, 'Formate oxidation fdhB': 100.0, 'Aerobic CO oxidation coxS': 100.0, 'Aerobic CO oxidation coxM': 100.0, 'DC/4-HB 4hbl': 100.0, 'N2 fixation nifD': 100.0, 'N2 fixation nifK': 100.0, 'N2 fixation nifH': 100.0, 'Nitrite reduction to ammonia nrfH': 100.0, 'Nitrite reduction to ammonia nrfA': 100.0, 'Sulfite reduction dsrD': 100.0, 'Sulfur oxidation dsrA': 100.0, 'Sulfur oxidation dsrB': 100.0, 'Sulfite reduction asrA': 100.0, 'Sulfite reduction asrC': 100.0, 'Sulfate reduction aprA': 100.0, 'Sulfate reduction sat': 100.0, 'Sulfate reduction apsA': 100.0, 'Sulfate reduction apsK': 100.0, 'Thiosulfate disproportionation phsA': 100.0, 'Ni-Fe Hydrogenase nife-group-1': 100.0, 'Ni-Fe Hydrogenase nife-group-4a-g': 100.0, 'Complex I (NADH-quinone oxidoreductase) nuoB': 100.0, 'Complex I (NADH-quinone oxidoreductase) nuoC': 100.0, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydA': 100.0, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydB': 100.0, 'Iron reduction DFE_0449': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrA': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrB': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrC': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrD': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrJ': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrK': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrM': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrO': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrP': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrT': 100.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrA': 100.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrC': 100.0, 'Dissimilatory sulfur metabolism | Sulfate reduction aprA': 100.0, 'Dissimilatory sulfur metabolism | Sulfate reduction aprB': 100.0, 'Dissimilatory sulfur metabolism | Sulfate reduction sat': 100.0, 'Dissimilatory sulfur metabolism | Thiosulfate disproportionation phsA': 100.0, "Assimilatory sulfur metabolism| APS (Adenosine 5'-phosphosulfate) <-> PAPS (3'-Phosphoadenosine 5'-phosphosulfate) cysC": 100.0, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) ddhA': 100.0, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) dmsA': 100.0, 'DMSO metabolism | DMS (Dimethyl sulfide) <-> Methanethiol dmoB': 100.0, 'Metabolism of organic sulfur | Homocysteine -> Methionine metH': 100.0, 'Metabolism of organic sulfur | Homocysteine <-> Methionine metE': 100.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> Homocysteine ahcY': 100.0}, 'sample_2': {'Aminotransferase class I and II aminotransferase class I and II': 800.0, 'Aspartate racemase aspartate racemase ': 800.0, 'Serine-pyruvate aminotransferase/archaeal aspartate aminotransferase serine-pyruvate aminotransferase/archaeal aspartate aminotransferase': 800.0, 'Fatty acid degradation acyl-CoA dehydrogenase': 0.0, 'Acetogenesis ack': 800.0, 'Formate oxidation fdwB': 0.0, 'Formate oxidation fdoH': 800.0, 'Wood Ljungdahl pathway (methyl branch) fthfs': 800.0, 'Wood Ljungdahl pathway (methyl branch) fhs': 800.0, 'Sulfur oxidation sdo': 800.0, 'Sulfite reduction asrB': 800.0, 'FeFe hydrogenase fefe-group-a2': 0.0, 'FeFe hydrogenase fefe-group-b': 0.0, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpA (V/A-type)': 0.0, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpB (V/A-type)': 0.0, 'Complex V (F-type H+-transporting ATPase) atpA (F-type)': 800.0, 'Complex V (F-type H+-transporting ATPase) atpD (F-type)': 800.0, 'Halogenated compounds breakdown E3.8.1.2': 800.0, 'Iron reduction FmnA': 0.0, 'Iron reduction FmnB': 800.0, 'Iron reduction EetA': 0.0, 'Iron reduction EetB': 0.0, 'Iron reduction DmkB': 800.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrB': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation sdo': 800.0, "Assimilatory sulfur metabolism| Sulfate <-> APS (Adenosine 5'-phosphosulfate) cysN": 800.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysK': 800.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysM': 800.0, 'Metabolism of organic sulfur | Cysteine <-> Cystathionine metB': 800.0, 'Metabolism of organic sulfur | Cysteine <-> 3-mercaptopyruvate aspB': 800.0, 'Metabolism of organic sulfur | OAHS (O-acetylhomoserine) + Sulfide -> Homocysteine metY': 800.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine malY': 0.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine metC': 800.0, 'Metabolism of organic sulfur | MHO (L-methionine (R)-S-oxide) -> Methionine) msrC': 800.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> SRH (S-ribosylhomocysteine) mtnN': 800.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysK': 800.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysM': 800.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide malY': 0.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide metC': 800.0, 'Sulfur-related amino acid metabolism | Methionine -> Methanethiol megL': 800.0, 'Sulfur-related amino acid metabolism | Methionine -> SAMe (S-adenosyl-L-methionine) metK': 800.0, '4-aminobutyrate aminotransferase and related aminotransferases 4-aminobutyrate aminotransferase and related aminotransferases': 800.0, 'Histidinol-phosphate/aromatic aminotransferase histidinol-phosphate/aromatic aminotransferase': 800.0, 'Phenol => Benzoyl-CoA ubiX': 800.0, 'Alcohol utilization adh': 800.0, 'Acetogenesis pta': 800.0, 'Formate <=> CO2 + H2 hycE': 800.0, 'Formate oxidation fdhB': 800.0, 'Aerobic CO oxidation coxS': 800.0, 'Aerobic CO oxidation coxM': 800.0, 'DC/4-HB 4hbl': 800.0, 'N2 fixation nifD': 800.0, 'N2 fixation nifK': 800.0, 'N2 fixation nifH': 800.0, 'Nitrite reduction to ammonia nrfH': 800.0, 'Nitrite reduction to ammonia nrfA': 800.0, 'Sulfite reduction dsrD': 800.0, 'Sulfur oxidation dsrA': 800.0, 'Sulfur oxidation dsrB': 800.0, 'Sulfite reduction asrA': 800.0, 'Sulfite reduction asrC': 800.0, 'Sulfate reduction aprA': 800.0, 'Sulfate reduction sat': 800.0, 'Sulfate reduction apsA': 800.0, 'Sulfate reduction apsK': 800.0, 'Thiosulfate disproportionation phsA': 800.0, 'Ni-Fe Hydrogenase nife-group-1': 800.0, 'Ni-Fe Hydrogenase nife-group-4a-g': 800.0, 'Complex I (NADH-quinone oxidoreductase) nuoB': 800.0, 'Complex I (NADH-quinone oxidoreductase) nuoC': 800.0, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydA': 800.0, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydB': 800.0, 'Iron reduction DFE_0449': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrA': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrB': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrC': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrD': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrJ': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrK': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrM': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrO': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrP': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrT': 800.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrA': 800.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrC': 800.0, 'Dissimilatory sulfur metabolism | Sulfate reduction aprA': 800.0, 'Dissimilatory sulfur metabolism | Sulfate reduction aprB': 800.0, 'Dissimilatory sulfur metabolism | Sulfate reduction sat': 800.0, 'Dissimilatory sulfur metabolism | Thiosulfate disproportionation phsA': 800.0, "Assimilatory sulfur metabolism| APS (Adenosine 5'-phosphosulfate) <-> PAPS (3'-Phosphoadenosine 5'-phosphosulfate) cysC": 800.0, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) ddhA': 800.0, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) dmsA': 800.0, 'DMSO metabolism | DMS (Dimethyl sulfide) <-> Methanethiol dmoB': 800.0, 'Metabolism of organic sulfur | Homocysteine -> Methionine metH': 800.0, 'Metabolism of organic sulfur | Homocysteine <-> Methionine metE': 800.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> Homocysteine ahcY': 800.0}, 'sample_3': {'Aminotransferase class I and II aminotransferase class I and II': 520.0, 'Aspartate racemase aspartate racemase ': 520.0, 'Serine-pyruvate aminotransferase/archaeal aspartate aminotransferase serine-pyruvate aminotransferase/archaeal aspartate aminotransferase': 520.0, 'Fatty acid degradation acyl-CoA dehydrogenase': 0.0, 'Acetogenesis ack': 520.0, 'Formate oxidation fdwB': 0.0, 'Formate oxidation fdoH': 520.0, 'Wood Ljungdahl pathway (methyl branch) fthfs': 520.0, 'Wood Ljungdahl pathway (methyl branch) fhs': 520.0, 'Sulfur oxidation sdo': 520.0, 'Sulfite reduction asrB': 520.0, 'FeFe hydrogenase fefe-group-a2': 0.0, 'FeFe hydrogenase fefe-group-b': 0.0, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpA (V/A-type)': 0.0, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpB (V/A-type)': 0.0, 'Complex V (F-type H+-transporting ATPase) atpA (F-type)': 520.0, 'Complex V (F-type H+-transporting ATPase) atpD (F-type)': 520.0, 'Halogenated compounds breakdown E3.8.1.2': 520.0, 'Iron reduction FmnA': 0.0, 'Iron reduction FmnB': 520.0, 'Iron reduction EetA': 0.0, 'Iron reduction EetB': 0.0, 'Iron reduction DmkB': 520.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrB': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation sdo': 520.0, "Assimilatory sulfur metabolism| Sulfate <-> APS (Adenosine 5'-phosphosulfate) cysN": 520.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysK': 520.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysM': 520.0, 'Metabolism of organic sulfur | Cysteine <-> Cystathionine metB': 520.0, 'Metabolism of organic sulfur | Cysteine <-> 3-mercaptopyruvate aspB': 520.0, 'Metabolism of organic sulfur | OAHS (O-acetylhomoserine) + Sulfide -> Homocysteine metY': 520.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine malY': 0.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine metC': 520.0, 'Metabolism of organic sulfur | MHO (L-methionine (R)-S-oxide) -> Methionine) msrC': 520.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> SRH (S-ribosylhomocysteine) mtnN': 520.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysK': 520.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysM': 520.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide malY': 0.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide metC': 520.0, 'Sulfur-related amino acid metabolism | Methionine -> Methanethiol megL': 520.0, 'Sulfur-related amino acid metabolism | Methionine -> SAMe (S-adenosyl-L-methionine) metK': 520.0, '4-aminobutyrate aminotransferase and related aminotransferases 4-aminobutyrate aminotransferase and related aminotransferases': 520.0, 'Histidinol-phosphate/aromatic aminotransferase histidinol-phosphate/aromatic aminotransferase': 520.0, 'Phenol => Benzoyl-CoA ubiX': 520.0, 'Alcohol utilization adh': 520.0, 'Acetogenesis pta': 520.0, 'Formate <=> CO2 + H2 hycE': 520.0, 'Formate oxidation fdhB': 520.0, 'Aerobic CO oxidation coxS': 520.0, 'Aerobic CO oxidation coxM': 520.0, 'DC/4-HB 4hbl': 520.0, 'N2 fixation nifD': 520.0, 'N2 fixation nifK': 520.0, 'N2 fixation nifH': 520.0, 'Nitrite reduction to ammonia nrfH': 520.0, 'Nitrite reduction to ammonia nrfA': 520.0, 'Sulfite reduction dsrD': 520.0, 'Sulfur oxidation dsrA': 520.0, 'Sulfur oxidation dsrB': 520.0, 'Sulfite reduction asrA': 520.0, 'Sulfite reduction asrC': 520.0, 'Sulfate reduction aprA': 520.0, 'Sulfate reduction sat': 520.0, 'Sulfate reduction apsA': 520.0, 'Sulfate reduction apsK': 520.0, 'Thiosulfate disproportionation phsA': 520.0, 'Ni-Fe Hydrogenase nife-group-1': 520.0, 'Ni-Fe Hydrogenase nife-group-4a-g': 520.0, 'Complex I (NADH-quinone oxidoreductase) nuoB': 520.0, 'Complex I (NADH-quinone oxidoreductase) nuoC': 520.0, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydA': 520.0, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydB': 520.0, 'Iron reduction DFE_0449': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrA': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrB': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrC': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrD': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrJ': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrK': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrM': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrO': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrP': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrT': 520.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrA': 520.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrC': 520.0, 'Dissimilatory sulfur metabolism | Sulfate reduction aprA': 520.0, 'Dissimilatory sulfur metabolism | Sulfate reduction aprB': 520.0, 'Dissimilatory sulfur metabolism | Sulfate reduction sat': 520.0, 'Dissimilatory sulfur metabolism | Thiosulfate disproportionation phsA': 520.0, "Assimilatory sulfur metabolism| APS (Adenosine 5'-phosphosulfate) <-> PAPS (3'-Phosphoadenosine 5'-phosphosulfate) cysC": 520.0, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) ddhA': 520.0, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) dmsA': 520.0, 'DMSO metabolism | DMS (Dimethyl sulfide) <-> Methanethiol dmoB': 520.0, 'Metabolism of organic sulfur | Homocysteine -> Methionine metH': 520.0, 'Metabolism of organic sulfur | Homocysteine <-> Methionine metE': 520.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> Homocysteine ahcY': 520.0}}
    expected_function_relative_abundance_samples = {'sample_1': {'Aminotransferase class I and II aminotransferase class I and II': 1.0, 'Aspartate racemase aspartate racemase ': 1.0, 'Serine-pyruvate aminotransferase/archaeal aspartate aminotransferase serine-pyruvate aminotransferase/archaeal aspartate aminotransferase': 1.0, 'Fatty acid degradation acyl-CoA dehydrogenase': 0.5, 'Acetogenesis ack': 1.0, 'Formate oxidation fdwB': 0.5, 'Formate oxidation fdoH': 1.0, 'Wood Ljungdahl pathway (methyl branch) fthfs': 1.0, 'Wood Ljungdahl pathway (methyl branch) fhs': 1.0, 'Sulfur oxidation sdo': 1.0, 'Sulfite reduction asrB': 1.0, 'FeFe hydrogenase fefe-group-a2': 0.5, 'FeFe hydrogenase fefe-group-b': 0.5, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpA (V/A-type)': 0.5, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpB (V/A-type)': 0.5, 'Complex V (F-type H+-transporting ATPase) atpA (F-type)': 1.0, 'Complex V (F-type H+-transporting ATPase) atpD (F-type)': 1.0, 'Halogenated compounds breakdown E3.8.1.2': 1.0, 'Iron reduction FmnA': 0.5, 'Iron reduction FmnB': 1.0, 'Iron reduction EetA': 0.5, 'Iron reduction EetB': 0.5, 'Iron reduction DmkB': 1.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrB': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation sdo': 1.0, "Assimilatory sulfur metabolism| Sulfate <-> APS (Adenosine 5'-phosphosulfate) cysN": 1.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysK': 1.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysM': 1.0, 'Metabolism of organic sulfur | Cysteine <-> Cystathionine metB': 1.0, 'Metabolism of organic sulfur | Cysteine <-> 3-mercaptopyruvate aspB': 1.0, 'Metabolism of organic sulfur | OAHS (O-acetylhomoserine) + Sulfide -> Homocysteine metY': 1.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine malY': 0.5, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine metC': 1.0, 'Metabolism of organic sulfur | MHO (L-methionine (R)-S-oxide) -> Methionine) msrC': 1.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> SRH (S-ribosylhomocysteine) mtnN': 1.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysK': 1.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysM': 1.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide malY': 0.5, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide metC': 1.0, 'Sulfur-related amino acid metabolism | Methionine -> Methanethiol megL': 1.0, 'Sulfur-related amino acid metabolism | Methionine -> SAMe (S-adenosyl-L-methionine) metK': 1.0, '4-aminobutyrate aminotransferase and related aminotransferases 4-aminobutyrate aminotransferase and related aminotransferases': 0.5, 'Histidinol-phosphate/aromatic aminotransferase histidinol-phosphate/aromatic aminotransferase': 0.5, 'Phenol => Benzoyl-CoA ubiX': 0.5, 'Alcohol utilization adh': 0.5, 'Acetogenesis pta': 0.5, 'Formate <=> CO2 + H2 hycE': 0.5, 'Formate oxidation fdhB': 0.5, 'Aerobic CO oxidation coxS': 0.5, 'Aerobic CO oxidation coxM': 0.5, 'DC/4-HB 4hbl': 0.5, 'N2 fixation nifD': 0.5, 'N2 fixation nifK': 0.5, 'N2 fixation nifH': 0.5, 'Nitrite reduction to ammonia nrfH': 0.5, 'Nitrite reduction to ammonia nrfA': 0.5, 'Sulfite reduction dsrD': 0.5, 'Sulfur oxidation dsrA': 0.5, 'Sulfur oxidation dsrB': 0.5, 'Sulfite reduction asrA': 0.5, 'Sulfite reduction asrC': 0.5, 'Sulfate reduction aprA': 0.5, 'Sulfate reduction sat': 0.5, 'Sulfate reduction apsA': 0.5, 'Sulfate reduction apsK': 0.5, 'Thiosulfate disproportionation phsA': 0.5, 'Ni-Fe Hydrogenase nife-group-1': 0.5, 'Ni-Fe Hydrogenase nife-group-4a-g': 0.5, 'Complex I (NADH-quinone oxidoreductase) nuoB': 0.5, 'Complex I (NADH-quinone oxidoreductase) nuoC': 0.5, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydA': 0.5, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydB': 0.5, 'Iron reduction DFE_0449': 0.5, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrA': 0.5, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrB': 0.5, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrC': 0.5, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrD': 0.5, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrJ': 0.5, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrK': 0.5, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrM': 0.5, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrO': 0.5, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrP': 0.5, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrT': 0.5, 'Dissimilatory sulfur metabolism | Sulfite reduction asrA': 0.5, 'Dissimilatory sulfur metabolism | Sulfite reduction asrC': 0.5, 'Dissimilatory sulfur metabolism | Sulfate reduction aprA': 0.5, 'Dissimilatory sulfur metabolism | Sulfate reduction aprB': 0.5, 'Dissimilatory sulfur metabolism | Sulfate reduction sat': 0.5, 'Dissimilatory sulfur metabolism | Thiosulfate disproportionation phsA': 0.5, "Assimilatory sulfur metabolism| APS (Adenosine 5'-phosphosulfate) <-> PAPS (3'-Phosphoadenosine 5'-phosphosulfate) cysC": 0.5, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) ddhA': 0.5, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) dmsA': 0.5, 'DMSO metabolism | DMS (Dimethyl sulfide) <-> Methanethiol dmoB': 0.5, 'Metabolism of organic sulfur | Homocysteine -> Methionine metH': 0.5, 'Metabolism of organic sulfur | Homocysteine <-> Methionine metE': 0.5, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> Homocysteine ahcY': 0.5}, 'sample_2': {'Aminotransferase class I and II aminotransferase class I and II': 1.0, 'Aspartate racemase aspartate racemase ': 1.0, 'Serine-pyruvate aminotransferase/archaeal aspartate aminotransferase serine-pyruvate aminotransferase/archaeal aspartate aminotransferase': 1.0, 'Fatty acid degradation acyl-CoA dehydrogenase': 0.0, 'Acetogenesis ack': 1.0, 'Formate oxidation fdwB': 0.0, 'Formate oxidation fdoH': 1.0, 'Wood Ljungdahl pathway (methyl branch) fthfs': 1.0, 'Wood Ljungdahl pathway (methyl branch) fhs': 1.0, 'Sulfur oxidation sdo': 1.0, 'Sulfite reduction asrB': 1.0, 'FeFe hydrogenase fefe-group-a2': 0.0, 'FeFe hydrogenase fefe-group-b': 0.0, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpA (V/A-type)': 0.0, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpB (V/A-type)': 0.0, 'Complex V (F-type H+-transporting ATPase) atpA (F-type)': 1.0, 'Complex V (F-type H+-transporting ATPase) atpD (F-type)': 1.0, 'Halogenated compounds breakdown E3.8.1.2': 1.0, 'Iron reduction FmnA': 0.0, 'Iron reduction FmnB': 1.0, 'Iron reduction EetA': 0.0, 'Iron reduction EetB': 0.0, 'Iron reduction DmkB': 1.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrB': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation sdo': 1.0, "Assimilatory sulfur metabolism| Sulfate <-> APS (Adenosine 5'-phosphosulfate) cysN": 1.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysK': 1.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysM': 1.0, 'Metabolism of organic sulfur | Cysteine <-> Cystathionine metB': 1.0, 'Metabolism of organic sulfur | Cysteine <-> 3-mercaptopyruvate aspB': 1.0, 'Metabolism of organic sulfur | OAHS (O-acetylhomoserine) + Sulfide -> Homocysteine metY': 1.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine malY': 0.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine metC': 1.0, 'Metabolism of organic sulfur | MHO (L-methionine (R)-S-oxide) -> Methionine) msrC': 1.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> SRH (S-ribosylhomocysteine) mtnN': 1.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysK': 1.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysM': 1.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide malY': 0.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide metC': 1.0, 'Sulfur-related amino acid metabolism | Methionine -> Methanethiol megL': 1.0, 'Sulfur-related amino acid metabolism | Methionine -> SAMe (S-adenosyl-L-methionine) metK': 1.0, '4-aminobutyrate aminotransferase and related aminotransferases 4-aminobutyrate aminotransferase and related aminotransferases': 1.0, 'Histidinol-phosphate/aromatic aminotransferase histidinol-phosphate/aromatic aminotransferase': 1.0, 'Phenol => Benzoyl-CoA ubiX': 1.0, 'Alcohol utilization adh': 1.0, 'Acetogenesis pta': 1.0, 'Formate <=> CO2 + H2 hycE': 1.0, 'Formate oxidation fdhB': 1.0, 'Aerobic CO oxidation coxS': 1.0, 'Aerobic CO oxidation coxM': 1.0, 'DC/4-HB 4hbl': 1.0, 'N2 fixation nifD': 1.0, 'N2 fixation nifK': 1.0, 'N2 fixation nifH': 1.0, 'Nitrite reduction to ammonia nrfH': 1.0, 'Nitrite reduction to ammonia nrfA': 1.0, 'Sulfite reduction dsrD': 1.0, 'Sulfur oxidation dsrA': 1.0, 'Sulfur oxidation dsrB': 1.0, 'Sulfite reduction asrA': 1.0, 'Sulfite reduction asrC': 1.0, 'Sulfate reduction aprA': 1.0, 'Sulfate reduction sat': 1.0, 'Sulfate reduction apsA': 1.0, 'Sulfate reduction apsK': 1.0, 'Thiosulfate disproportionation phsA': 1.0, 'Ni-Fe Hydrogenase nife-group-1': 1.0, 'Ni-Fe Hydrogenase nife-group-4a-g': 1.0, 'Complex I (NADH-quinone oxidoreductase) nuoB': 1.0, 'Complex I (NADH-quinone oxidoreductase) nuoC': 1.0, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydA': 1.0, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydB': 1.0, 'Iron reduction DFE_0449': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrA': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrB': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrC': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrD': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrJ': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrK': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrM': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrO': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrP': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrT': 1.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrA': 1.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrC': 1.0, 'Dissimilatory sulfur metabolism | Sulfate reduction aprA': 1.0, 'Dissimilatory sulfur metabolism | Sulfate reduction aprB': 1.0, 'Dissimilatory sulfur metabolism | Sulfate reduction sat': 1.0, 'Dissimilatory sulfur metabolism | Thiosulfate disproportionation phsA': 1.0, "Assimilatory sulfur metabolism| APS (Adenosine 5'-phosphosulfate) <-> PAPS (3'-Phosphoadenosine 5'-phosphosulfate) cysC": 1.0, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) ddhA': 1.0, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) dmsA': 1.0, 'DMSO metabolism | DMS (Dimethyl sulfide) <-> Methanethiol dmoB': 1.0, 'Metabolism of organic sulfur | Homocysteine -> Methionine metH': 1.0, 'Metabolism of organic sulfur | Homocysteine <-> Methionine metE': 1.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> Homocysteine ahcY': 1.0}, 'sample_3': {'Aminotransferase class I and II aminotransferase class I and II': 1.0, 'Aspartate racemase aspartate racemase ': 1.0, 'Serine-pyruvate aminotransferase/archaeal aspartate aminotransferase serine-pyruvate aminotransferase/archaeal aspartate aminotransferase': 1.0, 'Fatty acid degradation acyl-CoA dehydrogenase': 0.0, 'Acetogenesis ack': 1.0, 'Formate oxidation fdwB': 0.0, 'Formate oxidation fdoH': 1.0, 'Wood Ljungdahl pathway (methyl branch) fthfs': 1.0, 'Wood Ljungdahl pathway (methyl branch) fhs': 1.0, 'Sulfur oxidation sdo': 1.0, 'Sulfite reduction asrB': 1.0, 'FeFe hydrogenase fefe-group-a2': 0.0, 'FeFe hydrogenase fefe-group-b': 0.0, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpA (V/A-type)': 0.0, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpB (V/A-type)': 0.0, 'Complex V (F-type H+-transporting ATPase) atpA (F-type)': 1.0, 'Complex V (F-type H+-transporting ATPase) atpD (F-type)': 1.0, 'Halogenated compounds breakdown E3.8.1.2': 1.0, 'Iron reduction FmnA': 0.0, 'Iron reduction FmnB': 1.0, 'Iron reduction EetA': 0.0, 'Iron reduction EetB': 0.0, 'Iron reduction DmkB': 1.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrB': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation sdo': 1.0, "Assimilatory sulfur metabolism| Sulfate <-> APS (Adenosine 5'-phosphosulfate) cysN": 1.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysK': 1.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysM': 1.0, 'Metabolism of organic sulfur | Cysteine <-> Cystathionine metB': 1.0, 'Metabolism of organic sulfur | Cysteine <-> 3-mercaptopyruvate aspB': 1.0, 'Metabolism of organic sulfur | OAHS (O-acetylhomoserine) + Sulfide -> Homocysteine metY': 1.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine malY': 0.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine metC': 1.0, 'Metabolism of organic sulfur | MHO (L-methionine (R)-S-oxide) -> Methionine) msrC': 1.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> SRH (S-ribosylhomocysteine) mtnN': 1.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysK': 1.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysM': 1.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide malY': 0.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide metC': 1.0, 'Sulfur-related amino acid metabolism | Methionine -> Methanethiol megL': 1.0, 'Sulfur-related amino acid metabolism | Methionine -> SAMe (S-adenosyl-L-methionine) metK': 1.0, '4-aminobutyrate aminotransferase and related aminotransferases 4-aminobutyrate aminotransferase and related aminotransferases': 1.0, 'Histidinol-phosphate/aromatic aminotransferase histidinol-phosphate/aromatic aminotransferase': 1.0, 'Phenol => Benzoyl-CoA ubiX': 1.0, 'Alcohol utilization adh': 1.0, 'Acetogenesis pta': 1.0, 'Formate <=> CO2 + H2 hycE': 1.0, 'Formate oxidation fdhB': 1.0, 'Aerobic CO oxidation coxS': 1.0, 'Aerobic CO oxidation coxM': 1.0, 'DC/4-HB 4hbl': 1.0, 'N2 fixation nifD': 1.0, 'N2 fixation nifK': 1.0, 'N2 fixation nifH': 1.0, 'Nitrite reduction to ammonia nrfH': 1.0, 'Nitrite reduction to ammonia nrfA': 1.0, 'Sulfite reduction dsrD': 1.0, 'Sulfur oxidation dsrA': 1.0, 'Sulfur oxidation dsrB': 1.0, 'Sulfite reduction asrA': 1.0, 'Sulfite reduction asrC': 1.0, 'Sulfate reduction aprA': 1.0, 'Sulfate reduction sat': 1.0, 'Sulfate reduction apsA': 1.0, 'Sulfate reduction apsK': 1.0, 'Thiosulfate disproportionation phsA': 1.0, 'Ni-Fe Hydrogenase nife-group-1': 1.0, 'Ni-Fe Hydrogenase nife-group-4a-g': 1.0, 'Complex I (NADH-quinone oxidoreductase) nuoB': 1.0, 'Complex I (NADH-quinone oxidoreductase) nuoC': 1.0, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydA': 1.0, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydB': 1.0, 'Iron reduction DFE_0449': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrA': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrB': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrC': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrD': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrJ': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrK': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrM': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrO': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrP': 1.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrT': 1.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrA': 1.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrC': 1.0, 'Dissimilatory sulfur metabolism | Sulfate reduction aprA': 1.0, 'Dissimilatory sulfur metabolism | Sulfate reduction aprB': 1.0, 'Dissimilatory sulfur metabolism | Sulfate reduction sat': 1.0, 'Dissimilatory sulfur metabolism | Thiosulfate disproportionation phsA': 1.0, "Assimilatory sulfur metabolism| APS (Adenosine 5'-phosphosulfate) <-> PAPS (3'-Phosphoadenosine 5'-phosphosulfate) cysC": 1.0, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) ddhA': 1.0, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) dmsA': 1.0, 'DMSO metabolism | DMS (Dimethyl sulfide) <-> Methanethiol dmoB': 1.0, 'Metabolism of organic sulfur | Homocysteine -> Methionine metH': 1.0, 'Metabolism of organic sulfur | Homocysteine <-> Methionine metE': 1.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> Homocysteine ahcY': 1.0}}
    expected_function_participation_samples = {'sample_1': {'tax_id_name_1': {'Aminotransferase class I and II aminotransferase class I and II': 100.0, 'Aspartate racemase aspartate racemase ': 100.0, 'Serine-pyruvate aminotransferase/archaeal aspartate aminotransferase serine-pyruvate aminotransferase/archaeal aspartate aminotransferase': 100.0, 'Fatty acid degradation acyl-CoA dehydrogenase': 100.0, 'Acetogenesis ack': 100.0, 'Formate oxidation fdwB': 100.0, 'Formate oxidation fdoH': 100.0, 'Wood Ljungdahl pathway (methyl branch) fthfs': 100.0, 'Wood Ljungdahl pathway (methyl branch) fhs': 100.0, 'Sulfur oxidation sdo': 100.0, 'Sulfite reduction asrB': 100.0, 'FeFe hydrogenase fefe-group-a2': 100.0, 'FeFe hydrogenase fefe-group-b': 100.0, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpA (V/A-type)': 100.0, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpB (V/A-type)': 100.0, 'Complex V (F-type H+-transporting ATPase) atpA (F-type)': 100.0, 'Complex V (F-type H+-transporting ATPase) atpD (F-type)': 100.0, 'Halogenated compounds breakdown E3.8.1.2': 100.0, 'Iron reduction FmnA': 100.0, 'Iron reduction FmnB': 100.0, 'Iron reduction EetA': 100.0, 'Iron reduction EetB': 100.0, 'Iron reduction DmkB': 100.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrB': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation sdo': 100.0, "Assimilatory sulfur metabolism| Sulfate <-> APS (Adenosine 5'-phosphosulfate) cysN": 100.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysK': 100.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysM': 100.0, 'Metabolism of organic sulfur | Cysteine <-> Cystathionine metB': 100.0, 'Metabolism of organic sulfur | Cysteine <-> 3-mercaptopyruvate aspB': 100.0, 'Metabolism of organic sulfur | OAHS (O-acetylhomoserine) + Sulfide -> Homocysteine metY': 100.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine malY': 100.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine metC': 100.0, 'Metabolism of organic sulfur | MHO (L-methionine (R)-S-oxide) -> Methionine) msrC': 100.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> SRH (S-ribosylhomocysteine) mtnN': 100.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysK': 100.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysM': 100.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide malY': 100.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide metC': 100.0, 'Sulfur-related amino acid metabolism | Methionine -> Methanethiol megL': 100.0, 'Sulfur-related amino acid metabolism | Methionine -> SAMe (S-adenosyl-L-methionine) metK': 100.0}, 'tax_id_name_2': {'4-aminobutyrate aminotransferase and related aminotransferases 4-aminobutyrate aminotransferase and related aminotransferases': 100.0, 'Aminotransferase class I and II aminotransferase class I and II': 100.0, 'Aspartate racemase aspartate racemase ': 100.0, 'Histidinol-phosphate/aromatic aminotransferase histidinol-phosphate/aromatic aminotransferase': 100.0, 'Serine-pyruvate aminotransferase/archaeal aspartate aminotransferase serine-pyruvate aminotransferase/archaeal aspartate aminotransferase': 100.0, 'Phenol => Benzoyl-CoA ubiX': 100.0, 'Alcohol utilization adh': 100.0, 'Acetogenesis ack': 100.0, 'Acetogenesis pta': 100.0, 'Formate <=> CO2 + H2 hycE': 100.0, 'Formate oxidation fdoH': 100.0, 'Formate oxidation fdhB': 100.0, 'Aerobic CO oxidation coxS': 100.0, 'Aerobic CO oxidation coxM': 100.0, 'DC/4-HB 4hbl': 100.0, 'Wood Ljungdahl pathway (methyl branch) fthfs': 100.0, 'Wood Ljungdahl pathway (methyl branch) fhs': 100.0, 'N2 fixation nifD': 100.0, 'N2 fixation nifK': 100.0, 'N2 fixation nifH': 100.0, 'Nitrite reduction to ammonia nrfH': 100.0, 'Nitrite reduction to ammonia nrfA': 100.0, 'Sulfite reduction dsrD': 100.0, 'Sulfur oxidation dsrA': 100.0, 'Sulfur oxidation dsrB': 100.0, 'Sulfur oxidation sdo': 100.0, 'Sulfite reduction asrA': 100.0, 'Sulfite reduction asrB': 100.0, 'Sulfite reduction asrC': 100.0, 'Sulfate reduction aprA': 100.0, 'Sulfate reduction sat': 100.0, 'Sulfate reduction apsA': 100.0, 'Sulfate reduction apsK': 100.0, 'Thiosulfate disproportionation phsA': 100.0, 'Ni-Fe Hydrogenase nife-group-1': 100.0, 'Ni-Fe Hydrogenase nife-group-4a-g': 100.0, 'Complex I (NADH-quinone oxidoreductase) nuoB': 100.0, 'Complex I (NADH-quinone oxidoreductase) nuoC': 100.0, 'Complex V (F-type H+-transporting ATPase) atpA (F-type)': 100.0, 'Complex V (F-type H+-transporting ATPase) atpD (F-type)': 100.0, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydA': 100.0, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydB': 100.0, 'Halogenated compounds breakdown E3.8.1.2': 100.0, 'Iron reduction FmnB': 100.0, 'Iron reduction DmkB': 100.0, 'Iron reduction DFE_0449': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrA': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrB': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrC': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrD': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrJ': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrK': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrM': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrO': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrP': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrT': 100.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrA': 100.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrB': 100.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrC': 100.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation sdo': 100.0, 'Dissimilatory sulfur metabolism | Sulfate reduction aprA': 100.0, 'Dissimilatory sulfur metabolism | Sulfate reduction aprB': 100.0, 'Dissimilatory sulfur metabolism | Sulfate reduction sat': 100.0, 'Dissimilatory sulfur metabolism | Thiosulfate disproportionation phsA': 100.0, "Assimilatory sulfur metabolism| Sulfate <-> APS (Adenosine 5'-phosphosulfate) cysN": 100.0, "Assimilatory sulfur metabolism| APS (Adenosine 5'-phosphosulfate) <-> PAPS (3'-Phosphoadenosine 5'-phosphosulfate) cysC": 100.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysK': 100.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysM': 100.0, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) ddhA': 100.0, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) dmsA': 100.0, 'DMSO metabolism | DMS (Dimethyl sulfide) <-> Methanethiol dmoB': 100.0, 'Metabolism of organic sulfur | Cysteine <-> Cystathionine metB': 100.0, 'Metabolism of organic sulfur | Cysteine <-> 3-mercaptopyruvate aspB': 100.0, 'Metabolism of organic sulfur | OAHS (O-acetylhomoserine) + Sulfide -> Homocysteine metY': 100.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine metC': 100.0, 'Metabolism of organic sulfur | MHO (L-methionine (R)-S-oxide) -> Methionine) msrC': 100.0, 'Metabolism of organic sulfur | Homocysteine -> Methionine metH': 100.0, 'Metabolism of organic sulfur | Homocysteine <-> Methionine metE': 100.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> Homocysteine ahcY': 100.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> SRH (S-ribosylhomocysteine) mtnN': 100.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysK': 100.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysM': 100.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide metC': 100.0, 'Sulfur-related amino acid metabolism | Methionine -> Methanethiol megL': 100.0, 'Sulfur-related amino acid metabolism | Methionine -> SAMe (S-adenosyl-L-methionine) metK': 100.0}}, 'sample_2': {'tax_id_name_1': {'Aminotransferase class I and II aminotransferase class I and II': 0.0, 'Aspartate racemase aspartate racemase ': 0.0, 'Serine-pyruvate aminotransferase/archaeal aspartate aminotransferase serine-pyruvate aminotransferase/archaeal aspartate aminotransferase': 0.0, 'Fatty acid degradation acyl-CoA dehydrogenase': 0.0, 'Acetogenesis ack': 0.0, 'Formate oxidation fdwB': 0.0, 'Formate oxidation fdoH': 0.0, 'Wood Ljungdahl pathway (methyl branch) fthfs': 0.0, 'Wood Ljungdahl pathway (methyl branch) fhs': 0.0, 'Sulfur oxidation sdo': 0.0, 'Sulfite reduction asrB': 0.0, 'FeFe hydrogenase fefe-group-a2': 0.0, 'FeFe hydrogenase fefe-group-b': 0.0, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpA (V/A-type)': 0.0, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpB (V/A-type)': 0.0, 'Complex V (F-type H+-transporting ATPase) atpA (F-type)': 0.0, 'Complex V (F-type H+-transporting ATPase) atpD (F-type)': 0.0, 'Halogenated compounds breakdown E3.8.1.2': 0.0, 'Iron reduction FmnA': 0.0, 'Iron reduction FmnB': 0.0, 'Iron reduction EetA': 0.0, 'Iron reduction EetB': 0.0, 'Iron reduction DmkB': 0.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrB': 0.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation sdo': 0.0, "Assimilatory sulfur metabolism| Sulfate <-> APS (Adenosine 5'-phosphosulfate) cysN": 0.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysK': 0.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysM': 0.0, 'Metabolism of organic sulfur | Cysteine <-> Cystathionine metB': 0.0, 'Metabolism of organic sulfur | Cysteine <-> 3-mercaptopyruvate aspB': 0.0, 'Metabolism of organic sulfur | OAHS (O-acetylhomoserine) + Sulfide -> Homocysteine metY': 0.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine malY': 0.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine metC': 0.0, 'Metabolism of organic sulfur | MHO (L-methionine (R)-S-oxide) -> Methionine) msrC': 0.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> SRH (S-ribosylhomocysteine) mtnN': 0.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysK': 0.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysM': 0.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide malY': 0.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide metC': 0.0, 'Sulfur-related amino acid metabolism | Methionine -> Methanethiol megL': 0.0, 'Sulfur-related amino acid metabolism | Methionine -> SAMe (S-adenosyl-L-methionine) metK': 0.0}, 'tax_id_name_2': {'4-aminobutyrate aminotransferase and related aminotransferases 4-aminobutyrate aminotransferase and related aminotransferases': 800.0, 'Aminotransferase class I and II aminotransferase class I and II': 800.0, 'Aspartate racemase aspartate racemase ': 800.0, 'Histidinol-phosphate/aromatic aminotransferase histidinol-phosphate/aromatic aminotransferase': 800.0, 'Serine-pyruvate aminotransferase/archaeal aspartate aminotransferase serine-pyruvate aminotransferase/archaeal aspartate aminotransferase': 800.0, 'Phenol => Benzoyl-CoA ubiX': 800.0, 'Alcohol utilization adh': 800.0, 'Acetogenesis ack': 800.0, 'Acetogenesis pta': 800.0, 'Formate <=> CO2 + H2 hycE': 800.0, 'Formate oxidation fdoH': 800.0, 'Formate oxidation fdhB': 800.0, 'Aerobic CO oxidation coxS': 800.0, 'Aerobic CO oxidation coxM': 800.0, 'DC/4-HB 4hbl': 800.0, 'Wood Ljungdahl pathway (methyl branch) fthfs': 800.0, 'Wood Ljungdahl pathway (methyl branch) fhs': 800.0, 'N2 fixation nifD': 800.0, 'N2 fixation nifK': 800.0, 'N2 fixation nifH': 800.0, 'Nitrite reduction to ammonia nrfH': 800.0, 'Nitrite reduction to ammonia nrfA': 800.0, 'Sulfite reduction dsrD': 800.0, 'Sulfur oxidation dsrA': 800.0, 'Sulfur oxidation dsrB': 800.0, 'Sulfur oxidation sdo': 800.0, 'Sulfite reduction asrA': 800.0, 'Sulfite reduction asrB': 800.0, 'Sulfite reduction asrC': 800.0, 'Sulfate reduction aprA': 800.0, 'Sulfate reduction sat': 800.0, 'Sulfate reduction apsA': 800.0, 'Sulfate reduction apsK': 800.0, 'Thiosulfate disproportionation phsA': 800.0, 'Ni-Fe Hydrogenase nife-group-1': 800.0, 'Ni-Fe Hydrogenase nife-group-4a-g': 800.0, 'Complex I (NADH-quinone oxidoreductase) nuoB': 800.0, 'Complex I (NADH-quinone oxidoreductase) nuoC': 800.0, 'Complex V (F-type H+-transporting ATPase) atpA (F-type)': 800.0, 'Complex V (F-type H+-transporting ATPase) atpD (F-type)': 800.0, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydA': 800.0, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydB': 800.0, 'Halogenated compounds breakdown E3.8.1.2': 800.0, 'Iron reduction FmnB': 800.0, 'Iron reduction DmkB': 800.0, 'Iron reduction DFE_0449': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrA': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrB': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrC': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrD': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrJ': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrK': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrM': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrO': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrP': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrT': 800.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrA': 800.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrB': 800.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrC': 800.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation sdo': 800.0, 'Dissimilatory sulfur metabolism | Sulfate reduction aprA': 800.0, 'Dissimilatory sulfur metabolism | Sulfate reduction aprB': 800.0, 'Dissimilatory sulfur metabolism | Sulfate reduction sat': 800.0, 'Dissimilatory sulfur metabolism | Thiosulfate disproportionation phsA': 800.0, "Assimilatory sulfur metabolism| Sulfate <-> APS (Adenosine 5'-phosphosulfate) cysN": 800.0, "Assimilatory sulfur metabolism| APS (Adenosine 5'-phosphosulfate) <-> PAPS (3'-Phosphoadenosine 5'-phosphosulfate) cysC": 800.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysK': 800.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysM': 800.0, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) ddhA': 800.0, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) dmsA': 800.0, 'DMSO metabolism | DMS (Dimethyl sulfide) <-> Methanethiol dmoB': 800.0, 'Metabolism of organic sulfur | Cysteine <-> Cystathionine metB': 800.0, 'Metabolism of organic sulfur | Cysteine <-> 3-mercaptopyruvate aspB': 800.0, 'Metabolism of organic sulfur | OAHS (O-acetylhomoserine) + Sulfide -> Homocysteine metY': 800.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine metC': 800.0, 'Metabolism of organic sulfur | MHO (L-methionine (R)-S-oxide) -> Methionine) msrC': 800.0, 'Metabolism of organic sulfur | Homocysteine -> Methionine metH': 800.0, 'Metabolism of organic sulfur | Homocysteine <-> Methionine metE': 800.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> Homocysteine ahcY': 800.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> SRH (S-ribosylhomocysteine) mtnN': 800.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysK': 800.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysM': 800.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide metC': 800.0, 'Sulfur-related amino acid metabolism | Methionine -> Methanethiol megL': 800.0, 'Sulfur-related amino acid metabolism | Methionine -> SAMe (S-adenosyl-L-methionine) metK': 800.0}}, 'sample_3': {'tax_id_name_1': {'Aminotransferase class I and II aminotransferase class I and II': 0.0, 'Aspartate racemase aspartate racemase ': 0.0, 'Serine-pyruvate aminotransferase/archaeal aspartate aminotransferase serine-pyruvate aminotransferase/archaeal aspartate aminotransferase': 0.0, 'Fatty acid degradation acyl-CoA dehydrogenase': 0.0, 'Acetogenesis ack': 0.0, 'Formate oxidation fdwB': 0.0, 'Formate oxidation fdoH': 0.0, 'Wood Ljungdahl pathway (methyl branch) fthfs': 0.0, 'Wood Ljungdahl pathway (methyl branch) fhs': 0.0, 'Sulfur oxidation sdo': 0.0, 'Sulfite reduction asrB': 0.0, 'FeFe hydrogenase fefe-group-a2': 0.0, 'FeFe hydrogenase fefe-group-b': 0.0, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpA (V/A-type)': 0.0, 'Complex V (ATP synthase: V/A-type H+/Na+-transporting ATPase) atpB (V/A-type)': 0.0, 'Complex V (F-type H+-transporting ATPase) atpA (F-type)': 0.0, 'Complex V (F-type H+-transporting ATPase) atpD (F-type)': 0.0, 'Halogenated compounds breakdown E3.8.1.2': 0.0, 'Iron reduction FmnA': 0.0, 'Iron reduction FmnB': 0.0, 'Iron reduction EetA': 0.0, 'Iron reduction EetB': 0.0, 'Iron reduction DmkB': 0.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrB': 0.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation sdo': 0.0, "Assimilatory sulfur metabolism| Sulfate <-> APS (Adenosine 5'-phosphosulfate) cysN": 0.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysK': 0.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysM': 0.0, 'Metabolism of organic sulfur | Cysteine <-> Cystathionine metB': 0.0, 'Metabolism of organic sulfur | Cysteine <-> 3-mercaptopyruvate aspB': 0.0, 'Metabolism of organic sulfur | OAHS (O-acetylhomoserine) + Sulfide -> Homocysteine metY': 0.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine malY': 0.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine metC': 0.0, 'Metabolism of organic sulfur | MHO (L-methionine (R)-S-oxide) -> Methionine) msrC': 0.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> SRH (S-ribosylhomocysteine) mtnN': 0.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysK': 0.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysM': 0.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide malY': 0.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide metC': 0.0, 'Sulfur-related amino acid metabolism | Methionine -> Methanethiol megL': 0.0, 'Sulfur-related amino acid metabolism | Methionine -> SAMe (S-adenosyl-L-methionine) metK': 0.0}, 'tax_id_name_2': {'4-aminobutyrate aminotransferase and related aminotransferases 4-aminobutyrate aminotransferase and related aminotransferases': 520.0, 'Aminotransferase class I and II aminotransferase class I and II': 520.0, 'Aspartate racemase aspartate racemase ': 520.0, 'Histidinol-phosphate/aromatic aminotransferase histidinol-phosphate/aromatic aminotransferase': 520.0, 'Serine-pyruvate aminotransferase/archaeal aspartate aminotransferase serine-pyruvate aminotransferase/archaeal aspartate aminotransferase': 520.0, 'Phenol => Benzoyl-CoA ubiX': 520.0, 'Alcohol utilization adh': 520.0, 'Acetogenesis ack': 520.0, 'Acetogenesis pta': 520.0, 'Formate <=> CO2 + H2 hycE': 520.0, 'Formate oxidation fdoH': 520.0, 'Formate oxidation fdhB': 520.0, 'Aerobic CO oxidation coxS': 520.0, 'Aerobic CO oxidation coxM': 520.0, 'DC/4-HB 4hbl': 520.0, 'Wood Ljungdahl pathway (methyl branch) fthfs': 520.0, 'Wood Ljungdahl pathway (methyl branch) fhs': 520.0, 'N2 fixation nifD': 520.0, 'N2 fixation nifK': 520.0, 'N2 fixation nifH': 520.0, 'Nitrite reduction to ammonia nrfH': 520.0, 'Nitrite reduction to ammonia nrfA': 520.0, 'Sulfite reduction dsrD': 520.0, 'Sulfur oxidation dsrA': 520.0, 'Sulfur oxidation dsrB': 520.0, 'Sulfur oxidation sdo': 520.0, 'Sulfite reduction asrA': 520.0, 'Sulfite reduction asrB': 520.0, 'Sulfite reduction asrC': 520.0, 'Sulfate reduction aprA': 520.0, 'Sulfate reduction sat': 520.0, 'Sulfate reduction apsA': 520.0, 'Sulfate reduction apsK': 520.0, 'Thiosulfate disproportionation phsA': 520.0, 'Ni-Fe Hydrogenase nife-group-1': 520.0, 'Ni-Fe Hydrogenase nife-group-4a-g': 520.0, 'Complex I (NADH-quinone oxidoreductase) nuoB': 520.0, 'Complex I (NADH-quinone oxidoreductase) nuoC': 520.0, 'Complex V (F-type H+-transporting ATPase) atpA (F-type)': 520.0, 'Complex V (F-type H+-transporting ATPase) atpD (F-type)': 520.0, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydA': 520.0, 'Oxygen metabolism - cytochrome (quinone) oxidase, bd type cydB': 520.0, 'Halogenated compounds breakdown E3.8.1.2': 520.0, 'Iron reduction FmnB': 520.0, 'Iron reduction DmkB': 520.0, 'Iron reduction DFE_0449': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrA': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrB': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrC': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrD': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrJ': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrK': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrM': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrO': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrP': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation/Sulfite reduction dsrT': 520.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrA': 520.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrB': 520.0, 'Dissimilatory sulfur metabolism | Sulfite reduction asrC': 520.0, 'Dissimilatory sulfur metabolism | Sulfur oxidation sdo': 520.0, 'Dissimilatory sulfur metabolism | Sulfate reduction aprA': 520.0, 'Dissimilatory sulfur metabolism | Sulfate reduction aprB': 520.0, 'Dissimilatory sulfur metabolism | Sulfate reduction sat': 520.0, 'Dissimilatory sulfur metabolism | Thiosulfate disproportionation phsA': 520.0, "Assimilatory sulfur metabolism| Sulfate <-> APS (Adenosine 5'-phosphosulfate) cysN": 520.0, "Assimilatory sulfur metabolism| APS (Adenosine 5'-phosphosulfate) <-> PAPS (3'-Phosphoadenosine 5'-phosphosulfate) cysC": 520.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysK': 520.0, 'Assimilatory sulfur metabolism| Sulfide -> Cysteine cysM': 520.0, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) ddhA': 520.0, 'DMSO metabolism | DMSO (Dimethyl sulfoxide) <-> DMS (Dimethyl sulfide) dmsA': 520.0, 'DMSO metabolism | DMS (Dimethyl sulfide) <-> Methanethiol dmoB': 520.0, 'Metabolism of organic sulfur | Cysteine <-> Cystathionine metB': 520.0, 'Metabolism of organic sulfur | Cysteine <-> 3-mercaptopyruvate aspB': 520.0, 'Metabolism of organic sulfur | OAHS (O-acetylhomoserine) + Sulfide -> Homocysteine metY': 520.0, 'Metabolism of organic sulfur | Cystathionine -> Homocysteine metC': 520.0, 'Metabolism of organic sulfur | MHO (L-methionine (R)-S-oxide) -> Methionine) msrC': 520.0, 'Metabolism of organic sulfur | Homocysteine -> Methionine metH': 520.0, 'Metabolism of organic sulfur | Homocysteine <-> Methionine metE': 520.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> Homocysteine ahcY': 520.0, 'Metabolism of organic sulfur | SAH (S-adenosyl-L-homocysteine) -> SRH (S-ribosylhomocysteine) mtnN': 520.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysK': 520.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide cysM': 520.0, 'Sulfur-related amino acid metabolism | Cysteine -> Sulfide metC': 520.0, 'Sulfur-related amino acid metabolism | Methionine -> Methanethiol megL': 520.0, 'Sulfur-related amino acid metabolism | Methionine -> SAMe (S-adenosyl-L-methionine) metK': 520.0}}}

    for sample in expected_function_abundance_samples:
        for function in expected_function_abundance_samples[sample]:
            assert function_abundance_samples[sample][function] == expected_function_abundance_samples[sample][function]

    for sample in expected_function_relative_abundance_samples:
        for function in expected_function_relative_abundance_samples[sample]:
            assert function_relative_abundance_samples[sample][function] == expected_function_relative_abundance_samples[sample][function]

    for sample in expected_function_participation_samples:
        for organism in expected_function_participation_samples[sample]:
            for function in expected_function_participation_samples[sample][organism]:
                assert function_participation_samples[sample][organism][function] == expected_function_participation_samples[sample][organism][function]


def test_compute_bigecyhmm_functions_occurrence_cycles():
    bigecyhmm_cycle_file = os.path.join('input_data', 'bigecyhmm_output_folder', 'pathway_presence.tsv')

    function_occurrence_organisms, all_studied_organisms = compute_bigecyhmm_functions_occurrence(bigecyhmm_cycle_file)

    expected_function_occurrence_organisms = {'C-S-01:Organic carbon oxidation': {'tax_id_name_1': 1, 'tax_id_name_2': 1}, 'C-S-02:Carbon fixation': {'tax_id_name_1': 1, 'tax_id_name_2': 1}, 'C-S-03:Ethanol oxidation': {'tax_id_name_1': 1}, 'C-S-04:Acetate oxidation': {'tax_id_name_1': 1, 'tax_id_name_2': 1}, 'C-S-05:Hydrogen generation': {'tax_id_name_2': 1}, 'C-S-06:Fermentation': {'tax_id_name_1': 1, 'tax_id_name_2': 1}, 'C-S-07:Methanogenesis': {'tax_id_name_1': 1}, 'C-S-09:Hydrogen oxidation': {'tax_id_name_2': 1}, 'N-S-01:Nitrogen fixation': {'tax_id_name_1': 1, 'tax_id_name_2': 1}, 'N-S-07:Nitrous oxide reduction': {'tax_id_name_1': 1}, 'N-S-08:Nitrite ammonification': {'tax_id_name_1': 1, 'tax_id_name_2': 1}, 'O-S-01:Iron reduction': {'tax_id_name_1': 1}, 'O-S-03:Arsenate reduction': {'tax_id_name_2': 1}, 'S-S-03:Sulfur oxidation': {'tax_id_name_1': 1, 'tax_id_name_2': 1}, 'S-S-04:Sulfite oxidation': {'tax_id_name_2': 1}, 'S-S-05:Sulfate reduction': {'tax_id_name_2': 1}, 'S-S-06:Sulfite reduction': {'tax_id_name_1': 1, 'tax_id_name_2': 1}, 'S-S-08:Thiosulfate disproportionation 1': {'tax_id_name_2': 1}}

    for function in expected_function_occurrence_organisms:
        for organism in expected_function_occurrence_organisms[function]:
            assert function_occurrence_organisms[function][organism] == expected_function_occurrence_organisms[function][organism]


def test_compute_bigecyhmm_functions_occurrence_cycles_from_esmecata():
    bigecyhmm_cycle_file = os.path.join('input_data', 'bigecyhmm_output_folder', 'pathway_presence.tsv')
    proteome_tax_id_file = os.path.join('input_data', 'esmecata_output_folder', '0_proteomes', 'proteome_tax_id.tsv')

    observation_names_tax_id_names, observation_names_tax_ranks = read_esmecata_proteome_file(proteome_tax_id_file)

    tax_id_names_observation_names = {}
    for observation_name in observation_names_tax_id_names:
        tax_id_name = observation_names_tax_id_names[observation_name]
        if tax_id_name not in tax_id_names_observation_names:
            tax_id_names_observation_names[tax_id_name] = [observation_name]
        else:
            tax_id_names_observation_names[tax_id_name].append(observation_name)
    function_occurrence_organisms, all_studied_organisms = compute_bigecyhmm_functions_occurrence(bigecyhmm_cycle_file, tax_id_names_observation_names)

    expected_function_occurrence_organisms = {'C-S-01:Organic carbon oxidation': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'C-S-02:Carbon fixation': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'C-S-03:Ethanol oxidation': {'org_1': 1}, 'C-S-04:Acetate oxidation': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'C-S-05:Hydrogen generation': {'org_2': 1, 'org_3': 1}, 'C-S-06:Fermentation': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'C-S-07:Methanogenesis': {'org_1': 1}, 'C-S-09:Hydrogen oxidation': {'org_2': 1, 'org_3': 1}, 'N-S-01:Nitrogen fixation': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'N-S-07:Nitrous oxide reduction': {'org_1': 1}, 'N-S-08:Nitrite ammonification': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'O-S-01:Iron reduction': {'org_1': 1}, 'O-S-03:Arsenate reduction': {'org_2': 1, 'org_3': 1}, 'S-S-03:Sulfur oxidation': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'S-S-04:Sulfite oxidation': {'org_2': 1, 'org_3': 1}, 'S-S-05:Sulfate reduction': {'org_2': 1, 'org_3': 1}, 'S-S-06:Sulfite reduction': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'S-S-08:Thiosulfate disproportionation 1': {'org_2': 1, 'org_3': 1}}

    for function in expected_function_occurrence_organisms:
        for organism in expected_function_occurrence_organisms[function]:
            assert function_occurrence_organisms[function][organism] == expected_function_occurrence_organisms[function][organism]


def test_compute_bigecyhmm_functions_abundance_cycles():
    abundance_file_path = os.path.join('input_data', 'abundance_file_from_genomes.tsv')
    bigecyhmm_cycle_file = os.path.join('input_data', 'bigecyhmm_output_folder', 'pathway_presence.tsv')

    sample_abundance, sample_tot_abundance = read_measures_file(abundance_file_path)

    function_abundance_samples, function_relative_abundance_samples, function_participation_samples = compute_bigecyhmm_functions_abundance(bigecyhmm_cycle_file, sample_abundance, sample_tot_abundance)

    expected_function_abundance_samples = {'sample_1': {'C-S-01:Organic carbon oxidation': 200, 'C-S-02:Carbon fixation': 200, 'C-S-03:Ethanol oxidation': 100, 'C-S-04:Acetate oxidation': 200, 'C-S-06:Fermentation': 200, 'C-S-07:Methanogenesis': 100, 'N-S-01:Nitrogen fixation': 200, 'N-S-07:Nitrous oxide reduction': 100, 'N-S-08:Nitrite ammonification': 200, 'O-S-01:Iron reduction': 100, 'S-S-03:Sulfur oxidation': 200, 'S-S-06:Sulfite reduction': 200, 'C-S-05:Hydrogen generation': 100, 'C-S-09:Hydrogen oxidation': 100, 'O-S-03:Arsenate reduction': 100, 'S-S-04:Sulfite oxidation': 100, 'S-S-05:Sulfate reduction': 100, 'S-S-08:Thiosulfate disproportionation 1': 100}, 'sample_2': {'C-S-01:Organic carbon oxidation': 800, 'C-S-02:Carbon fixation': 800, 'C-S-03:Ethanol oxidation': 0, 'C-S-04:Acetate oxidation': 800, 'C-S-06:Fermentation': 800, 'C-S-07:Methanogenesis': 0, 'N-S-01:Nitrogen fixation': 800, 'N-S-07:Nitrous oxide reduction': 0, 'N-S-08:Nitrite ammonification': 800, 'O-S-01:Iron reduction': 0, 'S-S-03:Sulfur oxidation': 800, 'S-S-06:Sulfite reduction': 800, 'C-S-05:Hydrogen generation': 800, 'C-S-09:Hydrogen oxidation': 800, 'O-S-03:Arsenate reduction': 800, 'S-S-04:Sulfite oxidation': 800, 'S-S-05:Sulfate reduction': 800, 'S-S-08:Thiosulfate disproportionation 1': 800}, 'sample_3': {'C-S-01:Organic carbon oxidation': 520, 'C-S-02:Carbon fixation': 520, 'C-S-03:Ethanol oxidation': 0, 'C-S-04:Acetate oxidation': 520, 'C-S-06:Fermentation': 520, 'C-S-07:Methanogenesis': 0, 'N-S-01:Nitrogen fixation': 520, 'N-S-07:Nitrous oxide reduction': 0, 'N-S-08:Nitrite ammonification': 520, 'O-S-01:Iron reduction': 0, 'S-S-03:Sulfur oxidation': 520, 'S-S-06:Sulfite reduction': 520, 'C-S-05:Hydrogen generation': 520, 'C-S-09:Hydrogen oxidation': 520, 'O-S-03:Arsenate reduction': 520, 'S-S-04:Sulfite oxidation': 520, 'S-S-05:Sulfate reduction': 520, 'S-S-08:Thiosulfate disproportionation 1': 520}}
    expected_function_relative_abundance_samples = {'sample_1': {'C-S-01:Organic carbon oxidation': 1.0, 'C-S-02:Carbon fixation': 1.0, 'C-S-03:Ethanol oxidation': 0.5, 'C-S-04:Acetate oxidation': 1.0, 'C-S-06:Fermentation': 1.0, 'C-S-07:Methanogenesis': 0.5, 'N-S-01:Nitrogen fixation': 1.0, 'N-S-07:Nitrous oxide reduction': 0.5, 'N-S-08:Nitrite ammonification': 1.0, 'O-S-01:Iron reduction': 0.5, 'S-S-03:Sulfur oxidation': 1.0, 'S-S-06:Sulfite reduction': 1.0, 'C-S-05:Hydrogen generation': 0.5, 'C-S-09:Hydrogen oxidation': 0.5, 'O-S-03:Arsenate reduction': 0.5, 'S-S-04:Sulfite oxidation': 0.5, 'S-S-05:Sulfate reduction': 0.5, 'S-S-08:Thiosulfate disproportionation 1': 0.5}, 'sample_2': {'C-S-01:Organic carbon oxidation': 1.0, 'C-S-02:Carbon fixation': 1.0, 'C-S-03:Ethanol oxidation': 0.0, 'C-S-04:Acetate oxidation': 1.0, 'C-S-06:Fermentation': 1.0, 'C-S-07:Methanogenesis': 0.0, 'N-S-01:Nitrogen fixation': 1.0, 'N-S-07:Nitrous oxide reduction': 0.0, 'N-S-08:Nitrite ammonification': 1.0, 'O-S-01:Iron reduction': 0.0, 'S-S-03:Sulfur oxidation': 1.0, 'S-S-06:Sulfite reduction': 1.0, 'C-S-05:Hydrogen generation': 1.0, 'C-S-09:Hydrogen oxidation': 1.0, 'O-S-03:Arsenate reduction': 1.0, 'S-S-04:Sulfite oxidation': 1.0, 'S-S-05:Sulfate reduction': 1.0, 'S-S-08:Thiosulfate disproportionation 1': 1.0}, 'sample_3': {'C-S-01:Organic carbon oxidation': 1.0, 'C-S-02:Carbon fixation': 1.0, 'C-S-03:Ethanol oxidation': 0.0, 'C-S-04:Acetate oxidation': 1.0, 'C-S-06:Fermentation': 1.0, 'C-S-07:Methanogenesis': 0.0, 'N-S-01:Nitrogen fixation': 1.0, 'N-S-07:Nitrous oxide reduction': 0.0, 'N-S-08:Nitrite ammonification': 1.0, 'O-S-01:Iron reduction': 0.0, 'S-S-03:Sulfur oxidation': 1.0, 'S-S-06:Sulfite reduction': 1.0, 'C-S-05:Hydrogen generation': 1.0, 'C-S-09:Hydrogen oxidation': 1.0, 'O-S-03:Arsenate reduction': 1.0, 'S-S-04:Sulfite oxidation': 1.0, 'S-S-05:Sulfate reduction': 1.0, 'S-S-08:Thiosulfate disproportionation 1': 1.0}}
    expected_function_participation_samples = {'sample_1': {'tax_id_name_1': {'C-S-01:Organic carbon oxidation': 100, 'C-S-02:Carbon fixation': 100, 'C-S-03:Ethanol oxidation': 100, 'C-S-04:Acetate oxidation': 100, 'C-S-06:Fermentation': 100, 'C-S-07:Methanogenesis': 100, 'N-S-01:Nitrogen fixation': 100, 'N-S-07:Nitrous oxide reduction': 100, 'N-S-08:Nitrite ammonification': 100, 'O-S-01:Iron reduction': 100, 'S-S-03:Sulfur oxidation': 100, 'S-S-06:Sulfite reduction': 100}, 'tax_id_name_2': {'C-S-01:Organic carbon oxidation': 100, 'C-S-02:Carbon fixation': 100, 'C-S-04:Acetate oxidation': 100, 'C-S-05:Hydrogen generation': 100, 'C-S-06:Fermentation': 100, 'C-S-09:Hydrogen oxidation': 100, 'N-S-01:Nitrogen fixation': 100, 'N-S-08:Nitrite ammonification': 100, 'O-S-03:Arsenate reduction': 100, 'S-S-03:Sulfur oxidation': 100, 'S-S-04:Sulfite oxidation': 100, 'S-S-05:Sulfate reduction': 100, 'S-S-06:Sulfite reduction': 100, 'S-S-08:Thiosulfate disproportionation 1': 100}}, 'sample_2': {'tax_id_name_1': {'C-S-01:Organic carbon oxidation': 0, 'C-S-02:Carbon fixation': 0, 'C-S-03:Ethanol oxidation': 0, 'C-S-04:Acetate oxidation': 0, 'C-S-06:Fermentation': 0, 'C-S-07:Methanogenesis': 0, 'N-S-01:Nitrogen fixation': 0, 'N-S-07:Nitrous oxide reduction': 0, 'N-S-08:Nitrite ammonification': 0, 'O-S-01:Iron reduction': 0, 'S-S-03:Sulfur oxidation': 0, 'S-S-06:Sulfite reduction': 0}, 'tax_id_name_2': {'C-S-01:Organic carbon oxidation': 800, 'C-S-02:Carbon fixation': 800, 'C-S-04:Acetate oxidation': 800, 'C-S-05:Hydrogen generation': 800, 'C-S-06:Fermentation': 800, 'C-S-09:Hydrogen oxidation': 800, 'N-S-01:Nitrogen fixation': 800, 'N-S-08:Nitrite ammonification': 800, 'O-S-03:Arsenate reduction': 800, 'S-S-03:Sulfur oxidation': 800, 'S-S-04:Sulfite oxidation': 800, 'S-S-05:Sulfate reduction': 800, 'S-S-06:Sulfite reduction': 800, 'S-S-08:Thiosulfate disproportionation 1': 800}}, 'sample_3': {'tax_id_name_1': {'C-S-01:Organic carbon oxidation': 0, 'C-S-02:Carbon fixation': 0, 'C-S-03:Ethanol oxidation': 0, 'C-S-04:Acetate oxidation': 0, 'C-S-06:Fermentation': 0, 'C-S-07:Methanogenesis': 0, 'N-S-01:Nitrogen fixation': 0, 'N-S-07:Nitrous oxide reduction': 0, 'N-S-08:Nitrite ammonification': 0, 'O-S-01:Iron reduction': 0, 'S-S-03:Sulfur oxidation': 0, 'S-S-06:Sulfite reduction': 0}, 'tax_id_name_2': {'C-S-01:Organic carbon oxidation': 520, 'C-S-02:Carbon fixation': 520, 'C-S-04:Acetate oxidation': 520, 'C-S-05:Hydrogen generation': 520, 'C-S-06:Fermentation': 520, 'C-S-09:Hydrogen oxidation': 520, 'N-S-01:Nitrogen fixation': 520, 'N-S-08:Nitrite ammonification': 520, 'O-S-03:Arsenate reduction': 520, 'S-S-03:Sulfur oxidation': 520, 'S-S-04:Sulfite oxidation': 520, 'S-S-05:Sulfate reduction': 520, 'S-S-06:Sulfite reduction': 520, 'S-S-08:Thiosulfate disproportionation 1': 520}}}

    for sample in expected_function_abundance_samples:
        for function in expected_function_abundance_samples[sample]:
            assert function_abundance_samples[sample][function] == expected_function_abundance_samples[sample][function]

    for sample in expected_function_relative_abundance_samples:
        for function in expected_function_relative_abundance_samples[sample]:
            assert function_relative_abundance_samples[sample][function] == expected_function_relative_abundance_samples[sample][function]

    for sample in expected_function_participation_samples:
        for organism in expected_function_participation_samples[sample]:
            for function in expected_function_participation_samples[sample][organism]:
                assert function_participation_samples[sample][organism][function] == expected_function_participation_samples[sample][organism][function]


def test_compute_bigecyhmm_functions_abundance_cycles_from_esmecata():
    proteome_tax_id_file = os.path.join('input_data', 'esmecata_output_folder', '0_proteomes', 'proteome_tax_id.tsv')
    abundance_file_path = os.path.join('input_data', 'proteome_tax_id_abundance.tsv')
    bigecyhmm_cycle_file = os.path.join('input_data', 'bigecyhmm_output_folder', 'pathway_presence.tsv')

    sample_abundance, sample_tot_abundance = read_measures_file(abundance_file_path)
    observation_names_tax_id_names, observation_names_tax_ranks = read_esmecata_proteome_file(proteome_tax_id_file)

    tax_id_names_observation_names = {}
    for observation_name in observation_names_tax_id_names:
        tax_id_name = observation_names_tax_id_names[observation_name]
        if tax_id_name not in tax_id_names_observation_names:
            tax_id_names_observation_names[tax_id_name] = [observation_name]
        else:
            tax_id_names_observation_names[tax_id_name].append(observation_name)

    function_abundance_samples, function_relative_abundance_samples, function_participation_samples = compute_bigecyhmm_functions_abundance(bigecyhmm_cycle_file, sample_abundance, sample_tot_abundance, tax_id_names_observation_names)

    expected_function_abundance_samples = {'sample_1': {'C-S-01:Organic carbon oxidation': 200, 'C-S-02:Carbon fixation': 200, 'C-S-03:Ethanol oxidation': 100, 'C-S-04:Acetate oxidation': 200, 'C-S-06:Fermentation': 200, 'C-S-07:Methanogenesis': 100, 'N-S-01:Nitrogen fixation': 200, 'N-S-07:Nitrous oxide reduction': 100, 'N-S-08:Nitrite ammonification': 200, 'O-S-01:Iron reduction': 100, 'S-S-03:Sulfur oxidation': 200, 'S-S-06:Sulfite reduction': 200, 'C-S-05:Hydrogen generation': 100, 'C-S-09:Hydrogen oxidation': 100, 'O-S-03:Arsenate reduction': 100, 'S-S-04:Sulfite oxidation': 100, 'S-S-05:Sulfate reduction': 100, 'S-S-08:Thiosulfate disproportionation 1': 100}, 'sample_2': {'C-S-01:Organic carbon oxidation': 800, 'C-S-02:Carbon fixation': 800, 'C-S-03:Ethanol oxidation': 0, 'C-S-04:Acetate oxidation': 800, 'C-S-06:Fermentation': 800, 'C-S-07:Methanogenesis': 0, 'N-S-01:Nitrogen fixation': 800, 'N-S-07:Nitrous oxide reduction': 0, 'N-S-08:Nitrite ammonification': 800, 'O-S-01:Iron reduction': 0, 'S-S-03:Sulfur oxidation': 800, 'S-S-06:Sulfite reduction': 800, 'C-S-05:Hydrogen generation': 800, 'C-S-09:Hydrogen oxidation': 800, 'O-S-03:Arsenate reduction': 800, 'S-S-04:Sulfite oxidation': 800, 'S-S-05:Sulfate reduction': 800, 'S-S-08:Thiosulfate disproportionation 1': 800}, 'sample_3': {'C-S-01:Organic carbon oxidation': 520, 'C-S-02:Carbon fixation': 520, 'C-S-03:Ethanol oxidation': 0, 'C-S-04:Acetate oxidation': 520, 'C-S-06:Fermentation': 520, 'C-S-07:Methanogenesis': 0, 'N-S-01:Nitrogen fixation': 520, 'N-S-07:Nitrous oxide reduction': 0, 'N-S-08:Nitrite ammonification': 520, 'O-S-01:Iron reduction': 0, 'S-S-03:Sulfur oxidation': 520, 'S-S-06:Sulfite reduction': 520, 'C-S-05:Hydrogen generation': 520, 'C-S-09:Hydrogen oxidation': 520, 'O-S-03:Arsenate reduction': 520, 'S-S-04:Sulfite oxidation': 520, 'S-S-05:Sulfate reduction': 520, 'S-S-08:Thiosulfate disproportionation 1': 520}}
    expected_function_relative_abundance_samples = {'sample_1': {'C-S-01:Organic carbon oxidation': 1.0, 'C-S-02:Carbon fixation': 1.0, 'C-S-03:Ethanol oxidation': 0.5, 'C-S-04:Acetate oxidation': 1.0, 'C-S-06:Fermentation': 1.0, 'C-S-07:Methanogenesis': 0.5, 'N-S-01:Nitrogen fixation': 1.0, 'N-S-07:Nitrous oxide reduction': 0.5, 'N-S-08:Nitrite ammonification': 1.0, 'O-S-01:Iron reduction': 0.5, 'S-S-03:Sulfur oxidation': 1.0, 'S-S-06:Sulfite reduction': 1.0, 'C-S-05:Hydrogen generation': 0.5, 'C-S-09:Hydrogen oxidation': 0.5, 'O-S-03:Arsenate reduction': 0.5, 'S-S-04:Sulfite oxidation': 0.5, 'S-S-05:Sulfate reduction': 0.5, 'S-S-08:Thiosulfate disproportionation 1': 0.5}, 'sample_2': {'C-S-01:Organic carbon oxidation': 1.0, 'C-S-02:Carbon fixation': 1.0, 'C-S-03:Ethanol oxidation': 0.0, 'C-S-04:Acetate oxidation': 1.0, 'C-S-06:Fermentation': 1.0, 'C-S-07:Methanogenesis': 0.0, 'N-S-01:Nitrogen fixation': 1.0, 'N-S-07:Nitrous oxide reduction': 0.0, 'N-S-08:Nitrite ammonification': 1.0, 'O-S-01:Iron reduction': 0.0, 'S-S-03:Sulfur oxidation': 1.0, 'S-S-06:Sulfite reduction': 1.0, 'C-S-05:Hydrogen generation': 1.0, 'C-S-09:Hydrogen oxidation': 1.0, 'O-S-03:Arsenate reduction': 1.0, 'S-S-04:Sulfite oxidation': 1.0, 'S-S-05:Sulfate reduction': 1.0, 'S-S-08:Thiosulfate disproportionation 1': 1.0}, 'sample_3': {'C-S-01:Organic carbon oxidation': 1.0, 'C-S-02:Carbon fixation': 1.0, 'C-S-03:Ethanol oxidation': 0.0, 'C-S-04:Acetate oxidation': 1.0, 'C-S-06:Fermentation': 1.0, 'C-S-07:Methanogenesis': 0.0, 'N-S-01:Nitrogen fixation': 1.0, 'N-S-07:Nitrous oxide reduction': 0.0, 'N-S-08:Nitrite ammonification': 1.0, 'O-S-01:Iron reduction': 0.0, 'S-S-03:Sulfur oxidation': 1.0, 'S-S-06:Sulfite reduction': 1.0, 'C-S-05:Hydrogen generation': 1.0, 'C-S-09:Hydrogen oxidation': 1.0, 'O-S-03:Arsenate reduction': 1.0, 'S-S-04:Sulfite oxidation': 1.0, 'S-S-05:Sulfate reduction': 1.0, 'S-S-08:Thiosulfate disproportionation 1': 1.0}}
    expected_function_participation_samples = {'sample_1': {'org_1': {'C-S-01:Organic carbon oxidation': 100, 'C-S-02:Carbon fixation': 100, 'C-S-03:Ethanol oxidation': 100, 'C-S-04:Acetate oxidation': 100, 'C-S-06:Fermentation': 100, 'C-S-07:Methanogenesis': 100, 'N-S-01:Nitrogen fixation': 100, 'N-S-07:Nitrous oxide reduction': 100, 'N-S-08:Nitrite ammonification': 100, 'O-S-01:Iron reduction': 100, 'S-S-03:Sulfur oxidation': 100, 'S-S-06:Sulfite reduction': 100}, 'org_2': {'C-S-01:Organic carbon oxidation': 100, 'C-S-02:Carbon fixation': 100, 'C-S-04:Acetate oxidation': 100, 'C-S-05:Hydrogen generation': 100, 'C-S-06:Fermentation': 100, 'C-S-09:Hydrogen oxidation': 100, 'N-S-01:Nitrogen fixation': 100, 'N-S-08:Nitrite ammonification': 100, 'O-S-03:Arsenate reduction': 100, 'S-S-03:Sulfur oxidation': 100, 'S-S-04:Sulfite oxidation': 100, 'S-S-05:Sulfate reduction': 100, 'S-S-06:Sulfite reduction': 100, 'S-S-08:Thiosulfate disproportionation 1': 100}, 'org_3': {'C-S-01:Organic carbon oxidation': 0, 'C-S-02:Carbon fixation': 0, 'C-S-04:Acetate oxidation': 0, 'C-S-05:Hydrogen generation': 0, 'C-S-06:Fermentation': 0, 'C-S-09:Hydrogen oxidation': 0, 'N-S-01:Nitrogen fixation': 0, 'N-S-08:Nitrite ammonification': 0, 'O-S-03:Arsenate reduction': 0, 'S-S-03:Sulfur oxidation': 0, 'S-S-04:Sulfite oxidation': 0, 'S-S-05:Sulfate reduction': 0, 'S-S-06:Sulfite reduction': 0, 'S-S-08:Thiosulfate disproportionation 1': 0}}, 'sample_2': {'org_1': {'C-S-01:Organic carbon oxidation': 0, 'C-S-02:Carbon fixation': 0, 'C-S-03:Ethanol oxidation': 0, 'C-S-04:Acetate oxidation': 0, 'C-S-06:Fermentation': 0, 'C-S-07:Methanogenesis': 0, 'N-S-01:Nitrogen fixation': 0, 'N-S-07:Nitrous oxide reduction': 0, 'N-S-08:Nitrite ammonification': 0, 'O-S-01:Iron reduction': 0, 'S-S-03:Sulfur oxidation': 0, 'S-S-06:Sulfite reduction': 0}, 'org_2': {'C-S-01:Organic carbon oxidation': 200, 'C-S-02:Carbon fixation': 200, 'C-S-04:Acetate oxidation': 200, 'C-S-05:Hydrogen generation': 200, 'C-S-06:Fermentation': 200, 'C-S-09:Hydrogen oxidation': 200, 'N-S-01:Nitrogen fixation': 200, 'N-S-08:Nitrite ammonification': 200, 'O-S-03:Arsenate reduction': 200, 'S-S-03:Sulfur oxidation': 200, 'S-S-04:Sulfite oxidation': 200, 'S-S-05:Sulfate reduction': 200, 'S-S-06:Sulfite reduction': 200, 'S-S-08:Thiosulfate disproportionation 1': 200}, 'org_3': {'C-S-01:Organic carbon oxidation': 600, 'C-S-02:Carbon fixation': 600, 'C-S-04:Acetate oxidation': 600, 'C-S-05:Hydrogen generation': 600, 'C-S-06:Fermentation': 600, 'C-S-09:Hydrogen oxidation': 600, 'N-S-01:Nitrogen fixation': 600, 'N-S-08:Nitrite ammonification': 600, 'O-S-03:Arsenate reduction': 600, 'S-S-03:Sulfur oxidation': 600, 'S-S-04:Sulfite oxidation': 600, 'S-S-05:Sulfate reduction': 600, 'S-S-06:Sulfite reduction': 600, 'S-S-08:Thiosulfate disproportionation 1': 600}}, 'sample_3': {'org_1': {'C-S-01:Organic carbon oxidation': 0, 'C-S-02:Carbon fixation': 0, 'C-S-03:Ethanol oxidation': 0, 'C-S-04:Acetate oxidation': 0, 'C-S-06:Fermentation': 0, 'C-S-07:Methanogenesis': 0, 'N-S-01:Nitrogen fixation': 0, 'N-S-07:Nitrous oxide reduction': 0, 'N-S-08:Nitrite ammonification': 0, 'O-S-01:Iron reduction': 0, 'S-S-03:Sulfur oxidation': 0, 'S-S-06:Sulfite reduction': 0}, 'org_2': {'C-S-01:Organic carbon oxidation': 120, 'C-S-02:Carbon fixation': 120, 'C-S-04:Acetate oxidation': 120, 'C-S-05:Hydrogen generation': 120, 'C-S-06:Fermentation': 120, 'C-S-09:Hydrogen oxidation': 120, 'N-S-01:Nitrogen fixation': 120, 'N-S-08:Nitrite ammonification': 120, 'O-S-03:Arsenate reduction': 120, 'S-S-03:Sulfur oxidation': 120, 'S-S-04:Sulfite oxidation': 120, 'S-S-05:Sulfate reduction': 120, 'S-S-06:Sulfite reduction': 120, 'S-S-08:Thiosulfate disproportionation 1': 120}, 'org_3': {'C-S-01:Organic carbon oxidation': 400, 'C-S-02:Carbon fixation': 400, 'C-S-04:Acetate oxidation': 400, 'C-S-05:Hydrogen generation': 400, 'C-S-06:Fermentation': 400, 'C-S-09:Hydrogen oxidation': 400, 'N-S-01:Nitrogen fixation': 400, 'N-S-08:Nitrite ammonification': 400, 'O-S-03:Arsenate reduction': 400, 'S-S-03:Sulfur oxidation': 400, 'S-S-04:Sulfite oxidation': 400, 'S-S-05:Sulfate reduction': 400, 'S-S-06:Sulfite reduction': 400, 'S-S-08:Thiosulfate disproportionation 1': 400}}}

    for sample in expected_function_abundance_samples:
        for function in expected_function_abundance_samples[sample]:
            assert function_abundance_samples[sample][function] == expected_function_abundance_samples[sample][function]

    for sample in expected_function_relative_abundance_samples:
        for function in expected_function_relative_abundance_samples[sample]:
            assert function_relative_abundance_samples[sample][function] == expected_function_relative_abundance_samples[sample][function]

    for sample in expected_function_participation_samples:
        for organism in expected_function_participation_samples[sample]:
            for function in expected_function_participation_samples[sample][organism]:
                assert function_participation_samples[sample][organism][function] == expected_function_participation_samples[sample][organism][function]


def test_create_visualisation():
    bigecyhmm_output_folder = os.path.join('input_data', 'bigecyhmm_output_folder')

    output_folder = 'output_folder'
    create_visualisation(bigecyhmm_output_folder, output_folder)

    heatmap_occurrence = os.path.join(output_folder, 'function_occurrence', 'heatmap_occurrence.png')
    assert os.path.exists(heatmap_occurrence)

    shutil.rmtree(output_folder)


def test_create_visualisation_cli():
    bigecyhmm_output_folder = os.path.join('input_data', 'bigecyhmm_output_folder')

    output_folder = 'output_folder'
    subprocess.call(['bigecyhmm_visualisation', 'genomes', '--bigecyhmm', bigecyhmm_output_folder, '-o', output_folder])

    heatmap_occurrence = os.path.join(output_folder, 'function_occurrence', 'heatmap_occurrence.png')
    assert os.path.exists(heatmap_occurrence)

    shutil.rmtree(output_folder)


def test_create_visualisation_ko_cli():
    ko_abundance_file = os.path.join('input_data', 'ko_file.tsv')
    output_folder = 'output_folder'

    subprocess.call(['bigecyhmm_visualisation', 'ko', '--ko', ko_abundance_file, '-o', output_folder])

    polar_plot = os.path.join(output_folder, 'diagram_visualisation', 'sample_1_carbon_cycle.png')
    assert os.path.exists(polar_plot)
    heatmap_occurrence = os.path.join(output_folder, 'diagram_visualisation', 'sample_2_nitrogen_cycle.png')
    assert os.path.exists(heatmap_occurrence)

    shutil.rmtree(output_folder)


def test_create_visualisation_abundance():
    abundance_file_path = os.path.join('input_data', 'abundance_file_from_genomes.tsv')
    bigecyhmm_output_folder = os.path.join('input_data', 'bigecyhmm_output_folder')

    output_folder = 'output_folder'
    create_visualisation(bigecyhmm_output_folder, output_folder, abundance_file_path=abundance_file_path)

    carbon_cycle_diagram = os.path.join(output_folder, 'function_abundance', 'cycle_diagrams_abundance', 'sample_1_carbon_cycle.png')
    assert os.path.exists(carbon_cycle_diagram)
    polar_plot = os.path.join(output_folder, 'function_abundance', 'polar_plot_abundance_sample_sample_1.png')
    assert os.path.exists(polar_plot)
    polar_plot = os.path.join(output_folder, 'function_abundance', 'polar_plot_abundance_sample_sample_2.png')
    assert os.path.exists(polar_plot)
    polar_plot = os.path.join(output_folder, 'function_abundance', 'polar_plot_abundance_sample_sample_3.png')
    assert os.path.exists(polar_plot)
    heatmap_occurrence = os.path.join(output_folder, 'function_occurrence', 'heatmap_occurrence.png')
    assert os.path.exists(heatmap_occurrence)

    shutil.rmtree(output_folder)


def test_create_visualisation_abundance_cli():
    abundance_file_path = os.path.join('input_data', 'abundance_file_from_genomes.tsv')
    bigecyhmm_output_folder = os.path.join('input_data', 'bigecyhmm_output_folder')

    output_folder = 'output_folder'
    subprocess.call(['bigecyhmm_visualisation', 'genomes', '--bigecyhmm', bigecyhmm_output_folder, '--abundance-file', abundance_file_path, '-o', output_folder])

    carbon_cycle_diagram = os.path.join(output_folder, 'function_abundance', 'cycle_diagrams_abundance', 'sample_1_carbon_cycle.png')
    assert os.path.exists(carbon_cycle_diagram)
    polar_plot = os.path.join(output_folder, 'function_abundance', 'polar_plot_abundance_sample_sample_1.png')
    assert os.path.exists(polar_plot)
    polar_plot = os.path.join(output_folder, 'function_abundance', 'polar_plot_abundance_sample_sample_2.png')
    assert os.path.exists(polar_plot)
    polar_plot = os.path.join(output_folder, 'function_abundance', 'polar_plot_abundance_sample_sample_3.png')
    assert os.path.exists(polar_plot)
    heatmap_occurrence = os.path.join(output_folder, 'function_occurrence', 'heatmap_occurrence.png')
    assert os.path.exists(heatmap_occurrence)

    shutil.rmtree(output_folder)


def test_create_visualisation_abundance_from_esmecata():
    esmecata_output_folder = os.path.join('input_data', 'esmecata_output_folder')
    abundance_file_path = os.path.join('input_data', 'proteome_tax_id_abundance.tsv')
    bigecyhmm_output_folder = os.path.join('input_data', 'bigecyhmm_output_folder')

    output_folder = 'output_folder'
    create_visualisation(bigecyhmm_output_folder, output_folder, esmecata_output_folder, abundance_file_path)

    carbon_cycle_diagram = os.path.join(output_folder, 'function_abundance', 'cycle_diagrams_abundance', 'sample_1_carbon_cycle.png')
    assert os.path.exists(carbon_cycle_diagram)
    polar_plot = os.path.join(output_folder, 'function_abundance', 'polar_plot_abundance_sample_sample_1.png')
    assert os.path.exists(polar_plot)
    polar_plot = os.path.join(output_folder, 'function_abundance', 'polar_plot_abundance_sample_sample_2.png')
    assert os.path.exists(polar_plot)
    polar_plot = os.path.join(output_folder, 'function_abundance', 'polar_plot_abundance_sample_sample_3.png')
    assert os.path.exists(polar_plot)
    heatmap_occurrence = os.path.join(output_folder, 'function_occurrence', 'heatmap_occurrence.png')
    assert os.path.exists(heatmap_occurrence)

    expected_function_occurrence_organisms = {'C-S-01:Organic carbon oxidation': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'C-S-02:Carbon fixation': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'C-S-03:Ethanol oxidation': {'org_1': 1.0, 'org_2': 0.0, 'org_3': 0.0}, 'C-S-04:Acetate oxidation': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'C-S-05:Hydrogen generation': {'org_1': 0.0, 'org_2': 1.0, 'org_3': 1.0}, 'C-S-06:Fermentation': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'C-S-07:Methanogenesis': {'org_1': 1.0, 'org_2': 0.0, 'org_3': 0.0}, 'C-S-09:Hydrogen oxidation': {'org_1': 0.0, 'org_2': 1.0, 'org_3': 1.0}, 'N-S-01:Nitrogen fixation': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'N-S-07:Nitrous oxide reduction': {'org_1': 1.0, 'org_2': 0.0, 'org_3': 0.0}, 'N-S-08:Nitrite ammonification': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'O-S-01:Iron reduction': {'org_1': 1.0, 'org_2': 0.0, 'org_3': 0.0}, 'O-S-03:Arsenate reduction': {'org_1': 0.0, 'org_2': 1.0, 'org_3': 1.0}, 'O-S-06:Aerobic respiration': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'P-S-01:PhnD': {'org_1': 0.0, 'org_2': 1.0, 'org_3': 1.0}, 'P-S-04:PstS': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'P-S-08:PtxD': {'org_1': 1.0, 'org_2': 0.0, 'org_3': 0.0}, 'P-S-09:PtxB': {'org_1': 1.0, 'org_2': 0.0, 'org_3': 0.0}, 'P-S-10:Phosphonate production': {'org_1': 0.0, 'org_2': 1.0, 'org_3': 1.0}, 'P-S-11:Phosphonate catabolism': {'org_1': 0.0, 'org_2': 1.0, 'org_3': 1.0}, 'P-S-12:Phytate degradation': {'org_1': 0.0, 'org_2': 1.0, 'org_3': 1.0}, 'P-S-13:Phosphatase': {'org_1': 1.0, 'org_2': 0.0, 'org_3': 0.0}, 'P-S-14:ppa': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'P-S-15:ppx': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'P-S-17:gcd and pqqC': {'org_1': 1.0, 'org_2': 0.0, 'org_3': 0.0}, 'P-S-18:Pho regulon': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'P-S-01:Immobilisation (P-rich)': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'P-S-02:Mineralisation': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'P-S-03:Dissolution': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'S-S-03:Sulfur oxidation': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'S-S-04:Sulfite oxidation': {'org_1': 0.0, 'org_2': 1.0, 'org_3': 1.0}, 'S-S-05:Sulfate reduction': {'org_1': 0.0, 'org_2': 1.0, 'org_3': 1.0}, 'S-S-06:Sulfite reduction': {'org_1': 1, 'org_2': 1, 'org_3': 1}, 'S-S-08:Thiosulfate disproportionation 1': {'org_1': 0.0, 'org_2': 1.0, 'org_3': 1.0}, 'S-S-07:Thiosulfate oxidation': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'N-S-02:Ammonia oxidation': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'N-S-04:Nitrate reduction': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'S-S-01:Sulfide oxidation': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'S-S-10:Polysulfide reduction': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'P-S-05:PNaS': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'N-S-05:Nitrite reduction': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'P-S-02:C-P lyase': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'S-S-02:Sulfur reduction': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'P-S-01:Immobilisation (P-poor)': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'O-S-05:Selenate reduction': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'S-S-09:Thiosulfate disproportionation 2': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'C-S-10:Acetogenesis WL': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'O-S-04:Arsenite oxidation': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'C-S-08:Methanotrophy': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'P-S-06:HtxB': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'O-S-02:Iron oxidation': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'P-S-16:ppk1': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'P-S-07:HtxA': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'P-S-03:PitA': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'N-S-03:Nitrite oxidation': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'N-S-10:Nitric oxide dismutase': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'N-S-06:Nitric oxide reduction': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}, 'N-S-09:Anammox': {'org_1': 0.0, 'org_2': 0.0, 'org_3': 0.0}}

    expected_df = pd.DataFrame(expected_function_occurrence_organisms)
    expected_df.index.name = 'function'
    expected_df.fillna(0, inplace=True)
    predicted_function_filepath = os.path.join(output_folder, 'function_occurrence', 'pathway_presence_in_organism.tsv')
    predicted_function_df = pd.read_csv(predicted_function_filepath, sep='\t')
    predicted_function_df.set_index('function', inplace=True)
    predicted_function_df = predicted_function_df.reindex(expected_df.index)
    predicted_function_df = predicted_function_df[expected_df.columns]
    pd.testing.assert_frame_equal(expected_df, predicted_function_df)

    shutil.rmtree(output_folder)


def test_create_visualisation_abundance_from_esmecata_cli():
    esmecata_output_folder = os.path.join('input_data', 'esmecata_output_folder')
    abundance_file_path = os.path.join('input_data', 'proteome_tax_id_abundance.tsv')
    bigecyhmm_output_folder = os.path.join('input_data', 'bigecyhmm_output_folder')

    output_folder = 'output_folder'
    subprocess.call(['bigecyhmm_visualisation', 'esmecata', '--esmecata', esmecata_output_folder, '--bigecyhmm', bigecyhmm_output_folder, '--abundance-file', abundance_file_path, '-o', output_folder])

    carbon_cycle_diagram = os.path.join(output_folder, 'function_abundance', 'cycle_diagrams_abundance', 'sample_1_carbon_cycle.png')
    assert os.path.exists(carbon_cycle_diagram)
    polar_plot = os.path.join(output_folder, 'function_abundance', 'polar_plot_abundance_sample_sample_1.png')
    assert os.path.exists(polar_plot)
    polar_plot = os.path.join(output_folder, 'function_abundance', 'polar_plot_abundance_sample_sample_2.png')
    assert os.path.exists(polar_plot)
    polar_plot = os.path.join(output_folder, 'function_abundance', 'polar_plot_abundance_sample_sample_3.png')
    assert os.path.exists(polar_plot)
    heatmap_occurrence = os.path.join(output_folder, 'function_occurrence', 'heatmap_occurrence.png')
    assert os.path.exists(heatmap_occurrence)

    shutil.rmtree(output_folder)


if __name__ == "__main__":
    test_create_visualisation_abundance_from_esmecata()