"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const nag_rules_1 = require("../../nag-rules");
/**
 *  CloudFront distributions use a security policy with minimum TLSv1.1 or TLSv1.2 and appropriate security ciphers for HTTPS viewer connections
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_cloudfront_1.CfnDistribution) {
        const distributionConfig = aws_cdk_lib_1.Stack.of(node).resolve(node.distributionConfig);
        const viewerCertificate = aws_cdk_lib_1.Stack.of(node).resolve(distributionConfig.viewerCertificate);
        if (viewerCertificate === undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const minimumProtocolVersion = aws_cdk_lib_1.Stack.of(node).resolve(viewerCertificate.minimumProtocolVersion);
        const sslSupportMethod = aws_cdk_lib_1.Stack.of(node).resolve(viewerCertificate.sslSupportMethod);
        const cloudFrontDefaultCertificate = aws_cdk_lib_1.Stack.of(node).resolve(viewerCertificate.cloudFrontDefaultCertificate);
        const outdatedProtocols = ['SSLv3', 'TLSv1', 'TLSv1_2016'];
        if (cloudFrontDefaultCertificate === true ||
            sslSupportMethod === undefined ||
            sslSupportMethod.toLowerCase() === 'vip' ||
            minimumProtocolVersion === undefined ||
            outdatedProtocols
                .map((x) => x.toLowerCase())
                .includes(minimumProtocolVersion.toLowerCase())) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: (0, path_1.parse)(__filename).name });
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiQ2xvdWRGcm9udERpc3RyaWJ1dGlvbkh0dHBzVmlld2VyTm9PdXRkYXRlZFNTTC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9ydWxlcy9jbG91ZGZyb250L0Nsb3VkRnJvbnREaXN0cmlidXRpb25IdHRwc1ZpZXdlck5vT3V0ZGF0ZWRTU0wudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQTs7O0VBR0U7QUFDRiwrQkFBNkI7QUFDN0IsNkNBQWlEO0FBQ2pELCtEQUE2RDtBQUM3RCwrQ0FBb0Q7QUFFcEQ7OztHQUdHO0FBQ0gsa0JBQWUsTUFBTSxDQUFDLGNBQWMsQ0FDbEMsQ0FBQyxJQUFpQixFQUFxQixFQUFFO0lBQ3ZDLElBQUksSUFBSSxZQUFZLGdDQUFlLEVBQUUsQ0FBQztRQUNwQyxNQUFNLGtCQUFrQixHQUFHLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FDL0MsSUFBSSxDQUFDLGtCQUFrQixDQUN4QixDQUFDO1FBQ0YsTUFBTSxpQkFBaUIsR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQzlDLGtCQUFrQixDQUFDLGlCQUFpQixDQUNyQyxDQUFDO1FBQ0YsSUFBSSxpQkFBaUIsS0FBSyxTQUFTLEVBQUUsQ0FBQztZQUNwQyxPQUFPLDZCQUFpQixDQUFDLGFBQWEsQ0FBQztRQUN6QyxDQUFDO1FBQ0QsTUFBTSxzQkFBc0IsR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQ25ELGlCQUFpQixDQUFDLHNCQUFzQixDQUN6QyxDQUFDO1FBQ0YsTUFBTSxnQkFBZ0IsR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQzdDLGlCQUFpQixDQUFDLGdCQUFnQixDQUNuQyxDQUFDO1FBQ0YsTUFBTSw0QkFBNEIsR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQ3pELGlCQUFpQixDQUFDLDRCQUE0QixDQUMvQyxDQUFDO1FBQ0YsTUFBTSxpQkFBaUIsR0FBRyxDQUFDLE9BQU8sRUFBRSxPQUFPLEVBQUUsWUFBWSxDQUFDLENBQUM7UUFDM0QsSUFDRSw0QkFBNEIsS0FBSyxJQUFJO1lBQ3JDLGdCQUFnQixLQUFLLFNBQVM7WUFDOUIsZ0JBQWdCLENBQUMsV0FBVyxFQUFFLEtBQUssS0FBSztZQUN4QyxzQkFBc0IsS0FBSyxTQUFTO1lBQ3BDLGlCQUFpQjtpQkFDZCxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztpQkFDM0IsUUFBUSxDQUFDLHNCQUFzQixDQUFDLFdBQVcsRUFBRSxDQUFDLEVBQ2pELENBQUM7WUFDRCxPQUFPLDZCQUFpQixDQUFDLGFBQWEsQ0FBQztRQUN6QyxDQUFDO1FBQ0QsT0FBTyw2QkFBaUIsQ0FBQyxTQUFTLENBQUM7SUFDckMsQ0FBQztTQUFNLENBQUM7UUFDTixPQUFPLDZCQUFpQixDQUFDLGNBQWMsQ0FBQztJQUMxQyxDQUFDO0FBQ0gsQ0FBQyxFQUNELE1BQU0sRUFDTixFQUFFLEtBQUssRUFBRSxJQUFBLFlBQUssRUFBQyxVQUFVLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FDbEMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qXG5Db3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cblNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4qL1xuaW1wb3J0IHsgcGFyc2UgfSBmcm9tICdwYXRoJztcbmltcG9ydCB7IENmblJlc291cmNlLCBTdGFjayB9IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IENmbkRpc3RyaWJ1dGlvbiB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1jbG91ZGZyb250JztcbmltcG9ydCB7IE5hZ1J1bGVDb21wbGlhbmNlIH0gZnJvbSAnLi4vLi4vbmFnLXJ1bGVzJztcblxuLyoqXG4gKiAgQ2xvdWRGcm9udCBkaXN0cmlidXRpb25zIHVzZSBhIHNlY3VyaXR5IHBvbGljeSB3aXRoIG1pbmltdW0gVExTdjEuMSBvciBUTFN2MS4yIGFuZCBhcHByb3ByaWF0ZSBzZWN1cml0eSBjaXBoZXJzIGZvciBIVFRQUyB2aWV3ZXIgY29ubmVjdGlvbnNcbiAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICovXG5leHBvcnQgZGVmYXVsdCBPYmplY3QuZGVmaW5lUHJvcGVydHkoXG4gIChub2RlOiBDZm5SZXNvdXJjZSk6IE5hZ1J1bGVDb21wbGlhbmNlID0+IHtcbiAgICBpZiAobm9kZSBpbnN0YW5jZW9mIENmbkRpc3RyaWJ1dGlvbikge1xuICAgICAgY29uc3QgZGlzdHJpYnV0aW9uQ29uZmlnID0gU3RhY2sub2Yobm9kZSkucmVzb2x2ZShcbiAgICAgICAgbm9kZS5kaXN0cmlidXRpb25Db25maWdcbiAgICAgICk7XG4gICAgICBjb25zdCB2aWV3ZXJDZXJ0aWZpY2F0ZSA9IFN0YWNrLm9mKG5vZGUpLnJlc29sdmUoXG4gICAgICAgIGRpc3RyaWJ1dGlvbkNvbmZpZy52aWV3ZXJDZXJ0aWZpY2F0ZVxuICAgICAgKTtcbiAgICAgIGlmICh2aWV3ZXJDZXJ0aWZpY2F0ZSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5OT05fQ09NUExJQU5UO1xuICAgICAgfVxuICAgICAgY29uc3QgbWluaW11bVByb3RvY29sVmVyc2lvbiA9IFN0YWNrLm9mKG5vZGUpLnJlc29sdmUoXG4gICAgICAgIHZpZXdlckNlcnRpZmljYXRlLm1pbmltdW1Qcm90b2NvbFZlcnNpb25cbiAgICAgICk7XG4gICAgICBjb25zdCBzc2xTdXBwb3J0TWV0aG9kID0gU3RhY2sub2Yobm9kZSkucmVzb2x2ZShcbiAgICAgICAgdmlld2VyQ2VydGlmaWNhdGUuc3NsU3VwcG9ydE1ldGhvZFxuICAgICAgKTtcbiAgICAgIGNvbnN0IGNsb3VkRnJvbnREZWZhdWx0Q2VydGlmaWNhdGUgPSBTdGFjay5vZihub2RlKS5yZXNvbHZlKFxuICAgICAgICB2aWV3ZXJDZXJ0aWZpY2F0ZS5jbG91ZEZyb250RGVmYXVsdENlcnRpZmljYXRlXG4gICAgICApO1xuICAgICAgY29uc3Qgb3V0ZGF0ZWRQcm90b2NvbHMgPSBbJ1NTTHYzJywgJ1RMU3YxJywgJ1RMU3YxXzIwMTYnXTtcbiAgICAgIGlmIChcbiAgICAgICAgY2xvdWRGcm9udERlZmF1bHRDZXJ0aWZpY2F0ZSA9PT0gdHJ1ZSB8fFxuICAgICAgICBzc2xTdXBwb3J0TWV0aG9kID09PSB1bmRlZmluZWQgfHxcbiAgICAgICAgc3NsU3VwcG9ydE1ldGhvZC50b0xvd2VyQ2FzZSgpID09PSAndmlwJyB8fFxuICAgICAgICBtaW5pbXVtUHJvdG9jb2xWZXJzaW9uID09PSB1bmRlZmluZWQgfHxcbiAgICAgICAgb3V0ZGF0ZWRQcm90b2NvbHNcbiAgICAgICAgICAubWFwKCh4KSA9PiB4LnRvTG93ZXJDYXNlKCkpXG4gICAgICAgICAgLmluY2x1ZGVzKG1pbmltdW1Qcm90b2NvbFZlcnNpb24udG9Mb3dlckNhc2UoKSlcbiAgICAgICkge1xuICAgICAgICByZXR1cm4gTmFnUnVsZUNvbXBsaWFuY2UuTk9OX0NPTVBMSUFOVDtcbiAgICAgIH1cbiAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5DT01QTElBTlQ7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5OT1RfQVBQTElDQUJMRTtcbiAgICB9XG4gIH0sXG4gICduYW1lJyxcbiAgeyB2YWx1ZTogcGFyc2UoX19maWxlbmFtZSkubmFtZSB9XG4pO1xuIl19