"""Interactively setup a config file"""

from __future__ import annotations

import logging
from pathlib import Path
from typing import Any

logger = logging.getLogger(__name__)

# Default directory structure configuration
DEFAULT_CONFIG = {
    "input_dir": "src",
    "output_dir": "out",
}

# Default settings for boolean flags
DEFAULT_FLAGS = {
    "verbose": False,
    "quiet": False,
}

# Template for the bash2gitlab.toml file
TOML_TEMPLATE = """# Configuration for bash2gitlab
# This file was generated by the 'bash2gitlab init' command.

# Directory settings
input_dir = "{input_dir}"
output_dir = "{output_dir}"

# Command-line flag defaults
verbose = {verbose}
quiet = {quiet}
"""

__all__ = ["prompt_for_config", "create_config_file"]


def get_str_input(prompt: str, default: str) -> str:
    """Prompts the user for string input with a default value."""
    prompt_with_default = f"{prompt} (default: {default}): "
    return input(prompt_with_default).strip() or default


def get_bool_input(prompt: str, default: bool) -> bool:
    """Prompts the user for boolean (y/n) input with a default."""
    default_str = "y" if default else "n"
    prompt_with_default = f"{prompt} (y/n, default: {default_str}): "
    response = input(prompt_with_default).strip().lower()
    if not response:
        return default
    return response in ["y", "yes"]


def prompt_for_config() -> dict[str, Any]:
    """
    Interactively prompts the user for project configuration details.
    This function is separate from file I/O to be easily testable.
    """
    print("Initializing a new bash2gitlab project.")
    print("Please confirm the directory structure (press Enter to accept defaults).")
    config: dict[str, Any] = {}
    for key, value in DEFAULT_CONFIG.items():
        user_value = get_str_input(f"  -> {key}", value)
        config[key] = user_value

    print("\nConfigure default behavior (press Enter to accept defaults).")
    for key, flag_value in DEFAULT_FLAGS.items():
        flag_user_value = get_bool_input(f"  -> Always use --{key}?", flag_value)
        config[key] = flag_user_value

    return config


def create_config_file(base_path: Path, config: dict[str, Any], dry_run: bool = False):
    """
    Creates the config file for a new project.
    This function performs all file system operations.
    """
    config_file_path = base_path / "bash2gitlab.toml"

    logger.info("\nThe following file will be created:")
    print(f"  - {config_file_path}")

    if dry_run:
        logger.warning("\nDRY RUN: No file will be created.")
        return

    # Write the config file
    # Lowercase boolean values for TOML compatibility
    formatted_config = config.copy()
    for key, value in formatted_config.items():
        if isinstance(value, bool):
            formatted_config[key] = str(value).lower()

    toml_content = TOML_TEMPLATE.format(**formatted_config)
    config_file_path.write_text(toml_content, encoding="utf-8")

    logger.info("\n✅ Project initialization complete.")
