# -*- coding: utf-8 -*-
# yapf

import sys
import io
from typing import Any, Tuple, List, Mapping, Callable

import distutils.cmd

import functools
from contextlib import contextmanager

import funcy  # type: ignore
import pytest  # type: ignore

from altered import state  # type: ignore

import pycodestyle  # type: ignore
import mypy.api  # type: ignore

import pyflakes.api  # type: ignore


class fixture(object):
    @staticmethod
    def params(namelist: str, *values: Any) -> Any:
        "Does params"
        return pytest.mark.parametrize(namelist, values)


def logcall(fn: Callable, realname: str = None) -> Callable:
    "A simple composition that will log parameters an results."

    @functools.wraps(fn)
    def _(*args: Any, **kwargs: Mapping[Any, Any]) -> Any:
        "Function logger, will log if enclosing functions is turned on"
        on = logcall._enabled
        try:
            name = realname or fn.__name__
            if kwargs:
                desc = f'{name}({args}, {kwargs}'
            else:
                desc = desc = f'{name}({args})'
            if on:
                print(desc)
            ret = fn(*args, **kwargs)
            if on:
                print(f'  -> {ret}')
            return ret
        except Exception as exc:
            if on:
                print(f'  ..raised {exc}')
            raise

    @contextmanager
    def on():
        logcall._enabled = True
        try:
            yield
        except Exception as exc:
            raise
        finally:
            logcall._enabled = False

        logcall._enabled = False

    logcall.on = on
    logcall._enabled = False
    return _


class ReviewProject(distutils.cmd.Command):
    user_options: List[str] = []

    def initialize_options(self: 'ReviewProject') -> None:
        pass

    def finalize_options(self: 'ReviewProject') -> None:
        pass

    @staticmethod
    def lint() -> Tuple[int, str, str]:
        print('     pyflakes...')
        warn, err = io.StringIO(), io.StringIO()
        code = pyflakes.api.checkRecursive(
            ['micropy'],  # package directory
            pyflakes.reporter.Reporter(warningStream=warn, errorStream=err))
        return (code, warn.getvalue(), err.getvalue())

    @staticmethod
    def style() -> Tuple[int, str, str]:
        """Runs pycodestyle, unpleasantly white-boxy call.  ..but it doesn't
         seem to be written for integration, so:

        """
        print('     pycodestyle...')
        code = 0
        warn, err = io.StringIO(), io.StringIO()
        with state(sys, argv=[], stdout=warn, stderr=err):
            try:
                pycodestyle._main()
            except SystemExit as exc:
                code = exc.code
        return code, warn.getvalue(), err.getvalue()

    @staticmethod
    def types() -> Tuple[int, str, str]:
        "Runs MyPy, the standard Python type checker."
        print('     MyPy...')
        warn, err, code = mypy.api.run(['.'])
        return code, warn, err

    @staticmethod
    def separator_line() -> None:
        "Prints a separating line of 79 characters wide."
        print()
        print(79 * '-')

    def run(self: 'ReviewProject') -> None:
        "Runs the setup task `'review'`."
        reports = {
            'pyflakes': ReviewProject.lint(),
            'pycodestyle': ReviewProject.style(),
            'mypy': ReviewProject.types(),
        }

        for tool in reports:
            code, warn, err = reports[tool]
            if code != 0:
                ReviewProject.separator_line()
                print(f'{tool}: code {code}')
                print('WARN')
                print(warn)
                print('ERROR')
                print(err)

        issues = sum(funcy.walk_values(lambda x: x[0], reports).values())
        if issues > 0:
            ReviewProject.separator_line()
            print()
            print(f'{issues} issues found.')
