#!/usr/bin/env python
from __future__ import print_function

'''
Peach CI Generic Integration Runner

This script provides generic integration with CI systems by
running a command that returns non-zero when testing did not pass.
The vast majority of CI systems support this method of integration.

If a specific integration is offered for your CI system that is
preferred over this generic integration.
'''

'''
For syntax help run:

  python peach_ci_runner.py --help

'''

'''
Copyright 2017 Peach Tech

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
'''

###############################################################
##     DO NOT EDIT THIS FILE.  REPORT ANY ISSUES TO 
##          SUPPORT@PEACH.TECH
###############################################################

import logging, os, sys, time, signal, errno
import os

try:
	import peachapisec
except:
	print("Detected missing dependency 'peachapisec' python module.")
	print("Install: python -m pip install peachapisec-api")
	exit(100)

try:
	from requests import get, post, delete
except:
	print("Detected missing dependency 'requests' python module.")
	print("Install: python -m pip install requests")
	exit(100)

try:
	import click
except:
	print("Detected missing dependency 'click' python module.")
	print("Install: python -m pip install click")
	exit(100)

import requests, json
import subprocess, signal
import logging, logging.handlers
from time import sleep
import sys, shlex

try:
	# only works on python 2
	reload(sys)
	sys.setdefaultencoding("utf-8")
except:
	pass

try:
	from subprocess import DEVNULL
except ImportError:
	import os
	DEVNULL = open(os.devnull, 'wb')

logger = logging.getLogger(__name__)

syslog_level = logging.ERROR
syslog_level = logging.INFO

logger.setLevel(syslog_level)
logFormatter = logging.Formatter("%(asctime)s [%(levelname)-5.5s] Peach Web CI: %(message)s")

consoleHandler = logging.StreamHandler()
consoleHandler.setFormatter(logFormatter)
logger.addHandler(consoleHandler)

class PeachGenericCi(object):

	def __init__(self):
		self.test_process = None
		self.peach_jobid = None
		self.code = None
		self.sessionSetup = False
		self.sessionTeardown = False

	def run(self):

		logger.info("Peach Web CI Generic Starting")

		logger.info(" ")
		logger.info("  Peach API Security UI: %s", self.peach_ui)
		logger.info(" ")

		logger.info("  project: %s", self.project)
		logger.info("  profile: %s", self.profile)
		logger.info("  peach_api: %s", self.peach_api)
		logger.info("  peach_api_token: %s", self.peach_api_token)
		logger.info("  automation_cmd: %s", self.automation_cmd)
		logger.info("  exit_code_ok: %d", self.exit_code_ok)
		logger.info("  exit_code_failure: %d", self.exit_code_failure)
		logger.info("  exit_code_error: %d", self.exit_code_error)
		logger.info("  junit: %s", self.junit)
		logger.info("  syslog_enabled: %s", self.syslog_enabled )
		logger.info("  issue_tracker_cwd: %s", self.issue_tracker_cwd)
		logger.info("  issue_tracker: %s", self.issue_tracker )
		if self.syslog_enabled:
			logger.info("  syslog_host: %s", self.syslog_host)
			logger.info("  syslog_port: %d", self.syslog_port)

		self.test_process = None
		self.peach_jobid = None
		self.code = None

		os.environ[str('PEACH_API')] = str(self.peach_api)
		os.environ[str('PEACH_API_TOKEN')] = str(self.peach_api_token)

		peachapisec.set_peach_api(self.peach_api)
		peachapisec.set_peach_api_token(self.peach_api_token)

		signal.signal(signal.SIGINT, self.stop_handler)
		signal.signal(signal.SIGTERM, self.stop_handler)

		self.startJob()
		try:
			
			# Launch test automation
			self.launchAutomation()
			
			# Wait for fuzzing to end
			self.waitForTestingCompletion()

			self.waitForJobToComplete()

		except Exception as ex:
			logger.info(str(ex))
		finally:
			logger.info("Ending session")
			
			ret = peachapisec.session_teardown()
			self.sessionTeardown = True
			
			if ret and self.junit:
				with open(self.junit, "wb+") as fd:
					fd.write(bytes(peachapisec.junit_xml().encode('utf-8')))

			if self.issue_tracker:
				logger.info("Sending findings to issue tracker")

				popen_kwargs = {
					'cwd' : self.issue_tracker_cwd
				}

				if sys.platform == 'win32':
					cmd = self.issue_tracker
					popen_kwargs['creationflags'] = subprocess.CREATE_NEW_PROCESS_GROUP

				else:
					cmd = shlex.split(self.issue_tracker)
					popen_kwargs['preexec_fn'] = os.setsid

				try:
					issue_cmd = subprocess.Popen(
						cmd,
						**popen_kwargs)

					issue_cmd.wait()

				except Exception as ex:
					logger.info(str(ex))

			peach_jobid = None

			if ret:
				logger.info("Session Teardown indicated failures during testing.")
				logger.info("Findings can be viewed at: %s/jobs/%s" % (self.peach_api, self.peach_jobid))
				self.eexit(self.exit_code_failure)
			
			logger.info("Session completed successfully, no faults detected.")
			self.eexit(self.exit_code_ok)

	def startJob(self):
		'''Start up a testing job and receive proxy info
		'''

		# Start job
		try:
			logger.info("Starting session")
			self.peach_jobid = None
			
			peachapisec.session_setup(self.project, self.profile, self.peach_api, self.tags)

			self.sessionSetup = True
			self.peach_jobid = str(peachapisec.session_id())
			self.peach_proxy = str(peachapisec.proxy_url())

		except Exception as ex:
			logger.critical("Error starting session: %s", ex)
			self.eexit(exit_code_failure)

	def terminate(self, pid):
		'''Send SIGTERM to a pid/gpid
		'''

		if sys.platform == 'win32':
			subprocess.call([
				'taskkill',
				'/t',               # Tree (including children)
				'/f',               # Force 
				'/pid', str(pid),
			])
		else:
			try:
				os.killpg(os.getpgid(pid), signal.SIGTERM)
			except:
				pass
		

	def kill(self, pid):
		'''Kill a process/process group on any platform
		'''

		if sys.platform == 'win32':
			subprocess.call([
				'taskkill',
				'/t',               # Tree (including children)
				'/f',               # Force 
				'/pid', str(pid),
			])
		else:
			try:
				os.killpg(os.getpgid(pid), signal.SIGKILL)
			except:
				pass
			
			# time.sleep(0.1)
			
			# try:
			# 	os.kill(pid, signal.SIGKILL)
			# except:
			# 	pass

	def eexit(self, code):
		'''Exit and perform any cleanup
		'''

		# Note: Cleanup moved to goodbye.
		
		if not self.code:
			self.code = code

		logger.info("eexit(%d)", self.code)
		exit(self.code)

	def goodbye(self):
		'''Make sure we cleanup after ourselves. Called by atexit.
		'''

		logger.info("Ensuring job has stopped")

		if self.test_process:
			try:
				self.terminate(self.test_process.pid)
				time.sleep(0.5)
				self.kill(self.test_process.pid)
				time.sleep(0.5)
				self.test_process.wait()

			except:
				pass
		
		if self.peach_jobid and (self.sessionSetup and not self.sessionTeardown):
			peachapisec.session_teardown()

	def ignore_handler(self, signal, frame):
		pass

	def stop_handler(self, num, frame):
		'''Signal handler for SIGINT/SIGTERM
		'''
		# Only catch SIGINT/SIGTERM once
		signal.signal(signal.SIGINT, self.ignore_handler)
		signal.signal(signal.SIGTERM, self.ignore_handler)

		logger.critical("Session terminated by user")

		# Calling eexit will cause goodbye to be called
		self.eexit(self.exit_code_error)

	def launchAutomation(self):
		'''Launch automation command setting self.test_process.
		'''
		
		if self.automation_cmd:
			
			logger.info("Launching test automation")
			logger.info("  PEACH_SESSIONID: %s", self.peach_jobid)
			logger.info("  PEACH_API: %s", self.peach_api)
			logger.info("  PEACH_PROXY: %s", self.peach_proxy)
			
			try:
				
				test_env = os.environ.copy()
				test_env[str("PEACH_SESSIONID")] = str(self.peach_jobid)
				test_env[str("PEACH_API")] = str(self.peach_api)
				test_env[str("PEACH_PROXY")] = str(self.peach_proxy)
				
				popen_kwargs = {
					'env':test_env,
				}

				if sys.platform == 'win32':
					cmd = self.automation_cmd
					popen_kwargs['creationflags'] = subprocess.CREATE_NEW_PROCESS_GROUP
				else:
					cmd = shlex.split(self.automation_cmd)
					popen_kwargs['preexec_fn'] = os.setsid

				if not self.verbose:
					popen_kwargs['stdin'] = subprocess.PIPE
					popen_kwargs['stdout'] = DEVNULL
					popen_kwargs['stderr'] = subprocess.STDOUT

				self.test_process = subprocess.Popen(
					cmd,
					**popen_kwargs
				)
				
				if not self.test_process:
					logger.critical("Unable to start test automation")
					self.eexit(self.exit_code_error)
				
				sleep(1)
				if self.test_process.poll():
					logger.critical("Unable to start test automation")
					self.eexit(self.exit_code_error)
			
			except OSError as ex:
				# This exception occurs prior to automation starting.
				# For example, file not found type things
				
				exe = args = ""

				if sys.platform == 'win32':
					msg = "Can't start automation command line '%s'." % self.automation_cmd

					parts = shlex.split(cmd)
					exe = parts[0]
					args = parts[1]

				else:
					if len(cmd) > 0: exe = cmd[0]
					if len(cmd) > 1: args = cmd[1]

					msg = "Can't start automation command '%s'." % exe

				if ex.errno == errno.EACCES:
					logger.critical("  Verify the user has permission to execute the file and try again")
				
				if ex.errno == errno.ENOENT:
					candidate = os.path.abspath(exe)
					while True:
						parts = os.path.split(candidate)
						if os.path.exists(parts[0]):
							logger.critical("  Could not locate '%s' in directory '%s'", parts[1], parts[0])
							break
						if parts[0] == candidate:
							logger.critical("  Verify '%s' is a valid path and try again", args[0])
							break
						candidate = parts[0]

				self.eexit(self.exit_code_error)

			except Exception as ex:
				logger.critical("Error starting test automation: %s", ex)
				self.eexit(self.exit_code_error)
		
		else:
			logger.critical("No automation configured.  Currently required.")
			self.eexit(self.exit_code_error)

	def waitForTestingCompletion(self):
		'''Wait for testing session to complete (suite teardown called).
		'''
		
		logger.info("Waiting for testing to complete")
		
		while True:
			time.sleep(10)
			
			state = peachapisec.session_state()
			logger.debug("Polling state....%s" % state)
			
			if state == 'Invalid':
				logger.critical("Session in invalid state")
				self.eexit(self.exit_code_error)
			
			elif state == 'Error':
				try:
					reason = peachapisec.session_error_reason()
					logger.error("Testing failed: %s", reason)
				except:
					logger.error("Session errored, unknwon reason")

				self.eexit(self.exit_code_error)
			
			elif state == 'Idle':
				logger.info("Session idle, testing completed")
				break
			
			elif state == 'Finished':
				logger.error("Session finished, incorrect state transfer")
				self.eexit(self.exit_code_error)
			
			ret = self.test_process.poll()
			if ret:
				self.test_process = None
				logger.warning("Automation process exitted with out calling suite teardown")
				break
	
	def waitForJobToComplete(self):
		'''Wait for test_process to exit. Will timeout if process
		doesn't exit in a 'reasonable' time.
		'''
		logger.info("Waiting for job to complete")

		if self.test_process:
			for i in range(1, 50):
				ret = self.test_process.poll()
				if ret != None:
					break

			time.sleep(i)

# ################################
# ################################
# ################################

app = PeachGenericCi()

import atexit
@atexit.register
def goodbye():
	app.goodbye()

@click.group(invoke_without_command=True, help="Peach CI Generic Integration Runner")
@click.option("-j", "--junit", help="Enable JUnit XML output and provide output filename. Defaults to PEACH_JUNIT environ.")
@click.option("--exit_code_ok", default=0, help="Exit code when no issues found. Defaults to 0, or PEACH_EXIT_CODE_OK environ.")
@click.option("--exit_code_failure", default=1, help="Exit code when issues found. Defaults to 1, or PEACH_EXIT_CODE_FAILURE environ.")
@click.option("--exit_code_error", default=100, help="Exit code when testing failed to complete. Defaults to 100, or PEACH_EXIT_CODE_ERROR environ.")
@click.option("-a", "--automation_cmd", help="Traffic automation command. Defaults to PEACH_AUTOMATION_CMD environ.")
@click.option("-p", "--project", help="Project configuration to use. Defaults to PEACH_PROJECT environ.")
@click.option("-o", "--profile", help="Project test profile to use. Defaults to PEACH_PROFILE environ.")
@click.option("-a","--api", help="Peach API Security API URL. Defaults to PEACH_API environ.")
@click.option("-t","--api_token", help="Peach API Security API Token. Defaults to PEACH_API_TOKEN environ.")
@click.option("--ui", help="Peach UI URL. Defaults to PEACH_UI environ.")
@click.option("--syslog_host", help="Syslog host, disabled if not provided. Defaults to PEACH_SYSLOG_HOST environ.")
@click.option("--syslog_port", default=514, help="Syslog port. Defaults to 514 or PEACH_SYSLOG_PORT environ.")
@click.option("--issue_tracker", help="Issue tracker integration script. Defaults to PEACH_ISSUE_TRACKER environ.")
@click.option("--issue_tracker_cwd", help="Issue tracker working directory. Defaults to PEACH_ISSUE_TRACKER_CWD environ.")
@click.option("-v","--verbose", is_flag=True, help="Enable verbose output. Defaults to off or PEACH_VERBOSE environ.")
@click.option("-D", "--tag", multiple=True, help="Use to specify a tag.  For multiple tags use this option multiple times, once per tag")
@click.pass_context
def cli(ctx,
	junit, 
	exit_code_ok, 
	exit_code_failure, 
	exit_code_error,
	automation_cmd, 
	project, 
	profile, 
	api, 
	api_token, 
	ui,
	syslog_host,
	syslog_port,
	issue_tracker,
	issue_tracker_cwd,
	verbose, tag, **kwargs):

	if syslog_host:
		syslog_enabled = True

		syslogHandler = logging.handlers.SysLogHandler(address=(syslog_host, syslog_port))
		syslogHandler.setFormatter(logFormatter)
		logger.addHandler(syslogHandler)
	else:
		syslog_enabled = False

	if not api:
		print("Error, set PEACH_API or provide --api argument")
		exit(1)

	if not api_token:
		print("Error, set PEACH_API_TOKEN or provide --api argument")
		exit(1)

	if not project:
		print("Error, set PEACH_PROJECT or provide --project argument")
		exit(1)

	if not profile:
		print("Error, set PEACH_PROFILE or provide --profile argument")
		exit(1)

	if not automation_cmd:
		print("Error, set PEACH_AUTOMATION_CMD or provide --automation_cmd argument")
		exit(1)

	if issue_tracker and not issue_tracker_cwd:
		print("Error, when providing issue_tracker, issue_tracker_cwd is also required.")
		exit(1)
		

	app.junit = junit
	app.exit_code_ok = exit_code_ok
	app.exit_code_failure = exit_code_failure
	app.exit_code_error = exit_code_error
	app.automation_cmd = automation_cmd
	app.project = project
	app.profile = profile
	app.peach_api = api
	app.peach_api_token = api_token
	app.peach_ui = ui
	app.syslog_enabled = syslog_enabled
	app.syslog_host = syslog_host
	app.syslog_port = syslog_port
	app.issue_tracker = issue_tracker
	app.issue_tracker_cwd = issue_tracker_cwd
	app.verbose = verbose
	app.tags = list(tag)
	app.run()

def run():
	'''Start Peach CI
	'''
	cli(obj={}, auto_envvar_prefix='PEACH')



if __name__ == '__main__':
	run()	

# end
