# -*- coding: utf-8 -*-
#
# connection_analyzer.py
#
# Stub implementation of ConnectionAnalyzer to fix import issues
# This provides minimal functionality to avoid breaking existing code
#

import logging
from typing import Dict, List, Tuple

logger = logging.getLogger(__name__)


class ConnectionAnalyzer:
    """
    Stub implementation of ConnectionAnalyzer.

    This provides basic functionality to analyze circuit connections
    for component placement optimization.
    """

    def __init__(self):
        self.connections: Dict[str, List[str]] = {}
        self.connection_weights: Dict[Tuple[str, str], int] = {}

    def analyze_circuit(self, circuit) -> None:
        """
        Analyze circuit connections.

        Args:
            circuit: Circuit object with components and nets
        """
        logger.debug(f"Analyzing connections for circuit: {circuit.name}")

        # Build connection graph from circuit nets
        # Handle blank circuits where nets might be empty or converted to list
        nets = circuit.nets
        if isinstance(nets, list):
            # If nets got converted to a list (shouldn't happen but defensive)
            nets = {}
        elif nets is None:
            nets = {}

        for net_name, net in nets.items():
            connected_components = []

            # Find all components connected to this net
            for comp in circuit.components:
                for pin in comp._pins.values():
                    if hasattr(pin, "net") and pin.net == net:
                        connected_components.append(comp.ref)
                        break

            # Record connections between components
            for i, comp_a in enumerate(connected_components):
                for comp_b in connected_components[i + 1 :]:
                    # Record bidirectional connection
                    if comp_a not in self.connections:
                        self.connections[comp_a] = []
                    if comp_b not in self.connections:
                        self.connections[comp_b] = []

                    if comp_b not in self.connections[comp_a]:
                        self.connections[comp_a].append(comp_b)
                    if comp_a not in self.connections[comp_b]:
                        self.connections[comp_b].append(comp_a)

                    # Track connection weight
                    key = tuple(sorted([comp_a, comp_b]))
                    self.connection_weights[key] = (
                        self.connection_weights.get(key, 0) + 1
                    )

    def get_connections(self, comp_ref: str) -> List[str]:
        """
        Get list of components connected to the given component.

        Args:
            comp_ref: Component reference (e.g., "R1")

        Returns:
            List of connected component references
        """
        return self.connections.get(comp_ref, [])

    def get_connection_weight(self, comp_a: str, comp_b: str) -> int:
        """
        Get connection weight between two components.

        Args:
            comp_a, comp_b: Component references

        Returns:
            Number of nets connecting the components
        """
        key = tuple(sorted([comp_a, comp_b]))
        return self.connection_weights.get(key, 0)

    def get_placement_connections(
        self, comp_ref: str, placed_components: Dict[str, Tuple[float, float]]
    ) -> List[Tuple[str, int]]:
        """
        Get connections to already-placed components.

        Args:
            comp_ref: Component being placed
            placed_components: Dict of {ref: (x, y)} for placed components

        Returns:
            List of (connected_ref, connection_weight) tuples
        """
        connections = []

        for connected_ref in self.get_connections(comp_ref):
            if connected_ref in placed_components:
                weight = self.get_connection_weight(comp_ref, connected_ref)
                connections.append((connected_ref, weight))

        return connections
    
    def get_placement_order(self, components) -> List[str]:
        """
        Get optimal placement order for components based on connections.
        
        Args:
            components: List of component references to order (can be strings or SchematicSymbol objects)
            
        Returns:
            List of component references in optimal placement order
        """
        # Handle both string component refs and SchematicSymbol objects
        def get_component_ref(comp):
            if isinstance(comp, str):
                return comp
            else:
                # Assume it's a SchematicSymbol with a ref attribute
                return getattr(comp, 'ref', str(comp))
        
        # Extract component references
        comp_refs = [get_component_ref(comp) for comp in components]
        
        # Simple implementation: sort by number of connections (most connected first)
        def connection_count(comp_ref):
            return len(self.get_connections(comp_ref))
        
        ordered = sorted(comp_refs, key=connection_count, reverse=True)
        logger.debug(f"Placement order: {ordered}")
        return ordered
