/*!
Module to provide implementation of drag force and simple atmospheric models.
 */

use nalgebra::{Vector3, Vector6};

use crate::constants::AngleFormat;
use crate::coordinates::SMatrix3;

#[cfg(test)]
use crate::constants::DEGREES;
use crate::{OMEGA_EARTH, position_ecef_to_geodetic};

const OMEGA_VECTOR: Vector3<f64> = Vector3::new(0.0, 0.0, OMEGA_EARTH);

/// Computes the perturbing, non-conservative acceleration caused by atmospheric
/// drag assuming that the ballistic properties of the spacecraft are captured by
/// the coefficient of drag.
///
/// Arguments:
///
/// - `x_object`: Satellite Cartesean state in the inertial reference frame [m; m/s]
/// - `density`: atmospheric density [kg/m^3]
/// - `mass`: Spacecraft mass [kg]
/// - `area`: Wind-facing cross-sectional area [m^2]
/// - `drag_coefficient`: coefficient of drag [dimensionless]
/// - `T`: Rotation matrix from the inertial to the true-of-date frame
///
/// Return:
///
/// - `a`: Acceleration due to drag in the X, Y, and Z inertial directions. [m/s^2]
///
/// References:
///
/// 1. O. Montenbruck, and E. Gill, _Satellite Orbits: Models, Methods and Applications_, 2012, p.83-86.
///
#[allow(non_snake_case)]
pub fn acceleration_drag(
    x_object: Vector6<f64>,
    density: f64,
    mass: f64,
    area: f64,
    drag_coefficient: f64,
    T: SMatrix3,
) -> Vector3<f64> {
    // Position and velocity in true-of-date system
    let r_tod: Vector3<f64> = T * x_object.fixed_rows::<3>(0);
    let v_tod: Vector3<f64> = T * x_object.fixed_rows::<3>(3);

    // Velocity relative to the Earth's atmosphere
    let v_rel = v_tod - OMEGA_VECTOR.cross(&r_tod);
    let v_abs = v_rel.norm();

    // Acceleration
    let a_tod = -0.5 * drag_coefficient * (area / mass) * density * v_abs * v_rel;

    T.transpose() * a_tod
}

/// Computes the atmospheric density for the modified Harris-Priester model from Montenbruck and Gill.
///
/// Arguments:
///
/// - r_tod: Satellite position in the true-of-date frame [m]
/// - r_sun: Sun position in the true-of-date frame [m]
///
/// Returns:
///
/// - density: Atmospheric density at the satellite position [kg/m^3]
pub fn density_harris_priester(r_tod: Vector3<f64>, r_sun: Vector3<f64>) -> f64 {
    // Harris-Priester Constants
    const HP_UPPER_LIMIT: f64 = 1000.0; // Upper height limit [km]
    const HP_LOWER_LIMIT: f64 = 100.0; // Lower height limit [km]
    #[allow(clippy::approx_constant)]
    const HP_RA_LAG: f64 = 0.523599; // Right ascension lag [rad]
    const HP_N_PRM: f64 = 3.0; // Harris-Priester parameter
    // 2(6) low(high) inclination
    const HP_N: usize = 50; // Number of coefficients

    // Height [km]
    const HP_H: [f64; 50] = [
        100.0, 120.0, 130.0, 140.0, 150.0, 160.0, 170.0, 180.0, 190.0, 200.0, 210.0, 220.0, 230.0,
        240.0, 250.0, 260.0, 270.0, 280.0, 290.0, 300.0, 320.0, 340.0, 360.0, 380.0, 400.0, 420.0,
        440.0, 460.0, 480.0, 500.0, 520.0, 540.0, 560.0, 580.0, 600.0, 620.0, 640.0, 660.0, 680.0,
        700.0, 720.0, 740.0, 760.0, 780.0, 800.0, 840.0, 880.0, 920.0, 960.0, 1000.0,
    ];

    // Minimum density [g/km^3]
    const HP_C_MIN: [f64; 50] = [
        4.974e+05, 2.490e+04, 8.377e+03, 3.899e+03, 2.122e+03, 1.263e+03, 8.008e+02, 5.283e+02,
        3.617e+02, 2.557e+02, 1.839e+02, 1.341e+02, 9.949e+01, 7.488e+01, 5.709e+01, 4.403e+01,
        3.430e+01, 2.697e+01, 2.139e+01, 1.708e+01, 1.099e+01, 7.214e+00, 4.824e+00, 3.274e+00,
        2.249e+00, 1.558e+00, 1.091e+00, 7.701e-01, 5.474e-01, 3.916e-01, 2.819e-01, 2.042e-01,
        1.488e-01, 1.092e-01, 8.070e-02, 6.012e-02, 4.519e-02, 3.430e-02, 2.632e-02, 2.043e-02,
        1.607e-02, 1.281e-02, 1.036e-02, 8.496e-03, 7.069e-03, 4.680e-03, 3.200e-03, 2.210e-03,
        1.560e-03, 1.150e-03,
    ];

    // Maximum density [g/km^3]
    const HP_C_MAX: [f64; 50] = [
        4.974e+05, 2.490e+04, 8.710e+03, 4.059e+03, 2.215e+03, 1.344e+03, 8.758e+02, 6.010e+02,
        4.297e+02, 3.162e+02, 2.396e+02, 1.853e+02, 1.455e+02, 1.157e+02, 9.308e+01, 7.555e+01,
        6.182e+01, 5.095e+01, 4.226e+01, 3.526e+01, 2.511e+01, 1.819e+01, 1.337e+01, 9.955e+00,
        7.492e+00, 5.684e+00, 4.355e+00, 3.362e+00, 2.612e+00, 2.042e+00, 1.605e+00, 1.267e+00,
        1.005e+00, 7.997e-01, 6.390e-01, 5.123e-01, 4.121e-01, 3.325e-01, 2.691e-01, 2.185e-01,
        1.779e-01, 1.452e-01, 1.190e-01, 9.776e-02, 8.059e-02, 5.741e-02, 4.210e-02, 3.130e-02,
        2.360e-02, 1.810e-02,
    ];

    // Satellite height
    let geod = position_ecef_to_geodetic(r_tod, AngleFormat::Radians);
    let height = geod[2] / 1.0e3; // height in [km]

    // Exit with zero density outside height model limits
    #[allow(clippy::manual_range_contains)]
    if height >= HP_UPPER_LIMIT || height <= HP_LOWER_LIMIT {
        return 0.0;
    }

    // Sun right ascension, declination
    let ra_sun = r_sun[1].atan2(r_sun[0]);
    let dec_sun = r_sun[2].atan2((r_sun[0].powi(2) + r_sun[1].powi(2)).sqrt());

    // Unit vector u towards the apex of the diurnal bulge
    // in inertial geocentric coordinates
    let c_dec = dec_sun.cos();
    let u = Vector3::new(
        c_dec * (ra_sun + HP_RA_LAG).cos(),
        c_dec * (ra_sun + HP_RA_LAG).sin(),
        dec_sun.sin(),
    );

    // Cosine of half angle between satellite position vector and
    // apex of diurnal bulge
    let c_psi2 = 0.5 + 0.5 * r_tod.dot(&u) / r_tod.norm();

    // Height index search and exponential density interpolation
    let mut ih = 0; // section index reset
    for i in 0..(HP_N - 1) {
        // loop over N_Coef height regimes
        if height >= HP_H[i] && height < HP_H[i + 1] {
            ih = i; // ih identifies height section
            break;
        }
    }

    let h_min = (HP_H[ih] - HP_H[ih + 1]) / (HP_C_MIN[ih + 1] / HP_C_MIN[ih]).ln();
    let h_max = (HP_H[ih] - HP_H[ih + 1]) / (HP_C_MAX[ih + 1] / HP_C_MAX[ih]).ln();

    let d_min = HP_C_MIN[ih] * ((HP_H[ih] - height) / h_min).exp();
    let d_max = HP_C_MAX[ih] * ((HP_H[ih] - height) / h_max).exp();

    // Density computation
    let density = d_min + (d_max - d_min) * c_psi2.powf(HP_N_PRM);

    // Convert from g/km^3 to kg/m^3 and return
    density * 1.0e-12
}

#[cfg(test)]
#[allow(clippy::too_many_arguments)]
mod tests {
    use approx::assert_abs_diff_eq;
    use rstest::rstest;

    use crate::constants::AngleFormat;
    use crate::constants::R_EARTH;
    use crate::coordinates::*;
    use crate::time::Epoch;
    use crate::{TimeSystem, sun_position};

    use super::*;

    #[test]
    fn test_acceleration_drag() {
        let oe = Vector6::new(R_EARTH + 500e3, 0.01, 97.3, 15.0, 30.0, 45.0);

        let x_object = state_osculating_to_cartesian(oe, DEGREES);

        let a = acceleration_drag(x_object, 1.0e-12, 1000.0, 1.0, 2.0, SMatrix3::identity());

        assert_abs_diff_eq!(a.norm(), 5.97601877277239e-8, epsilon = 1.0e-10);
    }

    #[rstest]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 6466752.314, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 6566752.314, 2.02686e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 6666752.314, 1.91155e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 6766752.314, 3.44149e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 6866752.314, 8.3078e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 6966752.314, 2.39134e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 7066752.314, 7.85043e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 7166752.314, 2.91971e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 7266752.314, 1.29753e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4595372.625, 0.000, - 4565130.155, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4666083.303, 0.000, - 4635840.833, 2.18111e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4736793.981, 0.000, - 4706551.511, 2.35614e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4807504.659, 0.000, - 4777262.189, 4.7302e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4878215.337, 0.000, - 4847972.867, 1.24008e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4948926.015, 0.000, - 4918683.545, 3.78277e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 5019636.693, 0.000, - 4989394.224, 1.28079e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 5090347.372, 0.000, - 5060104.902, 4.79319e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 5161058.050, 0.000, - 5130815.580, 2.16372e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 6488137.000, 0.000, 0.000, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 6588137.000, 0.000, 0.000, 2.1227e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 6688137.000, 0.000, 0.000, 2.1878e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 6788137.000, 0.000, 0.000, 4.24226e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 6888137.000, 0.000, 0.000, 1.08512e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 6988137.000, 0.000, 0.000, 3.25598e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 7088137.000, 0.000, 0.000, 1.09311e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 7188137.000, 0.000, 0.000, 4.08396e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 7288137.000, 0.000, 0.000, 1.83582e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4595372.625, 0.000, 4565130.155, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4666083.303, 0.000, 4635840.833, 1.94414e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4736793.981, 0.000, 4706551.511, 1.67309e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4807504.659, 0.000, 4777262.189, 2.75018e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4878215.337, 0.000, 4847972.867, 6.11189e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4948926.015, 0.000, 4918683.545, 1.64474e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 5019636.693, 0.000, 4989394.224, 5.19005e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 5090347.372, 0.000, 5060104.902, 1.9142e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 5161058.050, 0.000, 5130815.580, 8.32597e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 6466752.314, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 6566752.314, 2.02686e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 6666752.314, 1.91155e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 6766752.314, 3.44149e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 6866752.314, 8.3078e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 6966752.314, 2.39134e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 7066752.314, 7.85043e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 7166752.314, 2.91971e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 7266752.314, 1.29753e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3249419.145, 3249419.145, - 4565130.155, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3299419.145, 3299419.145, - 4635840.833, 1.96699e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3349419.145, 3349419.145, - 4706551.511, 1.73901e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3399419.145, 3399419.145, - 4777262.189, 2.94137e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3449419.145, 3449419.145, - 4847972.867, 6.71949e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3499419.145, 3499419.145, - 4918683.545, 1.85142e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3549419.145, 3549419.145, - 4989394.224, 5.92685e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3599419.145, 3599419.145, - 5060104.902, 2.19281e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3649419.145, 3649419.145, - 5130815.580, 9.61476e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4587805.670, 4587805.670, 0.000, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4658516.348, 4658516.348, 0.000, 1.89456e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4729227.026, 4729227.026, 0.000, 1.53023e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4799937.704, 4799937.704, 0.000, 2.33613e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4870648.382, 4870648.382, 0.000, 4.79705e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4941359.061, 4941359.061, 0.000, 1.19782e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 5012069.739, 5012069.739, 0.000, 3.59797e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 5082780.417, 5082780.417, 0.000, 1.31263e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 5153491.095, 5153491.095, 0.000, 5.54503e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3249419.145, 3249419.145, 4565130.155, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3299419.145, 3299419.145, 4635840.833, 1.86015e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3349419.145, 3349419.145, 4706551.511, 1.43103e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3399419.145, 3399419.145, 4777262.189, 2.04857e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3449419.145, 3449419.145, 4847972.867, 3.88368e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3499419.145, 3499419.145, 4918683.545, 8.87301e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3549419.145, 3549419.145, 4989394.224, 2.49158e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3599419.145, 3599419.145, 5060104.902, 8.94486e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3649419.145, 3649419.145, 5130815.580, 3.61172e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 6466752.314, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 6566752.314, 2.02686e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 6666752.314, 1.91155e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 6766752.314, 3.44149e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 6866752.314, 8.3078e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 6966752.314, 2.39134e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 7066752.314, 7.85043e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 7166752.314, 2.91971e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 0.000, - 7266752.314, 1.29753e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 4595372.625, - 4565130.155, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 4666083.303, - 4635840.833, 1.87205e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 4736793.981, - 4706551.511, 1.46534e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 4807504.659, - 4777262.189, 2.14808e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 4878215.337, - 4847972.867, 4.19989e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 4948926.015, - 4918683.545, 9.94849e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 5019636.693, - 4989394.224, 2.87493e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 5090347.372, - 5060104.902, 1.03942e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 5161058.050, - 5130815.580, 4.28211e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 6488137.000, 0.000, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 6588137.000, 0.000, 1.84089e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 6688137.000, 0.000, 1.37551e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 6788137.000, 0.000, 1.88764e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 6888137.000, 0.000, 3.37258e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 6988137.000, 0.000, 7.13557e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 7088137.000, 0.000, 1.87256e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 7188137.000, 0.000, 6.6056e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 7288137.000, 0.000, 2.53021e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 4595372.625, 4565130.155, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 4666083.303, 4635840.833, 1.83982e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 4736793.981, 4706551.511, 1.37243e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 4807504.659, 4777262.189, 1.87871e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 4878215.337, 4847972.867, 3.34422e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 4948926.015, 4918683.545, 7.03917e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 5019636.693, 4989394.224, 1.83822e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 5090347.372, 5060104.902, 6.47581e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, 5161058.050, 5130815.580, 2.47021e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, 0.000, - 6466752.314, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, 0.000, - 6566752.314, 2.02686e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, 0.000, - 6666752.314, 1.91155e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, 0.000, - 6766752.314, 3.44149e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, 0.000, - 6866752.314, 8.3078e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, 0.000, - 6966752.314, 2.39134e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, 0.000, - 7066752.314, 7.85043e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, 0.000, - 7166752.314, 2.91971e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, 0.000, - 7266752.314, 1.29753e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3249419.145, 3249419.145, - 4565130.155, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3299419.145, 3299419.145, - 4635840.833, 1.85607e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3349419.145, 3349419.145, - 4706551.511, 1.41928e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3399419.145, 3399419.145, - 4777262.189, 2.01455e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3449419.145, 3449419.145, - 4847972.867, 3.77574e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3499419.145, 3499419.145, - 4918683.545, 8.50645e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3549419.145, 3549419.145, - 4989394.224, 2.36111e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3599419.145, 3599419.145, - 5060104.902, 8.45229e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3649419.145, 3649419.145, - 5130815.580, 3.3842e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4587805.670, 4587805.670, 0.000, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4658516.348, 4658516.348, 0.000, 1.83904e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4729227.026, 4729227.026, 0.000, 1.37018e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4799937.704, 4799937.704, 0.000, 1.87221e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4870648.382, 4870648.382, 0.000, 3.32358e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4941359.061, 4941359.061, 0.000, 6.96899e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 5012069.739, 5012069.739, 0.000, 1.81321e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 5082780.417, 5082780.417, 0.000, 6.38129e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 5153491.095, 5153491.095, 0.000, 2.42651e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3249419.145, 3249419.145, 4565130.155, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3299419.145, 3299419.145, 4635840.833, 1.83903e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3349419.145, 3349419.145, 4706551.511, 1.37015e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3399419.145, 3399419.145, 4777262.189, 1.87211e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3449419.145, 3449419.145, 4847972.867, 3.32325e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3499419.145, 3499419.145, 4918683.545, 6.96786e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3549419.145, 3549419.145, 4989394.224, 1.81281e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3599419.145, 3599419.145, 5060104.902, 6.37978e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3649419.145, 3649419.145, 5130815.580, 2.42581e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, 0.000, - 6466752.314, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, 0.000, - 6566752.314, 2.02686e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, 0.000, - 6666752.314, 1.91155e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, 0.000, - 6766752.314, 3.44149e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, 0.000, - 6866752.314, 8.3078e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, 0.000, - 6966752.314, 2.39134e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, 0.000, - 7066752.314, 7.85043e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, 0.000, - 7166752.314, 2.91971e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, 0.000, - 7266752.314, 1.29753e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4595372.625, 0.000, - 4565130.155, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4666083.303, 0.000, - 4635840.833, 1.88217e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4736793.981, 0.000, - 4706551.511, 1.49452e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4807504.659, 0.000, - 4777262.189, 2.23267e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4878215.337, 0.000, - 4847972.867, 4.46856e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4948926.015, 0.000, - 4918683.545, 1.08619e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 5019636.693, 0.000, - 4989394.224, 3.20038e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 5090347.372, 0.000, - 5060104.902, 1.16242e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 5161058.050, 0.000, - 5130815.580, 4.85081e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 6488137.000, 0.000, 0.000, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 6588137.000, 0.000, 0.000, 1.84355e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 6688137.000, 0.000, 0.000, 1.38318e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 6788137.000, 0.000, 0.000, 1.90989e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 6888137.000, 0.000, 0.000, 3.44326e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 6988137.000, 0.000, 0.000, 7.37582e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 7088137.000, 0.000, 0.000, 1.95816e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 7188137.000, 0.000, 0.000, 6.9291e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 7288137.000, 0.000, 0.000, 2.67978e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4595372.625, 0.000, 4565130.155, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4666083.303, 0.000, 4635840.833, 1.84088e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4736793.981, 0.000, 4706551.511, 1.37548e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4807504.659, 0.000, 4777262.189, 1.88758e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4878215.337, 0.000, 4847972.867, 3.37239e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4948926.015, 0.000, 4918683.545, 7.13492e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 5019636.693, 0.000, 4989394.224, 1.87233e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 5090347.372, 0.000, 5060104.902, 6.60475e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 5161058.050, 0.000, 5130815.580, 2.52983e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 0.000, - 6466752.314, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 0.000, - 6566752.314, 2.02686e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 0.000, - 6666752.314, 1.91155e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 0.000, - 6766752.314, 3.44149e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 0.000, - 6866752.314, 8.3078e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 0.000, - 6966752.314, 2.39134e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 0.000, - 7066752.314, 7.85043e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 0.000, - 7166752.314, 2.91971e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 0.000, - 7266752.314, 1.29753e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3249419.145, - 3249419.145, - 4565130.155, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3299419.145, - 3299419.145, - 4635840.833, 2.00199e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3349419.145, - 3349419.145, - 4706551.511, 1.83986e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3399419.145, - 3399419.145, - 4777262.189, 3.23371e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3449419.145, - 3449419.145, - 4847972.867, 7.64797e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3499419.145, - 3499419.145, - 4918683.545, 2.16706e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3549419.145, - 3549419.145, - 4989394.224, 7.05143e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3599419.145, - 3599419.145, - 5060104.902, 2.6178e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3649419.145, - 3649419.145, - 5130815.580, 1.15796e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4587805.670, - 4587805.670, 0.000, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4658516.348, - 4658516.348, 0.000, 1.92488e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4729227.026, - 4729227.026, 0.000, 1.6176e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4799937.704, - 4799937.704, 0.000, 2.58939e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4870648.382, - 4870648.382, 0.000, 5.60144e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 4941359.061, - 4941359.061, 0.000, 1.47128e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 5012069.739, - 5012069.739, 0.000, 4.5723e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 5082780.417, - 5082780.417, 0.000, 1.68084e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 5153491.095, - 5153491.095, 0.000, 7.24747e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3249419.145, - 3249419.145, 4565130.155, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3299419.145, - 3299419.145, 4635840.833, 1.87141e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3349419.145, - 3349419.145, 4706551.511, 1.46347e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3399419.145, - 3399419.145, 4777262.189, 2.14262e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3449419.145, - 3449419.145, 4847972.867, 4.18237e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3499419.145, - 3499419.145, 4918683.545, 9.88836e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3549419.145, - 3549419.145, 4989394.224, 2.85331e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3599419.145, - 3599419.145, 5060104.902, 1.03118e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 3649419.145, - 3649419.145, 5130815.580, 4.24368e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 0.000, - 6466752.314, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 0.000, - 6566752.314, 2.02686e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 0.000, - 6666752.314, 1.91155e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 0.000, - 6766752.314, 3.44149e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 0.000, - 6866752.314, 8.3078e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 0.000, - 6966752.314, 2.39134e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 0.000, - 7066752.314, 7.85043e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 0.000, - 7166752.314, 2.91971e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 0.000, - 7266752.314, 1.29753e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 4595372.625, - 4565130.155, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 4666083.303, - 4635840.833, 2.22688e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 4736793.981, - 4706551.511, 2.48806e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 4807504.659, - 4777262.189, 5.11258e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 4878215.337, - 4847972.867, 1.36153e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 4948926.015, - 4918683.545, 4.19561e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 5019636.693, - 4989394.224, 1.42788e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 5090347.372, - 5060104.902, 5.34903e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 5161058.050, - 5130815.580, 2.42069e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 6488137.000, 0.000, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 6588137.000, 0.000, 2.18086e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 6688137.000, 0.000, 2.35544e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 6788137.000, 0.000, 4.72819e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 6888137.000, 0.000, 1.23946e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 6988137.000, 0.000, 3.78068e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 7088137.000, 0.000, 1.28006e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 7188137.000, 0.000, 4.79047e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 7288137.000, 0.000, 2.16247e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 4595372.625, 4565130.155, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 4666083.303, 4635840.833, 1.96541e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 4736793.981, 4706551.511, 1.73439e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 4807504.659, 4777262.189, 2.92786e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 4878215.337, 4847972.867, 6.67618e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 4948926.015, 4918683.545, 1.83655e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 5019636.693, 4989394.224, 5.87341e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 5090347.372, 5060104.902, 2.17244e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, - 0.000, - 5161058.050, 5130815.580, 9.51979e-15)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, - 0.000, - 6466752.314, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, - 0.000, - 6566752.314, 2.02686e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, - 0.000, - 6666752.314, 1.91155e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, - 0.000, - 6766752.314, 3.44149e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, - 0.000, - 6866752.314, 8.3078e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, - 0.000, - 6966752.314, 2.39134e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, - 0.000, - 7066752.314, 7.85043e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, - 0.000, - 7166752.314, 2.91971e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 0.000, - 0.000, - 7266752.314, 1.29753e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3249419.145, - 3249419.145, - 4565130.155, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3299419.145, - 3299419.145, - 4635840.833, 2.33717e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3349419.145, - 3349419.145, - 4706551.511, 2.80592e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3399419.145, - 3399419.145, - 4777262.189, 6.0339e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3449419.145, - 3449419.145, - 4847972.867, 1.65413e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3499419.145, - 3499419.145, - 4918683.545, 5.1903e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3549419.145, - 3549419.145, - 4989394.224, 1.78226e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3599419.145, - 3599419.145, - 5060104.902, 6.68824e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3649419.145, - 3649419.145, - 5130815.580, 3.03984e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4587805.670, - 4587805.670, 0.000, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4658516.348, - 4658516.348, 0.000, 2.32952e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4729227.026, - 4729227.026, 0.000, 2.78392e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4799937.704, - 4799937.704, 0.000, 5.97024e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4870648.382, - 4870648.382, 0.000, 1.63395e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 4941359.061, - 4941359.061, 0.000, 5.12179e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 5012069.739, - 5012069.739, 0.000, 1.75789e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 5082780.417, - 5082780.417, 0.000, 6.59629e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 5153491.095, - 5153491.095, 0.000, 2.99739e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3249419.145, - 3249419.145, 4565130.155, 1.11289e-07)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3299419.145, - 3299419.145, 4635840.833, 2.01966e-10)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3349419.145, - 3349419.145, 4706551.511, 1.89075e-11)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3399419.145, - 3399419.145, 4777262.189, 3.38104e-12)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3449419.145, - 3449419.145, 4847972.867, 8.11538e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3499419.145, - 3499419.145, 4918683.545, 2.32578e-13)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3549419.145, - 3549419.145, 4989394.224, 7.61632e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3599419.145, - 3599419.145, 5060104.902, 2.83105e-14)]
    #[case(24622331959.580, - 133060326832.922, - 57688711921.833, 3649419.145, - 3649419.145, 5130815.580, 1.25646e-14)]
    fn test_harris_priester_cross_validation(
        #[case] rsun_x: f64,
        #[case] rsun_y: f64,
        #[case] rsun_z: f64,
        #[case] r_x: f64,
        #[case] r_y: f64,
        #[case] r_z: f64,
        #[case] rho_expected: f64,
    ) {
        let r_sun = Vector3::new(rsun_x, rsun_y, rsun_z);
        let r = Vector3::new(r_x, r_y, r_z);

        let rho = density_harris_priester(r, r_sun);

        assert_abs_diff_eq!(rho, rho_expected, epsilon = 1.0e-12);
    }

    #[test]
    fn test_harris_priester_bounds() {
        let r_sun = Vector3::new(24622331959.580, -133060326832.922, -57688711921.833);

        // Test below 100 km threshold
        let r = position_geodetic_to_ecef(Vector3::new(0.0, 0.0, 50.0e3), AngleFormat::Degrees)
            .unwrap();
        let rho = density_harris_priester(r, r_sun);

        assert_eq!(rho, 0.0);

        // Test above 1000 km threshold
        let r = position_geodetic_to_ecef(Vector3::new(0.0, 0.0, 1100.0e3), AngleFormat::Degrees)
            .unwrap();
        let rho = density_harris_priester(r, r_sun);

        assert_eq!(rho, 0.0);
    }

    #[rstest]
    #[case(60310.000, 1.000, 100.000, 2.300, 4.884992303790e+06, 4.553508537449e+06, 1.330313604797e+06, - 5.300020580854e+03, 4.925709387424e+03, 2.601867699687e+03, 6.475291922546e-07, - 5.955905823290e-07, - 3.391292982024e-07)]
    #[case(60310.000, 1.000, 100.000, 2.300, 3.854234369244e+06, 5.331427650549e+06, 1.762510444136e+06, - 6.281746044740e+03, 3.815301677691e+03, 2.254442602385e+03, 6.650957772889e-07, - 3.988840098278e-07, - 2.544421167373e-07)]
    #[case(60310.000, 1.000, 100.000, 2.300, 2.670937897492e+06, 5.898362795150e+06, 2.124959710177e+06, - 7.013918036984e+03, 2.555322690457e+03, 1.818275122888e+03, 6.409667332941e-07, - 2.298120374578e-07, - 1.770183557869e-07)]
    #[case(60310.000, 1.000, 100.000, 2.300, 1.382105394865e+06, 6.232286239143e+06, 2.403465989137e+06, - 7.467690875201e+03, 1.197415434669e+03, 1.311370423462e+03, 5.842585088334e-07, - 9.135822408135e-08, - 1.092477881395e-07)]
    #[case(60310.000, 1.000, 100.000, 2.300, 3.879697748585e+04, 6.320698885147e+06, 2.587294936269e+06, - 7.626311735293e+03, - 2.030743556792e+02, 7.545803726769e+02, 5.026456239033e-07, 1.444392491547e-08, - 5.293306533864e-08)]
    #[case(60310.000, 1.000, 100.000, 2.300, - 1.306071545499e+06, 6.161054495134e+06, 2.669589120058e+06, - 7.485641457809e+03, - 1.589726432986e+03, 1.705831662309e+02, 4.081446766290e-07, 8.668767563667e-08, - 9.894677441492e-09)]
    #[case(60310.000, 1.000, 100.000, 2.300, - 2.599961458555e+06, 5.760720193579e+06, 2.647597126838e+06, - 7.053988043184e+03, - 2.907616196131e+03, - 4.171887600156e+02, 3.131156928240e-07, 1.282887478988e-07, 1.969100563219e-08)]
    #[case(60310.000, 1.000, 100.000, 2.300, - 3.792857897424e+06, 5.136498372385e+06, 2.522713383918e+06, - 6.351318581368e+03, - 4.105664983317e+03, - 9.855989555158e+02, 2.275007278096e-07, 1.457511811933e-07, 3.751606537078e-08)]
    #[case(60310.000, 1.000, 100.000, 2.300, - 4.839229618329e+06, 4.313760582551e+06, 2.300338349966e+06, - 5.407960526895e+03, - 5.138647350964e+03, - 1.512772642576e+03, 1.559048510435e-07, 1.464885073622e-07, 4.630478305728e-08)]
    #[case(60310.000, 1.000, 100.000, 2.300, - 5.699723056095e+06, 3.325261742120e+06, 1.989578439892e+06, - 4.262928650995e+03, - 5.968779101105e+03, - 1.978931630018e+03, 9.954914915777e-08, 1.374999502478e-07, 4.899977063788e-08)]
    #[case(60310.000, 1.000, 100.000, 2.300, - 6.342536886568e+06, 2.209712299398e+06, 1.602811608208e+06, - 2.962019299787e+03, - 6.566856713480e+03, - 2.367070862012e+03, 5.686726682414e-08, 1.239386286371e-07, 4.805940901794e-08)]
    #[case(60310.000, 1.000, 100.000, 2.300, - 6.744447614457e+06, 1.010187496333e+06, 1.155147802510e+06, - 1.555801915035e+03, - 6.912958990231e+03, - 2.663466756131e+03, 2.527983956238e-08, 1.095212143544e-07, 4.542898162750e-08)]
    #[case(60310.000, 1.000, 100.000, 2.300, - 6.891477821536e+06, - 2.275518102869e+05, 6.638138965866e+05, - 9.761444639723e+01, - 6.996750915333e+03, - 2.858022017467e+03, 1.713172708983e-09, 9.741641119267e-08, 4.287172531695e-08)]
    #[case(60310.000, 1.000, 100.000, 2.300, - 6.779215094915e+06, - 1.456761075547e+06, 1.474909195700e+05, 1.358359721155e+03, - 6.817446444227e+03, - 2.944461654893e+03, - 1.773818124483e-08, 8.957550077534e-08, 4.171240578084e-08)]
    #[case(60310.000, 1.000, 100.000, 2.300, - 6.412800799786e+06, - 2.631381709006e+06, - 3.743717496543e+05, 2.758884958527e+03, - 6.383491905142e+03, - 2.920400175922e+03, - 3.788560161627e-08, 8.731043787305e-08, 4.310130707932e-08)]
    #[case(60310.000, 1.000, 100.000, 2.300, - 5.806614110708e+06, - 3.707634060799e+06, - 8.822442375822e+05, 4.053313697470e+03, - 5.712027156556e+03, - 2.787300862703e+03, - 6.446074751998e-08, 9.011680293095e-08, 4.749509376747e-08)]
    #[case(60310.000, 1.000, 100.000, 2.300, - 4.983679016998e+06, - 4.645498608912e+06, - 1.357188627116e+06, 5.195069678263e+03, - 4.828170587673e+03, - 2.550345567020e+03, - 1.036692911994e-07, 9.531054328296e-08, 5.444304022724e-08)]
    #[case(60310.000, 1.000, 100.000, 2.300, - 3.974821657117e+06, - 5.410057126297e+06, - 1.781501428157e+06, 6.142981046602e+03, - 3.764159581142e+03, - 2.218228747299e+03, - 1.614433532925e-07, 9.757449849799e-08, 6.229798340648e-08)]
    #[case(60310.000, 1.000, 100.000, 2.300, - 2.817603714143e+06, - 5.972669872748e+06, - 2.139313418925e+06, 6.862473412589e+03, - 2.558363041530e+03, - 1.802883666286e+03, - 2.421864827659e-07, 8.866437983603e-08, 6.793810291675e-08)]
    #[case(60310.000, 1.000, 100.000, 2.300, - 1.555056064850e+06, - 6.311966594700e+06, - 2.417135091026e+06, 7.326629008929e+03, - 1.254169479678e+03, - 1.319142854603e+03, - 3.470644698157e-07, 5.766114271947e-08, 6.667697200344e-08)]
    #[case(60310.000, 1.000, 100.000, 2.300, - 2.342365879764e+05, - 6.414628848619e+06, - 2.604335853094e+06, 7.517108189447e+03, 1.012550540356e+02, - 7.843300873158e+02, - 4.725061343330e-07, - 7.931859928099e-09, 5.257236492762e-08)]
    #[case(60310.000, 1.000, 100.000, 2.300, 1.095362092957e+06, - 6.275939733234e+06, - 2.693544355988e+06, 7.424915414211e+03, 1.458348575476e+03, - 2.177780628923e+02, - 6.065981611401e-07, - 1.200154294953e-07, 1.896063186108e-08)]
    #[case(60310.000, 1.000, 100.000, 2.300, 2.383524570841e+06, - 5.900075611853e+06, - 2.680956181964e+06, 7.050975739560e+03, 2.766680692859e+03, 3.597345651486e+02, - 7.304906266784e-07, - 2.860812404277e-07, - 3.969086848276e-08)]
    #[case(60310.000, 1.000, 100.000, 2.300, 3.581106522323e+06, - 5.300114080501e+06, - 2.566534189861e+06, 6.406472303857e+03, 3.976622102432e+03, 9.266306232258e+02, - 8.157042028116e-07, - 5.034459753379e-07, - 1.255579403572e-07)]
    #[case(60310.000, 1.000, 100.000, 2.300, 4.641862770238e+06, - 4.497734593543e+06, - 2.354086603151e+06, 5.512883447615e+03, 5.041134556879e+03, 1.461264246671e+03, - 8.323328365445e-07, - 7.549157872248e-07, - 2.345768393719e-07)]
    #[case(60310.000, 1.000, 100.000, 2.300, 5.524250386626e+06, - 3.522593991176e+06, - 2.051209062029e+06, 4.401653269803e+03, 5.917636197580e+03, 1.942721880224e+03, - 7.566927617697e-07, - 1.006810656914e-06, - 3.546733618769e-07)]
    #[case(60310.000, 1.000, 100.000, 2.300, 6.193136243056e+06, - 2.411369562038e+06, - 1.669079863560e+06, 3.113434801059e+03, 6.569869917653e+03, 2.351671474407e+03, - 5.822225786598e-07, - 1.212620773785e-06, - 4.660943067555e-07)]
    #[case(60310.000, 1.000, 100.000, 2.300, 6.621329023944e+06, - 1.206477537446e+06, - 1.222102840276e+06, 1.696862974319e+03, 6.969675450702e+03, 2.671223865848e+03, - 3.295192521274e-07, - 1.328583760690e-06, - 5.470991336177e-07)]
    #[case(60310.000, 1.000, 100.000, 2.300, 6.790850714079e+06, 4.550542743298e+04, - 7.273998381722e+05, 2.068459741129e+02, 7.098544211312e+03, 2.887757995344e+03, - 4.232160931968e-08, - 1.329742026701e-06, - 5.815191957405e-07)]
    #[case(60310.000, 1.000, 100.000, 2.300, 6.693864225551e+06, 1.295667609670e+06, - 2.041642113971e+05, - 1.297593298229e+03, 6.948825108710e+03, 2.991652446262e+03, 2.271126461274e-07, - 1.219593187243e-06, - 5.647373599657e-07)]
    #[case(60310.000, 1.000, 100.000, 2.300, 6.333183868415e+06, 2.494761038735e+06, 3.271026349663e+05, - 2.755843531779e+03, 6.524454555312e+03, 2.977862301791e+03, 4.385586274848e-07, - 1.032983373613e-06, - 5.073840303605e-07)]
    #[case(60310.000, 1.000, 100.000, 2.300, 5.722314467058e+06, 3.595279736803e+06, 8.454390492652e+05, - 4.108466612970e+03, 5.841107621980e+03, 2.846284516417e+03, 5.767674716446e-07, - 8.133252603937e-07, - 4.268119157331e-07)]
    fn validate_acceleration_drag(
        #[case] mjd_tt: f64,
        #[case] area: f64,
        #[case] mass: f64,
        #[case] cd: f64,
        #[case] r_x: f64,
        #[case] r_y: f64,
        #[case] r_z: f64,
        #[case] v_x: f64,
        #[case] v_y: f64,
        #[case] v_z: f64,
        #[case] a_x_expected: f64,
        #[case] a_y_expected: f64,
        #[case] a_z_expected: f64,
    ) {
        let x = Vector6::new(r_x, r_y, r_z, v_x, v_y, v_z);

        let epc = Epoch::from_mjd(mjd_tt, TimeSystem::TT);
        let r_sun = sun_position(epc);

        let rho = density_harris_priester(Vector3::new(r_x, r_y, r_z), r_sun);

        let a = acceleration_drag(x, rho, mass, area, cd, SMatrix3::identity());

        let tol = 1.0e-12;
        assert_abs_diff_eq!(a[0], a_x_expected, epsilon = tol);
        assert_abs_diff_eq!(a[1], a_y_expected, epsilon = tol);
        assert_abs_diff_eq!(a[2], a_z_expected, epsilon = tol);
    }
}
