/*!
Models of solar radiation pressure.
 */

use nalgebra::Vector3;

use crate::constants::{AU, R_EARTH, R_SUN};

/// Calculate the acceleration due to solar radiation pressure.
///
/// # Arguments
///
/// - `r_object`: Position vector of the object.
/// - `r_sun`: Position vector of the sun. If the sun is at the origin, this is the zero vector.
/// - `mass`: Mass of the object.
/// - `cr`: Coefficient of reflectivity.
/// - `area`: Cross-sectional area of the object.
/// - `p0`: Solar radiation pressure at 1 AU.
///
/// # Returns
///
/// - `a_srp` : Acceleration due to solar radiation pressure.
///
/// # Examples
///
/// ```
/// use brahe::orbit_dynamics::acceleration_solar_radiation_pressure;
/// use nalgebra::Vector3;
/// use brahe::constants::AU;
///
/// let r_object = Vector3::new(AU, 0.0, 0.0);
/// let r_sun = Vector3::new(0.0, 0.0, 0.0);
///
/// let a_srp = acceleration_solar_radiation_pressure(r_object, r_sun, 1.0, 1.0, 1.0, 4.5e-6);
///
/// // Acceleration should be in the positive x-direction and magnitude should be 4.5e-6
/// assert!((a_srp[0] - 4.5e-6).abs() < 1e-12);
/// assert!((a_srp[1] - 0.0).abs() < 1e-12);
/// assert!((a_srp[2] - 0.0).abs() < 1e-12);
/// ```
pub fn acceleration_solar_radiation_pressure(
    r_object: Vector3<f64>,
    r_sun: Vector3<f64>,
    mass: f64,
    cr: f64,
    area: f64,
    p0: f64,
) -> Vector3<f64> {
    let d = r_object - r_sun;

    d * cr * (area / mass) * p0 * AU.powi(2) / d.norm().powi(3)
}

/// Calculate the fraction of the object that is illuminated by the sun using a conical model
/// for Earth shadowing.
///
/// # Arguments
///
/// - `r_object`: Position vector of the object in the ECI frame.
/// - `r_sun`: Position vector of the sun. If the sun is at the origin, this is the zero vector.
///
/// # Returns
///
/// - `nu`: Illumination fraction of the object (0.0 = fully shadowed, 1.0 = fully illuminated).
///
/// # Examples
///
/// ```
/// use brahe::orbit_dynamics::eclipse_conical;
/// use nalgebra::Vector3;
/// use brahe::constants::R_EARTH;
///
/// let r_object = Vector3::new(R_EARTH, 0.0, 0.0);
/// let r_sun = Vector3::new(0.0, 0.0, 0.0);
///
/// let nu = eclipse_conical(r_object, r_sun);
///
/// // The object is shadowed, so the illumination fraction should be 0.0
/// assert_eq!(nu, 0.0);
/// ```
#[allow(non_snake_case)] // To better comply with the literature
pub fn eclipse_conical(r_object: Vector3<f64>, r_sun: Vector3<f64>) -> f64 {
    // Occultation Geometry
    let a = (R_SUN / (r_sun - r_object).norm()).asin();
    let b = (R_EARTH / r_object.norm()).asin();
    let c =
        (-r_object.dot(&(r_sun - r_object)) / (r_object.norm() * (r_sun - r_object).norm())).acos();

    // Test Occulation Conditions and return illumination fraction
    if (a - b).abs() < c && c < (a + b) {
        // Partial occultation

        let x = (c.powi(2) + a.powi(2) - b.powi(2)) / (2.0 * c);
        let y = (a.powi(2) - x.powi(2)).sqrt();
        let A = a.powi(2) * (x / a).acos() + b.powi(2) * ((c - x) / b).acos() - c * y;

        1.0 - A / (std::f64::consts::PI * a.powi(2))
    } else if (a + b) <= c {
        // No occultation
        1.0
    } else {
        // Full occultation
        0.0
    }
}

/// Calculate the fraction of the object that is illuminated by the sun using a cylindrical model
/// for Earth shadowing.
///
/// # Arguments
///
/// - `r_object`: Position vector of the object in the ECI frame.
/// - `r_sun`: Position vector of the sun. If the sun is at the origin, this is the zero vector..
///
/// # Returns
///
/// - `nu`: Illumination fraction of the object.
///
/// # Examples
///
/// ```
/// use brahe::orbit_dynamics::eclipse_cylindrical;
/// use nalgebra::Vector3;
/// use brahe::constants::R_EARTH;
///
/// let r_object = Vector3::new(R_EARTH, 0.0, 0.0);
/// let r_sun = Vector3::new(0.0, 0.0, 0.0);
///
/// let nu = eclipse_cylindrical(r_object, r_sun);
///
/// // The object is shadowed, so the illumination fraction should be 0.0
/// assert_eq!(nu, 0.0);
/// ```
pub fn eclipse_cylindrical(r_object: Vector3<f64>, r_sun: Vector3<f64>) -> f64 {
    // Unit vector in the direction of the sun
    let e_sun = r_sun / r_sun.norm();

    // Projection of spacecraft position vector onto the sun vector
    let r_proj = r_object.dot(&e_sun);

    // Compute illumination fraction
    if r_proj >= 1.0 || (r_object - r_proj * e_sun).norm() > R_EARTH {
        1.0
    } else {
        0.0
    }
}

#[cfg(test)]
#[allow(clippy::too_many_arguments)]
mod tests {
    use approx::assert_abs_diff_eq;
    use rstest::rstest;

    use crate::{Epoch, TimeSystem, sun_position};

    use super::*;

    #[test]
    fn test_acceleration_solar_radiation_pressure() {
        let r_object = Vector3::new(AU, 0.0, 0.0);
        let r_sun = Vector3::new(0.0, 0.0, 0.0);

        let a_srp = acceleration_solar_radiation_pressure(r_object, r_sun, 1.0, 1.0, 1.0, 4.5e-6);

        // Acceleration should be in the negative x-direction and magnitude should be 4.5e-6 AU^2
        assert_abs_diff_eq!(a_srp[0], 4.5e-6, epsilon = 1e-12);
        assert_abs_diff_eq!(a_srp[1], 0.0, epsilon = 1e-12);
        assert_abs_diff_eq!(a_srp[2], 0.0, epsilon = 1e-12);
        assert_abs_diff_eq!(a_srp.norm(), 4.5e-6, epsilon = 1e-12);
    }

    #[rstest]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 0.0, 4884992.30378986, 4553508.53744864, 1330313.60479734, - 1.42047171104736e-08, 7.67808694826067e-08, 3.3288209420537e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 0.0, 3854234.36924427, 5331427.65054898, 1762510.44413639, - 1.42050089628377e-08, 7.677968107271e-08, 3.32877489635206e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 0.0, 2670937.8974923, 5898362.79515022, 2124959.71017719, - 1.42054446252169e-08, 7.67786726161509e-08, 3.3287379057795e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 0.0, 1382105.39486525, 6232286.23914348, 2403465.98913669, - 1.42060066401701e-08, 7.67778836164616e-08, 3.32871141500211e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 0.0, 38796.9774858514, 6320698.88514676, 2587294.93626938, - 1.42066725214415e-08, 7.67773445197101e-08, 3.32869643739751e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 0.0, - 1306071.54549889, 6161054.49513375, 2669589.12005785, - 1.42074157583338e-08, 7.67770755291348e-08, 3.32869351702894e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 0.0, - 2599961.45855466, 5760720.19357889, 2647597.12683792, - 1.42082069782729e-08, 7.67770859146354e-08, 3.32870271289161e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 0.0, - 3792857.8974237, 5136498.37238468, 2522713.38391818, - 1.42090151998075e-08, 7.67773738278905e-08, 3.32872360512713e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 0.0, - 4839229.61832879, 4313760.58255103, 2300338.34996557, - 1.42098091087349e-08, 7.6777926601063e-08, 3.32875532154675e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 0.0, - 5699723.05609476, 3325261.74212029, 1989578.43989231, - 1.42105582960492e-08, 7.67787214805961e-08, 3.32879658177311e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 0.0, - 6342536.88656784, 2209712.29939824, 1602811.60820791, - 1.42112344063431e-08, 7.67797267294272e-08, 3.32884575566685e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, - 6744447.61445741, 1010187.49633257, 1155147.8025102, - 1.42118121572608e-08, 7.67809030214275e-08, 3.32890093244216e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, - 6891477.8215365, - 227551.810286937, 663813.896586629, - 1.42122702028037e-08, 7.67822050500889e-08, 3.32895999693665e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, - 6779215.09491482, - 1456761.07554748, 147490.919569996, - 1.42125918243053e-08, 7.67835832776195e-08, 3.32902070979423e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, - 6412800.79978623, - 2631381.70900648, - 374371.749654303, - 1.42127654418044e-08, 7.67849857584842e-08, 3.32908078874545e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, - 5806614.11070755, - 3707634.06079869, - 882244.237582166, - 1.42127849449683e-08, 7.67863599808602e-08, 3.32913798863175e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, - 4983679.01699774, - 4645498.60891225, - 1357188.62711648, - 1.42126498466962e-08, 7.6787654678628e-08, 3.3291901782435e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, - 3974821.65711715, - 5410057.12629704, - 1781501.42815682, - 1.42123652644725e-08, 7.67888215740229e-08, 3.32923541237474e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, - 2817603.71414268, - 5972669.87274763, - 2139313.41892538, - 1.42119417350478e-08, 7.67898170161077e-08, 3.32927199771167e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, - 1555056.0648496, - 6311966.5947002, - 2417135.09102613, - 1.42113948678381e-08, 7.67906034825116e-08, 3.32929855126456e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, - 234236.587976406, - 6414628.84861909, - 2604335.85309436, - 1.42107448422921e-08, 7.67911509116001e-08, 3.32931405003975e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, 1095362.09295707, - 6275939.73323432, - 2693544.35598794, - 1.42100157551191e-08, 7.67914378300387e-08, 3.32931787056417e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, 2383524.57084058, - 5900075.61185268, - 2680956.18196418, - 1.42092348252888e-08, 7.6791452237584e-08, 3.32930981676729e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, 3581106.52232317, - 5300114.08050051, - 2566534.18986112, - 1.42084314685694e-08, 7.6791192208215e-08, 3.32929013465542e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, 4641862.77023787, - 4497734.59354263, - 2354086.60315067, - 1.42076362592396e-08, 7.67906661659354e-08, 3.32925951224595e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, 5524250.38662626, - 3522593.99117576, - 2051209.06202945, - 1.42068798043775e-08, 7.67898927963389e-08, 3.32921906342941e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, 6193136.2430559, - 2411369.56203787, - 1669079.86356028, - 1.42061915652471e-08, 7.67889005627843e-08, 3.32917029485019e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, 6621329.02394446, - 1206477.53744592, - 1222102.84027575, - 1.42055986698172e-08, 7.67877268097691e-08, 3.32911505557152e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, 6790850.71407875, 45505.4274329756, - 727399.838172203, - 1.42051247690004e-08, 7.67864164560291e-08, 3.32905547021242e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, 6693864.22555059, 1295667.60967006, - 204164.211397098, - 1.4204788995187e-08, 7.67850203052004e-08, 3.32899385735956e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 1.0, 6333183.86841522, 2494761.03873549, 327102.634966258, - 1.42046050834611e-08, 7.67835930304877e-08, 3.32893263626367e-08)]
    #[case(60310.0, 1.0, 100.0, 1.8, 4.56e-06, 149597870000.0, 0.0, 5722314.46705829, 3595279.7368028, 845439.049265208, - 1.42045807122508e-08, 7.67821909185259e-08, 3.32887422598158e-08)]
    fn test_cross_validation_srp_and_eclipse(
        #[case] mjd_tt: f64,
        #[case] area: f64,
        #[case] mass: f64,
        #[case] cr: f64,
        #[case] p0: f64,
        #[case] _au: f64,
        #[case] illum: f64,
        #[case] rx: f64,
        #[case] ry: f64,
        #[case] rz: f64,
        #[case] ax: f64,
        #[case] ay: f64,
        #[case] az: f64,
    ) {
        let epc = Epoch::from_mjd(mjd_tt, TimeSystem::TT);
        let r_object = Vector3::new(rx, ry, rz);

        let r_sun = sun_position(epc);

        let a_srp = acceleration_solar_radiation_pressure(r_object, r_sun, mass, cr, area, p0);
        let nu = eclipse_cylindrical(r_object, r_sun);

        let tol = 1.0e-9;
        assert_abs_diff_eq!(a_srp[0], ax, epsilon = tol);
        assert_abs_diff_eq!(a_srp[1], ay, epsilon = tol);
        assert_abs_diff_eq!(a_srp[2], az, epsilon = tol);
        assert_abs_diff_eq!(nu, illum, epsilon = tol);
    }

    #[rstest]
    #[case(60310.0, 4884992.30378986, 4553508.53744864, 1330313.60479734, 0.0)]
    #[case(60310.0, 4800651.83916825, 4630465.12411231, 1371132.4924634, 0.0)]
    #[case(60310.0, 4714804.29690742, 4705968.06024955, 1411520.93796545, 0.0)]
    #[case(60310.0, 4627476.64814936, 4779993.66398564, 1451466.26836184, 0.0)]
    #[case(60310.0, 4538696.34158385, 4852518.7320042, 1490955.95446072, 0.0)]
    #[case(60310.0, 4448491.29401321, 4923520.54744123, 1529977.61509333, 0.0)]
    #[case(60310.0, 4356889.88074156, 4992976.88759316, 1568519.02133019, 0.0)]
    #[case(60310.0, 4263920.92579301, 5060866.03143528, 1606568.10063842, 0.0)]
    #[case(60310.0, 4169613.69196325, 5127166.76694747, 1644112.94097834, 0.0)]
    #[case(60310.0, 4073997.8707091, 5191858.39824401, 1681141.79483748, 0.0)]
    #[case(60310.0, 3977103.57188077, 5254920.75250437, 1717643.08320035, 0.0)]
    #[case(60310.0, 3878961.31330151, 5316334.18670211, 1753605.39945219, 0.0)]
    #[case(60310.0, 3779602.01019953, 5376079.59412914, 1789017.51321495, 0.0)]
    #[case(60310.0, 3679056.96449705, 5434138.4107125, 1823868.37411406, 0.0)]
    #[case(60310.0, 3577357.85396155, 5490492.62112132, 1858147.11547413, 0.0)]
    #[case(60310.0, 3474536.72122421, 5545124.76466131, 1891843.05794228, 0.0)]
    #[case(60310.0, 3370625.96267067, 5598017.94095464, 1924945.7130374, 0.0)]
    #[case(60310.0, 3265658.31720927, 5649155.81540292, 1957444.78662401, 0.0)]
    #[case(60310.0, 3159666.85492207, 5698522.62443138, 1989330.18230924, 0.0)]
    #[case(60310.0, 3052684.96560382, 5746103.18051216, 2020592.00476154, 0.0)]
    #[case(60310.0, 2944746.34719436, 5791882.876965, 2051220.56294994, 0.0)]
    #[case(60310.0, 2835884.99410962, 5835847.6925338, 2081206.37330236, 0.0)]
    #[case(60310.0, 2726135.18547687, 5877984.1957373, 2110540.16278205, 0.0)]
    #[case(60310.0, 2615531.47327951, 5918279.54899273, 2139212.87188067, 0.0)]
    #[case(60310.0, 2504108.67041685, 5956721.51251097, 2167215.65752723, 0.0)]
    #[case(60310.0, 2391901.83868453, 5993298.44796229, 2194539.89591148, 0.0)]
    #[case(60310.0, 2278946.27668098, 6027999.32191159, 2221177.1852211, 0.0)]
    #[case(60310.0, 2165277.50764549, 6060813.7090223, 2247119.34829143, 0.0)]
    #[case(60310.0, 2050931.26723346, 6091731.79502823, 2272358.43516701, 0.0)]
    #[case(60310.0, 1935943.49123437, 6120744.37947287, 2296886.72557404, 0.0)]
    #[case(60310.0, 1820350.30323804, 6147842.87821555, 2320696.73130298, 0.0)]
    #[case(60310.0, 1704188.00225474, 6173019.32570429, 2343781.19850047, 0.0)]
    #[case(60310.0, 1587493.05029476, 6196266.37701509, 2366133.10987006, 0.0)]
    #[case(60310.0, 1470302.05991283, 6217577.30965757, 2387745.6867809, 0.0)]
    #[case(60310.0, 1352651.78172317, 6236946.02514722, 2408612.39128401, 0.0)]
    #[case(60310.0, 1234579.09189045, 6254367.05034419, 2428726.92803547, 0.0)]
    #[case(60310.0, 1116120.97960232, 6269835.53855927, 2448083.24612618, 0.0)]
    #[case(60310.0, 997314.534528933, 6283347.27042723, 2466675.54081769, 0.0)]
    #[case(60310.0, 878196.934274843, 6294898.6545484, 2484498.25518375, 0.0)]
    #[case(60310.0, 758805.431828819, 6304486.7278989, 2501546.08165738, 0.0)]
    #[case(60310.0, 639177.343016822, 6312109.15601069, 2517813.96348305, 0.0)]
    #[case(60310.0, 519350.033963561, 6317764.23292203, 2533297.09607398, 0.0)]
    #[case(60310.0, 399360.908567879, 6321450.88089982, 2547990.92827414, 0.0)]
    #[case(60310.0, 279247.395997234, 6323168.64993457, 2561891.1635252, 0.0)]
    #[case(60310.0, 159046.938206441, 6322917.71700979, 2574993.76093809, 0.0)]
    #[case(60310.0, 38796.9774858514, 6320698.88514676, 2587294.93626938, 0.0)]
    #[case(60310.0, - 81465.0559559881, 6316513.5822265, 2598791.16280245, 0.0)]
    #[case(60310.0, - 201701.756370153, 6310363.85959048, 2609479.17213364, 0.0)]
    #[case(60310.0, - 321875.754800686, 6302252.39042182, 2619355.95486346, 0.0)]
    #[case(60310.0, - 441949.731368581, 6292182.46790874, 2628418.76119317, 0.0)]
    #[case(60310.0, - 561886.427501581, 6280158.00319233, 2636665.1014269, 0.0)]
    #[case(60310.0, - 681648.658105034, 6266183.52310061, 2644092.74637971, 0.0)]
    #[case(60310.0, - 801199.323669165, 6250264.16767096, 2650699.72769191, 0.0)]
    #[case(60310.0, - 920501.422308105, 6232405.68746327, 2656484.33805005, 0.0)]
    #[case(60310.0, - 1039518.06172624, 6212614.44066596, 2661445.13131497, 0.0)]
    #[case(60310.0, - 1158212.47110736, 6190897.38999741, 2665580.92255757, 0.0)]
    #[case(60310.0, - 1276548.01292232, 6167262.09940514, 2668890.78800264, 0.0)]
    #[case(60310.0, - 1394488.19465092, 6141716.73056553, 2671374.06488146, 0.0)]
    #[case(60310.0, - 1511996.68041384, 6114270.03918634, 2673030.35119373, 0.0)]
    #[case(60310.0, - 1629037.30251045, 6084931.37111513, 2673859.50537957, 0.0)]
    #[case(60310.0, - 1745574.07285872, 6053710.65825613, 2673861.64590216, 0.0)]
    #[case(60310.0, - 1861571.19433303, 6020618.41429842, 2673037.15074187, 0.0)]
    #[case(60310.0, - 1976993.07199631, 5985665.73025848, 2671386.65680262, 0.0)]
    #[case(60310.0, - 2091804.32422266, 5948864.26983985, 2668911.05923129, 0.0)]
    #[case(60310.0, - 2205969.7937069, 5910226.26461318, 2665611.51065095, 0.0)]
    #[case(60310.0, - 2319454.55835748, 5869764.50901954, 2661489.42030895, 0.0)]
    #[case(60310.0, - 2432223.94206934, 5827492.35520033, 2656546.4531406, 0.0)]
    #[case(60310.0, - 2544243.52537343, 5783423.70765677, 2650784.52874953, 0.0)]
    #[case(60310.0, - 2655479.15595961, 5737573.0177424, 2644205.82030557, 0.0)]
    #[case(60310.0, - 2765896.95906985, 5689955.27799172, 2636812.75336128, 0.0)]
    #[case(60310.0, - 2875463.34775871, 5640586.01628837, 2628608.00458809, 0.0)]
    #[case(60310.0, - 2984145.03301812, 5589481.28987619, 2619594.50043307, 0.0)]
    #[case(60310.0, - 3091909.03376373, 5536657.67921655, 2609775.41569747, 0.0)]
    #[case(60310.0, - 3198722.68668002, 5482132.28169526, 2599154.17203816, 0.0)]
    #[case(60310.0, - 3304553.65592167, 5425922.7051828, 2587734.43639298, 0.0)]
    #[case(60310.0, - 3409369.94266855, 5368047.06145092, 2575520.11933129, 0.0)]
    #[case(60310.0, - 3513139.89453199, 5308523.95944951, 2562515.37333072, 0.0)]
    #[case(60310.0, - 3615832.21481003, 5247372.49844698, 2548724.59098152, 0.0)]
    #[case(60310.0, - 3717415.97158941, 5184612.26103781, 2534152.4031195, 0.0)]
    #[case(60310.0, - 3817860.60669221, 5120263.30602075, 2518803.67688891, 0.0)]
    #[case(60310.0, - 3917135.9444651, 5054346.16115129, 2502683.51373651, 0.0)]
    #[case(60310.0, - 4015212.20040932, 4986881.81577186, 2485797.24733796, 0.0)]
    #[case(60310.0, - 4112059.98964964, 4917891.71332349, 2468150.44145791, 0.0)]
    #[case(60310.0, - 4207650.33524044, 4847397.74374225, 2449748.88774503, 0.0)]
    #[case(60310.0, - 4301954.67630739, 4775422.23574439, 2430598.60346324, 0.0)]
    #[case(60310.0, - 4394944.87602328, 4701987.94900334, 2410705.82916049, 0.0)]
    #[case(60310.0, - 4486593.22941641, 4627118.06622251, 2390077.02627631, 0.0)]
    #[case(60310.0, - 4576872.47101033, 4550836.18510718, 2368718.87468955, 0.0)]
    #[case(60310.0, - 4665755.78229365, 4473166.3102392, 2346638.27020756, 0.0)]
    #[case(60310.0, - 4753216.79901884, 4394132.84485794, 2323842.32199813, 0.0)]
    #[case(60310.0, - 4839229.61832879, 4313760.58255103, 2300338.34996557, 0.0)]
    #[case(60310.0, - 4923768.80571041, 4232074.69885842, 2276133.88207221, 0.0)]
    #[case(60310.0, - 5006809.4017742, 4149100.7427933, 2251236.65160667, 0.0)]
    #[case(60310.0, - 5088326.92885902, 4064864.62828321, 2225654.59440028, 0.0)]
    #[case(60310.0, - 5168297.39746148, 3979392.62553495, 2199395.84599289, 0.0)]
    #[case(60310.0, - 5246697.31248915, 3892711.35232661, 2172468.73874957, 0.0)]
    #[case(60310.0, - 5323503.67933717, 3804847.76523019, 2144881.7989293, 0.0)]
    #[case(60310.0, - 5398694.00978772, 3715829.15076812, 2116643.7437072, 0.0)]
    #[case(60310.0, - 5472246.32773204, 3625683.1165071, 2087763.47815158, 0.0)]
    #[case(60310.0, - 5544139.17471455, 3534437.58209254, 2058250.09215698, 0.0)]
    #[case(60310.0, - 5614351.61529901, 3442120.77022694, 2028112.85733478, 0.0)]
    #[case(60310.0, - 5682863.24225639, 3348761.19759532, 1997361.22386248, 0.0)]
    #[case(60310.0, - 5749654.18157448, 3254387.66574114, 1966004.81729307, 0.0)]
    #[case(60310.0, - 5814705.09728913, 3159029.25189573, 1934053.4353258, 0.0)]
    #[case(60310.0, - 5877997.19613716, 3062715.29976449, 1901517.04453956, 0.0)]
    #[case(60310.0, - 5939512.23203117, 2965475.41027285, 1868405.77709022, 0.0)]
    #[case(60310.0, - 5999232.51035617, 2867339.43227523, 1834729.92737331, 0.0)]
    #[case(60310.0, - 6057140.89208857, 2768337.45322993, 1800499.94865307, 0.0)]
    #[case(60310.0, - 6113220.79773754, 2668499.78984294, 1765726.44965935, 0.0)]
    #[case(60310.0, - 6167456.21110922, 2567856.97868377, 1730420.1911535, 0.0)]
    #[case(60310.0, - 6219831.68289426, 2466439.76677597, 1694592.08246458, 0.0)]
    #[case(60310.0, - 6270332.33407886, 2364279.10216552, 1658253.17799699, 0.0)]
    #[case(60310.0, - 6318943.85918023, 2261406.12446969, 1621414.67371083, 0.0)]
    #[case(60310.0, - 6365652.52930657, 2157852.15540921, 1584087.90357623, 0.0)]
    #[case(60310.0, - 6410445.19504258, 2053648.68932664, 1546284.33600274, 0.0)]
    #[case(60310.0, - 6453309.28916086, 1948827.38369344, 1508015.57024498, 0.0)]
    #[case(60310.0, - 6494232.82916003, 1843420.04960854, 1469293.33278585, 0.0)]
    #[case(60310.0, - 6533204.4196303, 1737458.64229101, 1430129.47369825, 0.0)]
    #[case(60310.0, - 6570213.25444718, 1630975.25156932, 1390535.96298663, 0.0)]
    #[case(60310.0, - 6605249.11879428, 1524002.09236984, 1350524.88690933, 0.0)]
    #[case(60310.0, - 6638302.39101588, 1416571.49520698, 1310108.44428303, 0.0)]
    #[case(60310.0, - 6669364.04430031, 1308715.8966774, 1269298.94277015, 0.0)]
    #[case(60310.0, - 6698425.64819495, 1200467.8299608, 1228108.79515055, 0.0)]
    #[case(60310.0, - 6725479.36995389, 1091859.91532942, 1186550.51557835, 1.0)]
    #[case(60310.0, - 6750517.97571907, 982924.850668897, 1144636.71582516, 1.0)]
    #[case(60310.0, - 6773534.83153615, 873695.402012287, 1102380.10151049, 1.0)]
    #[case(60310.0, - 6794523.90420585, 764204.394089938, 1059793.46832069, 1.0)]
    #[case(60310.0, - 6813479.76197206, 654484.70089706, 1016889.69821708, 1.0)]
    #[case(60310.0, - 6830397.57504768, 544569.23628123, 973681.755634503, 1.0)]
    #[case(60310.0, - 6845273.11597931, 434490.944551965, 930182.683671247, 1.0)]
    #[case(60310.0, - 6858102.75985182, 324282.791114343, 886405.600271169, 1.0)]
    #[case(60310.0, - 6868883.48433419, 213977.753128695, 842363.694399103, 1.0)]
    #[case(60310.0, - 6877612.86956743, 103608.810198338, 798070.22221042, 1.0)]
    #[case(60310.0, - 6884289.09789602, - 6791.06491274318, 753538.503215667, 1.0)]
    #[case(60310.0, - 6888910.95344396, - 117188.91553036, 708781.916441193, 1.0)]
    #[case(60310.0, - 6891477.8215365, - 227551.810286937, 663813.896586629, 1.0)]
    #[case(60310.0, - 6891989.68796902, - 337846.852302819, 618647.930180118, 1.0)]
    #[case(60310.0, - 6890447.138124, - 448041.188330369, 573297.55173213, 1.0)]
    #[case(60310.0, - 6886851.35593751, - 558102.017859074, 527776.339888713, 1.0)]
    #[case(60310.0, - 6881204.12271629, - 667996.602180035, 482097.913585, 1.0)]
    #[case(60310.0, - 6873507.81580677, - 777692.273408086, 436275.928199817, 1.0)]
    #[case(60310.0, - 6863765.40711726, - 887156.443460088, 390324.071712115, 1.0)]
    #[case(60310.0, - 6851980.46149442, - 996356.612987639, 344256.060860108, 1.0)]
    #[case(60310.0, - 6838157.13495549, - 1105260.38026278, 298085.637303814, 1.0)]
    #[case(60310.0, - 6822300.17277729, - 1213835.45001513, 251826.563791789, 1.0)]
    #[case(60310.0, - 6804414.90744345, - 1322049.64221898, 205492.620332795, 1.0)]
    #[case(60310.0, - 6784507.2564509, - 1429870.90082886, 159097.600373133, 1.0)]
    #[case(60310.0, - 6762583.71997712, - 1537267.3024623, 112655.306980352, 1.0)]
    #[case(60310.0, - 6738651.37840917, - 1644207.0650281, 66179.5490340657, 1.0)]
    #[case(60310.0, - 6712717.88973588, - 1750658.55629913, 19684.1374245311, 1.0)]
    #[case(60310.0, - 6684791.48680442, - 1856590.30242801, - 26817.1187402823, 1.0)]
    #[case(60310.0, - 6654880.97444244, - 1961970.99640451, - 73310.4159184861, 1.0)]
    #[case(60310.0, - 6622995.72644706, - 2066769.50645336, - 119781.959908273, 1.0)]
    #[case(60310.0, - 6589145.68244193, - 2170954.88437124, - 166217.969611452, 1.0)]
    #[case(60310.0, - 6553341.34460361, - 2274496.37380163, - 212604.680785896, 1.0)]
    #[case(60310.0, - 6515593.77425837, - 2377363.41844644, - 258928.349786259, 1.0)]
    #[case(60310.0, - 6475914.5883508, - 2479525.67021308, - 305175.257292363, 1.0)]
    #[case(60310.0, - 6434315.95578531, - 2580952.99729596, - 351331.712024609, 1.0)]
    #[case(60310.0, - 6390810.59364174, - 2681615.49219117, - 397384.054445851, 1.0)]
    #[case(60310.0, - 6345411.76326628, - 2781483.47964316, - 443318.6604491, 1.0)]
    #[case(60310.0, - 6298133.2662389, - 2880527.5245225, - 489121.945030504, 1.0)]
    #[case(60310.0, - 6248989.44021843, - 2978718.43963345, - 534780.365947011, 1.0)]
    #[case(60310.0, - 6197995.15466649, - 3076027.29345032, - 580280.427358143, 1.0)]
    #[case(60310.0, - 6145165.80645134, - 3172425.41778155, - 625608.683451324, 1.0)]
    #[case(60310.0, - 6090517.31533301, - 3267884.41536048, - 670751.742050204, 1.0)]
    #[case(60310.0, - 6034066.11933053, - 3362376.16736178, - 715696.268205432, 1.0)]
    #[case(60310.0, - 5975829.16997272, - 3455872.8408425, - 760428.987767324, 1.0)]
    #[case(60310.0, - 5915823.92743341, - 3548346.89610663, - 804936.690939891, 1.0)]
    #[case(60310.0, - 5854068.35555238, - 3639771.0939924, - 849206.23581571, 1.0)]
    #[case(60310.0, - 5790580.916743, - 3730118.50308112, - 893224.551891081, 1.0)]
    #[case(60310.0, - 5725380.56678777, - 3819362.50682659, - 936978.643560963, 1.0)]
    #[case(60310.0, - 5658486.74952277, - 3907476.81060431, - 980455.593593165, 1.0)]
    #[case(60310.0, - 5589919.39141212, - 3994435.44867915, - 1023642.56658128, 1.0)]
    #[case(60310.0, - 5519698.89601366, - 4080212.79109095, - 1066526.81237583, 1.0)]
    #[case(60310.0, - 5447846.1383366, - 4164783.55045672, - 1109095.66949314, 1.0)]
    #[case(60310.0, - 5374382.4590926, - 4248122.78868869, - 1151336.56850141, 1.0)]
    #[case(60310.0, - 5299329.65884103, - 4330205.92362719, - 1193237.03538349, 1.0)]
    #[case(60310.0, - 5222709.99202957, - 4411008.73558749, - 1234784.69487582, 1.0)]
    #[case(60310.0, - 5144546.16093126, - 4490507.37381943, - 1275967.27378311, 1.0)]
    #[case(60310.0, - 5064861.30947885, - 4568678.36287923, - 1316772.60426818, 1.0)]
    #[case(60310.0, - 4983679.01699774, - 4645498.60891225, - 1357188.62711648, 1.0)]
    #[case(60310.0, - 4901023.29183828, - 4720945.40584587, - 1397203.39497484, 1.0)]
    #[case(60310.0, - 4816918.56490867, - 4794996.44149157, - 1436805.07556383, 1.0)]
    #[case(60310.0, - 4731389.68310934, - 4867629.80355516, - 1475981.95486337, 1.0)]
    #[case(60310.0, - 4644461.90266988, - 4938823.98555431, - 1514722.440271, 1.0)]
    #[case(60310.0, - 4556160.88238956, - 5008557.89264242, - 1553015.06373233, 1.0)]
    #[case(60310.0, - 4466512.67678242, - 5076810.84733777, - 1590848.48484317, 1.0)]
    #[case(60310.0, - 4375543.72912799, - 5143562.59515713, - 1628211.4939228, 1.0)]
    #[case(60310.0, - 4283280.86442865, - 5208793.3101527, - 1665093.01505796, 1.0)]
    #[case(60310.0, - 4189751.28227455, - 5272483.60035165, - 1701482.10911691, 1.0)]
    #[case(60310.0, - 4094982.54961725, - 5334614.513097, - 1737367.97673325, 1.0)]
    #[case(60310.0, - 3999002.59345312, - 5395167.54028909, - 1772739.96125872, 1.0)]
    #[case(60310.0, - 3901839.69341732, - 5454124.62352651, - 1807587.55168472, 1.0)]
    #[case(60310.0, - 3803522.47428963, - 5511468.15914559, - 1841900.38553189, 1.0)]
    #[case(60310.0, - 3704079.89841303, - 5567181.0031574, - 1875668.2517072, 1.0)]
    #[case(60310.0, - 3603541.25802614, - 5621246.47608126, - 1908881.09332819, 1.0)]
    #[case(60310.0, - 3501936.16751049, - 5673648.36767372, - 1941529.01051356, 1.0)]
    #[case(60310.0, - 3399294.55555381, - 5724370.94155212, - 1973602.26313989, 1.0)]
    #[case(60310.0, - 3295646.65723028, - 5773398.93971144, - 2005091.27356368, 1.0)]
    #[case(60310.0, - 3191023.00599889, - 5820717.58693375, - 2035986.6293083, 1.0)]
    #[case(60310.0, - 3085454.42562099, - 5866312.59508888, - 2066279.08571534, 1.0)]
    #[case(60310.0, - 2978972.02199807, - 5910170.16732549, - 2095959.56855966, 1.0)]
    #[case(60310.0, - 2871607.17493107, - 5952277.00215135, - 2125019.17662772, 1.0)]
    #[case(60310.0, - 2763391.52980204, - 5992620.29740188, - 2153449.18425858, 1.0)]
    #[case(60310.0, - 2654356.98917965, - 6031187.75409573, - 2181241.04384697, 1.0)]
    #[case(60310.0, - 2544535.70434941, - 6067967.58017642, - 2208386.38830793, 1.0)]
    #[case(60310.0, - 2433960.06676993, - 6102948.49413895, - 2234877.03350237, 1.0)]
    #[case(60310.0, - 2322662.69945643, - 6136119.72854021, - 2260704.98062302, 1.0)]
    #[case(60310.0, - 2210676.44829263, - 6167471.03339213, - 2285862.41854016, 1.0)]
    #[case(60310.0, - 2098034.37327231, - 6196992.6794365, - 2310341.72610653, 1.0)]
    #[case(60310.0, - 1984769.73967177, - 6224675.46130025, - 2334135.47442085, 1.0)]
    #[case(60310.0, - 1870916.00915452, - 6250510.70053013, - 2357236.42904935, 1.0)]
    #[case(60310.0, - 1756506.83080939, - 6274490.2485056, - 2379637.55220464, 1.0)]
    #[case(60310.0, - 1641576.03212357, - 6296606.48922893, - 2401332.0048814, 1.0)]
    #[case(60310.0, - 1526157.60989173, - 6316852.34199111, - 2422313.14894823, 1.0)]
    #[case(60310.0, - 1410285.72106275, - 6335221.26391278, - 2442574.54919497, 1.0)]
    #[case(60310.0, - 1293994.67352536, - 6351707.25235866, - 2462109.975335, 1.0)]
    #[case(60310.0, - 1177318.91683423, - 6366304.84722456, - 2480913.40396182, 1.0)]
    #[case(60310.0, - 1060293.03287791, - 6379009.1330958, - 2498979.02045921, 1.0)]
    #[case(60310.0, - 942951.726490069, - 6389815.74127577, - 2516301.22086447, 1.0)]
    #[case(60310.0, - 825329.816005711, - 6398720.85168352, - 2532874.61368395, 1.0)]
    #[case(60310.0, - 707462.223763752, - 6405721.19461929, - 2548694.02166035, 1.0)]
    #[case(60310.0, - 589383.966557698, - 6410814.05239672, - 2563754.48349106, 1.0)]
    #[case(60310.0, - 471130.146035953, - 6413997.26084061, - 2578051.25549689, 1.0)]
    #[case(60310.0, - 352735.93905346, - 6415269.2106491, - 2591579.81324058, 1.0)]
    #[case(60310.0, - 234236.587976406, - 6414628.84861909, - 2604335.85309436, 1.0)]
    #[case(60310.0, - 115667.390941652, - 6412075.67873373, - 2616315.29375598, 1.0)]
    #[case(60310.0, 2936.30792726699, - 6407609.76311089, - 2627514.27771249, 1.0)]
    #[case(60310.0, 121539.128345764, - 6401231.72281138, - 2637929.17265103, 1.0)]
    #[case(60310.0, 240105.663733653, - 6392942.73850591, - 2647556.5728162, 1.0)]
    #[case(60310.0, 358600.491118462, - 6382744.5509995, - 2656393.30031303, 1.0)]
    #[case(60310.0, 476988.181074679, - 6370639.46161239, - 2664436.40635511, 1.0)]
    #[case(60310.0, 595233.307697021, - 6356630.33241623, - 2671683.17245714, 1.0)]
    #[case(60310.0, 713300.45860561, - 6340720.58632447, - 2678131.11157114, 1.0)]
    #[case(60310.0, 831154.244981074, - 6322914.2070359, - 2683777.96916581, 1.0)]
    #[case(60310.0, 948759.311627373, - 6303215.73883028, - 2688621.72424827, 1.0)]
    #[case(60310.0, 1066080.34706031, - 6281630.28621496, - 2692660.59032746, 1.0)]
    #[case(60310.0, 1183082.0936194, - 6258163.51342141, - 2695893.0163187, 1.0)]
    #[case(60310.0, 1299729.35760097, - 6232821.64375085, - 2698317.68738856, 1.0)]
    #[case(60310.0, 1415987.01941018, - 6205611.45876768, - 2699933.52573955, 1.0)]
    #[case(60310.0, 1531820.04372942, - 6176540.29734, - 2700739.69133379, 1.0)]
    #[case(60310.0, 1647193.48970116, - 6145616.05452605, - 2700735.58255517, 1.0)]
    #[case(60310.0, 1762072.52112221, - 6112847.18030586, - 2699920.83680928, 1.0)]
    #[case(60310.0, 1876422.41664753, - 6078242.67815699, - 2698295.33106045, 1.0)]
    #[case(60310.0, 1990208.58000058, - 6041812.10347359, - 2695859.18230529, 1.0)]
    #[case(60310.0, 2103396.55018789, - 6003565.56182795, - 2692612.7479821, 1.0)]
    #[case(60310.0, 2215952.01171518, - 5963513.70707367, - 2688556.62631548, 1.0)]
    #[case(60310.0, 2327840.80480229, - 5921667.73928969, - 2683691.65659565, 1.0)]
    #[case(60310.0, 2439028.93559433, - 5878039.40256436, - 2678018.91939167, 1.0)]
    #[case(60310.0, 2549482.58636614, - 5832640.98261903, - 2671539.73669817, 1.0)]
    #[case(60310.0, 2659168.12571736, - 5785485.30427016, - 2664255.67201489, 1.0)]
    #[case(60310.0, 2768052.11875522, - 5736585.72872965, - 2656168.53035843, 1.0)]
    #[case(60310.0, 2876101.33726203, - 5685956.15074255, - 2647280.3582057, 1.0)]
    #[case(60310.0, 2983282.76984467, - 5633610.9955617, - 2637593.44336859, 1.0)]
    #[case(60310.0, 3089563.63206281, - 5579565.21575876, - 2627110.31479912, 1.0)]
    #[case(60310.0, 3194911.3765329, - 5523834.28787116, - 2615833.74232471, 1.0)]
    #[case(60310.0, 3299293.70300504, - 5466434.20888458, - 2603766.73631301, 1.0)]
    #[case(60310.0, 3402678.56840919, - 5407381.49255054, - 2590912.5472658, 1.0)]
    #[case(60310.0, 3505034.19686793, - 5346693.16553886, - 2577274.66534145, 1.0)]
    #[case(60310.0, 3606329.08967233, - 5284386.76342465, - 2562856.81980553, 1.0)]
    #[case(60310.0, 3706532.03521769, - 5220480.32650961, - 2547662.97840906, 1.0)]
    #[case(60310.0, 3805612.11889587, - 5154992.39547765, - 2531697.34669402, 1.0)]
    #[case(60310.0, 3903538.73294095, - 5087942.00688447, - 2514964.36722575, 1.0)]
    #[case(60310.0, 4000281.5862246, - 5019348.68848137, - 2497468.71875177, 1.0)]
    #[case(60310.0, 4095810.71399796, - 4949232.45437303, - 2479215.31528666, 1.0)]
    #[case(60310.0, 4190096.48757652, - 4877613.80000955, - 2460209.3051228, 1.0)]
    #[case(60310.0, 4283109.62396429, - 4804513.69701289, - 2440456.06976643, 1.0)]
    #[case(60310.0, 4374821.19541412, - 4729953.58783782, - 2419961.2227989, 1.0)]
    #[case(60310.0, 4465202.6389202, - 4653955.38026781, - 2398730.60866285, 1.0)]
    #[case(60310.0, 4554225.76563943, - 4576541.44174611, - 2376770.30137285, 1.0)]
    #[case(60310.0, 4641862.77023787, - 4497734.59354263, - 2354086.60315067, 1.0)]
    #[case(60310.0, 4728086.24015868, - 4417558.10475688, - 2330686.04298453, 1.0)]
    #[case(60310.0, 4812869.16480787, - 4336035.68615782, - 2306575.3751126, 1.0)]
    #[case(60310.0, 4896184.94465412, - 4253191.48386111, - 2281761.57743029, 1.0)]
    #[case(60310.0, 4978007.40023896, - 4169050.07284456, - 2256251.84982142, 1.0)]
    #[case(60310.0, 5058310.78109358, - 4083636.45030262, - 2230053.61241307, 1.0)]
    #[case(60310.0, 5137069.77455842, - 3996976.02884082, - 2203174.5037542, 1.0)]
    #[case(60310.0, 5214259.51450193, - 3909094.62951107, - 2175622.37891787, 1.0)]
    #[case(60310.0, 5289855.5899344, - 3820018.47468904, - 2147405.30752726, 1.0)]
    #[case(60310.0, 5363834.05351347, - 3729774.18079457, - 2118531.57170535, 1.0)]
    #[case(60310.0, 5436171.42993702, - 3638388.75085658, - 2089009.66394856, 1.0)]
    #[case(60310.0, 5506844.72422006, - 3545889.56692366, - 2058848.28492427, 1.0)]
    #[case(60310.0, 5575831.42985142, - 3452304.38232174, - 2028056.34119252, 1.0)]
    #[case(60310.0, 5643109.53682671, - 3357661.31376053, - 1996642.94285209, 1.0)]
    #[case(60310.0, 5708657.53955344, - 3261988.8332901, - 1964617.40111105, 1.0)]
    #[case(60310.0, 5772454.44462473, - 3165315.76010946, - 1931989.22578225, 1.0)]
    #[case(60310.0, 5834479.77845755, - 3067671.25222882, - 1898768.12270389, 1.0)]
    #[case(60310.0, 5894713.59479185, - 2969084.79798746, - 1864963.99108564, 1.0)]
    #[case(60310.0, 5953136.48204672, - 2869586.20742913, - 1830586.92078061, 1.0)]
    #[case(60310.0, 6009729.57052975, - 2769205.60353717, - 1795647.18948366, 1.0)]
    #[case(60310.0, 6064474.53949583, - 2667973.41333126, - 1760155.25985653, 1.0)]
    #[case(60310.0, 6117353.62405173, - 2565920.35882829, - 1724121.77658018, 1.0)]
    #[case(60310.0, 6168349.62190262, - 2463077.44786959, - 1687557.56333509, 1.0)]
    #[case(60310.0, 6217445.89993686, - 2359475.96481695, - 1650473.6197099, 1.0)]
    #[case(60310.0, 6264626.40064548, - 2255147.46111996, - 1612881.11803919, 1.0)]
    #[case(60310.0, 6309875.64837253, - 2150123.74575739, - 1574791.40017106, 1.0)]
    #[case(60310.0, 6353178.75539291, - 2044436.87555519, - 1536215.97416502, 1.0)]
    #[case(60310.0, 6394521.42781401, - 1938119.14538401, - 1497166.51092135, 1.0)]
    #[case(60310.0, 6433889.97129754, - 1831203.07823922, - 1457654.84074231, 1.0)]
    #[case(60310.0, 6471271.29659837, - 1723721.41520624, - 1417692.94982641, 1.0)]
    #[case(60310.0, 6506652.92491664, - 1615707.1053145, - 1377292.97669633, 1.0)]
    #[case(60310.0, 6540022.99305999, - 1507193.29528307, - 1336467.2085617, 1.0)]
    #[case(60310.0, 6571370.25841249, - 1398213.31916126, - 1295228.07761755, 1.0)]
    #[case(60310.0, 6600684.10370707, - 1288800.68786768, - 1253588.15727942, 1.0)]
    #[case(60310.0, 6627954.54159822, - 1178989.07863104, - 1211560.15835645, 1.0)]
    #[case(60310.0, 6653172.21903188, - 1068812.32433641, - 1169156.92516324, 1.0)]
    #[case(60310.0, 6676328.42140944, - 958304.402780476, - 1126391.43157187, 1.0)]
    #[case(60310.0, 6697415.07654287, - 847499.425839565, - 1083276.77700521, 1.0)]
    #[case(60310.0, 6716424.75839808, - 736431.628554201, - 1039826.18237267, 1.0)]
    #[case(60310.0, 6733350.69062368, - 625135.358134166, - 996052.985949897, 1.0)]
    #[case(60310.0, 6748186.74986244, - 513645.062887964, - 951970.639203466, 1.0)]
    #[case(60310.0, 6760927.4688427, - 401995.281080765, - 907592.702562192, 1.0)]
    #[case(60310.0, 6771568.03924728, - 290220.629725004, - 862932.841136309, 1.0)]
    #[case(60310.0, 6780104.3143573, - 178355.79330776, - 818004.820386037, 1.0)]
    #[case(60310.0, 6786532.81146867, - 66435.5124592978, - 772822.501741027, 1.0)]
    #[case(60310.0, 6790850.71407875, 45505.4274329756, - 727399.838172203, 1.0)]
    #[case(60310.0, 6793055.87384129, 157432.207660593, - 681750.869717597, 1.0)]
    #[case(60310.0, 6793146.81228732, 269309.987675864, - 635889.71896374, 1.0)]
    #[case(60310.0, 6791122.72231014, 381103.916614606, - 589830.586484354, 1.0)]
    #[case(60310.0, 6786983.46941255, 492779.144855482, - 543587.746237917, 1.0)]
    #[case(60310.0, 6780729.59271457, 604300.835611008, - 497175.540925928, 1.0)]
    #[case(60310.0, 6772362.30571994, 715634.176545665, - 450608.377313569, 1.0)]
    #[case(60310.0, 6761883.49684005, 826744.391416264, - 403900.721514573, 1.0)]
    #[case(60310.0, 6749295.7296737, 937596.751729688, - 357067.094242167, 1.0)]
    #[case(60310.0, 6734602.24304164, 1048156.58841325, - 310122.066027878, 1.0)]
    #[case(60310.0, 6717806.95077452, 1158389.30349264, - 263080.252410163, 1.0)]
    #[case(60310.0, 6698914.44125344, 1268260.38177248, - 215956.309094749, 1.0)]
    #[case(60310.0, 6677929.97670206, 1377735.40251462, - 168764.927088658, 1.0)]
    #[case(60310.0, 6654859.49222957, 1486780.05110894, - 121520.827809896, 1.0)]
    #[case(60310.0, 6629709.59462389, 1595360.13073179, - 74238.7581747628, 1.0)]
    #[case(60310.0, 6602487.56089461, 1703441.57398683, - 26933.4856649414, 1.0)]
    #[case(60310.0, 6573201.33656528, 1810990.45452318, 20380.2066237267, 1.0)]
    #[case(60310.0, 6541859.53371494, 1917972.99862581, 67687.5249485644, 1.0)]
    #[case(60310.0, 6508471.42876861, 2024355.59677293, 114973.669901488, 1.0)]
    #[case(60310.0, 6473046.960037, 2130104.81515533, 162223.841402323, 1.0)]
    #[case(60310.0, 6435596.72500555, 2235187.40715227, 209423.243701911, 1.0)]
    #[case(60310.0, 6396131.9773731, 2339570.3247591, 256557.090392876, 1.0)]
    #[case(60310.0, 6354664.62384083, 2443220.72996103, 303610.609425789, 1.0)]
    #[case(60310.0, 6311207.22065185, 2546106.0060482, 350569.048128634, 1.0)]
    #[case(60310.0, 6265772.96988252, 2648193.76886672, 397417.678227298, 1.0)]
    #[case(60310.0, 6218375.7154861, 2749451.87800054, 444141.800864936, 1.0)]
    #[case(60310.0, 6169029.93909007, 2849848.44787906, 490726.751617905, 1.0)]
    #[case(60310.0, 6117750.75554811, 2949351.85880538, 537157.905506151, 1.0)]
    #[case(60310.0, 6064553.90824823, 3047930.76789992, 583420.681995684, 1.0)]
    #[case(60310.0, 6009455.76417851, 3145554.1199546, 629500.549990982, 1.0)]
    #[case(60310.0, 5952473.30875196, 3242191.1581924, 675383.032815101, 1.0)]
    #[case(60310.0, 5893624.14039259, 3337811.43492722, 721053.713175118, 1.0)]
    #[case(60310.0, 5832926.4648842, 3432384.82211934, 766498.238110826, 0.0)]
    #[case(60310.0, 5770399.0894843, 3525881.52182125, 811702.323924327, 0.0)]
    #[case(60310.0, 5706061.4168052, 3618272.07650926, 856651.761088322, 0.0)]
    #[case(60310.0, 5639933.43846468, 3709527.37929585, 901332.419130853, 0.0)]
    #[case(60310.0, 5572035.72850859, 3799618.68401827, 945730.25149433, 0.0)]
    #[case(60310.0, 5502389.43660825, 3888517.61519825, 989831.300366499, 0.0)]
    #[case(60310.0, 5431016.28103513, 3976196.17786876, 1033621.70148133, 0.0)]
    #[case(60310.0, 5357938.54141586, 4062626.76726277, 1077087.68888745, 0.0)]
    #[case(60310.0, 5283179.0512706, 4147782.17835971, 1120215.59968207, 0.0)]
    #[case(60310.0, 5206761.19033779, 4231635.6152853, 1162991.87870824, 0.0)]
    #[case(60310.0, 5128708.87668871, 4314160.70056012, 1205403.08321316, 0.0)]
    #[case(60310.0, 5049046.55863509, 4395331.48419301, 1247435.88746564, 0.0)]
    #[case(60310.0, 4967799.20643349, 4475122.45261482, 1289077.08733042, 0.0)]
    fn test_eclipse_cross_validation(
        #[case] mjd_tt: f64,
        #[case] rx: f64,
        #[case] ry: f64,
        #[case] rz: f64,
        #[case] illum: f64,
    ) {
        let r_object = Vector3::new(rx, ry, rz);

        let epc = Epoch::from_mjd(mjd_tt, TimeSystem::TT);
        let r_sun = sun_position(epc);

        let nu_cyl = eclipse_cylindrical(r_object, r_sun);
        let nu_con = eclipse_conical(r_object, r_sun);

        assert_eq!(nu_cyl, illum);

        // Confirm that conical eclipse follows the same pattern as cylindrical
        // Specifically
        //  - if: 0.0 <= nu_con < 1.0, then nu_cyl = 0.0;
        //  - nu_con = 0.0, then nu_cyl = 0.0;
        // - nu_con = 1.0, then nu_cyl = 1.0;
        assert_eq!(nu_con.floor(), nu_cyl.floor());
    }
}
