from http import HTTPStatus
from typing import Any, Optional, Union

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.backup_restore_request import BackupRestoreRequest
from ...models.error_response import ErrorResponse
from ...models.http_validation_error import HTTPValidationError
from ...types import UNSET, Response, Unset


def _get_kwargs(
  graph_id: str,
  *,
  body: BackupRestoreRequest,
  authorization: Union[None, Unset, str] = UNSET,
  auth_token: Union[None, Unset, str] = UNSET,
) -> dict[str, Any]:
  headers: dict[str, Any] = {}
  if not isinstance(authorization, Unset):
    headers["authorization"] = authorization

  cookies = {}
  if auth_token is not UNSET:
    cookies["auth-token"] = auth_token

  _kwargs: dict[str, Any] = {
    "method": "post",
    "url": f"/v1/{graph_id}/backup/restore",
    "cookies": cookies,
  }

  _kwargs["json"] = body.to_dict()

  headers["Content-Type"] = "application/json"

  _kwargs["headers"] = headers
  return _kwargs


def _parse_response(
  *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Any, ErrorResponse, HTTPValidationError]]:
  if response.status_code == 202:
    response_202 = response.json()
    return response_202
  if response.status_code == 400:
    response_400 = ErrorResponse.from_dict(response.json())

    return response_400
  if response.status_code == 402:
    response_402 = ErrorResponse.from_dict(response.json())

    return response_402
  if response.status_code == 403:
    response_403 = ErrorResponse.from_dict(response.json())

    return response_403
  if response.status_code == 404:
    response_404 = ErrorResponse.from_dict(response.json())

    return response_404
  if response.status_code == 500:
    response_500 = ErrorResponse.from_dict(response.json())

    return response_500
  if response.status_code == 422:
    response_422 = HTTPValidationError.from_dict(response.json())

    return response_422
  if client.raise_on_unexpected_status:
    raise errors.UnexpectedStatus(response.status_code, response.content)
  else:
    return None


def _build_response(
  *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Any, ErrorResponse, HTTPValidationError]]:
  return Response(
    status_code=HTTPStatus(response.status_code),
    content=response.content,
    headers=response.headers,
    parsed=_parse_response(client=client, response=response),
  )


def sync_detailed(
  graph_id: str,
  *,
  client: AuthenticatedClient,
  body: BackupRestoreRequest,
  authorization: Union[None, Unset, str] = UNSET,
  auth_token: Union[None, Unset, str] = UNSET,
) -> Response[Union[Any, ErrorResponse, HTTPValidationError]]:
  """Restore Encrypted Backup

   Restore a graph database from an encrypted backup.

  Restores a complete Kuzu database from an encrypted backup:
  - **Format**: Only full_dump backups can be restored
  - **Encryption**: Only encrypted backups can be restored (security requirement)
  - **System Backup**: Creates automatic backup of existing database before restore
  - **Verification**: Optionally verifies database integrity after restore

  **Restore Features:**
  - **Atomic Operation**: Complete replacement of database
  - **Rollback Protection**: System backup created before restore
  - **Data Integrity**: Verification ensures successful restore
  - **Security**: Only encrypted backups to prevent data tampering

  **Progress Monitoring:**
  Use the returned operation_id to connect to the SSE stream:
  ```javascript
  const eventSource = new EventSource('/v1/operations/{operation_id}/stream');
  eventSource.addEventListener('operation_progress', (event) => {
    const data = JSON.parse(event.data);
    console.log('Restore progress:', data.message);
  });
  ```

  **SSE Connection Limits:**
  - Maximum 5 concurrent SSE connections per user
  - Rate limited to 10 new connections per minute
  - Automatic circuit breaker for Redis failures
  - Graceful degradation if event system unavailable

  **Important Notes:**
  - Only encrypted backups can be restored (security measure)
  - Existing database is backed up to S3 before restore
  - Restore is a destructive operation - existing data is replaced
  - System backups are stored separately for recovery

  **Credit Consumption:**
  - Base cost: 100.0 credits
  - Large databases (>10GB): 200.0 credits
  - Multiplied by graph tier

  Returns operation details for SSE monitoring.

  Args:
      graph_id (str): Graph database identifier
      authorization (Union[None, Unset, str]):
      auth_token (Union[None, Unset, str]):
      body (BackupRestoreRequest): Request model for restoring from a backup.

  Raises:
      errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
      httpx.TimeoutException: If the request takes longer than Client.timeout.

  Returns:
      Response[Union[Any, ErrorResponse, HTTPValidationError]]
  """

  kwargs = _get_kwargs(
    graph_id=graph_id,
    body=body,
    authorization=authorization,
    auth_token=auth_token,
  )

  response = client.get_httpx_client().request(
    **kwargs,
  )

  return _build_response(client=client, response=response)


def sync(
  graph_id: str,
  *,
  client: AuthenticatedClient,
  body: BackupRestoreRequest,
  authorization: Union[None, Unset, str] = UNSET,
  auth_token: Union[None, Unset, str] = UNSET,
) -> Optional[Union[Any, ErrorResponse, HTTPValidationError]]:
  """Restore Encrypted Backup

   Restore a graph database from an encrypted backup.

  Restores a complete Kuzu database from an encrypted backup:
  - **Format**: Only full_dump backups can be restored
  - **Encryption**: Only encrypted backups can be restored (security requirement)
  - **System Backup**: Creates automatic backup of existing database before restore
  - **Verification**: Optionally verifies database integrity after restore

  **Restore Features:**
  - **Atomic Operation**: Complete replacement of database
  - **Rollback Protection**: System backup created before restore
  - **Data Integrity**: Verification ensures successful restore
  - **Security**: Only encrypted backups to prevent data tampering

  **Progress Monitoring:**
  Use the returned operation_id to connect to the SSE stream:
  ```javascript
  const eventSource = new EventSource('/v1/operations/{operation_id}/stream');
  eventSource.addEventListener('operation_progress', (event) => {
    const data = JSON.parse(event.data);
    console.log('Restore progress:', data.message);
  });
  ```

  **SSE Connection Limits:**
  - Maximum 5 concurrent SSE connections per user
  - Rate limited to 10 new connections per minute
  - Automatic circuit breaker for Redis failures
  - Graceful degradation if event system unavailable

  **Important Notes:**
  - Only encrypted backups can be restored (security measure)
  - Existing database is backed up to S3 before restore
  - Restore is a destructive operation - existing data is replaced
  - System backups are stored separately for recovery

  **Credit Consumption:**
  - Base cost: 100.0 credits
  - Large databases (>10GB): 200.0 credits
  - Multiplied by graph tier

  Returns operation details for SSE monitoring.

  Args:
      graph_id (str): Graph database identifier
      authorization (Union[None, Unset, str]):
      auth_token (Union[None, Unset, str]):
      body (BackupRestoreRequest): Request model for restoring from a backup.

  Raises:
      errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
      httpx.TimeoutException: If the request takes longer than Client.timeout.

  Returns:
      Union[Any, ErrorResponse, HTTPValidationError]
  """

  return sync_detailed(
    graph_id=graph_id,
    client=client,
    body=body,
    authorization=authorization,
    auth_token=auth_token,
  ).parsed


async def asyncio_detailed(
  graph_id: str,
  *,
  client: AuthenticatedClient,
  body: BackupRestoreRequest,
  authorization: Union[None, Unset, str] = UNSET,
  auth_token: Union[None, Unset, str] = UNSET,
) -> Response[Union[Any, ErrorResponse, HTTPValidationError]]:
  """Restore Encrypted Backup

   Restore a graph database from an encrypted backup.

  Restores a complete Kuzu database from an encrypted backup:
  - **Format**: Only full_dump backups can be restored
  - **Encryption**: Only encrypted backups can be restored (security requirement)
  - **System Backup**: Creates automatic backup of existing database before restore
  - **Verification**: Optionally verifies database integrity after restore

  **Restore Features:**
  - **Atomic Operation**: Complete replacement of database
  - **Rollback Protection**: System backup created before restore
  - **Data Integrity**: Verification ensures successful restore
  - **Security**: Only encrypted backups to prevent data tampering

  **Progress Monitoring:**
  Use the returned operation_id to connect to the SSE stream:
  ```javascript
  const eventSource = new EventSource('/v1/operations/{operation_id}/stream');
  eventSource.addEventListener('operation_progress', (event) => {
    const data = JSON.parse(event.data);
    console.log('Restore progress:', data.message);
  });
  ```

  **SSE Connection Limits:**
  - Maximum 5 concurrent SSE connections per user
  - Rate limited to 10 new connections per minute
  - Automatic circuit breaker for Redis failures
  - Graceful degradation if event system unavailable

  **Important Notes:**
  - Only encrypted backups can be restored (security measure)
  - Existing database is backed up to S3 before restore
  - Restore is a destructive operation - existing data is replaced
  - System backups are stored separately for recovery

  **Credit Consumption:**
  - Base cost: 100.0 credits
  - Large databases (>10GB): 200.0 credits
  - Multiplied by graph tier

  Returns operation details for SSE monitoring.

  Args:
      graph_id (str): Graph database identifier
      authorization (Union[None, Unset, str]):
      auth_token (Union[None, Unset, str]):
      body (BackupRestoreRequest): Request model for restoring from a backup.

  Raises:
      errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
      httpx.TimeoutException: If the request takes longer than Client.timeout.

  Returns:
      Response[Union[Any, ErrorResponse, HTTPValidationError]]
  """

  kwargs = _get_kwargs(
    graph_id=graph_id,
    body=body,
    authorization=authorization,
    auth_token=auth_token,
  )

  response = await client.get_async_httpx_client().request(**kwargs)

  return _build_response(client=client, response=response)


async def asyncio(
  graph_id: str,
  *,
  client: AuthenticatedClient,
  body: BackupRestoreRequest,
  authorization: Union[None, Unset, str] = UNSET,
  auth_token: Union[None, Unset, str] = UNSET,
) -> Optional[Union[Any, ErrorResponse, HTTPValidationError]]:
  """Restore Encrypted Backup

   Restore a graph database from an encrypted backup.

  Restores a complete Kuzu database from an encrypted backup:
  - **Format**: Only full_dump backups can be restored
  - **Encryption**: Only encrypted backups can be restored (security requirement)
  - **System Backup**: Creates automatic backup of existing database before restore
  - **Verification**: Optionally verifies database integrity after restore

  **Restore Features:**
  - **Atomic Operation**: Complete replacement of database
  - **Rollback Protection**: System backup created before restore
  - **Data Integrity**: Verification ensures successful restore
  - **Security**: Only encrypted backups to prevent data tampering

  **Progress Monitoring:**
  Use the returned operation_id to connect to the SSE stream:
  ```javascript
  const eventSource = new EventSource('/v1/operations/{operation_id}/stream');
  eventSource.addEventListener('operation_progress', (event) => {
    const data = JSON.parse(event.data);
    console.log('Restore progress:', data.message);
  });
  ```

  **SSE Connection Limits:**
  - Maximum 5 concurrent SSE connections per user
  - Rate limited to 10 new connections per minute
  - Automatic circuit breaker for Redis failures
  - Graceful degradation if event system unavailable

  **Important Notes:**
  - Only encrypted backups can be restored (security measure)
  - Existing database is backed up to S3 before restore
  - Restore is a destructive operation - existing data is replaced
  - System backups are stored separately for recovery

  **Credit Consumption:**
  - Base cost: 100.0 credits
  - Large databases (>10GB): 200.0 credits
  - Multiplied by graph tier

  Returns operation details for SSE monitoring.

  Args:
      graph_id (str): Graph database identifier
      authorization (Union[None, Unset, str]):
      auth_token (Union[None, Unset, str]):
      body (BackupRestoreRequest): Request model for restoring from a backup.

  Raises:
      errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
      httpx.TimeoutException: If the request takes longer than Client.timeout.

  Returns:
      Union[Any, ErrorResponse, HTTPValidationError]
  """

  return (
    await asyncio_detailed(
      graph_id=graph_id,
      client=client,
      body=body,
      authorization=authorization,
      auth_token=auth_token,
    )
  ).parsed
