from http import HTTPStatus
from typing import Any, Optional, Union

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.error_response import ErrorResponse
from ...models.schema_validation_request import SchemaValidationRequest
from ...models.schema_validation_response import SchemaValidationResponse
from ...types import UNSET, Response, Unset


def _get_kwargs(
  graph_id: str,
  *,
  body: SchemaValidationRequest,
  authorization: Union[None, Unset, str] = UNSET,
  auth_token: Union[None, Unset, str] = UNSET,
) -> dict[str, Any]:
  headers: dict[str, Any] = {}
  if not isinstance(authorization, Unset):
    headers["authorization"] = authorization

  cookies = {}
  if auth_token is not UNSET:
    cookies["auth-token"] = auth_token

  _kwargs: dict[str, Any] = {
    "method": "post",
    "url": f"/v1/{graph_id}/schema/validate",
    "cookies": cookies,
  }

  _kwargs["json"] = body.to_dict()

  headers["Content-Type"] = "application/json"

  _kwargs["headers"] = headers
  return _kwargs


def _parse_response(
  *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[ErrorResponse, SchemaValidationResponse]]:
  if response.status_code == 200:
    response_200 = SchemaValidationResponse.from_dict(response.json())

    return response_200
  if response.status_code == 400:
    response_400 = ErrorResponse.from_dict(response.json())

    return response_400
  if response.status_code == 402:
    response_402 = ErrorResponse.from_dict(response.json())

    return response_402
  if response.status_code == 403:
    response_403 = ErrorResponse.from_dict(response.json())

    return response_403
  if response.status_code == 422:
    response_422 = ErrorResponse.from_dict(response.json())

    return response_422
  if response.status_code == 500:
    response_500 = ErrorResponse.from_dict(response.json())

    return response_500
  if client.raise_on_unexpected_status:
    raise errors.UnexpectedStatus(response.status_code, response.content)
  else:
    return None


def _build_response(
  *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[ErrorResponse, SchemaValidationResponse]]:
  return Response(
    status_code=HTTPStatus(response.status_code),
    content=response.content,
    headers=response.headers,
    parsed=_parse_response(client=client, response=response),
  )


def sync_detailed(
  graph_id: str,
  *,
  client: AuthenticatedClient,
  body: SchemaValidationRequest,
  authorization: Union[None, Unset, str] = UNSET,
  auth_token: Union[None, Unset, str] = UNSET,
) -> Response[Union[ErrorResponse, SchemaValidationResponse]]:
  """Validate Schema

   Validate a custom schema definition before deployment.

  This endpoint performs comprehensive validation including:
  - **Structure Validation**: Ensures proper JSON/YAML format
  - **Type Checking**: Validates data types (STRING, INT, DOUBLE, etc.)
  - **Constraint Verification**: Checks primary keys and unique constraints
  - **Relationship Integrity**: Validates node references in relationships
  - **Naming Conventions**: Ensures valid identifiers
  - **Compatibility**: Checks against existing extensions if specified

  Supported formats:
  - JSON schema definitions
  - YAML schema definitions
  - Direct dictionary format

  Validation helps prevent:
  - Schema deployment failures
  - Data integrity issues
  - Performance problems
  - Naming conflicts

  Credit consumption:
  - Base cost: 5.0 credits
  - Multiplied by graph tier

  Args:
      graph_id (str): Graph database identifier
      authorization (Union[None, Unset, str]):
      auth_token (Union[None, Unset, str]):
      body (SchemaValidationRequest): Request model for schema validation.

  Raises:
      errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
      httpx.TimeoutException: If the request takes longer than Client.timeout.

  Returns:
      Response[Union[ErrorResponse, SchemaValidationResponse]]
  """

  kwargs = _get_kwargs(
    graph_id=graph_id,
    body=body,
    authorization=authorization,
    auth_token=auth_token,
  )

  response = client.get_httpx_client().request(
    **kwargs,
  )

  return _build_response(client=client, response=response)


def sync(
  graph_id: str,
  *,
  client: AuthenticatedClient,
  body: SchemaValidationRequest,
  authorization: Union[None, Unset, str] = UNSET,
  auth_token: Union[None, Unset, str] = UNSET,
) -> Optional[Union[ErrorResponse, SchemaValidationResponse]]:
  """Validate Schema

   Validate a custom schema definition before deployment.

  This endpoint performs comprehensive validation including:
  - **Structure Validation**: Ensures proper JSON/YAML format
  - **Type Checking**: Validates data types (STRING, INT, DOUBLE, etc.)
  - **Constraint Verification**: Checks primary keys and unique constraints
  - **Relationship Integrity**: Validates node references in relationships
  - **Naming Conventions**: Ensures valid identifiers
  - **Compatibility**: Checks against existing extensions if specified

  Supported formats:
  - JSON schema definitions
  - YAML schema definitions
  - Direct dictionary format

  Validation helps prevent:
  - Schema deployment failures
  - Data integrity issues
  - Performance problems
  - Naming conflicts

  Credit consumption:
  - Base cost: 5.0 credits
  - Multiplied by graph tier

  Args:
      graph_id (str): Graph database identifier
      authorization (Union[None, Unset, str]):
      auth_token (Union[None, Unset, str]):
      body (SchemaValidationRequest): Request model for schema validation.

  Raises:
      errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
      httpx.TimeoutException: If the request takes longer than Client.timeout.

  Returns:
      Union[ErrorResponse, SchemaValidationResponse]
  """

  return sync_detailed(
    graph_id=graph_id,
    client=client,
    body=body,
    authorization=authorization,
    auth_token=auth_token,
  ).parsed


async def asyncio_detailed(
  graph_id: str,
  *,
  client: AuthenticatedClient,
  body: SchemaValidationRequest,
  authorization: Union[None, Unset, str] = UNSET,
  auth_token: Union[None, Unset, str] = UNSET,
) -> Response[Union[ErrorResponse, SchemaValidationResponse]]:
  """Validate Schema

   Validate a custom schema definition before deployment.

  This endpoint performs comprehensive validation including:
  - **Structure Validation**: Ensures proper JSON/YAML format
  - **Type Checking**: Validates data types (STRING, INT, DOUBLE, etc.)
  - **Constraint Verification**: Checks primary keys and unique constraints
  - **Relationship Integrity**: Validates node references in relationships
  - **Naming Conventions**: Ensures valid identifiers
  - **Compatibility**: Checks against existing extensions if specified

  Supported formats:
  - JSON schema definitions
  - YAML schema definitions
  - Direct dictionary format

  Validation helps prevent:
  - Schema deployment failures
  - Data integrity issues
  - Performance problems
  - Naming conflicts

  Credit consumption:
  - Base cost: 5.0 credits
  - Multiplied by graph tier

  Args:
      graph_id (str): Graph database identifier
      authorization (Union[None, Unset, str]):
      auth_token (Union[None, Unset, str]):
      body (SchemaValidationRequest): Request model for schema validation.

  Raises:
      errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
      httpx.TimeoutException: If the request takes longer than Client.timeout.

  Returns:
      Response[Union[ErrorResponse, SchemaValidationResponse]]
  """

  kwargs = _get_kwargs(
    graph_id=graph_id,
    body=body,
    authorization=authorization,
    auth_token=auth_token,
  )

  response = await client.get_async_httpx_client().request(**kwargs)

  return _build_response(client=client, response=response)


async def asyncio(
  graph_id: str,
  *,
  client: AuthenticatedClient,
  body: SchemaValidationRequest,
  authorization: Union[None, Unset, str] = UNSET,
  auth_token: Union[None, Unset, str] = UNSET,
) -> Optional[Union[ErrorResponse, SchemaValidationResponse]]:
  """Validate Schema

   Validate a custom schema definition before deployment.

  This endpoint performs comprehensive validation including:
  - **Structure Validation**: Ensures proper JSON/YAML format
  - **Type Checking**: Validates data types (STRING, INT, DOUBLE, etc.)
  - **Constraint Verification**: Checks primary keys and unique constraints
  - **Relationship Integrity**: Validates node references in relationships
  - **Naming Conventions**: Ensures valid identifiers
  - **Compatibility**: Checks against existing extensions if specified

  Supported formats:
  - JSON schema definitions
  - YAML schema definitions
  - Direct dictionary format

  Validation helps prevent:
  - Schema deployment failures
  - Data integrity issues
  - Performance problems
  - Naming conflicts

  Credit consumption:
  - Base cost: 5.0 credits
  - Multiplied by graph tier

  Args:
      graph_id (str): Graph database identifier
      authorization (Union[None, Unset, str]):
      auth_token (Union[None, Unset, str]):
      body (SchemaValidationRequest): Request model for schema validation.

  Raises:
      errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
      httpx.TimeoutException: If the request takes longer than Client.timeout.

  Returns:
      Union[ErrorResponse, SchemaValidationResponse]
  """

  return (
    await asyncio_detailed(
      graph_id=graph_id,
      client=client,
      body=body,
      authorization=authorization,
      auth_token=auth_token,
    )
  ).parsed
