import datetime
from collections.abc import Mapping
from typing import Any, TypeVar

from attrs import define as _attrs_define
from attrs import field as _attrs_field
from dateutil.parser import isoparse

T = TypeVar("T", bound="OAuthInitResponse")


@_attrs_define
class OAuthInitResponse:
  """Response with OAuth authorization URL.

  Attributes:
      auth_url (str): URL to redirect user for authorization
      state (str): OAuth state for security
      expires_at (datetime.datetime): When this OAuth request expires
  """

  auth_url: str
  state: str
  expires_at: datetime.datetime
  additional_properties: dict[str, Any] = _attrs_field(init=False, factory=dict)

  def to_dict(self) -> dict[str, Any]:
    auth_url = self.auth_url

    state = self.state

    expires_at = self.expires_at.isoformat()

    field_dict: dict[str, Any] = {}
    field_dict.update(self.additional_properties)
    field_dict.update(
      {
        "auth_url": auth_url,
        "state": state,
        "expires_at": expires_at,
      }
    )

    return field_dict

  @classmethod
  def from_dict(cls: type[T], src_dict: Mapping[str, Any]) -> T:
    d = dict(src_dict)
    auth_url = d.pop("auth_url")

    state = d.pop("state")

    expires_at = isoparse(d.pop("expires_at"))

    o_auth_init_response = cls(
      auth_url=auth_url,
      state=state,
      expires_at=expires_at,
    )

    o_auth_init_response.additional_properties = d
    return o_auth_init_response

  @property
  def additional_keys(self) -> list[str]:
    return list(self.additional_properties.keys())

  def __getitem__(self, key: str) -> Any:
    return self.additional_properties[key]

  def __setitem__(self, key: str, value: Any) -> None:
    self.additional_properties[key] = value

  def __delitem__(self, key: str) -> None:
    del self.additional_properties[key]

  def __contains__(self, key: str) -> bool:
    return key in self.additional_properties
