use std::path::PathBuf;
use crate::dna::compiler::{Compiler, OptimizationLevel};
use anyhow::Context;



pub fn compile_command(
    input: PathBuf,
    output: Option<PathBuf>,
    compress: bool,
    optimize: u8,
    cache: bool,
    verbose: bool,
    _quiet: bool,
) -> Result<(), Box<dyn std::error::Error>> {
    let output_path = output
        .unwrap_or_else(|| {
            let mut path = input.clone();
            path.set_extension("hlxb");
            path
        });
    if verbose {
        println!("📦 Compiling: {}", input.display());
        println!("  Optimization: Level {}", optimize);
        println!("  Compression: {}", if compress { "Enabled" } else { "Disabled" });
        println!("  Cache: {}", if cache { "Enabled" } else { "Disabled" });
    }
    let compiler = Compiler::new(OptimizationLevel::from(optimize));
    let binary = compiler.compile_file(&input)?;
    let serializer = crate::dna::mds::serializer::BinarySerializer::new(compress);
    serializer.write_to_file(&binary, &output_path)?;
    println!("✅ Compiled successfully: {}", output_path.display());
    println!("  Size: {} bytes", binary.size());
    if verbose {
        let stats = binary.symbol_table.stats();
        println!(
            "  Strings: {} (unique: {})", stats.total_strings, stats.unique_strings
        );
        println!("  Agents: {}", stats.agents);
        println!("  Workflows: {}", stats.workflows);
    }
    Ok(())
}

fn compile_with_progress(
    input: PathBuf,
    output: Option<PathBuf>,
    compress: bool,
    optimize: u8,
    cache: bool,
    verbose: bool,
) -> Result<(), Box<dyn std::error::Error>> {
    use indicatif::{ProgressBar, ProgressStyle};
    use crate::dna::mds::semantic::SemanticAnalyzer;
    use crate::dna::mds::lint::lint_files;
    use crate::dna::mds::fmt::format_files;
    use crate::dna::mds::optimizer::OptimizationLevel;
    let pb = ProgressBar::new(100);
    pb.set_style(
        ProgressStyle::default_bar()
            .template(
                "{spinner:.green} [{elapsed_precise}] [{bar:40.cyan/blue}] {pos}/{len} {msg}",
            )
            .unwrap()
            .progress_chars("#>-"),
    );
    pb.set_message("🔍 Running semantic analysis...");
    pb.inc(10);
    let analyzer = SemanticAnalyzer::new();
    if verbose {
        println!("  📊 Semantic analysis: Analyzing code structure...");
    }
    pb.set_message("🔧 Running lint checks...");
    pb.inc(10);
    if verbose {
        println!("  🔧 Linting: Checking code quality...");
    }
    lint_files(vec![input.clone()], verbose)?;
    pb.set_message("✨ Formatting code...");
    pb.inc(10);
    if verbose {
        println!("  ✨ Formatting: Ensuring code consistency...");
    }
    format_files(vec![input.clone()], false, verbose)?;
    pb.set_message("⚙️ Initializing compiler...");
    pb.inc(10);
    let mut compiler = Compiler::new(OptimizationLevel::Two);
    pb.set_message("📖 Loading file...");
    pb.inc(10);
    let content = std::fs::read_to_string(&input)
        .context(format!("Failed to read file: {}", input.display()))?;
    pb.set_message("🔍 Parsing configuration...");
    pb.inc(15);
    let ast = crate::parse(&content).map_err(|e| anyhow::anyhow!("Failed to parse Helix configuration: {}", e))?;
    pb.set_message("⚡ Compiling with optimizations...");
    pb.inc(20);
    let result = compiler.compile_file(&input).context("Failed to compile file")?;
    pb.set_message("🎯 Finalizing compilation...");
    pb.inc(15);
    pb.finish_with_message("✅ Enhanced compilation completed successfully!");
    if verbose {
        println!("🚀 Enhanced compilation completed using all Helix modules!");
        println!("  📊 Semantic analysis: ✅");
        println!("  🔧 Linting: ✅");
        println!("  ✨ Formatting: ✅");
        println!("  ⚡ Optimization: Level {}", optimize);
        println!("  📦 Result: {:?}", result);
    }
    Ok(())
}