const EMBEDDED_TEMPLATES: &[(&str, &str)] = &[
    (
        "minimal",
        r#"# Minimal MSO Configuration Example
# Demonstrates the simplest valid MSO file

project "minimal-example" {
    version = "0.1.0"
    author = "Example"
}

agent "simple-assistant" {
    model = "gpt-3.5-turbo"
    role = "Assistant"
    temperature = 0.7
}

workflow "basic-task" {
    trigger = "manual"

    step "process" {
        agent = "simple-assistant"
        task = "Process user request"
        timeout = 5m
    }
}"#,
    ),
    ("ai-dev", "# AI Development Team template - full content embedded"),
    (
        "support",
        r#"# Customer Support AI Configuration
# AI-powered customer service system

project "customer-support-system" {
    version = "2.0.0"
    author = "Support Team"
    description = "AI-driven customer support with multi-channel capabilities"
}

agent "support-specialist" {
    model = "claude-3-sonnet"
    role = "Customer Support Specialist"
    temperature = 0.7
    max_tokens = 100000

    capabilities [
        "customer-service"
        "problem-solving"
        "empathy"
        "multi-language"
        "escalation-handling"
    ]

    backstory {
        8 years in customer support leadership
        Handled 100K+ customer interactions
        Expert in de-escalation techniques
        Trained support teams worldwide
    }

    tools = [
        "zendesk"
        "intercom"
        "slack"
        "email-client"
        "knowledge-base"
    ]
}

agent "technical-expert" {
    model = "gpt-4"
    role = "Technical Support Engineer"
    temperature = 0.6
    max_tokens = 80000

    capabilities [
        "technical-troubleshooting"
        "bug-analysis"
        "system-diagnostics"
        "code-review"
        "api-debugging"
    ]

    backstory {
        12 years in software engineering
        Specialized in distributed systems
        Published technical documentation
        Led incident response teams
    }

    tools = [
        "terminal"
        "database-client"
        "monitoring-tools"
        "api-tester"
        "log-analyzer"
    ]
}

workflow "customer-inquiry-handling" {
    trigger = "webhook"

    step "triage" {
        agent = "support-specialist"
        task = "Analyze customer inquiry and determine priority level"
        timeout = 5m
    }

    step "initial-response" {
        agent = "support-specialist"
        task = "Provide immediate acknowledgment and gather more details"
        timeout = 10m
        depends_on = ["triage"]
    }

    step "technical-analysis" {
        agent = "technical-expert"
        task = "Investigate technical aspects of the issue"
        timeout = 15m
        depends_on = ["triage"]

        retry {
            max_attempts = 2
            delay = 2m
            backoff = "exponential"
        }
    }

    step "resolution" {
        crew = ["support-specialist", "technical-expert"]
        task = "Develop and implement solution"
        timeout = 30m
        depends_on = ["initial-response", "technical-analysis"]
    }

    step "follow-up" {
        agent = "support-specialist"
        task = "Ensure customer satisfaction and document resolution"
        timeout = 10m
        depends_on = ["resolution"]
    }

    pipeline {
        triage -> initial-response -> technical-analysis -> resolution -> follow-up
    }
}

crew "support-team" {
    agents [
        "support-specialist"
        "technical-expert"
    ]

    process = "hierarchical"
    manager = "technical-expert"
    max_iterations = 5
    verbose = true
}

memory {
    provider = "redis"
    connection = "redis:

    embeddings {
        model = "text-embedding-ada-002"
        dimensions = 1536
        batch_size = 50
    }

    cache_size = 5000
    persistence = false
}

context "production" {
    environment = "prod"
    debug = false
    max_tokens = 150000

    secrets {
        zendesk_token = $ZENDESK_API_TOKEN
        intercom_token = $INTERCOM_API_TOKEN
        slack_token = $SLACK_API_TOKEN
    }

    variables {
        support_email = "support@company.com"
        response_timeout = 4h
        escalation_threshold = 24h
        max_concurrent_tickets = 50
    }
}"#,
    ),
    ("data-pipeline", "# Data Pipeline template - full content embedded"),
    ("research", "# Research Assistant template - full content embedded"),
];

pub fn get_embedded_templates() -> &'static [(&'static str, &'static str)] {
    EMBEDDED_TEMPLATES
}