# AUTOGENERATED! DO NOT EDIT! File to edit: ../../../nbs/agents/benchmark_agents/03_ERM_agents.ipynb.

# %% auto 0
__all__ = ['FakePolicy', 'NewsvendorData', 'LinearModel', 'MLP', 'MLP_BLOCK', 'RNN', 'SGDBase', 'LERMsgdAgent', 'MLPsgdAgent',
           'RNNsgdAgent', 'RNNsgdMetaAgent', 'LlamaRotaryEmbedding', 'rotate_half', 'apply_rotary_pos_emb',
           'CausalSelfAttention', 'find_multiple', 'MLP_block', 'RMSNorm', 'Block', 'LagLlama', 'LagLlamasgdAgent',
           'NewsvendorDataMeta', 'LagLlamasgdMetaAgent']

# %% ../../../nbs/agents/benchmark_agents/03_ERM_agents.ipynb 4
# General libraries:
import numpy as np
from scipy.stats import norm
from tqdm import trange, tqdm
from time import sleep

# Torch
import torch
import torch.nn as nn
import torch.nn.functional as F
from torch.utils.data import Dataset, DataLoader
from ..processors.processors import GetTimeSeriesAndStaticFeatures


# Mushroom libraries
from mushroom_rl.core import Agent

# %% ../../../nbs/agents/benchmark_agents/03_ERM_agents.ipynb 6
class FakePolicy():
    def reset(*args, **kwargs):
        pass

class NewsvendorData(Dataset):

    def __init__(self, x, y, mask=None, feature_map=None):
        # create torch tensors
        self.x=torch.from_numpy(x)
        self.y=torch.from_numpy(y)
        if mask is not None:
            self.mask = torch.from_numpy(mask)
            self.mask = self.mask.bool()
            if len(self.mask.shape) == 1:
                self.mask = self.mask.unsqueeze(1)
        else:
            self.mask = None
        self.feature_map = feature_map
        
        # convert to torch float32
        self.x=self.x.float()
        self.y=self.y.float()

        if self.feature_map is not None:
            self.n_samples=y.shape[0]*y.shape[1]
            self.index_mapping = dict()
            index_used=0
            for i in range(y.shape[0]):
                for j in range(y.shape[1]):
                    self.index_mapping[index_used] = (i,j)
                    index_used+=1
        else:
            self.n_samples=y.shape[0]

    def __getitem__(self, index):
        
        if self.feature_map is not None:
            
            coordinates = self.index_mapping[index]

            time = coordinates[0]
            product = coordinates[1]

            y = self.y[time, product]

            # print("time: ", time, "product: ", product, "y: ", y)

            # print(self.feature_map)
            # print(self.feature_map[:, product])
            # print(self.x[time])

            x = self.x[time, self.feature_map[:, product].astype(bool)]

            if self.mask is not None:
                mask = self.mask[time, product]
        
        else:
            x = self.x[index]
            y = self.y[index]
            if self.mask is not None:
                mask = self.mask[index]
        
        if self.feature_map is not None:
            if self.mask is not None:
                return x, y, mask, product
            else:
                return x, y, product
        else:
            if self.mask is not None:
                return x, y, mask
            else:
                return x, y

    def __len__(self):
        return self.n_samples

class LinearModel(nn.Module):
    def __init__(self, input_size, output_size, relu_output=False):
        super().__init__()
        self.l1=nn.Linear(input_size, output_size)
        if relu_output:
            self.final_activation = nn.ReLU()
        else:
            self.final_activation = nn.Identity()
            
    def forward(self, x):
        out=self.l1(x)
        out=self.final_activation(out)
        return out

class MLP(nn.Module):
    def __init__(self, input_size, hidden_size, output_size, num_hidden_layers=3, drop_prob=0.0, relu_output=False):
        super().__init__()

        # List of layers
        layers = []

        # Input layer
        layers.append(nn.Linear(input_size, hidden_size))
        layers.append(nn.ReLU())
        layers.append(nn.Dropout(p=drop_prob))
        
        # Hidden layers
        for _ in range(num_hidden_layers-1): 
            layers.append(nn.Linear(hidden_size, hidden_size))
            layers.append(nn.ReLU())
            layers.append(nn.Dropout(p=drop_prob))

        # Output layer
        layers.append(nn.Linear(hidden_size, output_size))
        if relu_output:
            layers.append(nn.ReLU()) # output is non-negative
        else:
            layers.append(nn.Identity())

        # Combine layers
        self.model = nn.Sequential(*layers)

    def forward(self, x):
        return self.model(x)

class MLP_BLOCK(nn.Module):
    def __init__(self, size_in, size_out, drop_prob):
        super(MLP_BLOCK, self).__init__()
        self.MLP_block = nn.Sequential(
            nn.Linear(size_in, size_out),
            nn.ReLU(),
            nn.Dropout(p=drop_prob),
            # add batch norm
            nn.BatchNorm1d(size_out)
        )
    
    def forward(self, x):
        return self.MLP_block(x)

class RNN(nn.Module):
    def __init__(self,
                input_size,
                hidden_size,
                output_size,
                num_hidden_layers=2,
                num_RNN_layers=2,
                drop_prob=0.0,
                num_time_series_features=None,
                lag_window=None,
                relu_output=False):
        super().__init__()

        self.input_processor = GetTimeSeriesAndStaticFeatures(num_time_series_features,lag_window)
        # self._n_output = output_shape[0]
        # self._n_features = n_features

        self._h1_recurrent = nn.GRU(int(input_size/lag_window), int(hidden_size/2), num_RNN_layers, batch_first = True)
        self._h1_dropout = nn.Dropout(p=drop_prob)

        self.mlp_blocks = nn.ModuleList()
        for i in range(num_hidden_layers):
            if i == 0:
                self.mlp_blocks.append(MLP_BLOCK(size_in = int(hidden_size/2), size_out = hidden_size, drop_prob=drop_prob))
            else:
                self.mlp_blocks.append(MLP_BLOCK(size_in = int(hidden_size/(2**(i-1))), size_out = int(hidden_size/(2**i)), drop_prob=drop_prob))

        if num_hidden_layers==0:
            self.output_layer = nn.Linear(int(hidden_size/2), output_size)
        else:
            self.output_layer = nn.Linear(int(hidden_size/(2**i)), output_size)

        if relu_output:
            self.final_activation = nn.ReLU()  # output is non-negative
        else:
            self.final_activation = nn.Identity()

    def forward(self, x):

        # if shape is 1 then expand
        if len(x.shape)==1:
            x=x.unsqueeze(0)
        x = self.input_processor(x)[0] # no static data returned as no inventory

        # RNN block
        x, _ = self._h1_recurrent(x)
        x = self._h1_dropout(x)
        x = x[:, -1, :] # only take the last time step

 
        # MLP block
        for mlp_block in self.mlp_blocks:
            x = mlp_block(x)  
        
        x = self.output_layer(x) 
        x = self.final_activation(x)

        # print(x.shape)
    
        return x    

class SGDBase(Agent):

    def __init__(self, config=None, input_size=None, hidden_size=64, output_size=1, learning_rate=0.01, num_hidden_layers=3, num_RNN_layers=None, drop_prob=0.0, l2_reg=0.0, num_time_series_features=None, lag_window=None, relu_output=False):
        if self.model_type=="Linear":
            self.model=LinearModel(input_size, output_size, relu_output=relu_output)
        elif self.model_type=="MLP":
            self.model=MLP(input_size, hidden_size, output_size, num_hidden_layers=num_hidden_layers, drop_prob=drop_prob, relu_output=relu_output)
        elif self.model_type=="RNN":
            self.model=RNN(input_size, hidden_size, output_size, num_hidden_layers=num_hidden_layers, num_RNN_layers=num_RNN_layers, drop_prob=drop_prob, num_time_series_features=num_time_series_features, lag_window=lag_window, relu_output=relu_output)
        elif self.model_type=="LagLlama":
            # print("config: ", config)
            self.model=LagLlama(**config)
        self.optimizer = torch.optim.Adam(self.model.parameters(), lr=learning_rate, weight_decay=l2_reg)
        self.criterion = nn.MSELoss()


    def fit(self, X_train, y_train, mask, cu, co, batch_size=64, learning_rate=0.01, device="cpu"):
        
        if y_train.ndim == 1:
            y_train = y.reshape(-1, 1)
        
        dataset_train=NewsvendorData(X_train, y_train, mask)

        self.model.to(device)

        train_loader=DataLoader(dataset=dataset_train, batch_size=batch_size, shuffle=True)

        self.model.train()

        total_loss = 0
        for i, (output) in enumerate(train_loader):

                if len(output)==2:
                    feat, labels = output
                    masks=None
                else:
                    feat, labels, masks = output
                    masks=masks.to(device)
                
                feat=feat.to(device)
                labels=labels.to(device)
                outputs=self.model(feat)

                loss = torch.mean(SGDBase.pinball_loss(cu, co, labels, outputs, masks))
                
                total_loss += loss.item()

                #backward
                self.optimizer.zero_grad()
                loss.backward()
                self.optimizer.step()

        print("training loss: ", total_loss)   
        self.model.eval()
        self.model.to("cpu")

        return self.model

    def predict(self, X):

        # if len(X.shape)==1:
        #     X = np.expand_dims(X, axis=0)
        
        self.model.eval()
        with torch.no_grad():
            X=torch.from_numpy(X)
            X=X.float()
            output=self.model(X)
            output=output.numpy()
        
        if len(output.shape)==2:
            output=output.reshape(-1)
        
        return output

    def train(self):
        self.model.train()
    
    def eval(self):
        self.model.eval()   
    
    ### Helper functions

    @staticmethod
    def max_or_zero(data):
        value = torch.max(data, torch.zeros_like(data))
        return value

    @staticmethod
    def pinball_loss(cu, co, demand, order_quantity, mask):

        # TODO: check that this works in termns of shapes

        if len(demand.shape)==1:
            demand = demand.unsqueeze(1)
        
        assert demand.shape == order_quantity.shape

        cu = torch.tensor(cu, dtype=torch.float32)
        co = torch.tensor(co, dtype=torch.float32)

        cu = cu.to(demand.device)
        co = co.to(demand.device)

        underage=cu*SGDBase.max_or_zero(demand-order_quantity)
        overage=co*SGDBase.max_or_zero(order_quantity-demand)

        loss=underage+overage
        
        if mask is not None:
            loss = loss*mask
        return loss

class LERMsgdAgent(SGDBase):
    def __init__(self,
                    input_size,
                    output_size,
                    cu,
                    co,
                    final_activation="identity",
                    batch_size=128,
                    learning_rate=0.01,
                    device="cpu",
                    agent_name = "LERM"
                    ):
        self.name=agent_name
        self.model_type="Linear"
        self.cu = cu
        self.co = co
        self.device = device
        self.batch_size=batch_size
        self.learning_rate=learning_rate
        self.device=device


        self.policy=FakePolicy()
        self._postprocessors = list()
        self._preprocessors = list()
        self.train_directly=True
        self.train_mode = "epochs"

        if final_activation=="identity":
            self.final_activation = False
        elif final_activation=="relu":
            self.final_activation = True

        super().__init__(input_size=input_size, hidden_size=None, output_size=output_size, learning_rate=learning_rate, relu_output=self.final_activation)

    def fit_epoch(self, features_train, demand_train, mask=None):
        super().fit(features_train, demand_train, mask=mask, cu=self.cu, co=self.co, batch_size=self.batch_size, learning_rate=self.learning_rate, device=self.device)

    def draw_action(self, X):
        return super().predict(X)

class MLPsgdAgent(SGDBase):
    def __init__(self,
                    input_size,
                    output_size,
                    cu,
                    co,
                    final_activation="identity",
                    hidden_size=64,
                    batch_size=128,
                    learning_rate=0.01,
                    device="cpu",
                    agent_name = "DLNV",
                    num_hidden_layers=3,
                    drop_prob=0.0,
                    l2_reg=0.0,
                    ): 
        
        self.name=agent_name
        self.model_type="MLP"
        self.cu = cu
        self.co = co
        self.device = device
        self.hidden_size = hidden_size
        self.batch_size=batch_size
        self.learning_rate=learning_rate
        self.device=device

        self.policy=FakePolicy()
        self._postprocessors = list()
        self._preprocessors = list()
        self.train_directly=True
        self.train_mode = "epochs"

        if final_activation=="identity":
            self.final_activation = False
        elif final_activation=="relu":
            self.final_activation = True

        super().__init__(input_size=input_size, hidden_size=hidden_size, output_size=output_size, learning_rate=learning_rate, num_hidden_layers=num_hidden_layers, drop_prob=drop_prob, l2_reg=l2_reg, relu_output=self.final_activation)

    def fit_epoch(self, features_train, demand_train, mask=None):
        super().fit(features_train, demand_train, mask=mask, cu=self.cu, co=self.co, batch_size=self.batch_size, learning_rate=self.learning_rate, device=self.device)

    def draw_action(self, X):
        return super().predict(X)

class RNNsgdAgent(SGDBase):
    def __init__(self,
                    input_size,
                    output_size,
                    cu,
                    co,
                    num_time_series_features,
                    lag_window,
                    final_activation="identity",
                    hidden_size=64,
                    batch_size=128,
                    learning_rate=0.01,
                    device="cpu",
                    agent_name = "DLNV_RNN",
                    num_RNN_layers = 2,
                    num_hidden_layers=3,
                    drop_prob=0.0,
                    l2_reg=0.0,
                    ): 
        
        self.name=agent_name
        self.model_type="RNN"
        self.cu = cu
        self.co = co
        self.device = device
        self.hidden_size = hidden_size
        self.batch_size=batch_size
        self.learning_rate=learning_rate
        self.device=device

        self.policy=FakePolicy()
        self._postprocessors = list()
        self._preprocessors = list()
        self.train_directly=True
        self.train_mode = "epochs"
    
        if final_activation=="identity":
            self.final_activation = False
        elif final_activation=="relu":
            self.final_activation = True

        super().__init__(input_size=input_size, hidden_size=hidden_size, output_size=output_size, learning_rate=learning_rate, num_hidden_layers=num_hidden_layers, num_RNN_layers=num_RNN_layers, drop_prob=drop_prob, l2_reg=l2_reg, num_time_series_features=num_time_series_features, lag_window=lag_window, relu_output=self.final_activation, )
        
    def fit_epoch(self, features_train, demand_train, mask=None):
        super().fit(features_train, demand_train, mask=mask, cu=self.cu, co=self.co, batch_size=self.batch_size, learning_rate=self.learning_rate, device=self.device)

    def draw_action(self, X):
        return super().predict(X)

class RNNsgdMetaAgent(SGDBase):
    def __init__(self,
                    feature_map,
                    input_size,
                    output_size,
                    cu,
                    co,
                    num_time_series_features,
                    lag_window,
                    final_activation="identity",
                    hidden_size=64,
                    batch_size=128,
                    learning_rate=0.01,
                    device="cpu",
                    agent_name = "DLNV_RNN",
                    num_RNN_layers = 2,
                    num_hidden_layers=3,
                    drop_prob=0.0,
                    l2_reg=0.0,
                    ): 
        
        self.feature_map = feature_map
        self.name=agent_name
        self.model_type="RNN"
        self.cu = cu
        self.co = co
        self.device = device
        self.hidden_size = hidden_size
        self.batch_size=batch_size
        self.learning_rate=learning_rate
        self.device=device

        self.policy=FakePolicy()
        self._postprocessors = list()
        self._preprocessors = list()
        self.train_directly=True
        self.train_mode = "epochs"

        # print("feature map: ", self.feature_map)
    
        if final_activation=="identity":
            self.final_activation = False
        elif final_activation=="relu":
            self.final_activation = True

        input_size = np.sum(self.feature_map[:,0])

        super().__init__(input_size=input_size, hidden_size=hidden_size, output_size=1, learning_rate=learning_rate, num_hidden_layers=num_hidden_layers, num_RNN_layers=num_RNN_layers, drop_prob=drop_prob, l2_reg=l2_reg, num_time_series_features=num_time_series_features, lag_window=lag_window, relu_output=self.final_activation, )

    def fit_epoch(self, features_train, demand_train, mask=None):
        self.fit(features_train, demand_train, mask=mask, cu=self.cu, co=self.co, batch_size=self.batch_size, learning_rate=self.learning_rate, device=self.device)
    
    def fit(self, X_train, y_train, mask, cu, co, batch_size=64, learning_rate=0.01, device="cpu"):
        
        if y_train.ndim == 1:
            y_train = y.reshape(-1, 1)
         
        dataset_train=NewsvendorData(X_train, y_train, mask, self.feature_map)

        self.model.to(device)
        
        train_loader=DataLoader(dataset=dataset_train, batch_size=batch_size, shuffle=False) # make shuffle true

        self.model.train()

        total_loss = 0
        for i, (output) in tqdm(enumerate(train_loader)):

                if len(output)==3:
                    feat, labels, product = output
                    masks=None
                else:
                    feat, labels, masks, product = output
                    masks=masks.to(device)   

                feat=feat.to(device)
                labels=labels.to(device)
                outputs=self.model(feat)

                cu_selected = cu[product]
                co_selected = co[product]
                
                loss_per_product = self.pinball_loss(cu_selected, co_selected, labels, outputs, masks)
                loss = torch.mean(loss_per_product)
                
                total_loss += loss.item()

                #backward
                self.optimizer.zero_grad()
                loss.backward()
                self.optimizer.step()

        print("training loss: ", total_loss)   
        self.model.eval()
        self.model.to("cpu")

        return self.model

    def draw_action(self, X):
        return self.predict(X)
    
    def predict(self, X):
        self.model.eval()
        self.model.to(self.device)

        if len(X.shape) == 1:
            X = np.expand_dims(X, axis=0)

        # Initialize a list to hold the X_i batches
        X_batches = []

        # Create batches
        for i in range(self.feature_map.shape[1]):
            X_i = X[:, self.feature_map[:, i].astype(bool)]
            X_batches.append(X_i)

        X_batch = np.concatenate([x for x in X_batches], axis=0)
        X_batch = torch.from_numpy(X_batch)
        X_batch = X_batch.float().to(self.device)

        with torch.no_grad():
            output = self.model(X_batch)
            output = output.cpu().numpy()

        # reduce output dimension

        output = output.squeeze(1)
        
        # print(outputs)

        # check if outputs need to got to cpu or handled by mushroomrl

        return output

    @staticmethod
    def pinball_loss(cu, co, demand, order_quantity, mask):

        if len(demand.shape)==1:
            demand = demand.unsqueeze(1)
        
        assert demand.shape == order_quantity.shape

        cu = torch.tensor(cu, dtype=torch.float32)
        co = torch.tensor(co, dtype=torch.float32)

        cu = cu.unsqueeze(1)
        co = co.unsqueeze(1)

        cu = cu.to(demand.device)
        co = co.to(demand.device)

        underage_quantity = SGDBase.max_or_zero(demand-order_quantity)
        overage_quantity = SGDBase.max_or_zero(order_quantity-demand)

        assert cu.shape == underage_quantity.shape
        assert co.shape == overage_quantity.shape

        underage=cu*underage_quantity
        overage=co*overage_quantity

        loss=underage+overage
 
        if mask is not None:
            loss = loss*mask
        return loss


# %% ../../../nbs/agents/benchmark_agents/03_ERM_agents.ipynb 8
class LlamaRotaryEmbedding(torch.nn.Module):

    # Rotary positional embeddings (RoPE) based on https://arxiv.org/abs/2104.09864, code exactly as implemented in https://github.com/time-series-foundation-models/lag-llama

    # TODO: potentially also include LlamaLinearScalingRotaryEmbedding and LlamaDynamicNTKScalingRotaryEmbedding

    def __init__(self, dim, max_position_embeddings=2048, base=10000, device=None):
        super().__init__()

        self.dim = dim
        self.max_position_embeddings = max_position_embeddings
        self.base = base
        inv_freq = 1.0 / (
            self.base ** (torch.arange(0, self.dim, 2).float().to(device) / self.dim)
        )
        self.register_buffer("inv_freq", inv_freq, persistent=False)

        # Build here to make `torch.jit.trace` work.
        self._set_cos_sin_cache(
            seq_len=max_position_embeddings,
            device=self.inv_freq.device,
            dtype=torch.get_default_dtype(),
        )

    def _set_cos_sin_cache(self, seq_len, device, dtype):
        self.max_seq_len_cached = seq_len
        t = torch.arange(
            self.max_seq_len_cached, device=device, dtype=self.inv_freq.dtype
        )

        freqs = torch.einsum("i,j->ij", t, self.inv_freq)
        # Different from paper, but it uses a different permutation in order to obtain the same calculation
        emb = torch.cat((freqs, freqs), dim=-1)
        self.register_buffer(
            "cos_cached", emb.cos()[None, None, :, :].to(dtype), persistent=False
        )
        self.register_buffer(
            "sin_cached", emb.sin()[None, None, :, :].to(dtype), persistent=False
        )

    def forward(self, device, dtype, seq_len=None):
        # x: [bs, num_attention_heads, seq_len, head_size]
        if seq_len > self.max_seq_len_cached:
            self._set_cos_sin_cache(seq_len=seq_len, device=device, dtype=dtype)

        return (
            self.cos_cached[:, :, :seq_len, ...].to(dtype=dtype),
            self.sin_cached[:, :, :seq_len, ...].to(dtype=dtype),
        )

def rotate_half(x):
    """Rotates half the hidden dims of the input."""
    x1 = x[..., : x.shape[-1] // 2]
    x2 = x[..., x.shape[-1] // 2 :]
    return torch.cat((-x2, x1), dim=-1)

def apply_rotary_pos_emb(q, k, cos, sin, position_ids):
    # The first two dimensions of cos and sin are always 1, so we can `squeeze` them.
    cos = cos.squeeze(1).squeeze(0)  # [seq_len, dim]
    sin = sin.squeeze(1).squeeze(0)  # [seq_len, dim]
    cos = cos[position_ids].unsqueeze(1)  # [bs, 1, seq_len, dim]
    sin = sin[position_ids].unsqueeze(1)  # [bs, 1, seq_len, dim]
    q_embed = (q * cos) + (rotate_half(q) * sin)
    k_embed = (k * cos) + (rotate_half(k) * sin)
    return q_embed, k_embed

class CausalSelfAttention(nn.Module):
    def __init__(self, n_embd_per_head, n_head, block_size, dropout) -> None:
        super().__init__()
        # query projections for all heads, but in a batch
        self.q_proj = nn.Linear(
            n_embd_per_head * n_head,
            n_embd_per_head * n_head,
            bias=False,
        )
        # key, value projections
        self.kv_proj = nn.Linear(
            n_embd_per_head * n_head,
            2 * n_embd_per_head * n_head,
            bias=False,
        )
        # output projection
        self.c_proj = nn.Linear(
            n_embd_per_head * n_head,
            n_embd_per_head * n_head,
            bias=False,
        )

        self.n_head = n_head
        self.n_embd_per_head = n_embd_per_head
        self.block_size = block_size
        self.dropout = dropout

        # self.rope_scaling = rope_scaling # so far on default RoPE implemented
        self.rope_scaling=None
        # self._rope_scaling_validation() # so far on default RoPE implemented

        self._init_rope()
        self.kv_cache = None

    def _init_rope(self):
        if self.rope_scaling is None:
            # print(self.n_embd_per_head, self.block_size)
            self.rotary_emb = LlamaRotaryEmbedding(
                self.n_embd_per_head, max_position_embeddings=self.block_size
            )
        # note: the following is not yet implemented 
        else:
            scaling_type = self.rope_scaling["type"]
            scaling_factor = self.rope_scaling["factor"]
            if scaling_type == "nope":
                self.rotary_emb = None
            elif scaling_type == "linear":
                self.rotary_emb = LlamaLinearScalingRotaryEmbedding(
                    self.n_embd_per_head,
                    max_position_embeddings=self.block_size,
                    scaling_factor=scaling_factor,
                )
            elif scaling_type == "dynamic":
                self.rotary_emb = LlamaDynamicNTKScalingRotaryEmbedding(
                    self.n_embd_per_head,
                    max_position_embeddings=self.block_size,
                    scaling_factor=scaling_factor,
                )
            else:
                raise ValueError(f"Unknown RoPE scaling type {scaling_type}")

    # def _rope_scaling_validation(self):
    #     """
    #     Validate the `rope_scaling` configuration.
    #     """
    #     if self.rope_scaling is None:
    #         return

    #     if not isinstance(self.rope_scaling, dict) or len(self.rope_scaling) != 2:
    #         raise ValueError(
    #             "`rope_scaling` must be a dictionary with with two fields, `name` and `factor`, "
    #             f"got {self.rope_scaling}"
    #         )
    #     rope_scaling_type = self.rope_scaling.get("type", None)
    #     rope_scaling_factor = self.rope_scaling.get("factor", None)
    #     if rope_scaling_type is None or rope_scaling_type not in [
    #         "linear",
    #         "dynamic",
    #         "nope",
    #     ]:
    #         raise ValueError(
    #             f"`rope_scaling`'s name field must be one of ['linear', 'dynamic'], got {rope_scaling_type}"
    #         )
    #     if rope_scaling_type in ["linear", "dynamic"]:
    #         if (
    #             rope_scaling_factor is None
    #             or not isinstance(rope_scaling_factor, float)
    #             or rope_scaling_factor < 1.0
    #         ):
    #             raise ValueError(
    #                 f"`rope_scaling`'s factor field must be an float >= 1, got {rope_scaling_factor}"
    #             )

    def forward(self, x: torch.Tensor, use_kv_cache: bool) -> torch.Tensor:
        # batch size, sequence length, embedding dimensionality (n_embd)

        B, T, C = x.size()

        # calculate query, key, values for all heads in batch and move head forward to be the batch dim
        q = self.q_proj(x)
        k, v = self.kv_proj(x).split(self.n_embd_per_head * self.n_head, dim=2)

        if use_kv_cache:
            # Optimized for single next prediction
            if self.kv_cache is not None:
                # Update cache
                k = torch.cat([self.kv_cache[0], k], dim=1)[:, 1:]
                v = torch.cat([self.kv_cache[1], v], dim=1)[:, 1:]
                self.kv_cache = k, v
            else:
                # Build cache
                self.kv_cache = k, v

        k = k.view(B, -1, self.n_head, self.n_embd_per_head).transpose(
            1, 2
        )  # (B, nh, T, hs)
        q = q.view(B, -1, self.n_head, self.n_embd_per_head).transpose(
            1, 2
        )  # (B, nh, T, hs)
        v = v.view(B, -1, self.n_head, self.n_embd_per_head).transpose(
            1, 2
        )  # (B, nh, T, hs)

        if self.rotary_emb is not None:
            cos, sin = self.rotary_emb(device=v.device, dtype=v.dtype, seq_len=T)
            q, k = apply_rotary_pos_emb(q, k, cos, sin, position_ids=None)

        # causal self-attention; Self-attend: (B, nh, T, hs) x (B, nh, hs, T) -> (B, nh, T, T)
        #  att = (q @ k.transpose(-2, -1)) * (1.0 / math.sqrt(k.size(-1)))
        #  att = att.masked_fill(self.bias[:,:,:T,:T] == 0, float('-inf'))
        #  att = F.softmax(att, dim=-1)
        #  y = att @ v # (B, nh, T, T) x (B, nh, T, hs) -> (B, nh, T, hs)

        # efficient attention using Flash Attention CUDA kernels
        # When using kv cache at inference, is_causal=False since decoder is causal, at each generation step we want
        # to avoid recalculating the same previous token attention

        # TODO: understand how causeal gets turned of in the encoder (we want to process the input squence at once)

        if use_kv_cache:
            y = F.scaled_dot_product_attention(
                q, k, v, attn_mask=None, dropout_p=self.dropout, is_causal=False
            )
        else:
            y = F.scaled_dot_product_attention(
                q, k, v, attn_mask=None, dropout_p=self.dropout, is_causal=True
            )

        # re-assemble all head outputs side by side
        y = y.transpose(1, 2).contiguous().view(B, T, C)

        # output projection
        y = self.c_proj(y)

        return y

def find_multiple(n: int, k: int) -> int:
    if n % k == 0:
        return n
    return n + k - (n % k)

class MLP_block(nn.Module):
    def __init__(self, n_embd_per_head, n_head, min_multiple = 256, n_mlp_layers=2) -> None:
        super().__init__()
        hidden_dim = 4 * n_embd_per_head * n_head
        n_hidden = int(2 * hidden_dim / 3)
        self.n_mlp_layers = n_mlp_layers

        # TODO: ensure the hidden dim could also work on small data (might need to scale down the hidden dim)
        n_hidden = find_multiple(n_hidden, min_multiple)

        print("hidden dimension in MLP block: ", n_hidden)

        self.c_fc1 = nn.Linear(
            n_embd_per_head * n_head, n_hidden, bias=False
        )
        if n_mlp_layers ==2:
            self.c_fc2 = nn.Linear(
                n_embd_per_head * n_head, n_hidden, bias=False
            )
        
        self.c_proj = nn.Linear(
            n_hidden, n_embd_per_head * n_head, bias=False
        )

    def forward(self, x: torch.Tensor) -> torch.Tensor:
        if self.n_mlp_layers == 2:
            x = F.silu(self.c_fc1(x)) * self.c_fc2(x)
        else:
            x = F.silu(self.c_fc1(x))
        x = self.c_proj(x)
        return x

class RMSNorm(nn.Module):
    """Root Mean Square Layer Normalization.

    Derived from https://github.com/bzhangGo/rmsnorm/blob/master/rmsnorm_torch.py. BSD 3-Clause License:
    https://github.com/bzhangGo/rmsnorm/blob/master/LICENSE.
    """

    def __init__(self, size: int, dim: int = -1, eps: float = 1e-5) -> None:
        super().__init__()
        self.scale = nn.Parameter(torch.ones(size))
        self.eps = eps
        self.dim = dim

    def forward(self, x: torch.Tensor) -> torch.Tensor:
        # NOTE: the original RMSNorm paper implementation is not equivalent
        # norm_x = x.norm(2, dim=self.dim, keepdim=True)
        # rms_x = norm_x * d_x ** (-1. / 2)
        # x_normed = x / (rms_x + self.eps)
        # keep RMSNorm in float32
        norm_x = x.to(torch.float32).pow(2).mean(dim=self.dim, keepdim=True)
        x_normed = x * torch.rsqrt(norm_x + self.eps)
        return (self.scale * x_normed).type_as(x)

class Block(nn.Module):
    def __init__(self, n_embd_per_head, n_head, block_size, dropout, min_multiple = 256, n_mlp_layers=2) -> None:
        super().__init__()
        self.rms_1 = RMSNorm(n_embd_per_head * n_head)
        self.attn = CausalSelfAttention(n_embd_per_head, n_head, block_size, dropout)
        self.rms_2 = RMSNorm(n_embd_per_head * n_head)
        self.mlp = MLP_block(n_embd_per_head, n_head, min_multiple = min_multiple, n_mlp_layers=n_mlp_layers)

    def forward(self, x: torch.Tensor, use_kv_cache: bool) -> torch.Tensor:
        x = x + self.attn(self.rms_1(x), use_kv_cache)
        y = x + self.mlp(self.rms_2(x))
        return y

class LagLlama(nn.Module):
    def __init__(self,
                input_size,
                output_size,

                context_length = 16,
                max_context_length = 16, # check what context and max_context do
                n_layer = 1, # default LagLlama paper: 32
                n_head = 4, # default LagLlama paper: 32
                n_embd_per_head = 32, # default LagLlama paper: 128
                rope_scaling = None, # not yet implemetned (scaled version of rotary embeddings)
                # distr_output: Only needed for probabilistic forecasting
                # num_parallel_samples: int = 100, # Only needed for probabilistic forecasting
                # time_feat: True, # Already included in our multi-variate time series input
                min_multiple = 256,
                n_mlp_layers = 2,
            
                drop_prob=0.0,
                num_time_series_features=None,
                lag_window=None,
                relu_output=False):

        super().__init__()

        block_size = max_context_length
        self.context_length = context_length


        # did not implement optional time_features as all inputs are day-level.
        # Any other time-features (e.g., weekday) can be included in preprocessing
        # and added as part of the feature vector

        # self.num_parallel_samples = num_parallel_samples
        self.input_processor = GetTimeSeriesAndStaticFeatures(num_time_series_features,lag_window)

        self.input_size = input_size # size of input vector (features + demand) on one timestep without positional embeddings
        

        print("input size: ", self.input_size)
        #! Note that no scaling is implemented in the model, needs to be addressed in pre-processing

        # self.distr_output = distr_output # TODO check later how to get rid of

        # changed from PropabilisticForecastingNetwork to sinlge output
        self.param_proj = nn.Linear(n_embd_per_head * n_head, output_size
        )

        self.transformer = nn.ModuleDict(
            dict(
                wte=nn.Linear(
                    self.input_size, n_embd_per_head * n_head
                ),
                h=nn.ModuleList([Block(n_embd_per_head, n_head, block_size, drop_prob, min_multiple = min_multiple, n_mlp_layers=n_mlp_layers) for _ in range(n_layer)]),
                ln_f=RMSNorm(n_embd_per_head * n_head),
            )
        )
        self.y_cache = False  # used at time of inference when kv cached is used

        if relu_output:
            self.final_activation = nn.ReLU()  # output is non-negative
        else:
            self.final_activation = nn.Identity()


        def _init_weights(self, module: nn.Module) -> None:
            if isinstance(module, nn.Linear):
                torch.nn.init.normal_(
                    module.weight, mean=0.0, std=0.02 / math.sqrt(2 * self.n_layer)
                )
            elif isinstance(module, nn.Embedding):
                torch.nn.init.normal_(
                    module.weight, mean=0.0, std=0.02 / math.sqrt(2 * self.n_layer)
                )
                
        # def prepare_input(
        #     self,
        #     past_target: torch.Tensor,
        #     past_observed_values: torch.Tensor,
        #     past_time_feat: Optional[torch.Tensor] = None,
        #     future_time_feat: Optional[torch.Tensor] = None,
        #     future_target: Optional[torch.Tensor] = None,
        # ):
            
        #     #####################
        #     #! Took scaling out as input is already scaled.
        #     # TODO: check if scaling can still improve performance (e.g., due to local sacles)

        #     # scaled_past_target, loc, scale = self.scaler(
        #     #     past_target, past_observed_values
        #     # )  # Data is standardized (past_observed_values is passed as "weights" parameter) # (bsz, context_length+max(self.lags_seq)
        #     #####################

        #     # In the below code, instead of max(self.lags_seq), it was previously -self.context_length
        #     if future_target is not None:
        #         input = torch.cat(
        #             (
        #                 scaled_past_target[..., max(self.lags_seq) :],  # Just the context
        #                 (future_target[..., :-1] - loc)
        #                 / scale,  # Not sure about the -1 here. Maybe so since the last value isn't used in the model for prediction of any new values. also if the prediction length is 1, this doesn't really affect anything
        #             ),
        #             dim=-1,
        #         )  # Shape is (bsz, context_length+(pred_len-1))
        #     else:
        #         input = scaled_past_target[..., max(self.lags_seq) :]
        #     if (past_time_feat is not None) and (future_time_feat is not None):
        #         time_feat = (
        #             torch.cat(
        #                 (
        #                     past_time_feat[..., max(self.lags_seq) :, :],
        #                     future_time_feat[..., :-1, :],
        #                 ),
        #                 dim=1,
        #             )
        #             if future_time_feat is not None
        #             else past_time_feat[..., max(self.lags_seq) :, :]
        #         )

        #     prior_input = (
        #         past_target[..., : max(self.lags_seq)] - loc
        #     ) / scale  # This the history used to construct lags.  # bsz, max(self.lags_seq)

        #     lags = lagged_sequence_values(
        #         self.lags_seq, prior_input, input, dim=-1
        #     )  # Lags are added as an extra dim. Shape is (bsz, context_length+(pred_len-1), len(self.lags_seq))

        #     static_feat = torch.cat(
        #         (loc.abs().log1p(), scale.log()), dim=-1
        #     )  # (bsz, 2) (loc and scale are concatenated)
        #     expanded_static_feat = unsqueeze_expand(
        #         static_feat, dim=-2, size=lags.shape[-2]
        #     )  # (bsz, context_length+(pred_len-1), 2)
        #     # expanded_static_feat: (bsz, context_length+(pred_len-1), len(self.lags_seq) + 2); (bsz, 1); (bsz, 1)

        #     if past_time_feat is not None:
        #         return (
        #             torch.cat((lags, expanded_static_feat, time_feat), dim=-1),
        #             loc,
        #             scale,
        #         )
        #     else:
        #         return torch.cat((lags, expanded_static_feat), dim=-1), loc, scale

    def forward(    self,
                    x: torch.Tensor,
                    use_kv_cache: bool = False,) -> torch.Tensor:

        #! not needed for meta agent. If standard-agent is needed put this processing step into the specific class.
        # # if shape is 1 then expand
        # if len(x.shape)==1:
        #     x=x.unsqueeze(0)
        # x = self.input_processor(x) # no static data returned as no inventory


        # TODO: Confirm size of x is btc
        
        (B, T, C) = x.size()

        # past_target = None
        # past_observed_values = None
        # past_time_feat = None
        # future_time_feat = None
        # future_target = None

        # transformer_input, loc, scale = self.prepare_input(
        #     past_target=past_target,
        #     past_observed_values=None, # not scaling applied
        #     future_target=future_target,
        #     past_time_feat=past_time_feat,
        #     future_time_feat=future_time_feat,
        # )

        transformer_input = x

        if use_kv_cache and self.y_cache:
            # Only use the most recent one, rest is in cache
            transformer_input = transformer_input[:, -1:]
        
        x = self.transformer.wte(
            transformer_input
        )  # token embeddings of shape (b, t, n_embd_per_head*n_head) # (bsz, context_length+(pred_len-1), n_embd_per_head*n_head)

        for block in self.transformer.h:
            x = block(x, use_kv_cache)
        x = self.transformer.ln_f(
            x
        )  # (bsz, context_length+(pred_len-1), n_embd_per_head*n_head)
        if use_kv_cache:
            self.y_cache = True
        output = self.param_proj(
            x
        )  # (bsz, context_length+(pred_len-1)) ; (bsz, context_length+(pred_len-1))

        output = self.final_activation(output)

        output = output[:, -1, :]
        # print(output.shape)
        return output
    
    def reset_cache(self) -> None:
        """
        Resets all cached key-values in attention.
        Has to be called after prediction loop in predictor
        """
        for block in self.transformer.h:
            block.y_cache = None
            block.attn.kv_cache = None

class LagLlamasgdAgent(SGDBase):

    def __init__(self,
                    input_size,
                    output_size,
                    cu,
                    co,
                    num_time_series_features,
                    lag_window,
                    agent_name = "DLNV_LagLlama",


                    final_activation="identity",

                    # TRansformer block params
                    context_length = None, # if context length is not provided, it is set to the lag_window
                    max_context_length = None,  # if max_context_length is not provided, it is set to the context_length
                                                # max context length used for RoPE - can be longer than lag_window during training
                    n_layer = 1, # default LagLlama paper: 32
                    n_head = 4, # default LagLlama paper: 32
                    n_embd_per_head = 32, # default LagLlama paper: 128
                    rope_scaling = None, # not yet implemented (scaled version of rotary embeddings)
                    min_multiple = 256,
                    n_mlp_layers = 2,
                    
                    # General params
                    drop_prob=0.0, 
                    batch_size=128,
                    learning_rate=0.01,
                    l2_reg=0.0,
                    device="cpu",

                    ):

        print("in init") 
        
        self.name=agent_name
        self.model_type="LagLlama"
        self.cu = cu
        self.co = co
        self.device = device

        if final_activation=="identity":
            final_activation = False
        elif final_activation=="relu":
            final_activation = True

        if context_length is None:
            context_length = lag_window
        if max_context_length is None:
            max_context_length = context_length

        config = {

            "input_size": input_size,
            "output_size": output_size,
            "num_time_series_features": num_time_series_features,

            "context_length": context_length,
            "max_context_length": max_context_length,
            "n_layer": n_layer,
            "n_head": n_head,
            "n_embd_per_head": n_embd_per_head,
            "rope_scaling": rope_scaling,
            "min_multiple": min_multiple,
            "n_mlp_layers": n_mlp_layers,

            "drop_prob": drop_prob,
            "num_time_series_features": num_time_series_features,
            "lag_window": lag_window,
            "relu_output": final_activation
        }

        
        self.batch_size=batch_size
        self.learning_rate=learning_rate

        self.policy=FakePolicy()
        self._postprocessors = list()
        self._preprocessors = list()
        self.train_directly=True
        self.train_mode = "epochs"
    
        if final_activation=="identity":
            self.final_activation = False
        elif final_activation=="relu":
            self.final_activation = True

        print("init parent")
        print("config: ", config)
        super().__init__(learning_rate = learning_rate, l2_reg=l2_reg, config=config)


    def fit_epoch(self, features_train, demand_train, mask=None):
        super().fit(features_train, demand_train, mask=mask, cu=self.cu, co=self.co, batch_size=self.batch_size, learning_rate=self.learning_rate, device=self.device)

    def draw_action(self, X):
        return super().predict(X)


class NewsvendorDataMeta(Dataset):

    def __init__(self, x_time_dependent, x_timeless, y, feature_map, lag_window, feature_size):

        # Function to handle mask not built in.

        self.x_time_dependent = x_time_dependent
        self.x_timeless = x_timeless
        self.y = y
        self.feature_map = feature_map

        self.lag_window = lag_window
        self.feature_size = feature_size
        self.num_products = y.shape[1]

        self.n_samples=(y.shape[0]-lag_window)*y.shape[1]
        self.index_mapping = dict()
        index_used=0
        for i in range(y.shape[0]-lag_window):
            for j in range(y.shape[1]):
                self.index_mapping[index_used] = (i,j)
                index_used+=1

    def __getitem__(self, index):
            
        coordinates = self.index_mapping[index]

        time = coordinates[0]+self.lag_window
        product = coordinates[1]

        y = self.y[time, product]

        x = np.zeros((self.lag_window, self.feature_size))

        ####### ADD LAG DEMAND
        relevant_demand = np.expand_dims(self.y[(time-self.lag_window):time, product], -1)
        x[:,0:1] = relevant_demand

        ####### ADD TIMELESS FEATURES
        features_timeless = np.expand_dims(self.x_timeless[product], axis=0)
        features_timeless = np.repeat(features_timeless, self.lag_window, axis=0)     
        x[:,1:1+features_timeless.shape[1]] = features_timeless  

        ####### ADD TIME DEPENDENT FEATURES
        relevant_features = self.x_time_dependent[(time-self.lag_window+1):time+1] # +1 because features of current period are visible to the agent.

        # get starting overarching features
        relevant_features_specific = relevant_features[:,:-self.num_products*2] #! TDOD make the 2 variable
        relevant_features_specific_mapped = relevant_features_specific[:, self.feature_map[product]]

        # get features unique per product
        relevant_features_product = relevant_features[:,-self.num_products*2:][:, [product, self.num_products+product]] #! TDOD make the 2 variable

        x[:,1+features_timeless.shape[1]:1+features_timeless.shape[1]+relevant_features_specific_mapped.shape[1]] = relevant_features_specific_mapped
        x[:,1+features_timeless.shape[1]+relevant_features_specific_mapped.shape[1]:1+features_timeless.shape[1]+relevant_features_specific_mapped.shape[1]+relevant_features_product.shape[1]] = relevant_features_product
          
        x = torch.tensor(x, dtype=torch.float32)
        y = torch.tensor(y, dtype=torch.float32)
        product = torch.tensor(product, dtype=torch.long)

        return x, y, product


    def __len__(self):
        return self.n_samples

class LagLlamasgdMetaAgent(SGDBase):

    def __init__(self,
                    feature_map,
                    input_size,
                    output_size,
                    cu,
                    co,
                    num_time_series_features,
                    lag_window,
                    agent_name = "DLNV_LagLlama",

                    final_activation="identity",

                    # TRansformer block params
                    context_length = None, # if context length is not provided, it is set to the lag_window
                    max_context_length = None,  # if max_context_length is not provided, it is set to the context_length
                                                # max context length used for RoPE - can be longer than lag_window during training
                    n_layer = 1, # default LagLlama paper: 32
                    n_head = 4, # default LagLlama paper: 32
                    n_embd_per_head = 32, # default LagLlama paper: 128
                    rope_scaling = None, # not yet implemented (scaled version of rotary embeddings)
                    min_multiple = 256,
                    n_mlp_layers = 2,
                    
                    # General params
                    drop_prob=0.0, 
                    batch_size=128, # batch size for training
                    learning_rate=0.01,
                    l2_reg=0.0,
                    device="cpu",

                    max_batch_size=2048, # relevant for inference only

                    weight_products = True
                    
                    ):

        print("in init") 
        
        self.feature_map = feature_map
        
        self.name=agent_name
        self.model_type="LagLlama"
        self.cu = cu
        self.co = co
        self.device = device

        print("in lag lama agent:", input_size)
        output_size = 1

        if final_activation=="identity":
            final_activation = False
        elif final_activation=="relu":
            final_activation = True

        if context_length is None:
            context_length = lag_window
        if max_context_length is None:
            max_context_length = context_length

        config = {

            "input_size": input_size,
            "output_size": output_size,
            "num_time_series_features": num_time_series_features,

            "context_length": context_length,
            "max_context_length": max_context_length,
            "n_layer": n_layer,
            "n_head": n_head,
            "n_embd_per_head": n_embd_per_head,
            "rope_scaling": rope_scaling,
            "min_multiple": min_multiple,
            "n_mlp_layers": n_mlp_layers,

            "drop_prob": drop_prob,
            "num_time_series_features": num_time_series_features,
            "lag_window": lag_window,
            "relu_output": final_activation
        }

        
        self.batch_size=batch_size
        self.learning_rate=learning_rate

        self.policy=FakePolicy()
        self._postprocessors = list()
        self._preprocessors = list()
        self.train_directly=True
        self.train_mode = "epochs"

        self.lag_window = lag_window
        self.input_size = input_size

        self.max_batch_size = max_batch_size
    
        if final_activation=="identity":
            self.final_activation = False
        elif final_activation=="relu":
            self.final_activation = True

        self.weight_products = weight_products

        super().__init__(learning_rate = learning_rate, l2_reg=l2_reg, config=config)

    def fit_epoch(self, features_train, demand_train, mask=None):
        self.fit(features_train, demand_train, mask=mask, cu=self.cu, co=self.co, batch_size=self.batch_size, learning_rate=self.learning_rate, device=self.device)

    def fit(self, X_train, y_train, mask, cu, co, batch_size=64, learning_rate=0.01, device="cpu"):

        features_time_dependent = X_train[0]
        features_timeless = X_train[1]
        
        dataset_train=NewsvendorDataMeta(features_time_dependent, features_timeless, y_train, self.feature_map, self.lag_window, self.input_size)

        self.model.to(device)
        
        train_loader=DataLoader(dataset=dataset_train, batch_size=batch_size, shuffle=True) # make shuffle true

        self.model.train()

        if self.weight_products:
            # !temporary:
            weights = [0.48, 0.58, 0.7, 0.93, 0.26, 0.64, 0.19, 0.67, 0.63, 0.54, 0.48, 0.39, 0.35, 0.22, 0.48, 0.73, 0.26, 0.91, 0.18, 0.36, 0.42, 0.42, 0.79, 0.52, 0.72, 0.62, 0.27, 0.09, 0.38, 0.1, 0.29, 0.52, 0.64, 0.11, 0.49, 0.41, 0.51, 0.26, 0.29, 0.25, 0.07, 0.11, 0.24, 0.9, 0.18, 0.15, 0.31, 0.27, 0.33, 0.21, 0.14, 0.22, 0.23, 0.8, 0.73, 0.04, 0.29, 0.24, 0.19, 0.59, 0.72, 0.7, 0.14, 0.42, 0.2, 0.08, 0.55, 0.18, 0.53, 0.77, 0.26, 0.45, 0.68, 0.9, 0.61, 0.5, 0.44, 0.57, 0.87, 0.03, 0.85, 0.52, 0.12, 0.7, 0.14, 0.43, 0.91, 0.7, 0.24, 0.74, 0.19, 0.16, 0.51, 0.3, 0.3, 0.41, 0.3, 0.54, 0.75, 0.54, 0.44, 0.34, 0.48, 0.26, 0.26, 0.61, 0.79, 0.14, 0.08, 0.44, 0.43, 0.69, 0.67, 0.62, 0.56, 0.66, 0.47, 0.74, 0.81, 0.25, 0.3, 0.22, 0.27, 0.65, 0.25, 0.42, 0.4, 0.73, 0.1, 0.59, 0.17, 0.31, 0.56, 0.35, 0.64, 0.39, 0.31, 0.87, 0.1, 0.1, 0.54, 0.4, 0.69, 0.23, 0.21, 0.72, 0.8, 0.79, 0.49, 0.29, 0.33, 0.62, 0.33, 0.09, 0.8, 0.09, 0.34, 0.21, 0.72, 0.74, 0.17, 0.79, 0.45, 0.28, 0.27, 0.11, 0.66, 0.41, 0.22, 0.13, 0.1, 0.07, 0.17, 0.04, 0.07, 0.37, 0.09, 0.07, 0.08, 0.17, 0.19, 0.12, 0.19, 0.04, 0.18, 0.09, 0.19, 0.15, 0.2, 0.68, 0.34, 0.43, 0.11, 0.55, 0.29, 0.76, 0.68, 0.77, 0.12, 0.86, 0.52, 0.44, 0.44, 0.19, 0.39, 0.34, 0.82, 0.62, 0.29, 0.25, 0.52, 0.53, 0.29, 0.07, 0.43, 0.29, 0.19, 0.34, 0.7, 0.23, 0.36, 0.54, 0.39, 0.24, 0.62, 0.17, 0.21, 0.47, 0.46, 0.2, 0.33, 0.24, 0.22, 0.5, 0.51, 0.46, 0.6, 0.61, 0.94, 0.29, 0.32, 0.33, 0.46, 0.25, 0.68, 0.13, 0.47, 0.37, 0.31, 0.25, 0.83, 0.68, 0.58, 0.24, 0.39, 0.25, 0.12, 0.33, 0.04, 0.22, 0.37, 0.19, 0.04, 0.44, 0.31, 0.46, 0.3, 0.2, 0.08, 0.46, 0.11, 0.11, 0.27, 0.49, 0.12, 0.05, 0.56, 0.11, 0.49, 0.22, 0.14, 0.18, 0.11, 0.26, 0.22, 0.15, 0.12, 0.04, 0.04, 0.21, 0.12, 0.19, 0.36, 0.25, 0.23, 0.05, 0.09, 0.26, 0.05, 0.14, 0.08, 0.07, 0.46, 0.56, 0.26, 0.08, 0.2, 0.06, 0.06, 0.1, 0.32, 0.37, 0.65, 0.05, 0.61, 0.2, 0.76, 0.1, 0.54, 0.15, 0.33, 0.43, 0.22, 0.7, 0.28, 0.82, 0.2, 0.39, 0.5, 0.59, 0.1, 0.89, 0.11, 0.14, 0.05, 0.63, 0.03, 0.02, 0.04, 0.06, 0.06, 0.54, 0.07, 0.05, 0.06, 0.07, 0.72, 0.13, 0.05, 0.06, 0.1, 0.06, 0.1, 0.12, 0.11, 0.07, 0.04, 0.09, 0.82, 0.14, 0.06, 0.17, 0.05, 0.08, 0.04, 0.13, 0.09, 0.17, 0.06, 0.04, 0.29, 0.39, 0.46, 0.61, 0.63, 0.12, 0.5, 0.49, 0.77, 0.15, 0.68, 0.1, 0.22, 0.18, 0.4, 0.66, 0.69, 0.39, 0.44, 0.61, 0.39, 0.63, 0.5, 0.24, 0.72, 0.32, 0.56, 0.33, 0.3, 0.1, 0.05, 0.96, 0.33, 0.33, 0.28, 0.1, 0.4, 0.63, 0.74, 0.17, 0.18, 0.34, 0.16, 0.77, 0.3, 0.67, 0.5, 0.22, 0.29, 0.24, 0.71, 0.15, 0.42, 0.51, 0.14, 0.64, 0.46, 0.31, 0.82, 0.28, 0.31, 0.4, 0.2, 0.31, 0.23, 0.17, 0.16, 0.29, 0.07, 0.25, 0.18, 0.38, 0.52, 0.04, 0.47, 0.31, 0.13, 0.22, 0.36, 0.65, 0.17, 0.17, 0.27, 0.15, 0.37, 0.44, 0.24, 0.05, 0.05, 0.07, 0.21, 0.2, 0.75, 0.18, 0.06, 0.29, 0.08, 0.08, 0.47, 0.49, 0.27, 0.43, 0.49, 0.33, 0.09, 0.61, 0.27, 0.08, 0.26, 0.08, 0.04, 0.11, 0.06, 0.04, 0.13, 0.1, 0.15, 0.1, 0.06, 0.2, 0.04, 0.16, 0.22, 0.27, 0.16, 0.42, 0.64, 0.3, 0.68, 0.11, 0.87, 0.49, 0.16, 0.41, 0.32, 0.15, 0.92, 0.64, 0.64, 0.6, 0.65, 0.08, 0.55, 0.45, 0.34, 0.16, 0.38, 0.49, 0.32, 0.11, 0.42, 0.36, 0.19, 0.05, 0.39, 0.13, 0.25, 0.73, 0.31, 0.48, 0.23, 0.05, 0.41, 0.21, 0.33, 0.45, 0.36, 0.17, 0.27, 0.23, 0.37, 0.37, 0.71, 0.3, 0.11, 0.11, 0.36, 0.17, 0.61, 0.33, 0.05, 0.14, 0.29, 0.31, 0.16, 0.07, 0.08, 0.13, 0.5, 0.07, 0.3, 0.07, 0.53, 0.13, 0.18, 0.25, 0.31, 0.27, 0.48, 0.38, 0.14, 0.31, 0.61, 0.27, 0.1, 0.29, 0.33, 0.28, 0.08, 0.35, 0.19, 0.06, 0.51, 0.61, 0.08, 0.53, 0.04, 0.43, 0.64, 0.49, 0.11, 0.17, 0.21, 0.19, 0.27, 0.35, 0.2, 0.13, 0.08, 0.54, 0.54, 0.75, 0.28, 0.39, 0.62, 0.18, 0.84, 0.47, 0.39, 0.43, 0.57, 0.32, 0.21, 0.08, 0.9, 0.83, 0.6, 0.55, 0.31, 0.42, 0.5, 0.04, 0.55, 0.21, 0.38, 0.66, 0.74, 0.6, 0.1, 0.36, 0.13, 0.62, 0.21, 0.35, 0.73, 0.65, 0.26, 0.74, 0.16, 0.18, 0.35, 0.57, 0.4, 0.3, 0.35, 0.37, 0.18, 0.36, 0.6, 0.32, 0.12, 0.99, 0.85, 0.56, 0.64, 0.59, 0.34, 0.24, 0.38, 0.19, 0.14, 0.68, 0.3, 0.49, 0.17, 0.49, 0.42, 0.55, 0.35, 0.28, 0.53, 0.41, 0.52, 0.36, 0.28, 0.67, 0.97, 0.73, 0.92, 0.44, 0.95, 0.59, 0.36, 0.58, 0.31, 0.34, 0.26, 0.73, 0.72, 0.47, 0.59, 0.14, 0.21, 0.34, 0.49, 0.52, 0.52, 0.54, 0.32, 0.75, 0.4, 0.37, 0.55, 0.96, 0.17, 0.21, 0.77, 0.48, 0.21, 0.29, 0.67, 0.87, 0.54, 0.06, 0.79, 0.9, 0.35, 0.64, 0.12, 0.38, 0.07, 0.57, 0.79, 0.54, 0.57, 0.24, 0.44, 0.66, 0.62, 0.44, 0.68, 0.85, 0.56, 0.38, 0.71, 0.18, 0.25, 0.63, 0.43, 0.98, 0.26, 0.81, 0.5, 0.76, 0.73, 0.93, 0.75, 0.23, 0.7, 0.5, 0.68, 0.69, 0.75, 0.06, 0.86, 0.7, 0.51, 0.54, 0.14, 0.14, 0.42, 0.22, 0.37, 0.29, 0.53, 0.11, 0.04, 0.53, 0.06, 0.27, 0.44, 0.77, 0.69, 0.22, 0.36, 0.15, 0.12, 0.06, 0.16, 0.17, 0.51, 0.31, 0.52, 0.13, 0.15, 0.15, 0.12, 0.43, 0.47, 0.72, 0.14, 0.18, 0.35, 0.72, 0.1, 0.23, 0.05, 0.08, 0.23, 0.21, 0.08, 0.08, 0.07, 0.64, 0.07, 0.11, 0.18, 0.06, 0.12, 0.15, 0.3, 0.52, 0.13, 0.57, 0.79, 0.13, 0.85, 0.86, 0.26, 0.55, 0.36, 0.67, 0.43, 0.47, 0.83, 0.36, 0.22, 0.16, 0.49, 0.22, 0.41, 0.13, 0.6, 0.41, 0.68, 0.45, 0.52, 0.06, 0.2, 0.48, 0.21, 0.65, 0.2, 0.89, 0.11, 0.31, 0.43, 0.82, 0.02, 0.31, 0.84, 0.55, 0.48, 0.12, 0.25, 0.84, 0.72, 0.33, 0.59, 0.23, 0.37, 0.29, 0.6, 0.49, 0.28, 0.29, 0.11, 0.52, 0.37, 0.48, 0.33, 0.48, 0.24, 0.11, 0.22, 0.12, 0.09, 0.09, 0.03, 0.2, 0.2, 0.26, 0.17, 0.1, 0.1, 0.43, 0.08, 0.58, 0.35, 0.69, 0.21, 0.27, 0.22, 0.68, 0.21, 0.34, 0.11, 0.44, 0.31, 0.24, 0.35, 0.16, 0.64, 0.06, 0.04, 0.1, 0.73, 0.79, 0.38, 0.08, 0.06, 0.39, 0.13, 0.65, 0.09, 0.37, 0.24, 0.39, 0.14, 0.2, 0.14, 0.16, 0.36, 0.74, 0.12, 0.27, 0.57, 0.24, 0.31, 0.13, 0.26, 0.05, 0.45, 0.4, 0.33, 0.2, 0.52, 0.67, 0.02, 0.18, 0.45, 0.34, 0.3, 0.05, 0.17, 0.07, 0.1, 0.31, 0.48, 0.15, 0.14, 0.27, 0.46, 0.36, 0.47, 0.58, 0.13, 0.11, 0.25, 0.37, 0.34, 0.58, 0.3, 0.13, 0.35, 0.15, 0.54, 0.58, 0.89, 0.4, 0.13, 0.06, 0.27, 0.54, 0.09, 0.62, 0.47, 0.12, 0.39, 0.36, 0.96, 0.61, 0.4, 0.26, 0.04, 0.65, 0.4, 0.57, 0.71, 0.4, 0.68, 0.44, 0.09, 0.32, 0.07, 0.08, 0.31, 0.72, 0.33, 0.28, 0.52, 0.65, 0.55, 0.32, 0.05, 0.05, 0.12, 0.51, 0.46, 0.61, 0.5, 0.72, 0.6, 0.28, 0.17, 0.07, 0.07, 0.43, 0.14, 0.32, 0.16, 0.48, 0.81, 0.68, 0.29, 0.17, 0.52, 0.98, 0.44, 0.33, 0.26, 0.54, 0.17, 0.35, 0.15, 0.19, 0.53, 0.57, 0.2, 0.4, 0.15, 0.36, 0.03, 0.03, 0.08, 0.22, 0.21, 0.82, 0.42, 0.97, 0.33, 0.66, 0.99, 0.82, 0.11, 0.96, 0.71, 0.93, 0.13, 0.06, 0.37, 0.48, 0.36, 0.15, 0.17, 0.24, 0.37, 0.05, 0.05, 0.18, 0.15, 0.14, 0.09, 0.36, 0.38, 0.32, 0.37, 0.18, 0.72, 0.08, 0.18, 0.06, 0.49, 0.37, 0.08, 0.09, 0.03, 0.13, 0.26, 0.3, 0.44, 0.39, 0.51, 0.18, 0.06, 0.21, 0.15, 0.46, 0.58, 0.19, 0.02, 0.05, 0.11, 0.15, 0.08, 0.06, 0.02, 0.01, 0.15, 0.02, 0.04, 0.06, 0.25, 0.57, 0.11, 0.07, 0.05, 0.68, 0.42, 0.5, 0.31, 0.61, 0.22, 0.24, 0.22, 0.2, 0.4, 0.19, 0.19, 0.29, 0.45, 0.26, 0.66, 0.18, 0.03, 0.16, 0.61, 0.19, 0.59, 0.28, 0.48, 0.29, 0.21, 0.4, 0.32, 0.16, 0.26, 0.21, 0.26, 0.57, 0.07, 0.58, 0.37, 0.12, 0.07, 0.3, 0.5, 0.41, 0.47, 0.72, 0.13, 0.04, 0.17, 0.25, 0.06, 0.16, 0.11, 0.18, 0.16, 0.03, 0.05, 0.07, 0.15, 0.15, 0.16, 0.07, 0.19, 0.21, 0.27, 0.07, 0.18, 0.23, 0.02, 0.21, 0.03, 0.15, 0.11, 0.05, 0.05, 0.38, 0.33, 0.08, 0.2, 0.16, 0.06, 0.14, 0.12, 0.1, 0.24, 0.13, 0.34, 0.21, 0.41, 0.54, 0.59, 0.73, 0.65, 0.26, 0.39, 0.21, 0.32, 0.02, 0.17, 0.54, 0.55, 0.53, 0.44, 0.59, 0.67, 0.3, 0.27, 0.32, 0.44, 0.53, 0.14, 0.69, 0.62, 0.05, 0.57, 0.22, 0.47, 0.45, 0.23, 0.13, 0.18, 0.64, 0.07, 0.46, 0.65, 0.11, 0.77, 0.31, 0.7, 0.51, 0.54, 0.79, 0.26, 0.11, 0.18, 0.16, 0.25, 0.22, 0.1, 0.16, 0.14, 0.14, 0.07, 0.43, 0.44, 0.06, 0.08, 0.21, 0.5, 0.41, 0.26, 0.11, 0.11, 0.74, 0.34, 0.21, 0.18, 0.25, 0.74, 0.43, 0.57, 0.23, 0.29, 0.46, 0.13, 0.3, 0.22, 0.54, 0.51, 0.67, 0.15, 0.66, 0.68, 0.22, 0.29, 0.4, 0.24, 0.11, 0.5, 0.11, 0.11, 0.2, 0.37, 0.09, 0.95, 0.66, 0.36, 0.05, 0.04, 0.18, 0.16, 0.99, 0.21, 0.64, 0.41, 0.18, 0.16, 0.15, 0.09, 0.03, 0.74, 0.48, 0.35, 0.18, 0.42, 0.25, 0.1, 0.84, 0.04, 0.23, 0.16, 0.84, 0.28, 0.76, 0.53, 0.33, 0.19, 0.45, 0.52, 0.82, 0.05, 0.73, 0.71, 0.14, 0.22, 0.55, 0.38, 0.3, 0.57, 0.16, 0.03, 0.54, 0.56, 0.37, 0.08, 0.28, 0.12, 0.1, 0.14, 0.11, 0.06, 0.45, 0.13, 0.13, 0.27, 0.06, 0.12, 0.03, 0.02, 0.11, 0.09, 0.06, 0.05, 0.05, 0.34, 0.06, 0.46, 0.5, 0.24, 0.11, 0.34, 0.88, 0.47, 0.39, 0.31, 0.34, 0.13, 0.09, 0.31, 0.09, 0.09, 0.03, 0.17, 0.04, 0.07, 0.15, 0.06, 0.28, 0.04, 0.02, 0.12, 0.08, 0.17, 0.05, 0.11, 0.1, 0.22, 0.1, 0.07, 0.07, 0.27, 0.29, 0.21, 0.42, 0.13, 0.54, 0.44, 0.56, 0.04, 0.51, 0.87, 0.6, 0.04, 0.59, 0.1, 0.07, 0.46, 0.47, 0.46, 0.79, 0.17, 0.36, 0.17, 0.43, 0.07, 0.35, 0.33, 0.06, 0.19, 0.29, 0.21, 0.17, 0.15, 0.09, 0.13, 0.07, 0.36, 0.46, 0.19, 0.03, 0.37, 0.51, 0.32, 0.18, 0.24, 0.19, 0.06, 0.35, 0.13, 0.77, 0.11, 0.11, 0.09, 0.26, 0.05, 0.43, 0.23, 0.27, 0.43, 0.22, 0.18, 0.03, 0.05, 0.28, 0.06, 0.04, 0.41, 0.08, 0.18, 0.1, 0.2, 0.1, 0.31, 0.03, 0.04, 0.16, 0.22, 0.1, 0.13, 0.23, 0.1, 0.16, 0.1, 0.05, 0.18, 0.29, 0.08, 0.06, 0.3, 0.07, 0.15, 0.1, 0.22, 0.11, 0.06, 0.08, 0.1, 0.1, 0.16, 0.3, 0.24, 0.18, 0.18, 0.15, 0.15, 0.34, 0.06, 0.19, 0.24, 0.25, 0.06, 0.64, 0.09, 0.31, 0.2, 0.38, 0.93, 0.14, 0.26, 0.25, 0.34, 0.4, 0.19, 0.12, 0.49, 0.26, 0.29, 0.18, 0.56, 0.47, 0.88, 0.44, 0.46, 0.32, 0.43, 0.43, 0.52, 0.51, 0.22, 0.25, 0.55, 0.28, 0.56, 0.49, 0.6, 0.17, 0.44, 0.17, 0.37, 0.52, 0.08, 0.47, 0.39, 0.63, 0.7, 0.17, 0.54, 0.25, 0.56, 0.13, 0.3, 0.23, 0.11, 0.48, 0.58, 0.31, 0.22, 0.16, 0.55, 0.13, 0.66, 0.31, 0.47, 0.46, 0.67, 0.39, 0.25, 0.56, 0.51, 0.44, 0.58, 0.48, 0.38, 0.72, 0.38, 0.26, 0.49, 0.41, 0.23, 0.2, 0.22, 0.21, 0.16, 0.04, 0.11, 0.23, 0.7, 0.22, 0.22, 0.38, 0.33, 0.87, 0.04, 0.2, 0.45, 0.77, 0.61, 0.09, 0.07, 0.3, 0.51, 0.56, 0.6, 0.26, 0.11, 0.34, 0.42, 0.72, 0.99, 0.62, 0.74, 0.17, 0.45, 0.41, 0.26, 0.2, 0.07, 0.46, 0.24, 0.62, 0.34, 0.08, 0.38, 0.3, 0.84, 0.44, 0.5, 0.12, 0.07, 0.82, 0.4, 0.36, 0.11, 0.71, 0.32, 0.73, 0.12, 0.73, 0.31, 0.12, 0.63, 0.33, 0.08, 0.54, 0.58, 0.13, 0.4, 0.18, 0.49, 0.25, 0.28, 0.1, 0.67, 0.18, 0.17, 0.05, 0.15, 0.22, 0.11, 0.31, 0.1, 0.68, 0.59, 0.38, 0.21, 0.19, 0.05, 0.59, 0.35, 0.12, 0.08, 0.25, 0.63, 0.1, 0.22, 0.19, 0.11, 0.35, 0.04, 0.17, 0.06, 0.04, 0.16, 0.09, 0.21, 0.03, 0.28, 0.46, 0.05, 0.14, 0.5, 0.06, 0.12, 0.32, 0.13, 0.19, 0.64, 0.16, 0.77, 0.39, 0.4, 0.52, 0.08, 0.17, 0.31, 0.64, 0.11, 0.41, 0.16, 0.13, 0.67, 0.19, 0.45, 0.31, 0.47, 0.18, 0.2, 0.66, 0.26, 0.22, 0.76, 0.65, 0.34, 0.19, 0.37, 0.27, 0.77, 0.1, 0.12, 0.07, 0.43, 0.87, 0.31, 0.63, 0.6, 0.05, 0.21, 0.19, 0.4, 0.07, 0.12, 0.2, 0.39, 0.43, 0.16, 0.18, 0.18, 0.17, 0.09, 0.11, 0.32, 0.5, 0.1, 0.09, 0.12, 0.04, 0.09, 0.13, 0.21, 0.67, 0.18, 0.2, 0.07, 0.13, 0.21, 0.09, 0.48, 0.15, 0.04, 0.26, 0.15, 0.26, 0.09, 0.06, 0.06, 0.05, 0.19, 0.34, 0.13, 0.15, 0.23, 0.22, 0.28, 0.1, 0.18, 0.17, 0.2, 0.69, 0.27, 0.55, 0.33, 0.54, 0.39, 0.3, 0.43, 0.08, 0.19, 0.48, 0.06, 0.35, 0.59, 0.42, 0.43, 0.5, 0.5, 0.08, 0.38, 0.02, 0.27, 0.61, 0.13, 0.4, 0.56, 0.85, 0.13, 0.05, 0.43, 0.17, 0.71, 0.11, 0.04, 0.07, 0.65, 0.38, 0.23, 0.16, 0.17, 0.11, 0.52, 0.05, 0.04, 0.09, 0.03, 0.61, 0.29, 0.02, 0.48, 0.2, 0.12, 0.06, 0.17, 0.42, 0.16, 0.33, 0.39, 0.34, 0.13, 0.69, 0.46, 0.23, 0.11, 0.87, 0.6, 0.16, 0.24, 0.71, 0.67, 0.21, 0.57, 0.64, 0.26, 0.08, 0.68, 0.72, 0.45, 0.42, 0.31, 0.17, 0.81, 0.07, 0.14, 1.0, 0.18, 0.43, 0.2, 0.25, 0.39, 0.45, 0.3, 0.56, 0.68, 0.13, 0.98, 0.74, 0.44, 0.42, 0.59, 0.18, 0.15, 0.09, 0.23, 0.76, 0.4, 0.19, 0.34, 0.31, 0.54, 0.62, 0.79, 0.16, 0.51, 0.36, 0.41, 0.14, 0.25, 0.66, 0.49, 0.69, 0.03, 0.4, 0.29, 1.0, 0.39, 0.6, 0.09, 0.25, 0.46, 0.24, 0.37, 0.06, 0.08, 0.49, 0.72, 0.21, 0.45, 0.29, 0.14, 0.34, 0.75, 0.23, 0.26, 0.29, 0.32, 0.06, 0.03, 0.2, 0.11, 0.74, 0.49, 0.16, 0.16, 0.01, 0.11, 0.17, 0.3, 0.6, 0.22, 0.1, 0.18, 0.07, 0.19, 0.22, 0.3, 0.24, 0.32, 0.2, 0.05, 0.43, 0.08, 0.44, 0.15, 0.13, 0.09, 0.52, 0.44, 0.21, 0.72, 0.13, 0.15, 0.17, 0.13, 0.17, 0.11, 0.02, 0.11, 0.03, 0.1, 0.22, 0.25, 0.4, 0.56, 0.72, 0.29, 0.2, 0.44, 0.11, 0.35, 0.69, 0.39, 0.04, 0.31, 0.28, 0.58, 0.17, 0.09, 0.2, 0.43, 0.34, 0.44, 0.35, 0.46, 0.2, 0.61, 0.37, 0.29, 0.35, 0.35, 0.22, 0.49, 0.67, 0.24, 0.41, 0.32, 0.49, 0.11, 0.26, 0.64, 0.17, 0.21, 0.11, 0.11, 0.06, 0.05, 0.36, 0.2, 0.04, 0.05, 0.07, 0.17, 0.34, 0.17, 0.32, 0.18, 0.09, 0.34, 0.25, 0.05, 0.45, 0.13, 0.15, 0.21, 0.07, 0.09, 0.21, 0.22, 0.14, 0.09, 0.05, 0.1, 0.07, 0.54, 0.08, 0.09, 0.14, 0.19, 0.12, 0.56, 0.2, 0.61, 0.26, 0.04, 0.1, 0.67, 0.24, 0.07, 0.39, 0.49, 0.43, 0.59, 0.52, 0.48, 0.73, 0.26, 0.41, 0.35, 0.36, 0.45, 0.54, 0.12, 0.53, 0.54, 0.27, 0.25, 0.55, 0.28, 0.67, 0.36, 0.44, 0.25, 0.57, 0.21, 0.16, 0.28, 0.53, 0.21, 0.37, 0.36, 0.15, 0.45, 0.3, 0.04, 0.03, 0.32, 0.38, 0.12, 0.16, 0.33, 0.55, 0.16, 0.3, 0.07, 0.34, 0.32, 0.27, 0.33, 0.61, 0.23, 0.28, 0.27, 0.09, 0.07, 0.46, 0.12, 0.34, 0.13, 0.36, 0.48, 0.12, 0.23, 0.57, 0.28, 0.69, 0.39, 0.29, 0.36, 0.43, 0.63, 0.41, 0.46, 0.27, 0.41, 0.33, 0.41, 0.45, 0.38, 0.4, 0.16, 0.14, 0.25, 0.23, 0.33, 0.58, 0.21, 0.46, 0.2, 0.13, 0.47, 0.34, 0.41, 0.27, 0.45, 0.56, 0.99, 0.68, 0.29, 0.37, 0.64, 0.33, 0.18, 0.08, 0.35, 0.73, 0.17, 0.56, 0.42, 0.26, 0.3, 0.1, 0.2, 0.42, 0.82, 0.5, 0.14, 0.42, 0.44, 0.8, 0.69, 0.53, 0.6, 0.28, 0.38, 0.25, 0.55, 0.52, 0.13, 0.47, 0.57, 0.54, 0.2, 0.38, 0.32, 0.46, 0.42, 0.5, 0.34, 0.38, 0.64, 0.42, 0.51, 0.18, 0.31, 0.49, 0.49, 0.39, 0.16, 0.33, 0.74, 0.63, 0.1, 0.12, 0.73, 0.1, 0.12, 0.66, 0.24, 0.44, 0.08, 0.59, 0.65, 0.11, 0.65, 0.46, 0.62, 0.65, 0.56, 0.28, 0.3, 0.2, 0.25, 0.19, 0.29, 0.39, 0.15, 0.04, 0.7, 0.21, 0.05, 0.19, 0.53, 0.17, 0.68, 0.2, 0.05, 0.77, 0.88, 0.18, 0.16, 0.78, 0.26, 0.15, 0.23, 0.35, 0.25, 0.19, 0.34, 0.48, 0.41, 0.17, 0.05, 0.74, 0.42, 0.57, 0.1, 0.17, 0.12, 0.04, 0.12, 0.02, 0.15, 0.12, 0.17, 0.09, 0.1, 0.41, 0.57, 0.29, 0.1, 0.27, 0.18, 0.03, 0.13, 0.25, 0.53, 0.56, 0.7, 0.8, 0.27, 0.54, 0.07, 0.45, 0.59, 0.22, 0.07, 0.09, 0.15, 0.53, 0.44, 0.47, 0.06, 0.62, 0.15, 0.21, 0.25, 0.3, 0.28, 0.48, 0.47, 0.17, 0.87, 0.26, 0.27, 0.21, 0.27, 0.2, 0.74, 0.1, 0.37, 0.21, 0.06, 0.37, 0.03, 0.29, 0.1, 0.09, 0.18, 0.18, 0.22, 0.38, 0.13, 0.04, 0.06, 0.12, 0.28, 0.19, 0.21, 0.14, 0.14, 0.14, 0.05, 0.13, 0.12, 0.15, 0.08, 0.11, 0.27, 0.13, 0.04, 0.14, 0.17, 0.25, 0.19, 0.2, 0.19, 0.1, 0.14, 0.35, 0.09, 0.06, 0.05, 0.06, 0.04, 0.05, 0.19, 0.16, 0.11, 0.1, 0.15, 0.09, 0.16, 0.11, 0.21, 0.35, 0.31, 0.23, 0.72, 0.27, 0.13, 0.31, 0.21, 0.31, 0.15, 0.94, 0.24, 0.37, 0.68, 0.4, 0.44, 0.19, 0.14, 0.28, 0.42, 0.87, 0.58, 0.42, 0.14, 0.42, 0.43, 0.12, 0.26, 0.47, 0.37, 0.31, 0.33, 0.41, 0.54, 0.45, 0.52, 0.25, 0.18, 0.18, 0.45, 0.32, 0.28, 0.61, 0.19, 0.55, 0.21, 0.34, 0.52, 0.5, 0.16, 0.34, 0.53, 0.23, 0.37, 0.2, 0.27, 0.65, 0.41, 0.38, 0.5, 0.23, 0.31, 0.47, 0.07, 0.31, 0.11, 0.24, 0.41, 0.3, 0.65, 0.49, 0.36, 0.39, 0.52, 0.65, 0.96, 0.5, 0.19, 0.88, 0.53, 0.15, 0.57, 0.19, 0.56, 0.15, 0.34, 0.48, 0.12, 0.72, 0.44, 0.41, 0.44, 0.54, 0.33, 0.51, 0.38, 0.7, 0.35, 0.35, 0.09, 0.47, 0.47, 0.07, 0.14, 0.24, 0.26, 0.62, 0.24, 0.13, 0.21, 0.57, 0.07, 0.76, 0.94, 0.68, 0.03, 0.21, 0.24, 0.13, 0.7, 0.52, 0.5, 0.16, 0.47, 0.23, 0.99, 0.23, 0.2, 0.16, 0.4, 0.51, 0.33, 0.2, 0.33, 0.5, 0.99, 0.84, 0.68, 0.41, 0.61, 0.34, 0.61, 0.25, 0.18, 0.32, 0.7, 0.98, 0.19, 0.21, 0.23, 0.19, 0.08, 0.37, 0.11, 0.22, 0.04, 0.37, 0.18, 0.02, 0.62, 0.15, 0.22, 0.11, 0.18, 0.07, 0.13, 0.29, 0.16, 0.06, 0.26, 0.19, 0.1, 0.04, 0.36, 0.13, 0.06, 0.27, 0.34, 0.1, 0.51, 0.14, 0.28, 0.09, 0.24, 0.16, 0.23, 0.14, 0.25, 0.09, 0.27, 0.02, 0.07, 0.13, 0.15, 0.16, 0.02, 0.19, 0.07, 0.04, 0.08, 0.12, 0.04, 0.05, 0.09, 0.09, 0.23, 0.32, 0.68, 0.45, 0.16, 0.24, 0.3, 0.2, 0.11, 0.42, 0.18, 0.34, 0.33, 0.34, 0.07, 0.45, 0.09, 0.26, 0.56, 0.22, 0.34, 0.1, 0.45, 0.19, 0.29, 0.27, 0.42, 0.17, 0.17, 0.26, 0.79, 0.4, 0.31, 0.18, 0.21, 0.46, 0.11, 0.54, 0.33, 0.22, 0.27, 0.82, 0.25, 0.68, 0.62, 0.68, 0.83, 0.65, 0.26, 0.19, 0.51, 0.39, 0.42, 0.61, 0.12, 0.13, 0.55, 0.53, 0.43, 0.23, 0.62, 0.12, 0.07, 0.08, 0.02, 0.29, 0.03, 0.13, 0.12, 0.16, 0.04, 0.29, 0.06, 0.01, 0.1, 0.22, 0.13, 0.03, 0.52, 0.04, 0.07, 0.04, 0.05, 0.11, 0.04, 0.44, 0.08, 0.34, 0.33, 0.04, 0.08, 0.09, 0.01, 0.15, 0.12, 0.25, 0.13, 0.16, 0.12, 0.23, 0.3, 0.04, 0.06, 0.3, 0.06, 0.43, 0.23, 0.06, 0.31, 0.65, 0.39, 0.23, 0.51, 0.31, 0.56, 0.31, 0.26, 0.21, 0.4, 0.37, 0.26, 0.11, 0.06, 0.14, 0.36, 0.18, 0.66, 0.6, 0.42, 0.2, 0.07, 0.38, 0.48, 0.44, 0.61, 0.1, 0.12, 0.35, 0.39, 0.07, 0.41, 0.38, 0.64, 0.61, 0.08, 0.24, 0.46, 0.51, 0.23, 0.45, 0.48, 0.08, 0.43, 0.54, 0.73, 0.16, 0.18, 0.78, 0.06, 0.51, 0.66, 0.52, 0.73, 0.8, 0.18, 0.16, 0.59, 0.34, 0.28, 0.65, 0.23, 0.35, 0.76, 0.6, 0.4, 0.41, 0.19, 0.63, 0.52, 0.35, 0.61, 0.17, 0.19, 0.71, 0.57, 0.25, 0.41, 0.23, 0.61, 0.24, 0.14, 0.26, 0.09, 0.1, 0.58, 0.63, 0.13, 0.61, 0.53, 0.64, 0.17, 0.57, 0.79, 0.46, 0.04, 0.3, 0.33, 0.58, 0.21, 0.66, 0.33, 0.18, 0.63, 0.39, 0.5, 0.18, 0.69, 0.1, 0.58, 0.75, 0.15, 0.52, 0.53, 0.57, 0.48, 0.4, 0.75, 0.47, 0.11, 0.58, 0.84, 0.98, 0.09, 0.26, 0.36, 0.75, 0.26, 0.24, 0.12, 0.34, 0.09, 0.08, 0.32, 0.4, 0.06, 0.23, 0.32, 0.13, 0.16, 0.12, 0.23, 0.03, 0.15, 0.15, 0.38, 0.29, 0.13, 0.05, 0.64, 0.21, 0.08, 0.06, 0.03, 0.07, 0.45, 0.2, 0.34, 0.25, 0.35, 0.12, 0.1, 0.12, 0.13, 0.13, 0.24, 0.21, 0.05, 0.49, 0.35, 0.25, 0.07, 0.19, 0.09, 0.06, 0.04, 0.12, 0.08, 0.1, 0.04, 0.03, 0.07, 0.14, 0.18, 0.06, 0.21, 0.06, 0.14, 0.64, 0.46, 0.52, 0.16, 0.64, 0.3, 0.43, 0.34, 0.47, 0.26, 0.72, 0.18, 0.23, 0.04, 0.27, 0.46, 0.61, 0.26, 0.57, 0.39, 0.24, 0.13, 0.83, 0.12, 0.42, 0.08, 0.24, 0.36, 0.24, 0.36, 0.71, 0.12, 0.64, 0.15, 0.16, 0.23, 0.4, 0.35, 0.14, 0.43, 0.08, 0.23, 0.1, 0.18, 0.1, 0.48, 0.8, 0.33, 0.23, 0.58, 0.18, 0.24, 0.41, 0.21, 0.39, 0.43, 0.5, 0.22, 0.84, 0.22, 0.2, 0.58, 0.09, 0.18, 0.05, 0.1, 0.09, 0.1, 0.12, 0.09, 0.09, 0.4, 0.04, 0.07, 0.14, 0.04, 0.47, 0.02, 0.05, 0.07, 0.18, 0.18, 0.27, 0.06, 0.14, 0.21, 0.07, 0.06, 0.12, 0.5, 0.04, 0.28, 0.13, 0.06, 0.11, 0.05, 0.06, 0.11, 0.1, 0.18, 0.04, 0.11, 0.23]
            weights = torch.tensor(weights, dtype=torch.float32).to(device)

        total_loss = 0
        for i, (output) in tqdm(enumerate(train_loader)):

                feat, labels, product = output

                feat=feat.to(device)
                labels=labels.to(device)
                outputs=self.model(feat)

                cu_selected = cu[product]
                co_selected = co[product]

                loss_per_product = self.pinball_loss(cu_selected, co_selected, labels, outputs)
                
                if self.weight_products:
                    # print(loss_per_product.shape)
                    # print(torch.sum(loss_per_product))
                    loss_per_product = loss_per_product * weights[product].unsqueeze(-1)
                    # print(loss_per_product.shape)
                    # print(torch.sum(loss_per_product))
                    
                loss = torch.mean(loss_per_product)
                
                total_loss += loss.item()

                #backward
                self.optimizer.zero_grad()
                loss.backward()
                self.optimizer.step()

        print("training loss: ", total_loss)   
        self.model.eval()
        self.model.to("cpu")

        return self.model

    def draw_action(self, X):

        # assert X does not contain nan values

        for i in X:
            if np.isnan(i).any():
                print("nan values in input")
                print(i)

                sleep(10)
            break
        
        action = self.predict(X)

        if np.isnan(action).any():
            print("nan values in action")
            print(action)

            for i in X[9]:
                print(i)

            sleep(10)

        return action
    
    # def predict(self, X):
    #     self.model.eval()
    #     self.model.to(self.device)

    #     if len(X.shape) == 1:
    #         X = np.expand_dims(X, axis=0)

    #     # Initialize a list to hold the X_i batches
    #     X_batches = []

    #     # Create batches
    #     for i in range(self.feature_map.shape[1]):
    #         X_i = X[:, self.feature_map[:, i].astype(bool)]
    #         X_batches.append(X_i)

    #     X_batch = np.concatenate([x for x in X_batches], axis=0)
    #     X_batch = torch.from_numpy(X_batch)
    #     X_batch = X_batch.float().to(self.device)

    #     with torch.no_grad():
    #         output = self.model(X_batch)
    #         output = output.cpu().numpy()

    #     # reduce output dimension

    #     output = output.squeeze(1)
        
    #     # print(outputs)

    #     # check if outputs need to got to cpu or handled by mushroomrl

    #     return output

    def predict(self, X):
        self.model.eval()
        self.model.to(self.device)

        X = torch.from_numpy(X).to(self.device)
        X = X.float()

        if X.size(0) > self.max_batch_size:
            batches = []
            for i in range(0, X.size(0), self.max_batch_size):
                batch = X[i:i+self.max_batch_size]
                batches.append(batch)
        else:
            batches = [X]
   
        outputs = []
        for X_batch in batches:
            with torch.no_grad():
                output_part = self.model(X_batch)
                outputs.append(output_part.cpu().numpy())
        
        # Concatenate the results from the smaller batches
        output = np.concatenate(outputs, axis=0)

        # Reduce output dimension
        output = output.squeeze()

        return output

    @staticmethod
    def pinball_loss(cu, co, demand, order_quantity):

        if len(demand.shape)==1:
            demand = demand.unsqueeze(1)
        
        assert demand.shape == order_quantity.shape

        cu = torch.tensor(cu, dtype=torch.float32)
        co = torch.tensor(co, dtype=torch.float32)

        cu = cu.unsqueeze(1)
        co = co.unsqueeze(1)

        cu = cu.to(demand.device)
        co = co.to(demand.device)

        underage_quantity = SGDBase.max_or_zero(demand-order_quantity)
        overage_quantity = SGDBase.max_or_zero(order_quantity-demand)

        assert cu.shape == underage_quantity.shape
        assert co.shape == overage_quantity.shape

        underage=cu*underage_quantity
        overage=co*overage_quantity

        loss=underage+overage
 
        return loss
