# OpenMeter Python SDK

[On PyPI](https://pypi.org/project/openmeter)

This package is generated by `@typespec/http-client-python` with Typespec.

## Prerequisites

- Python 3.9 or later is required to use this package.

## Install

> The Python SDK is in preview mode.

```sh
pip install openmeter
```

## Examples

### Setup

#### Synchronous Client

```python
from openmeter import Client

client = Client(
    endpoint="https://openmeter.cloud",
    token="your-api-token",
)
```

#### Async Client

```python
from openmeter.aio import Client

client = Client(
    endpoint="https://openmeter.cloud",
    token="your-api-token",
)
```

### Ingest an Event

#### Synchronous

```python
import datetime
import uuid

from openmeter.models import Event

# Create an Event instance (following CloudEvents specification)
event = Event(
    id=str(uuid.uuid4()),
    source="my-app",
    specversion="1.0",
    type="prompt",
    subject="customer-1",
    time=datetime.datetime.now(datetime.timezone.utc),
    data={
        "tokens": 100,
        "model": "gpt-4o",
        "type": "input",
    },
)

# Ingest the event
client.events.ingest_event(event)
```

#### Async

```python
import datetime
import uuid
import asyncio

from openmeter.aio import Client
from openmeter.models import Event

async def main():
    async with Client(
        endpoint="https://openmeter.cloud",
        token="your-api-token",
    ) as client:
        # Create an Event instance (following CloudEvents specification)
        event = Event(
            id=str(uuid.uuid4()),
            source="my-app",
            specversion="1.0",
            type="prompt",
            subject="customer-1",
            time=datetime.datetime.now(datetime.timezone.utc),
            data={
                "tokens": 100,
                "model": "gpt-4o",
                "type": "input",
            },
        )

        # Ingest the event
        await client.events.ingest_event(event)

asyncio.run(main())
```

### Query Meter

#### Synchronous

```python
from openmeter.models import MeterQueryResult

# Query total values
r: MeterQueryResult = client.meters.query_json(meter_id_or_slug="tokens_total")
print("Query total values:", r.data[0].value)
```

#### Async

```python
import asyncio

from openmeter.aio import Client
from openmeter.models import MeterQueryResult

async def main():
    async with Client(
        endpoint="https://openmeter.cloud",
        token="your-api-token",
    ) as client:
        # Query total values
        r: MeterQueryResult = await client.meters.query_json(
            meter_id_or_slug="tokens_total"
        )
        print("Query total values:", r.data[0].value)

asyncio.run(main())
```

## Client API Reference

The OpenMeter Python SDK provides a comprehensive client interface organized into logical operation groups. Below is a complete reference of all available methods.

### Overview

| Namespace                    | Operation | Method                                                                                                                         | Description                                        |
| ---------------------------- | --------- | ------------------------------------------------------------------------------------------------------------------------------ | -------------------------------------------------- |
| **Events**                   |           |                                                                                                                                | Track usage by ingesting events                    |
|                              | Create    | `client.events.ingest_event(event)`                                                                                            | Ingest a single event                              |
|                              | Create    | `client.events.ingest_events(events)`                                                                                          | Ingest batch of events                             |
|                              | Create    | `client.events.ingest_events_json(events_json)`                                                                                | Ingest events from JSON                            |
|                              | Read      | `client.events.list(**kwargs)`                                                                                                 | List ingested events with filtering                |
|                              | Read      | `client.events_v2.list(**kwargs)`                                                                                              | List ingested events with advanced filtering (V2)  |
| **Meters**                   |           |                                                                                                                                | Track and aggregate usage data from events         |
|                              | Create    | `client.meters.create(meter)`                                                                                                  | Create a new meter                                 |
|                              | Read      | `client.meters.get(meter_id_or_slug)`                                                                                          | Get a meter by ID or slug                          |
|                              | Read      | `client.meters.list(**kwargs)`                                                                                                 | List all meters                                    |
|                              | Read      | `client.meters.query_json(meter_id_or_slug, **kwargs)`                                                                         | Query usage data in JSON format                    |
|                              | Read      | `client.meters.query_csv(meter_id_or_slug, **kwargs)`                                                                          | Query usage data in CSV format                     |
|                              | Read      | `client.meters.query(meter_id_or_slug, **kwargs)`                                                                              | Query usage data                                   |
|                              | Read      | `client.meters.list_subjects(meter_id_or_slug, **kwargs)`                                                                      | List subjects for a meter                          |
|                              | Read      | `client.meters.list_group_by_values(meter_id_or_slug, **kwargs)`                                                               | List group-by values for a meter                   |
|                              | Update    | `client.meters.update(meter_id_or_slug, meter)`                                                                                | Update a meter by ID or slug                       |
|                              | Delete    | `client.meters.delete(meter_id_or_slug)`                                                                                       | Delete a meter by ID or slug                       |
| **Subjects**                 |           |                                                                                                                                | Manage entities that consume resources             |
|                              | Create    | `client.subjects.upsert(subjects)`                                                                                             | Create or update one or multiple subjects          |
|                              | Read      | `client.subjects.get(subject_id_or_key)`                                                                                       | Get a subject by ID or key                         |
|                              | Read      | `client.subjects.list()`                                                                                                       | List all subjects                                  |
|                              | Delete    | `client.subjects.delete(subject_id_or_key)`                                                                                    | Delete a subject by ID or key                      |
| **Customers**                |           |                                                                                                                                | Manage customer information and lifecycles         |
|                              | Create    | `client.customers.create(customer)`                                                                                            | Create a new customer                              |
|                              | Read      | `client.customers.get(customer_id_or_key, **kwargs)`                                                                           | Get a customer by ID or key                        |
|                              | Read      | `client.customers.list(**kwargs)`                                                                                              | List all customers                                 |
|                              | Read      | `client.customers.list_customer_subscriptions(customer_id_or_key, **kwargs)`                                                   | List customer subscriptions                        |
|                              | Update    | `client.customers.update(customer_id_or_key, customer)`                                                                        | Update a customer                                  |
|                              | Delete    | `client.customers.delete(customer_id_or_key)`                                                                                  | Delete a customer                                  |
| **Customer (Single)**        |           |                                                                                                                                | Customer-specific operations                       |
|                              | Read      | `client.customer.get_customer_access(customer_id_or_key)`                                                                      | Get customer access information                    |
| **Customer Apps**            |           |                                                                                                                                | Manage customer app integrations                   |
|                              | Read      | `client.customer_apps.list_app_data(customer_id_or_key, **kwargs)`                                                             | List app data for a customer                       |
|                              | Update    | `client.customer_apps.upsert_app_data(customer_id_or_key, app_data)`                                                           | Upsert app data for a customer                     |
|                              | Delete    | `client.customer_apps.delete_app_data(customer_id_or_key, app_id)`                                                             | Delete app data for a customer                     |
| **Customer Stripe**          |           |                                                                                                                                | Manage Stripe integration for customers            |
|                              | Read      | `client.customer_stripe.get(customer_id_or_key)`                                                                               | Get Stripe customer data                           |
|                              | Update    | `client.customer_stripe.upsert(customer_id_or_key, data)`                                                                      | Upsert Stripe customer data                        |
|                              | Create    | `client.customer_stripe.create_portal_session(customer_id_or_key, **kwargs)`                                                   | Create a Stripe customer portal session            |
| **Customer Entitlement**     |           |                                                                                                                                | Single customer entitlement operations             |
|                              | Read      | `client.customer_entitlement.get_customer_entitlement_value(customer_id_or_key, **kwargs)`                                     | Get customer entitlement value                     |
| **Customer Overrides**       |           |                                                                                                                                | Manage customer-specific pricing overrides         |
|                              | Read      | `client.customer_overrides.list(customer_id_or_key)`                                                                           | List customer overrides                            |
|                              | Read      | `client.customer_overrides.get(customer_id_or_key, override_id)`                                                               | Get a customer override                            |
|                              | Update    | `client.customer_overrides.upsert(customer_id_or_key, override)`                                                               | Upsert a customer override                         |
|                              | Delete    | `client.customer_overrides.delete(customer_id_or_key, override_id)`                                                            | Delete a customer override                         |
| **Features**                 |           |                                                                                                                                | Define application capabilities and services       |
|                              | Create    | `client.features.create(feature)`                                                                                              | Create a new feature                               |
|                              | Read      | `client.features.get(feature_id)`                                                                                              | Get a feature by ID                                |
|                              | Read      | `client.features.list(**kwargs)`                                                                                               | List all features                                  |
|                              | Delete    | `client.features.delete(feature_id)`                                                                                           | Delete a feature by ID                             |
| **Plans**                    |           |                                                                                                                                | Manage subscription plans and pricing              |
|                              | Create    | `client.plans.create(request)`                                                                                                 | Create a new plan                                  |
|                              | Read      | `client.plans.get(plan_id, **kwargs)`                                                                                          | Get a plan by ID                                   |
|                              | Read      | `client.plans.list(**kwargs)`                                                                                                  | List all plans                                     |
|                              | Update    | `client.plans.update(plan_id, body)`                                                                                           | Update a plan                                      |
|                              | Delete    | `client.plans.delete(plan_id)`                                                                                                 | Delete a plan by ID                                |
|                              | Other     | `client.plans.publish(plan_id)`                                                                                                | Publish a plan                                     |
|                              | Other     | `client.plans.archive(plan_id)`                                                                                                | Archive a plan version                             |
|                              | Other     | `client.plans.next(plan_id_or_key)`                                                                                            | Create new draft plan version                      |
| **Plan Addons**              |           |                                                                                                                                | Manage addons assigned to plans                    |
|                              | Create    | `client.plan_addons.create(plan_id, body)`                                                                                     | Create addon assignment for plan                   |
|                              | Read      | `client.plan_addons.get(plan_id, plan_addon_id)`                                                                               | Get addon assignment for plan                      |
|                              | Read      | `client.plan_addons.list(plan_id, **kwargs)`                                                                                   | List addon assignments for plan                    |
|                              | Update    | `client.plan_addons.update(plan_id, plan_addon_id, body)`                                                                      | Update addon assignment for plan                   |
|                              | Delete    | `client.plan_addons.delete(plan_id, plan_addon_id)`                                                                            | Delete addon assignment for plan                   |
| **Addons**                   |           |                                                                                                                                | Manage standalone addons available across plans    |
|                              | Create    | `client.addons.create(request)`                                                                                                | Create a new addon                                 |
|                              | Read      | `client.addons.get(addon_id, **kwargs)`                                                                                        | Get an addon by ID                                 |
|                              | Read      | `client.addons.list(**kwargs)`                                                                                                 | List all addons                                    |
|                              | Update    | `client.addons.update(addon_id, request)`                                                                                      | Update an addon                                    |
|                              | Delete    | `client.addons.delete(addon_id)`                                                                                               | Delete an addon by ID                              |
|                              | Other     | `client.addons.publish(addon_id)`                                                                                              | Publish an addon                                   |
|                              | Other     | `client.addons.archive(addon_id)`                                                                                              | Archive an addon                                   |
| **Subscriptions**            |           |                                                                                                                                | Manage customer subscriptions                      |
|                              | Create    | `client.subscriptions.create(body)`                                                                                            | Create a new subscription                          |
|                              | Read      | `client.subscriptions.get_expanded(subscription_id, **kwargs)`                                                                 | Get a subscription with expanded details           |
|                              | Update    | `client.subscriptions.edit(subscription_id, body)`                                                                             | Edit a subscription                                |
|                              | Update    | `client.subscriptions.change(subscription_id, body)`                                                                           | Change a subscription                              |
|                              | Update    | `client.subscriptions.migrate(subscription_id, body)`                                                                          | Migrate subscription to a new plan version         |
|                              | Update    | `client.subscriptions.restore(subscription_id)`                                                                                | Restore a canceled subscription                    |
|                              | Delete    | `client.subscriptions.cancel(subscription_id, body)`                                                                           | Cancel a subscription                              |
|                              | Delete    | `client.subscriptions.delete(subscription_id)`                                                                                 | Delete a subscription                              |
|                              | Other     | `client.subscriptions.unschedule_cancelation(subscription_id)`                                                                 | Unschedule a subscription cancelation              |
| **Subscription Addons**      |           |                                                                                                                                | Manage addons on subscriptions                     |
|                              | Create    | `client.subscription_addons.create(subscription_id, body)`                                                                     | Add an addon to a subscription                     |
|                              | Read      | `client.subscription_addons.get(subscription_id, subscription_addon_id)`                                                       | Get a subscription addon                           |
|                              | Read      | `client.subscription_addons.list(subscription_id, **kwargs)`                                                                   | List addons on a subscription                      |
|                              | Update    | `client.subscription_addons.update(subscription_id, subscription_addon_id, body)`                                              | Update a subscription addon                        |
| **Entitlements**             |           |                                                                                                                                | Admin entitlements management                      |
|                              | Read      | `client.entitlements.list(**kwargs)`                                                                                           | List all entitlements (admin)                      |
|                              | Read      | `client.entitlements.get(entitlement_id)`                                                                                      | Get an entitlement by ID                           |
| **Entitlements V2**          |           |                                                                                                                                | V2 Admin entitlements management                   |
|                              | Read      | `client.entitlements_v2.list(**kwargs)`                                                                                        | List all entitlements V2 (admin)                   |
|                              | Read      | `client.entitlements_v2.get(entitlement_id_or_feature_key, **kwargs)`                                                          | Get an entitlement V2 by ID or feature key         |
| **Customer Entitlements V2** |           |                                                                                                                                | Manage customer entitlements (V2)                  |
|                              | Create    | `client.customer_entitlements_v2.post(customer_id_or_key, body)`                                                               | Create a customer entitlement                      |
|                              | Read      | `client.customer_entitlements_v2.list(customer_id_or_key, **kwargs)`                                                           | List customer entitlements                         |
|                              | Read      | `client.customer_entitlements_v2.get(customer_id_or_key, entitlement_id_or_feature_key)`                                       | Get a customer entitlement                         |
|                              | Delete    | `client.customer_entitlements_v2.delete(customer_id_or_key, entitlement_id)`                                                   | Delete a customer entitlement                      |
|                              | Update    | `client.customer_entitlements_v2.override(customer_id_or_key, entitlement_id_or_feature_key, override)`                        | Override a customer entitlement                    |
| **Customer Entitlement V2**  |           |                                                                                                                                | Single customer entitlement operations (V2)        |
|                              | Read      | `client.customer_entitlement_v2.get_grants(customer_id_or_key, entitlement_id_or_feature_key, **kwargs)`                       | List grants for a customer entitlement             |
|                              | Read      | `client.customer_entitlement_v2.get_customer_entitlement_value(customer_id_or_key, entitlement_id_or_feature_key, **kwargs)`   | Get customer entitlement value                     |
|                              | Read      | `client.customer_entitlement_v2.get_customer_entitlement_history(customer_id_or_key, entitlement_id_or_feature_key, **kwargs)` | Get customer entitlement history                   |
|                              | Create    | `client.customer_entitlement_v2.create_customer_entitlement_grant(customer_id_or_key, entitlement_id_or_feature_key, grant)`   | Create a grant for customer entitlement            |
|                              | Update    | `client.customer_entitlement_v2.reset_customer_entitlement(customer_id_or_key, entitlement_id, **kwargs)`                      | Reset customer entitlement usage                   |
| **Grants**                   |           |                                                                                                                                | Admin grants management                            |
|                              | Read      | `client.grants.list(**kwargs)`                                                                                                 | List all grants (admin)                            |
|                              | Delete    | `client.grants.delete(grant_id)`                                                                                               | Delete (void) a grant                              |
| **Grants V2**                |           |                                                                                                                                | V2 Admin grants management                         |
|                              | Read      | `client.grants_v2.list(**kwargs)`                                                                                              | List all grants V2 (admin)                         |
| **Billing Profiles**         |           |                                                                                                                                | Manage billing profiles                            |
|                              | Create    | `client.billing_profiles.create(profile)`                                                                                      | Create a billing profile                           |
|                              | Read      | `client.billing_profiles.get(id)`                                                                                              | Get a billing profile by ID                        |
|                              | Read      | `client.billing_profiles.list(**kwargs)`                                                                                       | List billing profiles                              |
|                              | Update    | `client.billing_profiles.update(id, profile)`                                                                                  | Update a billing profile                           |
|                              | Delete    | `client.billing_profiles.delete(id)`                                                                                           | Delete a billing profile                           |
| **Invoices**                 |           |                                                                                                                                | Manage invoices                                    |
|                              | Read      | `client.invoices.list(**kwargs)`                                                                                               | List invoices                                      |
|                              | Other     | `client.invoices.invoice_pending_lines_action(customer_id, **kwargs)`                                                          | Invoice pending lines for customer                 |
| **Invoice**                  |           |                                                                                                                                | Single invoice operations                          |
|                              | Read      | `client.invoice.get_invoice(id, **kwargs)`                                                                                     | Get an invoice by ID                               |
|                              | Update    | `client.invoice.update_invoice(id, invoice)`                                                                                   | Update an invoice                                  |
|                              | Delete    | `client.invoice.delete_invoice(id)`                                                                                            | Delete an invoice                                  |
|                              | Other     | `client.invoice.advance_action(id)`                                                                                            | Advance invoice to next status                     |
|                              | Other     | `client.invoice.approve_action(id)`                                                                                            | Approve an invoice                                 |
|                              | Other     | `client.invoice.retry_action(id, body)`                                                                                        | Retry advancing invoice after failure              |
|                              | Other     | `client.invoice.void_invoice_action(id)`                                                                                       | Void an invoice                                    |
|                              | Other     | `client.invoice.recalculate_tax_action(id)`                                                                                    | Recalculate invoice tax amounts                    |
|                              | Other     | `client.invoice.snapshot_quantities_action(id)`                                                                                | Snapshot invoice quantities                        |
| **Customer Invoice**         |           |                                                                                                                                | Customer-specific invoice operations               |
|                              | Create    | `client.customer_invoice.create_pending_invoice_line(customer_id, body)`                                                       | Create pending invoice line for customer           |
|                              | Other     | `client.customer_invoice.simulate_invoice(customer_id, **kwargs)`                                                              | Simulate an invoice for a customer                 |
| **Apps**                     |           |                                                                                                                                | Manage integrations and app installations          |
|                              | Read      | `client.apps.list(**kwargs)`                                                                                                   | List installed apps                                |
|                              | Read      | `client.apps.get(id)`                                                                                                          | Get an app by ID                                   |
|                              | Update    | `client.apps.update(id, app)`                                                                                                  | Update an app                                      |
|                              | Delete    | `client.apps.uninstall(id)`                                                                                                    | Uninstall an app                                   |
| **App Stripe**               |           |                                                                                                                                | Stripe app integration                             |
|                              | Create    | `client.app_stripe.webhook(id, body)`                                                                                          | Handle Stripe webhook event                        |
|                              | Update    | `client.app_stripe.update_stripe_api_key(id, request)`                                                                         | Update Stripe API key                              |
|                              | Create    | `client.app_stripe.create_checkout_session(body)`                                                                              | Create Stripe checkout session                     |
| **App Custom Invoicing**     |           |                                                                                                                                | Custom invoicing app integration                   |
|                              | Other     | `client.app_custom_invoicing.draft_syncronized(id, invoice_number, **kwargs)`                                                  | Notify when draft invoice synchronized             |
|                              | Other     | `client.app_custom_invoicing.finalized(id, invoice_number, **kwargs)`                                                          | Notify when invoice finalized                      |
|                              | Other     | `client.app_custom_invoicing.payment_status(id, invoice_number, body)`                                                         | Update invoice payment status                      |
| **Marketplace**              |           |                                                                                                                                | App marketplace operations                         |
|                              | Read      | `client.marketplace.list(**kwargs)`                                                                                            | List marketplace apps                              |
|                              | Read      | `client.marketplace.get(app_type)`                                                                                             | Get marketplace app                                |
|                              | Read      | `client.marketplace.get_o_auth2_install_url(app_type, **kwargs)`                                                               | Get OAuth2 install URL                             |
|                              | Create    | `client.marketplace.authorize_o_auth2_install(app_type, **kwargs)`                                                             | Authorize OAuth2 installation                      |
|                              | Create    | `client.marketplace.install_with_api_key(app_type, body)`                                                                      | Install app with API key                           |
|                              | Create    | `client.marketplace.install(app_type, body)`                                                                                   | Install marketplace app                            |
| **Notification Channels**    |           |                                                                                                                                | Manage notification channels                       |
|                              | Create    | `client.notification_channels.create(channel)`                                                                                 | Create a notification channel                      |
|                              | Read      | `client.notification_channels.get(channel_id)`                                                                                 | Get a notification channel by ID                   |
|                              | Read      | `client.notification_channels.list(**kwargs)`                                                                                  | List notification channels                         |
|                              | Update    | `client.notification_channels.update(channel_id, channel)`                                                                     | Update a notification channel                      |
|                              | Delete    | `client.notification_channels.delete(channel_id)`                                                                              | Delete a notification channel                      |
| **Notification Rules**       |           |                                                                                                                                | Manage notification rules                          |
|                              | Create    | `client.notification_rules.create(rule)`                                                                                       | Create a notification rule                         |
|                              | Read      | `client.notification_rules.get(rule_id)`                                                                                       | Get a notification rule by ID                      |
|                              | Read      | `client.notification_rules.list(**kwargs)`                                                                                     | List notification rules                            |
|                              | Update    | `client.notification_rules.update(rule_id, rule)`                                                                              | Update a notification rule                         |
|                              | Delete    | `client.notification_rules.delete(rule_id)`                                                                                    | Delete a notification rule                         |
|                              | Other     | `client.notification_rules.test(rule_id)`                                                                                      | Test a notification rule                           |
| **Notification Events**      |           |                                                                                                                                | View notification events                           |
|                              | Read      | `client.notification_events.get(event_id)`                                                                                     | Get a notification event by ID                     |
|                              | Read      | `client.notification_events.list(**kwargs)`                                                                                    | List notification events                           |
| **Progress**                 |           |                                                                                                                                | Track long-running operations                      |
|                              | Read      | `client.progress.get_progress(id)`                                                                                             | Get progress of a long-running operation           |
| **Currencies**               |           |                                                                                                                                | Currency information                               |
|                              | Read      | `client.currencies.list_currencies()`                                                                                          | List all supported currencies                      |
| **Debug**                    |           |                                                                                                                                | Debug utilities for monitoring and troubleshooting |
|                              | Read      | `client.debug.metrics()`                                                                                                       | Get event ingestion metrics                        |
