# Scientific Tutorials using HPFRACC

This directory contains a comprehensive series of scientific tutorials demonstrating how to use the HPFRACC (High-Performance Fractional Calculus) library to solve real-world scientific problems. These tutorials are based on cutting-edge research and provide practical implementations of fractional calculus methods in various scientific domains.

## 📚 Tutorial Overview

### Tutorial 01: Anomalous Diffusion Analysis
**File**: `tutorial_01_anomalous_diffusion.py`

**Description**: Comprehensive analysis of anomalous diffusion processes using fractional calculus. This tutorial covers:
- Fractional diffusion equation solutions
- Analysis of subdiffusion and superdiffusion
- Green's function methods for fractional diffusion
- Comparison with analytical solutions
- Applications to biological and physical systems

**Key Features**:
- Analytical solutions to fractional diffusion equations
- Mean Square Displacement (MSD) analysis
- Green's function computation for diffusion, wave, and advection equations
- Performance benchmarking and validation
- Real-world applications in physics and biology

**References**:
- Metzler, R., & Klafter, J. (2000). The random walk's guide to anomalous diffusion
- Richardson, L. F. (1928). Atmospheric diffusion shown on a distance-neighbour graph
- Barkai, E., et al. (2000). From continuous time random walks to the fractional Fokker-Planck equation

### Tutorial 02: EEG Signal Analysis using Fractional Calculus
**File**: `tutorial_02_eeg_fractional_analysis.py`

**Description**: Advanced EEG signal analysis using fractional calculus methods for understanding neural dynamics and cognitive states. This tutorial covers:
- Fractional state space reconstruction for EEG signals
- Long-range dependence analysis in neural oscillations
- Memory characterization in neural dynamics
- Feature extraction for non-stationary EEG
- Applications to cognitive state classification

**Key Features**:
- Fractional-Order State Space (FOSS) reconstruction
- Hurst exponent estimation using R/S and DFA methods
- Fractal dimension computation
- Comprehensive feature extraction
- Cognitive state classification
- Real-time EEG analysis capabilities

**References**:
- Xie, Y., et al. (2024). Fractional-Order State Space (FOSS) reconstruction method
- Becker, R., et al. (2018). Alpha oscillations actively modulate long-range dependence
- Allegrini, P., et al. (2010). Spontaneous EEG undergoes rapid transition processes
- Linkenkaer-Hansen, K., et al. (2001). Long-range temporal correlations in brain oscillations
- Ramirez-Arellano, A., et al. (2023). Spatio-temporal fractal dimension analysis for PD detection

### Tutorial 03: Fractional State Space Modeling
**File**: `tutorial_03_fractional_state_space.py`

**Description**: Advanced fractional state space modeling techniques for complex dynamical systems. This tutorial covers:
- Fractional-Order State Space (FOSS) reconstruction
- Multi-span Transition Entropy Component Method (MTECM-FOSS)
- Stability analysis of fractional state space systems
- Parameter estimation for fractional state space models
- Applications to complex dynamical systems

**Key Features**:
- MTECM-FOSS complexity analysis
- Stability analysis with eigenvalue computation
- Parameter estimation using least squares and Kalman filtering
- System simulation and validation
- Applications to Lorenz and other chaotic systems

**References**:
- Xie, Y., et al. (2024). Fractional-Order State Space (FOSS) reconstruction method
- Chen, Y., et al. (2023). FPGA Implementation of Non-Commensurate Fractional-Order State-Space Models
- Wang, Y., et al. (2023). Parameter estimation in fractional-order Hammerstein state space systems
- Busłowicz, M. (2023). Practical stability of discrete fractional-order state space models
- Zhang, Y., et al. (2025). Fractional-order Wiener state space systems

## 🚀 Getting Started

### Prerequisites

1. **Install HPFRACC**:
   ```bash
   pip install hpfracc[ml]
   ```

2. **Install Additional Dependencies**:
   ```bash
   pip install scipy matplotlib scikit-learn torch numpy
   ```

3. **Verify Installation**:
   ```python
   import hpfracc
   print(f"HPFRACC version: {hpfracc.__version__}")
   ```

### Running the Tutorials

Each tutorial can be run independently:

```bash
# Run anomalous diffusion tutorial
python tutorial_01_anomalous_diffusion.py

# Run EEG analysis tutorial
python tutorial_02_eeg_fractional_analysis.py

# Run fractional state space tutorial
python tutorial_03_fractional_state_space.py
```

### Expected Outputs

Each tutorial provides:
- **Comprehensive Analysis**: Detailed mathematical analysis and results
- **Visualization**: Multiple plots showing different aspects of the analysis
- **Performance Metrics**: Timing and accuracy measurements
- **Validation**: Comparison with analytical solutions where available
- **Real-world Applications**: Practical examples and use cases

## 🔬 Scientific Background

### Why Fractional Calculus?

Fractional calculus extends classical calculus to non-integer orders, enabling the modeling of systems with:
- **Memory Effects**: Systems that remember their past states
- **Long-range Dependencies**: Correlations that extend over large time/space scales
- **Anomalous Dynamics**: Behavior that deviates from classical predictions
- **Complex Scaling**: Power-law relationships and fractal structures

### Applications in Science

1. **Physics**: Anomalous diffusion, viscoelastic materials, quantum mechanics
2. **Biology**: Protein dynamics, neural networks, population growth
3. **Engineering**: Control systems, signal processing, materials science
4. **Finance**: Asset price modeling, risk assessment, option pricing
5. **Medicine**: Drug delivery, tissue mechanics, brain dynamics

## 📊 Tutorial Features

### Advanced Mathematical Methods

- **Fractional Derivatives**: Riemann-Liouville, Caputo, Grünwald-Letnikov
- **Fractional Integrals**: Complete implementation with validation
- **Special Functions**: Gamma, Beta, Mittag-Leffler functions
- **Green's Functions**: Analytical solutions for various equations
- **Analytical Methods**: HPM and VIM for solving fractional differential equations

### Performance Optimization

- **GPU Acceleration**: PyTorch and JAX backend support
- **Parallel Processing**: Multi-core CPU optimization
- **Memory Management**: Efficient algorithms for large datasets
- **Real-time Processing**: Optimized for streaming data

### Validation and Testing

- **Analytical Solutions**: Comparison with known exact solutions
- **Convergence Analysis**: Verification of numerical methods
- **Error Estimation**: Comprehensive error analysis
- **Benchmarking**: Performance comparison with standard methods

## 🎯 Use Cases

### Research Applications

1. **Academic Research**: PhD and postdoctoral research in applied mathematics
2. **Industrial R&D**: Development of new materials and processes
3. **Medical Research**: Analysis of biological signals and systems
4. **Financial Modeling**: Risk assessment and portfolio optimization

### Educational Purposes

1. **Graduate Courses**: Advanced mathematics and engineering courses
2. **Workshops**: Hands-on training in fractional calculus
3. **Self-study**: Learning fractional calculus through practical examples
4. **Reference Implementation**: Benchmark for algorithm development

## 📈 Performance Benchmarks

### Computational Performance

- **Fractional Derivatives**: 10,000+ operations/second on CPU, 50,000+ on GPU
- **Fractional Integrals**: 5,000+ operations/second on CPU, 25,000+ on GPU
- **Special Functions**: 100,000+ operations/second for gamma/beta functions
- **Analytical Methods**: HPM and VIM solving complex FDEs in <1 second

### Accuracy Validation

- **Analytical Solutions**: Agreement within 10^-6 for standard test cases
- **Convergence**: Verified convergence for all implemented methods
- **Stability**: Robust performance across parameter ranges
- **Reproducibility**: Deterministic results with proper seeding

## 🔧 Customization

### Extending the Tutorials

Each tutorial is designed to be easily extensible:

```python
# Example: Adding new fractional orders
alpha_values = [0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0]

# Example: Custom feature extraction
def custom_feature_extractor(signal):
    # Add your custom features here
    features = {}
    features['custom_measure'] = compute_custom_measure(signal)
    return features
```

### Integration with Other Libraries

HPFRACC integrates seamlessly with popular scientific computing libraries:

```python
# NumPy integration
import numpy as np
from hpfracc.core.derivatives import create_fractional_derivative

# PyTorch integration
import torch
from hpfracc.ml import FractionalNeuralNetwork

# SciPy integration
from scipy import signal
from hpfracc.special import gamma, beta
```

## 📚 Additional Resources

### Documentation

- **HPFRACC Documentation**: [https://hpfracc.readthedocs.io](https://hpfracc.readthedocs.io)
- **API Reference**: Complete function and class documentation
- **Examples Gallery**: Additional code examples and use cases
- **Theory Guide**: Mathematical foundations and derivations

### Research Papers

The tutorials are based on cutting-edge research papers. See the bibliography files:
- `fractional_calculus_bibliography.bib`: General fractional calculus references
- `fractional_state_space_eeg_bibliography.bib`: EEG and state space specific references

### Community

- **GitHub Repository**: [https://github.com/dave2k77/fractional_calculus_library](https://github.com/dave2k77/fractional_calculus_library)
- **Issue Tracker**: Report bugs and request features
- **Discussions**: Community forum for questions and ideas
- **Contributions**: Guidelines for contributing to the project

## 🎓 Academic Use

### Citation

If you use these tutorials in your research, please cite:

```bibtex
@software{hpfracc2025,
  title={HPFRACC: High-Performance Fractional Calculus Library with Machine Learning Integration},
  author={Chin, Davian R.},
  year={2025},
  url={https://github.com/dave2k77/fractional_calculus_library},
  note={Department of Biomedical Engineering, University of Reading}
}
```

### Collaboration

For academic collaborations and research partnerships:
- **Email**: d.r.chin@pgr.reading.ac.uk
- **Institution**: Department of Biomedical Engineering, University of Reading
- **Research Areas**: Fractional calculus, machine learning, biomedical engineering

## 🔮 Future Developments

### Planned Tutorials

1. **Tutorial 04**: Fractional Control Systems
2. **Tutorial 05**: Fractional Image Processing
3. **Tutorial 06**: Fractional Financial Modeling
4. **Tutorial 07**: Fractional Materials Science
5. **Tutorial 08**: Fractional Quantum Mechanics

### Advanced Features

- **Distributed Computing**: Multi-node computation support
- **Real-time Processing**: Streaming data analysis
- **Cloud Integration**: AWS, Azure, and GCP deployment
- **Interactive Notebooks**: Jupyter notebook versions
- **Web Interface**: Browser-based analysis tools

---

**HPFRACC Scientific Tutorials** - *Empowering Research with High-Performance Fractional Calculus*

*Last Updated: January 2025*
*Version: 1.2.0*
