#!/usr/bin/env python3
"""
Tests for FFT-based fractional methods.

Tests the AdvancedFFTMethods class and its various methods.
"""

import pytest
import numpy as np
from hpfracc.algorithms.optimized_methods import AdvancedFFTMethods


class TestAdvancedFFTMethods:
    """Test AdvancedFFTMethods class."""

    def test_advanced_fft_methods_creation(self):
        """Test creating AdvancedFFTMethods instances."""
        # Test with spectral method
        fft_spectral = AdvancedFFTMethods(method="spectral")
        assert fft_spectral.method == "spectral"

        # Test with fractional Fourier method
        fft_frft = AdvancedFFTMethods(method="fractional_fourier")
        assert fft_frft.method == "fractional_fourier"

    def test_advanced_fft_methods_validation(self):
        """Test AdvancedFFTMethods validation."""
        # Test valid methods
        AdvancedFFTMethods("spectral")
        AdvancedFFTMethods("fractional_fourier")
        AdvancedFFTMethods("wavelet")

        # Test invalid method
        with pytest.raises(ValueError):
            AdvancedFFTMethods("invalid_method")

    def test_advanced_fft_methods_compute_derivative_scalar(self):
        """Test computing FFT derivative for scalar input."""
        fft = AdvancedFFTMethods(method="spectral")

        # Test with array input (FFT methods expect arrays)
        t = np.array([1.0])
        f = np.array([1.0])
        alpha = 0.5
        h = 0.1

        result = fft.compute_derivative(f, t, alpha, h)
        assert isinstance(result, np.ndarray)
        assert len(result) == 1
        assert not np.isnan(result[0])
        assert not np.isinf(result[0])

    def test_advanced_fft_methods_compute_derivative_array(self):
        """Test computing FFT derivative for array input."""
        fft = AdvancedFFTMethods(method="spectral")

        # Test with array function values
        t = np.linspace(0.1, 2.0, 50)
        f = t  # Simple linear function
        alpha = 0.5
        h = t[1] - t[0]

        result = fft.compute_derivative(f, t, alpha, h)
        assert isinstance(result, np.ndarray)
        assert result.shape == t.shape
        assert not np.any(np.isnan(result))
        assert not np.any(np.isinf(result))

    def test_advanced_fft_methods_different_methods(self):
        """Test different computation methods."""
        alpha = 0.5
        t = np.linspace(0.1, 2.0, 50)
        f = t**2  # Quadratic function

        # Test spectral method
        fft_spectral = AdvancedFFTMethods(method="spectral")
        h = t[1] - t[0]
        result_spectral = fft_spectral.compute_derivative(f, t, alpha, h)

        # Test fractional Fourier method
        fft_frft = AdvancedFFTMethods(method="fractional_fourier")
        result_frft = fft_frft.compute_derivative(f, t, alpha, h)

        # Results should be different but both valid
        assert not np.allclose(result_spectral, result_frft)
        assert not np.any(np.isnan(result_spectral))
        assert not np.any(np.isnan(result_frft))

    def test_advanced_fft_methods_analytical_comparison(self):
        """Test against known analytical results."""
        # For f(t) = t, the fractional derivative of order α is:
        # D^α f(t) = t^(1-α) / Γ(2-α)
        from scipy.special import gamma

        alpha = 0.5
        t = np.linspace(0.1, 2.0, 50)
        f = t

        fft = AdvancedFFTMethods(method="spectral")
        h = t[1] - t[0]
        numerical = fft.compute_derivative(f, t, alpha, h)

        # Analytical solution
        analytical = t ** (1 - alpha) / gamma(2 - alpha)

        # Check that numerical result is reasonable
        # (exact match not expected due to discretization)
        # FFT methods are experimental and may not match analytical solutions exactly
        # Just check that the result is finite and has reasonable magnitude
        assert not np.any(np.isnan(numerical))
        assert not np.any(np.isinf(numerical))
        assert np.any(np.abs(numerical) > 0)  # At least some non-zero values

    def test_advanced_fft_methods_edge_cases(self):
        """Test edge cases and boundary conditions."""
        fft = AdvancedFFTMethods(method="spectral")

        # Test with zero function
        t = np.linspace(0.1, 2.0, 10)
        f = np.zeros_like(t)
        alpha = 0.5
        h = t[1] - t[0]

        result = fft.compute_derivative(f, t, alpha, h)
        assert np.allclose(result, 0, atol=1e-10)

        # Test with constant function
        f = np.ones_like(t)
        result = fft.compute_derivative(f, t, alpha, h)
        # FFT derivative of constant should be zero
        assert np.allclose(result, 0, atol=1e-10)

    def test_advanced_fft_methods_performance(self):
        """Test performance characteristics."""
        fft = AdvancedFFTMethods(method="spectral")

        # Test with larger arrays
        t = np.linspace(0.1, 10.0, 1000)
        f = t**3
        alpha = 0.5
        h = t[1] - t[0]

        result = fft.compute_derivative(f, t, alpha, h)
        assert isinstance(result, np.ndarray)
        assert result.shape == t.shape
        assert not np.any(np.isnan(result))
        assert not np.any(np.isinf(result))

    def test_advanced_fft_methods_method_consistency(self):
        """Test that different methods give consistent results for same input."""
        alpha = 0.5
        t = np.linspace(0.1, 2.0, 50)
        f = t**2

        # Test both methods
        fft_spectral = AdvancedFFTMethods(method="spectral")
        fft_frft = AdvancedFFTMethods(method="fractional_fourier")
        h = t[1] - t[0]

        result_spectral = fft_spectral.compute_derivative(f, t, alpha, h)
        result_frft = fft_frft.compute_derivative(f, t, alpha, h)

        # Both should be finite and valid
        assert not np.any(np.isnan(result_spectral))
        assert not np.any(np.isnan(result_frft))
        assert not np.any(np.isinf(result_spectral))
        assert not np.any(np.isinf(result_frft))

        # Results should be different (different methods)
        assert not np.allclose(result_spectral, result_frft)

    def test_advanced_fft_methods_alpha_validation(self):
        """Test alpha parameter validation."""
        # Test valid alpha values
        for alpha in [0.1, 0.5, 1.0, 1.5, 2.0]:
            fft = AdvancedFFTMethods(method="spectral")
            t = np.linspace(0.1, 2.0, 10)
            f = t**2
            h = t[1] - t[0]
            result = fft.compute_derivative(f, t, alpha, h)
            assert isinstance(result, np.ndarray)
            assert not np.any(np.isnan(result))

        # Test invalid alpha values
        fft = AdvancedFFTMethods(method="spectral")
        t = np.linspace(0.1, 2.0, 10)
        f = t**2

        h = t[1] - t[0]
        with pytest.raises(ValueError):
            fft.compute_derivative(f, t, -0.1, h)

        with pytest.raises(ValueError):
            fft.compute_derivative(f, t, 0.0, h)

    def test_advanced_fft_methods_input_validation(self):
        """Test input validation."""
        fft = AdvancedFFTMethods(method="spectral")
        t = np.linspace(0.1, 2.0, 10)
        f = t**2
        alpha = 0.5

        # Test with mismatched array lengths
        f_wrong = t[:-1]  # One element shorter
        h = t[1] - t[0]
        with pytest.raises(ValueError):
            fft.compute_derivative(f_wrong, t, alpha, h)

        # Test with empty arrays
        with pytest.raises(ValueError):
            fft.compute_derivative(np.array([]), np.array([]), alpha, h)

    def test_advanced_fft_methods_spectral_optimization(self):
        """Test spectral method optimization."""
        alpha = 0.5
        t = np.linspace(0.1, 2.0, 100)
        f = t**2

        # Test spectral method specifically
        fft_spectral = AdvancedFFTMethods(method="spectral")
        h = t[1] - t[0]
        result_spectral = fft_spectral.compute_derivative(f, t, alpha, h)

        assert isinstance(result_spectral, np.ndarray)
        assert result_spectral.shape == t.shape
        assert not np.any(np.isnan(result_spectral))
        assert not np.any(np.isinf(result_spectral))

    def test_advanced_fft_methods_convolution_optimization(self):
        """Test convolution method optimization."""
        alpha = 0.5
        t = np.linspace(0.1, 2.0, 100)
        f = t**2

        # Test wavelet method specifically
        fft_wavelet = AdvancedFFTMethods(method="wavelet")
        h = t[1] - t[0]
        result_wavelet = fft_wavelet.compute_derivative(f, t, alpha, h)

        assert isinstance(result_wavelet, np.ndarray)
        assert result_wavelet.shape == t.shape
        assert not np.any(np.isnan(result_wavelet))
        assert not np.any(np.isinf(result_wavelet))

    def test_advanced_fft_methods_convergence(self):
        """Test convergence with increasing array size."""
        alpha = 0.5
        t_max = 1.0

        # Test with different grid sizes
        grid_sizes = [50, 100, 200]
        results = []

        for N in grid_sizes:
            t = np.linspace(0.1, t_max, N)
            f = t

            fft = AdvancedFFTMethods(method="spectral")
            h = t[1] - t[0]
            result = fft.compute_derivative(f, t, alpha, h)
            results.append(result[-1])  # Take last point

        # Results should converge (get more stable)
        assert len(results) == len(grid_sizes)
        assert all(not np.isnan(r) for r in results)

    def test_advanced_fft_methods_power_of_two_optimization(self):
        """Test power-of-two optimization for FFT."""
        alpha = 0.5

        # Test with power-of-two sizes
        for N in [64, 128, 256]:
            t = np.linspace(0.1, 2.0, N)
            f = t**2

            fft = AdvancedFFTMethods(method="spectral")
            h = t[1] - t[0]
            result = fft.compute_derivative(f, t, alpha, h)

            assert isinstance(result, np.ndarray)
            assert result.shape == t.shape
            assert not np.any(np.isnan(result))
            assert not np.any(np.isinf(result))

    def test_advanced_fft_methods_complex_functions(self):
        """Test with complex functions."""
        fft = AdvancedFFTMethods(method="spectral")

        # Test with exponential function
        t = np.linspace(0.1, 2.0, 50)
        f = np.exp(-t)
        alpha = 0.5

        h = t[1] - t[0]
        result = fft.compute_derivative(f, t, alpha, h)
        assert isinstance(result, np.ndarray)
        assert not np.any(np.isnan(result))

        # Test with trigonometric function
        f_trig = np.sin(t)
        result_trig = fft.compute_derivative(f_trig, t, alpha, h)
        assert isinstance(result_trig, np.ndarray)
        assert not np.any(np.isnan(result_trig))


if __name__ == "__main__":
    pytest.main([__file__])
