"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaRuntime = exports.LambdaFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const case_1 = require("case");
const component_1 = require("../component");
const javascript_1 = require("../javascript");
const source_code_1 = require("../source-code");
const internal_1 = require("./internal");
/**
 * Generates a pre-bundled AWS Lambda function construct from handler code.
 *
 * To use this, create an AWS Lambda handler file under your source tree with
 * the `.lambda.ts` extension and add a `LambdaFunction` component to your
 * typescript project pointing to this entrypoint.
 *
 * This will add a task to your "compile" step which will use `esbuild` to
 * bundle the handler code into the build directory. It will also generate a
 * file `src/foo-function.ts` with a custom AWS construct called `FooFunction`
 * which extends `@aws-cdk/aws-lambda.Function` which is bound to the bundled
 * handle through an asset.
 *
 * @example
 *
 * new LambdaFunction(myProject, {
 *   srcdir: myProject.srcdir,
 *   entrypoint: 'src/foo.lambda.ts',
 * });
 */
class LambdaFunction extends component_1.Component {
    /**
     * Defines a pre-bundled AWS Lambda function construct from handler code.
     *
     * @param project The project to use
     * @param options Options
     */
    constructor(project, options) {
        super(project);
        const cdkDeps = options.cdkDeps;
        const bundler = javascript_1.Bundler.of(project);
        if (!bundler) {
            throw new Error("No bundler found. Please add a Bundler component to your project.");
        }
        const runtime = options.runtime ?? LambdaRuntime.NODEJS_14_X;
        // allow Lambda handler code to import dev-deps since they are only needed
        // during bundling
        const eslint = javascript_1.Eslint.of(project);
        eslint?.allowDevDeps(options.entrypoint);
        const entrypoint = options.entrypoint;
        if (!entrypoint.endsWith(internal_1.TYPESCRIPT_LAMBDA_EXT) &&
            !entrypoint.endsWith(internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT)) {
            throw new Error(`${entrypoint} must have a ${internal_1.TYPESCRIPT_LAMBDA_EXT} or ${internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT} extension`);
        }
        const basePath = path_1.join(path_1.dirname(entrypoint), path_1.basename(entrypoint, options.edgeLambda ? internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT : internal_1.TYPESCRIPT_LAMBDA_EXT));
        const constructFile = options.constructFile ?? `${basePath}-function.ts`;
        if (path_1.extname(constructFile) !== ".ts") {
            throw new Error(`Construct file name "${constructFile}" must have a .ts extension`);
        }
        // type names
        const constructName = options.constructName ?? case_1.pascal(path_1.basename(basePath)) + "Function";
        const propsType = `${constructName}Props`;
        const bundle = bundler.addBundle(entrypoint, {
            target: runtime.esbuildTarget,
            platform: runtime.esbuildPlatform,
            externals: ["aws-sdk"],
            ...options.bundlingOptions,
            tsconfigPath: project?.tsconfigDev?.fileName,
        });
        // calculate the relative path between the directory containing the
        // generated construct source file to the directory containing the bundle
        // index.js by resolving them as absolute paths first.
        // e.g:
        //  - outfileAbs => `/project-outdir/assets/foo/bar/baz/foo-function/index.js`
        //  - constructAbs => `/project-outdir/src/foo/bar/baz/foo-function.ts`
        const outfileAbs = path_1.join(project.outdir, bundle.outfile);
        const constructAbs = path_1.join(project.outdir, constructFile);
        const relativeOutfile = path_1.relative(path_1.dirname(constructAbs), path_1.dirname(outfileAbs));
        const src = new source_code_1.SourceCode(project, constructFile);
        if (src.marker) {
            src.line(`// ${src.marker}`);
        }
        src.line("import * as path from 'path';");
        if (cdkDeps.cdkMajorVersion === 1) {
            if (options.edgeLambda) {
                src.line("import * as cloudfront from '@aws-cdk/aws-cloudfront';");
                cdkDeps.addV1Dependencies("@aws-cdk/aws-cloudfront");
            }
            src.line("import * as lambda from '@aws-cdk/aws-lambda';");
            src.line("import { Construct } from '@aws-cdk/core';");
            cdkDeps.addV1Dependencies("@aws-cdk/aws-lambda");
            cdkDeps.addV1Dependencies("@aws-cdk/core");
        }
        else {
            if (options.edgeLambda) {
                src.line("import * as cloudfront from 'aws-cdk-lib/aws-cloudfront';");
            }
            src.line("import * as lambda from 'aws-cdk-lib/aws-lambda';");
            src.line("import { Construct } from 'constructs';");
        }
        src.line();
        src.line("/**");
        src.line(` * Props for ${constructName}`);
        src.line(" */");
        if (options.edgeLambda) {
            src.open(`export interface ${propsType} extends cloudfront.experimental.EdgeFunctionProps {`);
        }
        else {
            src.open(`export interface ${propsType} extends lambda.FunctionOptions {`);
        }
        src.close("}");
        src.line();
        src.line("/**");
        src.line(` * An AWS Lambda function which executes ${internal_1.convertToPosixPath(basePath)}.`);
        src.line(" */");
        if (options.edgeLambda) {
            src.open(`export class ${constructName} extends cloudfront.experimental.EdgeFunction {`);
        }
        else {
            src.open(`export class ${constructName} extends lambda.Function {`);
        }
        src.open(`constructor(scope: Construct, id: string, props?: ${propsType}) {`);
        src.open("super(scope, id, {");
        src.line(`description: '${internal_1.convertToPosixPath(entrypoint)}',`);
        src.line("...props,");
        src.line(`runtime: new lambda.Runtime('${runtime.functionRuntime}', lambda.RuntimeFamily.NODEJS),`);
        src.line("handler: 'index.handler',");
        src.line(`code: lambda.Code.fromAsset(path.join(__dirname, '${internal_1.convertToPosixPath(relativeOutfile)}')),`);
        src.close("});");
        if ((options.awsSdkConnectionReuse ?? true) && !options.edgeLambda) {
            src.line("this.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });");
        }
        src.close("}");
        src.close("}");
        this.project.logger.verbose(`${basePath}: construct "${constructName}" generated under "${constructFile}"`);
        this.project.logger.verbose(`${basePath}: bundle task "${bundle.bundleTask.name}"`);
        if (bundle.watchTask) {
            this.project.logger.verbose(`${basePath}: bundle watch task "${bundle.watchTask.name}"`);
        }
    }
}
exports.LambdaFunction = LambdaFunction;
_a = JSII_RTTI_SYMBOL_1;
LambdaFunction[_a] = { fqn: "projen.awscdk.LambdaFunction", version: "0.65.85" };
/**
 * The runtime for the AWS Lambda function.
 */
class LambdaRuntime {
    constructor(
    /**
     * The Node.js runtime to use
     */
    functionRuntime, 
    /**
     * The esbuild setting to use.
     */
    esbuildTarget) {
        this.functionRuntime = functionRuntime;
        this.esbuildTarget = esbuildTarget;
        this.esbuildPlatform = "node";
    }
}
exports.LambdaRuntime = LambdaRuntime;
_b = JSII_RTTI_SYMBOL_1;
LambdaRuntime[_b] = { fqn: "projen.awscdk.LambdaRuntime", version: "0.65.85" };
/**
 * Node.js 10.x
 * @deprecated NodeJS10 has been deprecated
 */
LambdaRuntime.NODEJS_10_X = new LambdaRuntime("nodejs10.x", "node10");
/**
 * Node.js 12.x
 */
LambdaRuntime.NODEJS_12_X = new LambdaRuntime("nodejs12.x", "node12");
/**
 * Node.js 14.x
 */
LambdaRuntime.NODEJS_14_X = new LambdaRuntime("nodejs14.x", "node14");
/**
 * Node.js 16.x
 */
LambdaRuntime.NODEJS_16_X = new LambdaRuntime("nodejs16.x", "node16");
/**
 * Node.js 18.x
 */
LambdaRuntime.NODEJS_18_X = new LambdaRuntime("nodejs18.x", "node18");
//# sourceMappingURL=data:application/json;base64,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