"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.cleanup = exports.FILE_MANIFEST = void 0;
const path = require("path");
const fs = require("fs-extra");
const fs_extra_1 = require("fs-extra");
const glob = require("glob");
const common_1 = require("./common");
const logging = require("./logging");
exports.FILE_MANIFEST = `${common_1.PROJEN_DIR}/files.json`;
function cleanup(dir, newFiles, exclude) {
    try {
        const manifestFiles = getFilesFromManifest(dir);
        if (manifestFiles.length > 0) {
            // Use `FILE_MANIFEST` to remove files that are no longer managed by projen
            removeFiles(findOrphanedFiles(dir, manifestFiles, newFiles));
        }
        else {
            // Remove all files managed by projen with legacy logic
            removeFiles(findGeneratedFiles(dir, exclude));
        }
    }
    catch (e) {
        logging.warn(`warning: failed to clean up generated files: ${e.stack}`);
    }
}
exports.cleanup = cleanup;
function removeFiles(files) {
    for (const file of files) {
        fs.removeSync(file);
    }
}
function findGeneratedFiles(dir, exclude) {
    const ignore = [
        ...readGitIgnore(dir),
        "node_modules/**",
        ...exclude,
        ".git/**",
    ];
    const files = glob.sync("**", {
        ignore,
        cwd: dir,
        dot: true,
        nodir: true,
        absolute: true,
    });
    const generated = new Array();
    for (const file of files) {
        const contents = fs.readFileSync(file, "utf-8");
        if (contents.includes(common_1.PROJEN_MARKER)) {
            generated.push(file);
        }
    }
    return generated;
}
function findOrphanedFiles(dir, oldFiles, newFiles) {
    return oldFiles
        .filter((old) => !newFiles.includes(old))
        .map((f) => path.resolve(dir, f));
}
function getFilesFromManifest(dir) {
    try {
        const fileManifestPath = path.resolve(dir, exports.FILE_MANIFEST);
        if (fs_extra_1.existsSync(fileManifestPath)) {
            const fileManifest = JSON.parse(fs_extra_1.readFileSync(fileManifestPath, "utf-8"));
            if (fileManifest.files) {
                return fileManifest.files;
            }
        }
    }
    catch (e) {
        logging.warn(`warning: unable to get files to clean from file manifest: ${e.stack}`);
    }
    return [];
}
function readGitIgnore(dir) {
    const filepath = path.join(dir, ".gitignore");
    if (!fs.pathExistsSync(filepath)) {
        return [];
    }
    return fs
        .readFileSync(filepath, "utf-8")
        .split("\n")
        .filter((x) => x?.trim() !== "")
        .filter((x) => !x.startsWith("#") && !x.startsWith("!"))
        .map((x) => x.replace(/^\//, "")) // remove "/" prefix
        .map((x) => `${x}\n${x}/**`)
        .join("\n")
        .split("\n");
}
//# sourceMappingURL=data:application/json;base64,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