"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const semver = require("semver");
const yargs = require("yargs");
const inventory = require("../../inventory");
const logging = require("../../logging");
const option_hints_1 = require("../../option-hints");
const projects_1 = require("../../projects");
const util_1 = require("../../util");
const macros_1 = require("../macros");
const util_2 = require("../util");
class Command {
    constructor() {
        this.command = "new [PROJECT-TYPE-NAME] [OPTIONS]";
        this.describe = "Creates a new projen project";
    }
    builder(args) {
        args.positional("PROJECT-TYPE-NAME", {
            describe: "optional only when --from is used and there is a single project type in the external module",
            type: "string",
        });
        args.option("synth", {
            type: "boolean",
            default: true,
            desc: "Synthesize after creating .projenrc.js",
        });
        args.option("comments", {
            type: "boolean",
            default: true,
            desc: "Include commented out options in .projenrc.js (use --no-comments to disable)",
        });
        args.option("from", {
            type: "string",
            alias: "f",
            desc: 'External jsii npm module to create project from. Supports any package spec supported by npm (such as "my-pack@^2.0")',
        });
        args.option("git", {
            type: "boolean",
            default: true,
            desc: "Run `git init` and create an initial commit (use --no-git to disable)",
        });
        args.example("projen new awscdk-app-ts", 'Creates a new project of built-in type "awscdk-app-ts"');
        args.example("projen new --from projen-vue@^2", 'Creates a new project from an external module "projen-vue" with the specified version');
        for (const type of inventory.discover()) {
            args.command(type.pjid, type.docs ?? "", {
                builder: (cargs) => {
                    cargs.showHelpOnFail(false);
                    for (const option of type.options ?? []) {
                        if (option.simpleType !== "string" &&
                            option.simpleType !== "number" &&
                            option.simpleType !== "boolean" &&
                            option.kind !== "enum") {
                            continue; // we only support primitive and enum fields as command line options
                        }
                        let desc = [option.docs?.replace(/\ *\.$/, "") ?? ""];
                        const required = !option.optional;
                        let defaultValue;
                        if (option.default && option.default !== "undefined") {
                            if (!required) {
                                // if the field is not required, just describe the default but don't actually assign a value
                                desc.push(`[default: ${option.default
                                    .replace(/^\ *-/, "")
                                    .replace(/\.$/, "")
                                    .trim()}]`);
                            }
                            else {
                                // if the field is required and we have a @default, then assign
                                // the value here so it appears in `--help`
                                defaultValue = renderDefault(process.cwd(), option.default);
                            }
                        }
                        const argType = option.kind === "enum" ? "string" : option.simpleType;
                        cargs.option(option.switch, {
                            group: required ? "Required:" : "Optional:",
                            type: argType,
                            description: desc.join(" "),
                            default: defaultValue,
                            required,
                        });
                    }
                    return cargs;
                },
                handler: (argv) => initProject(process.cwd(), type, argv),
            });
        }
        return args;
    }
    async handler(args) {
        // handle --from which means we want to first install a jsii module and then
        // create a project defined within this module.
        if (args.from) {
            return initProjectFromModule(process.cwd(), args.from, args);
        }
        // project type is defined but was not matched by yargs, so print the list of supported types
        if (args.projectTypeName) {
            console.log(`Invalid project type ${args.projectTypeName}. Supported types:`);
            for (const pjid of inventory.discover().map((x) => x.pjid)) {
                console.log(`  ${pjid}`);
            }
            return;
        }
        // Handles the use case that nothing was specified since PROJECT-TYPE is now an optional positional parameter
        yargs.showHelp();
    }
}
/**
 * Given a value from "@default", processes macros and returns a stringified
 * (quoted) result.
 *
 * @returns a javascript primitive (could be a string, number or boolean)
 */
function renderDefault(cwd, value) {
    return macros_1.tryProcessMacro(cwd, value) ?? JSON.parse(value);
}
/**
 * Converts yargs command line switches to project type props.
 * @param type Project type
 * @param argv Command line switches
 */
function commandLineToProps(cwd, type, argv) {
    const props = {};
    // initialize props with default values
    for (const prop of type.options) {
        if (prop.default && prop.default !== "undefined" && !prop.optional) {
            props[prop.name] = renderDefault(cwd, prop.default);
        }
    }
    for (const [arg, value] of Object.entries(argv)) {
        for (const prop of type.options) {
            if (prop.switch === arg) {
                let curr = props;
                const queue = [...prop.path];
                while (true) {
                    const p = queue.shift();
                    if (!p) {
                        break;
                    }
                    if (queue.length === 0) {
                        curr[p] = value;
                    }
                    else {
                        curr[p] = curr[p] ?? {};
                        curr = curr[p];
                    }
                }
            }
        }
    }
    return props;
}
/**
 * Generates a new project from an external module.
 *
 * @param spec The name of the external module to load
 * @param args Command line arguments (incl. project type)
 */
async function initProjectFromModule(baseDir, spec, args) {
    const projenVersion = args.projenVersion ?? "latest";
    const installCommand = util_2.renderInstallCommand(baseDir, `projen@${projenVersion}`);
    if (args.projenVersion) {
        util_1.exec(installCommand, { cwd: baseDir });
    }
    else {
        // do not overwrite existing installation
        util_1.exec(`npm ls --prefix=${baseDir} --depth=0 --pattern projen || ${installCommand}`, { cwd: baseDir });
    }
    const moduleName = util_2.installPackage(baseDir, spec);
    // Find the just installed package and discover the rest recursively from this package folder
    const moduleDir = path.dirname(require.resolve(`${moduleName}/.jsii`, {
        paths: [baseDir],
    }));
    // Only leave projects from the main (requested) package
    const projects = inventory
        .discover(moduleDir)
        .filter((x) => x.moduleName === moduleName); // Only list project types from the requested 'from' module
    if (projects.length < 1) {
        throw new Error(`No projects found after installing ${spec}. The module must export at least one class which extends projen.Project`);
    }
    const requested = args.projectTypeName;
    const types = projects.map((p) => p.pjid);
    // if user did not specify a project type but the module has more than one, we need them to tell us which one...
    if (!requested && projects.length > 1) {
        throw new Error(`Multiple projects found after installing ${spec}: ${types.join(",")}. Please specify a project name.\nExample: npx projen new --from ${spec} ${types[0]}`);
    }
    // if user did not specify a type (and we know we have only one), the select it. otherwise, search by pjid.
    const type = !requested
        ? projects[0]
        : projects.find((p) => p.pjid === requested);
    if (!type) {
        throw new Error(`Project type ${requested} not found. Found ${types.join(",")}`);
    }
    for (const option of type.options ?? []) {
        if (option.simpleType !== "string" &&
            option.simpleType !== "number" &&
            option.simpleType !== "boolean") {
            continue; // we don't support non-primitive fields as command line options
        }
        if (args[option.name] !== undefined) {
            if (option.simpleType === "number") {
                args[option.name] = parseInt(args[option.name]);
                args[option.switch] = args[option.name];
            }
            else if (option.simpleType === "boolean") {
                const raw = args[option.name];
                const safe = typeof raw === "string" ? util_1.isTruthy(raw) : raw;
                args[option.name] = safe;
                args[option.switch] = safe;
            }
            continue; // do not overwrite passed arguments
        }
        if (option.default && option.default !== "undefined") {
            if (!option.optional) {
                const defaultValue = renderDefault(baseDir, option.default);
                args[option.name] = defaultValue;
                args[option.switch] = defaultValue;
            }
        }
    }
    // include a dev dependency for the external module
    args.devDeps = [spec];
    args["dev-deps"] = [spec];
    await initProject(baseDir, type, args);
}
/**
 * Generates a new project.
 * @param type Project type
 * @param args Command line arguments
 * @param additionalProps Additional parameters to include in .projenrc.js
 */
async function initProject(baseDir, type, args) {
    // convert command line arguments to project props using type information
    const props = commandLineToProps(baseDir, type, args);
    projects_1.Projects.createProject({
        dir: props.outdir ?? baseDir,
        projectFqn: type.fqn,
        projectOptions: props,
        optionHints: args.comments
            ? option_hints_1.InitProjectOptionHints.FEATURED
            : option_hints_1.InitProjectOptionHints.NONE,
        synth: args.synth,
        post: args.post,
    });
    if (fs.existsSync(path.join(baseDir, "package.json")) && args.post) {
        util_1.exec("npm run eslint --if-present", { cwd: baseDir });
    }
    if (args.git) {
        const git = (cmd) => util_1.exec(`git ${cmd}`, { cwd: baseDir });
        const gitversion = util_1.getGitVersion(util_1.execCapture("git --version", { cwd: baseDir }).toString());
        logging.debug("system using git version ", gitversion);
        if (gitversion && semver.gte(gitversion, "2.28.0")) {
            git("init -b main");
            git("add .");
            git('commit --allow-empty -m "chore: project created with projen"');
            logging.debug("default branch name set to main");
        }
        else {
            git("init");
            git("add .");
            git('commit --allow-empty -m "chore: project created with projen"');
            logging.debug("older version of git detected, changed default branch name to main");
            git("branch -M main");
        }
    }
}
module.exports = new Command();
//# sourceMappingURL=data:application/json;base64,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