import { Component } from "./component";
import { Project } from "./project";
/**
 * Props for DockerCompose.
 */
export interface DockerComposeProps {
    /**
     * A name to add to the docker-compose.yml filename.
     * @example 'myname' yields 'docker-compose.myname.yml'
     * @default - no name is added
     */
    readonly nameSuffix?: string;
    /**
     * Docker Compose schema version do be used
     * @default 3.3
     */
    readonly schemaVersion?: string;
    /**
     * Service descriptions.
     */
    readonly services?: Record<string, DockerComposeServiceDescription>;
}
/**
 * Options for port mappings.
 */
export interface DockerComposePortMappingOptions {
    /**
     * Port mapping protocol.
     * @default DockerComposeProtocol.TCP
     */
    readonly protocol?: DockerComposeProtocol;
}
/**
 * Create a docker-compose YAML file.
 */
export declare class DockerCompose extends Component {
    /**
     * Depends on a service name.
     */
    static serviceName(serviceName: string): IDockerComposeServiceName;
    /**
     * Create a port mapping.
     * @param publishedPort Published port number
     * @param targetPort Container's port number
     * @param options Port mapping options
     */
    static portMapping(publishedPort: number, targetPort: number, options?: DockerComposePortMappingOptions): DockerComposeServicePort;
    /**
     * Create a bind volume that binds a host path to the target path in the container.
     * @param sourcePath Host path name
     * @param targetPath Target path name
     */
    static bindVolume(sourcePath: string, targetPath: string): IDockerComposeVolumeBinding;
    /**
     * Create a named volume and mount it to the target path. If you use this
     * named volume in several services, the volume will be shared. In this
     * case, the volume configuration of the first-provided options are used.
     *
     * @param volumeName Name of the volume
     * @param targetPath Target path
     * @param options volume configuration (default: docker compose defaults)
     */
    static namedVolume(volumeName: string, targetPath: string, options?: DockerComposeVolumeConfig): IDockerComposeVolumeBinding;
    private readonly services;
    private readonly version;
    constructor(project: Project, props?: DockerComposeProps);
    /**
     * Add a service to the docker-compose file.
     * @param serviceName name of the service
     * @param description a service description
     */
    addService(serviceName: string, description: DockerComposeServiceDescription): DockerComposeService;
    /**
     * @internal
     */
    _synthesizeDockerCompose(): object;
}
/**
 * An interface providing the name of a docker compose service.
 */
export interface IDockerComposeServiceName {
    /**
     * The name of the docker compose service.
     */
    readonly serviceName: string;
}
/**
 * Description of a docker-compose.yml service.
 */
export interface DockerComposeServiceDescription {
    /**
     * Use a docker image.
     * Note: You must specify either `build` or `image` key.
     * @see imageBuild
     */
    readonly image?: string;
    /**
     * Build a docker image.
     * Note: You must specify either `imageBuild` or `image` key.
     * @see image
     */
    readonly imageBuild?: DockerComposeBuild;
    /**
     * Provide a command to the docker container.
     * @default - use the container's default command
     */
    readonly command?: string[];
    /**
     * Names of other services this service depends on.
     * @default - no dependencies
     */
    readonly dependsOn?: IDockerComposeServiceName[];
    /**
     * Mount some volumes into the service.
     * Use one of the following to create volumes:
     * @see DockerCompose.bindVolume() to mount a host path
     * @see DockerCompose.namedVolume() to create & mount a named volume
     */
    readonly volumes?: IDockerComposeVolumeBinding[];
    /**
     * Map some ports.
     * @default - no ports are mapped
     */
    readonly ports?: DockerComposeServicePort[];
    /**
     * Add environment variables.
     * @default - no environment variables are provided
     */
    readonly environment?: Record<string, string>;
}
/**
 * A docker-compose service.
 */
export declare class DockerComposeService implements IDockerComposeServiceName {
    /**
     * Name of the service.
     */
    readonly serviceName: string;
    /**
     * Docker image.
     */
    readonly image?: string;
    /**
     * Docker image build instructions.
     */
    readonly imageBuild?: DockerComposeBuild;
    /**
     * Command to run in the container.
     */
    readonly command?: string[];
    /**
     * Other services that this service depends on.
     */
    readonly dependsOn: IDockerComposeServiceName[];
    /**
     * Volumes mounted in the container.
     */
    readonly volumes: IDockerComposeVolumeBinding[];
    /**
     * Published ports.
     */
    readonly ports: DockerComposeServicePort[];
    /**
     * Environment variables.
     */
    readonly environment: Record<string, string>;
    constructor(serviceName: string, serviceDescription: DockerComposeServiceDescription);
    /**
     * Add a port mapping
     * @param publishedPort Published port number
     * @param targetPort Container's port number
     * @param options Port mapping options
     */
    addPort(publishedPort: number, targetPort: number, options?: DockerComposePortMappingOptions): void;
    /**
     * Add an environment variable
     * @param name environment variable name
     * @param value value of the environment variable
     */
    addEnvironment(name: string, value: string): void;
    /**
     * Make the service depend on another service.
     * @param serviceName
     */
    addDependsOn(serviceName: IDockerComposeServiceName): void;
    /**
     * Add a volume to the service.
     * @param volume
     */
    addVolume(volume: IDockerComposeVolumeBinding): void;
}
/**
 * A service port mapping
 */
export interface DockerComposeServicePort {
    /**
     * Published port number
     */
    readonly published: number;
    /**
     * Target port number
     */
    readonly target: number;
    /**
     * Network protocol
     */
    readonly protocol: DockerComposeProtocol;
    /**
     * Port mapping mode.
     */
    readonly mode: string;
}
/**
 * Network protocol for port mapping
 */
export declare enum DockerComposeProtocol {
    /**
     * TCP protocol
     */
    TCP = "tcp",
    /**
     * UDP protocol
     */
    UDP = "udp"
}
/**
 * Build arguments for creating a docker image.
 */
export interface DockerComposeBuild {
    /**
     * Docker build context directory.
     */
    readonly context: string;
    /**
     * A dockerfile to build from.
     * @default "Dockerfile"
     */
    readonly dockerfile?: string;
    /**
     * Build args.
     * @default - none are provided
     */
    readonly args?: Record<string, string>;
}
/**
 * Volume configuration
 */
export interface DockerComposeVolumeConfig {
    /**
     * Driver to use for the volume
     * @default - value is not provided
     */
    readonly driver?: string;
    /**
     * Options to provide to the driver.
     */
    readonly driverOpts?: Record<string, string>;
    /**
     * Set to true to indicate that the volume is externally created.
     * @default - unset, indicating that docker-compose creates the volume
     */
    readonly external?: boolean;
    /**
     * Name of the volume for when the volume name isn't going to work in YAML.
     * @default - unset, indicating that docker-compose creates volumes as usual
     */
    readonly name?: string;
}
/**
 * Volume binding information.
 */
export interface IDockerComposeVolumeBinding {
    /**
     * Binds the requested volume to the docker-compose volume configuration and
     * provide mounting instructions for synthesis.
     * @param volumeConfig the volume configuration
     * @returns mounting instructions for the service.
     */
    bind(volumeConfig: IDockerComposeVolumeConfig): DockerComposeVolumeMount;
}
/**
 * Storage for volume configuration.
 */
export interface IDockerComposeVolumeConfig {
    /**
     * Add volume configuration to the repository.
     * @param volumeName
     * @param configuration
     */
    addVolumeConfiguration(volumeName: string, configuration: DockerComposeVolumeConfig): void;
}
/**
 * Service volume mounting information.
 */
export interface DockerComposeVolumeMount {
    /**
     * Type of volume.
     */
    readonly type: string;
    /**
     * Volume source
     */
    readonly source: string;
    /**
     * Volume target
     */
    readonly target: string;
}
