"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkflowActions = void 0;
const GIT_PATCH_FILE = ".repo.patch";
const RUNNER_TEMP = "${{ runner.temp }}";
/**
 * A set of utility functions for creating GitHub actions in workflows.
 */
class WorkflowActions {
    /**
     * Creates a .patch file from the current git diff and uploads it as an
     * artifact. Use `checkoutWithPatch` to download and apply in another job.
     *
     * If a patch was uploaded,
     *
     * @param options Options
     * @returns Job steps
     */
    static createUploadGitPatch(options) {
        const MUTATIONS_FOUND = `steps.${options.stepId}.outputs.${options.outputName}`;
        const steps = [
            {
                id: options.stepId,
                name: "Find mutations",
                run: [
                    "git add .",
                    `git diff --staged --patch --exit-code > ${GIT_PATCH_FILE} || echo "${options.outputName}=true" >> $GITHUB_OUTPUT`,
                ].join("\n"),
            },
            {
                if: MUTATIONS_FOUND,
                name: "Upload patch",
                uses: "actions/upload-artifact@v3",
                with: { name: GIT_PATCH_FILE, path: GIT_PATCH_FILE },
            },
        ];
        if (options.mutationError) {
            steps.push({
                name: "Fail build on mutation",
                if: MUTATIONS_FOUND,
                run: [
                    `echo "::error::${options.mutationError}"`,
                    `cat ${GIT_PATCH_FILE}`,
                    "exit 1",
                ].join("\n"),
            });
        }
        return steps;
    }
    /**
     * Checks out a repository and applies a git patch that was created using
     * `createUploadGitPatch`.
     *
     * @param options Options
     * @returns Job steps
     */
    static checkoutWithPatch(options = {}) {
        return [
            {
                name: "Checkout",
                uses: "actions/checkout@v3",
                with: {
                    token: options.token,
                    ref: options.ref,
                    repository: options.repository,
                },
            },
            {
                name: "Download patch",
                uses: "actions/download-artifact@v3",
                with: { name: GIT_PATCH_FILE, path: RUNNER_TEMP },
            },
            {
                name: "Apply patch",
                run: `[ -s ${RUNNER_TEMP}/${GIT_PATCH_FILE} ] && git apply ${RUNNER_TEMP}/${GIT_PATCH_FILE} || echo "Empty patch. Skipping."`,
            },
        ];
    }
    /**
     * Configures the git identity (user name and email).
     * @param id The identity to use
     * @returns Job steps
     */
    static setGitIdentity(id) {
        return [
            {
                name: "Set git identity",
                run: [
                    `git config user.name "${id.name}"`,
                    `git config user.email "${id.email}"`,
                ].join("\n"),
            },
        ];
    }
}
exports.WorkflowActions = WorkflowActions;
//# sourceMappingURL=data:application/json;base64,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