"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Pom = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const semver_1 = require("../util/semver");
const xmlfile_1 = require("../xmlfile");
const POM_XML_ATTRS = {
    "@xsi:schemaLocation": "http://maven.apache.org/POM/4.0.0 http://maven.apache.org/xsd/maven-4.0.0.xsd",
    "@xmlns": "http://maven.apache.org/POM/4.0.0",
    "@xmlns:xsi": "http://www.w3.org/2001/XMLSchema-instance",
};
/**
 * A Project Object Model or POM is the fundamental unit of work in Maven. It is
 * an XML file that contains information about the project and configuration
 * details used by Maven to build the project.
 */
class Pom extends component_1.Component {
    constructor(project, options) {
        super(project);
        this.properties = {};
        this.repositories = [];
        this.fileName = "pom.xml";
        this.groupId = options.groupId;
        this.artifactId = options.artifactId;
        this.version = options.version;
        this.packaging = options.packaging ?? "jar";
        this.name = project.name;
        this.description = options.description;
        this.url = options.url;
        new xmlfile_1.XmlFile(project, this.fileName, { obj: () => this.synthPom() });
    }
    /**
     * Adds a key/value property to the pom.
     * @param key the key
     * @param value the value
     */
    addProperty(key, value) {
        this.properties[key] = value;
    }
    /**
     * Adds a runtime dependency.
     *
     * @param spec Format `<groupId>/<artifactId>@<semver>`
     */
    addDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.RUNTIME);
    }
    /**
     * Adds a test dependency.
     *
     * @param spec Format `<groupId>/<artifactId>@<semver>`
     */
    addTestDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.TEST);
    }
    /**
     * Adds a build plugin to the pom.
     *
     * The plug in is also added as a BUILD dep to the project.
     *
     * @param spec dependency spec (`group/artifact@version`)
     * @param options plugin options
     */
    addPlugin(spec, options = {}) {
        for (const dep of options.dependencies ?? []) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
        }
        return this.project.deps.addDependency(spec, dependencies_1.DependencyType.BUILD, options);
    }
    /**
     * Adds a repository to the pom
     * @param repository the repository to add
     */
    addRepository(repository) {
        this.repositories.push(repository);
    }
    synthPom() {
        return _resolve_1.resolve({
            project: {
                ...POM_XML_ATTRS,
                modelVersion: "4.0.0",
                groupId: this.groupId,
                artifactId: this.artifactId,
                version: this.version,
                packaging: this.packaging,
                name: this.name,
                description: this.description,
                url: this.url,
                properties: this.properties,
                ...this.synthRepositories(),
                ...this.synthDependencies(),
            },
        }, { omitEmpty: true });
    }
    synthDependencies() {
        const deps = this.project.deps.all;
        if (deps.length === 0) {
            return;
        }
        const dependencies = [];
        const plugins = [];
        for (const dep of deps) {
            switch (dep.type) {
                case dependencies_1.DependencyType.PEER:
                case dependencies_1.DependencyType.RUNTIME:
                    dependencies.push(mavenCoords(dep));
                    break;
                case dependencies_1.DependencyType.TEST:
                    dependencies.push({
                        ...mavenCoords(dep),
                        scope: "test",
                    });
                    break;
                // build maps to plugins
                case dependencies_1.DependencyType.BUILD:
                    plugins.push({
                        ...mavenCoords(dep),
                        ...pluginConfig(dep.metadata),
                    });
                    break;
                default:
                    throw new Error(`unsupported dependency type: ${dep.type}`);
            }
        }
        return {
            build: { plugins: { plugin: plugins } },
            dependencies: { dependency: dependencies },
        };
    }
    synthRepositories() {
        if (this.repositories.length === 0) {
            return;
        }
        return {
            repositories: { repository: this.repositories },
        };
    }
}
exports.Pom = Pom;
_a = JSII_RTTI_SYMBOL_1;
Pom[_a] = { fqn: "projen.java.Pom", version: "0.65.85" };
/**
 * Parses maven groupId and artifactId from a dependency name.
 *
 *     name    <=> <groupId>/<artifactId>
 *     version <=> <version>
 */
function mavenCoords(dep) {
    const name = dep.name;
    const parts = name.split("/");
    if (parts.length !== 2) {
        throw new Error(`invalid maven coordinates in dependency named "${name}". format is "<groupId>/<artifactId>". For example "org.junit.jupiter/junit-jupiter-engine"`);
    }
    return {
        groupId: parts[0],
        artifactId: parts[1],
        version: dep.version ? semver_1.toMavenVersionRange(dep.version) : undefined,
    };
}
function pluginConfig(options = {}) {
    return {
        configuration: options.configuration,
        dependencies: options.dependencies && options.dependencies.length > 0
            ? {
                dependency: options.dependencies?.map((d) => mavenCoords(dependencies_1.Dependencies.parseDependency(d))),
            }
            : undefined,
        executions: options.executions?.map((e) => ({
            execution: {
                id: e.id,
                goals: e.goals.map((goal) => ({ goal })),
                phase: e.phase,
            },
        })),
    };
}
//# sourceMappingURL=data:application/json;base64,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