"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.packageResolutionsFieldName = exports.minVersion = exports.extractCodeArtifactDetails = exports.codeArtifactRegex = exports.renderBundleName = void 0;
const path_1 = require("path");
const semver = require("semver");
const node_package_1 = require("./node-package");
function renderBundleName(entrypoint) {
    const parts = path_1.join(entrypoint).split(path_1.sep);
    if (parts[0] === "src") {
        parts.shift(); // just remove 'src' if its the first element for ergonomics
    }
    const p = parts.join(path_1.sep);
    const dir = path_1.dirname(p);
    const base = path_1.basename(p, path_1.extname(p));
    return path_1.join(dir, base);
}
exports.renderBundleName = renderBundleName;
/**
 * Regex for AWS CodeArtifact registry
 */
exports.codeArtifactRegex = /^https:\/\/(?<registry>(?<domain>[^\.]+)-(?<accountId>\d{12})\.d\.codeartifact\.(?<region>[^\.]+).*\.amazonaws\.com\/.*\/(?<repository>\w+)\/)/;
/**
 * gets AWS details from the Code Artifact registry URL
 * throws exception if not matching expected pattern
 * @param registryUrl Code Artifact registry URL
 * @returns object containing the (domain, accountId, region, repository)
 */
function extractCodeArtifactDetails(registryUrl) {
    const match = registryUrl.match(exports.codeArtifactRegex);
    if (match?.groups) {
        const { domain, accountId, region, repository, registry } = match.groups;
        return { domain, accountId, region, repository, registry };
    }
    throw new Error("Could not get CodeArtifact details from npm Registry");
}
exports.extractCodeArtifactDetails = extractCodeArtifactDetails;
function minVersion(version) {
    if (semver.validRange(version)) {
        return semver.minVersion(version)?.version;
    }
    else {
        return version;
    }
}
exports.minVersion = minVersion;
function packageResolutionsFieldName(packageManager = node_package_1.NodePackageManager.YARN) {
    if (packageManager == node_package_1.NodePackageManager.NPM) {
        return "overrides";
    }
    if (packageManager == node_package_1.NodePackageManager.PNPM) {
        return "pnpm.overrides";
    }
    return "resolutions";
}
exports.packageResolutionsFieldName = packageResolutionsFieldName;
//# sourceMappingURL=data:application/json;base64,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