"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.bump = void 0;
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const logging = require("../logging");
const util_1 = require("../util");
/**
 * Resolves the latest version from git tags and uses `standard-version` to bump
 * to the next version based on commits.
 *
 * This expects `standard-version` to be installed in the path.
 *
 * @param cwd working directory (git repository)
 * @param options options
 */
async function bump(cwd, options) {
    const versionFile = path_1.join(cwd, options.versionFile);
    const prerelease = options.prerelease;
    const major = options.majorVersion;
    const minMajorVersion = options.minMajorVersion;
    const prefix = options.tagPrefix ?? "";
    const bumpFile = path_1.join(cwd, options.bumpFile);
    const changelogFile = path_1.join(cwd, options.changelog);
    const releaseTagFile = path_1.join(cwd, options.releaseTagFile);
    if (major && minMajorVersion) {
        throw new Error(`minMajorVersion and majorVersion cannot be used together.`);
    }
    await fs_extra_1.mkdirp(path_1.dirname(bumpFile));
    await fs_extra_1.mkdirp(path_1.dirname(changelogFile));
    await fs_extra_1.mkdirp(path_1.dirname(releaseTagFile));
    const { latestVersion, latestTag, isFirstRelease } = determineLatestTag({
        cwd,
        major,
        prerelease,
        prefix,
    });
    const content = await tryReadVersionFile(versionFile);
    // update version
    content.version = latestVersion;
    logging.info(`Update ${versionFile} to latest resolved version: ${latestVersion}`);
    await fs_extra_1.writeFile(versionFile, JSON.stringify(content, undefined, 2));
    // check if the latest commit already has a version tag
    const currentTags = util_1.execCapture("git tag --points-at HEAD", { cwd })
        .toString("utf8")
        .split("\n");
    logging.info(`Tags listed on current commit: ${currentTags}`);
    let skipBump = false;
    if (currentTags.includes(latestTag)) {
        logging.info("Skipping bump...");
        skipBump = true;
        // delete the existing tag (locally)
        // if we don't do this, standard-version generates an empty changelog
        util_1.exec(`git tag --delete ${latestTag}`, { cwd });
    }
    // create a standard-version configuration file
    const rcfile = path_1.join(cwd, ".versionrc.json");
    await generateVersionrcFile(rcfile, versionFile, changelogFile, skipBump, prerelease, options.versionrcOptions);
    const cmd = ["npx", "standard-version@^9"];
    if (isFirstRelease) {
        cmd.push("--first-release");
    }
    if (prefix) {
        cmd.push(`--tag-prefix ${prefix}v`);
    }
    if (minMajorVersion) {
        const [majorVersion] = latestVersion.split(".");
        const majorVersionNumber = parseInt(majorVersion, 10);
        if (majorVersionNumber < minMajorVersion) {
            cmd.push(`--release-as ${minMajorVersion}.0.0`);
        }
    }
    util_1.exec(cmd.join(" "), { cwd });
    // add the tag back if it was previously removed
    if (currentTags.includes(latestTag)) {
        util_1.exec(`git tag ${latestTag}`, { cwd });
    }
    await fs_extra_1.remove(rcfile);
    const newVersion = (await tryReadVersionFile(versionFile)).version;
    if (!newVersion) {
        throw new Error(`bump failed: ${versionFile} does not have a version set`);
    }
    // if MAJOR is defined, ensure that the new version is within the same major version
    if (major) {
        if (!newVersion.startsWith(`${major}.`)) {
            throw new Error(`bump failed: this branch is configured to only publish v${major} releases - bump resulted in ${newVersion}`);
        }
    }
    await fs_extra_1.writeFile(bumpFile, newVersion);
    const newTag = `${prefix}v${newVersion}`;
    await fs_extra_1.writeFile(releaseTagFile, newTag);
}
exports.bump = bump;
async function tryReadVersionFile(versionFile) {
    if (!(await fs_extra_1.pathExists(versionFile))) {
        return {};
    }
    return JSON.parse(await fs_extra_1.readFile(versionFile, "utf8"));
}
function generateVersionrcFile(rcfile, versionFile, changelogFile, skipBump, prerelease, configOptions) {
    return fs_extra_1.writeFile(rcfile, JSON.stringify({
        ...{
            packageFiles: [
                {
                    filename: versionFile,
                    type: "json",
                },
            ],
            bumpFiles: [
                {
                    filename: versionFile,
                    type: "json",
                },
            ],
            commitAll: false,
            infile: changelogFile,
            prerelease: prerelease,
            header: "",
            skip: {
                commit: true,
                tag: true,
                bump: skipBump,
            },
            ...configOptions,
        },
    }, undefined, 2));
}
/**
 * Determines the latest release tag.
 * @param major (optional) A major version line to select from
 * @param prerelease (optional) A pre-release suffix.
 * @returns the latest tag, and whether it is the first release or not
 */
function determineLatestTag(options) {
    const { cwd, major, prerelease, prefix } = options;
    // filter only tags for this prefix and major version if specified (start with "vNN.").
    const prefixFilter = major !== undefined ? `${prefix}v${major}.*` : `${prefix}v*`;
    const listGitTags = [
        "git",
        '-c "versionsort.suffix=-"',
        "tag",
        '--sort="-version:refname"',
        "--list",
        `"${prefixFilter}"`,
    ].join(" ");
    const stdout = util_1.execCapture(listGitTags, { cwd }).toString("utf8");
    let tags = stdout?.split("\n");
    // if prerelease is set and there are existing prerelease tags, filter versions that end with "-PRE.ddd".
    const prereleaseTags = tags.filter((x) => new RegExp(`-${prerelease}\.[0-9]+$`).test(x));
    if (prerelease && prereleaseTags.length > 0) {
        tags = prereleaseTags;
    }
    tags = tags.filter((x) => x);
    // if a pre-release tag is used, then add it to the initial version
    let isFirstRelease = false;
    let latestTag;
    if (tags.length > 0) {
        latestTag = tags[0];
    }
    else {
        const initial = `${prefix}v${major ?? 0}.0.0`;
        latestTag = prerelease ? `${initial}-${prerelease}.0` : initial;
        isFirstRelease = true;
    }
    // remove tag prefix (if exists)
    let latestVersion = latestTag;
    if (prefix && latestVersion.startsWith(prefix)) {
        latestVersion = latestVersion.substr(prefix.length);
    }
    // remove "v" prefix (if exists)
    if (latestVersion.startsWith("v")) {
        latestVersion = latestVersion.substr(1);
    }
    return { latestVersion, latestTag, isFirstRelease };
}
//# sourceMappingURL=data:application/json;base64,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