"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Resolves the latest version from git tags and uses `standard-version` to bump
 * to the next version based on commits.
 *
 * This expects `standard-version` to be installed in the path.
 *
 * Environment variables:
 *
 * - OUTFILE: (required) the name of the JSON output file (the "version" field
 *   will be updated with the latest version)
 * - PRERELEASE: (optional) a prerelease tag to use (e.g. "beta")
 * - MAJOR: major version number NN to filter (tags are filtered by "vNN."
 *   prefix). if not specified, the last major version is selected
 * - MIN_MAJOR: minimum major version number to use
 * - CHANGELOG: name of changelog file to create
 * - RELEASE_TAG_PREFIX: (optional) a prefix to apply to the release tag
 *
 */
const logging = require("../logging");
const bump_version_1 = require("./bump-version");
const versionFile = process.env.OUTFILE;
const prerelease = process.env.PRERELEASE;
const major = process.env.MAJOR;
const minMajor = process.env.MIN_MAJOR;
const changelog = process.env.CHANGELOG;
const bumpFile = process.env.BUMPFILE;
const releaseTagFile = process.env.RELEASETAG;
const prefix = process.env.RELEASE_TAG_PREFIX;
const versionrcOptions = process.env.VERSIONRCOPTIONS;
if (!versionFile) {
    throw new Error("OUTFILE is required");
}
if (!changelog) {
    throw new Error("CHANGELOG is required");
}
if (!bumpFile) {
    throw new Error("BUMPFILE is required");
}
if (!releaseTagFile) {
    throw new Error("RELEASETAG is required");
}
const majorVersion = major == null || major === "" ? undefined : parseInt(major);
if (Number.isNaN(majorVersion)) {
    throw new Error(`MAJOR must be a number: ${majorVersion}`);
}
const minMajorVersion = minMajor == null || minMajor === "" ? undefined : parseInt(minMajor);
if (Number.isNaN(minMajorVersion)) {
    throw new Error(`minMajor must be a number: ${minMajorVersion}`);
}
const opts = {
    versionFile: versionFile,
    changelog: changelog,
    majorVersion: majorVersion,
    minMajorVersion: minMajorVersion,
    prerelease: prerelease,
    bumpFile: bumpFile,
    releaseTagFile: releaseTagFile,
    tagPrefix: prefix,
    // doesn't work with long customization
    versionrcOptions: JSON.parse(versionrcOptions ?? "{}"),
};
logging.debug(opts);
bump_version_1.bump(process.cwd(), opts).catch((e) => {
    console.log(e.stack);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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