import { Component } from "../component";
import { Job, JobStep, Tools } from "../github/workflows-model";
import { Project } from "../project";
import { BranchOptions } from "./release";
/**
 * Options for `Publisher`.
 */
export interface PublisherOptions {
    /**
     * The job ID that produces the build artifacts. All publish jobs will take a dependency on this job.
     */
    readonly buildJobId: string;
    /**
     * A GitHub workflow expression used as a condition for publishers.
     *
     * @default - no condition
     */
    readonly condition?: string;
    /**
     * The name of the artifact to download (e.g. `dist`).
     *
     * The artifact is expected to include a subdirectory for each release target:
     * `go` (GitHub), `dotnet` (NuGet), `java` (Maven), `js` (npm), `python`
     * (PyPI).
     *
     * @see https://github.com/aws/publib
     */
    readonly artifactName: string;
    /**
     * @deprecated use `publibVersion` instead
     */
    readonly jsiiReleaseVersion?: string;
    /**
     * Node version to setup in GitHub workflows if any node-based CLI utilities
     * are needed. For example `publib`, the CLI projen uses to publish releases,
     * is an npm library.
     *
     * @default 14.x
     */
    readonly workflowNodeVersion?: string;
    /**
     * Version requirement for `publib`.
     *
     * @default "latest"
     */
    readonly publibVersion?: string;
    /**
     * Create an issue when a publish task fails.
     *
     * @default false
     */
    readonly failureIssue?: boolean;
    /**
     * The label to apply to the issue marking failed publish tasks.
     * Only applies if `failureIssue` is true.
     *
     * @default "failed-release"
     */
    readonly failureIssueLabel?: string;
    /**
     * Github Runner selection labels
     * @default ["ubuntu-latest"]
     */
    readonly workflowRunsOn?: string[];
    /**
     * Define publishing tasks that can be executed manually as well as workflows.
     *
     * Normally, publishing only happens within automated workflows. Enable this
     * in order to create a publishing task for each publishing activity.
     *
     * @default false
     */
    readonly publishTasks?: boolean;
    /**
     * Do not actually publish, only print the commands that would be executed instead.
     *
     * Useful if you wish to block all publishing from a single option.
     */
    readonly dryRun?: boolean;
}
/**
 * Implements GitHub jobs for publishing modules to package managers.
 *
 * Under the hood, it uses https://github.com/aws/publib
 */
export declare class Publisher extends Component {
    static readonly PUBLISH_GIT_TASK_NAME = "publish:git";
    readonly buildJobId: string;
    readonly artifactName: string;
    readonly publibVersion: string;
    readonly condition?: string;
    /** @deprecated use `publibVersion` */
    readonly jsiiReleaseVersion: string;
    private readonly failureIssue;
    private readonly failureIssueLabel;
    private readonly runsOn;
    private readonly publishTasks;
    private readonly _jobFactories;
    private readonly _gitHubPrePublishing;
    private readonly dryRun;
    private readonly workflowNodeVersion;
    constructor(project: Project, options: PublisherOptions);
    /**
     * Called by `Release` to add the publishing jobs to a release workflow
     * associated with a specific branch.
     * @param branch The branch name
     * @param options Branch options
     *
     * @internal
     */
    _renderJobsForBranch(branch: string, options: Partial<BranchOptions>): Record<string, Job>;
    /**
     * Adds pre publishing steps for the GitHub release job.
     *
     * @param steps The steps.
     */
    addGitHubPrePublishingSteps(...steps: JobStep[]): void;
    /**
     * Publish to git.
     *
     * This includes generating a project-level changelog and release tags.
     *
     * @param options Options
     */
    publishToGit(options: GitPublishOptions): import("..").Task;
    /**
     * Creates a GitHub Release.
     * @param options Options
     */
    publishToGitHubReleases(options: GitHubReleasesPublishOptions): void;
    /**
     * Publishes artifacts from `js/**` to npm.
     * @param options Options
     */
    publishToNpm(options?: NpmPublishOptions): void;
    /**
     * Publishes artifacts from `dotnet/**` to NuGet Gallery.
     * @param options Options
     */
    publishToNuget(options?: NugetPublishOptions): void;
    /**
     * Publishes artifacts from `java/**` to Maven.
     * @param options Options
     */
    publishToMaven(options?: MavenPublishOptions): void;
    /**
     * Publishes wheel artifacts from `python` to PyPI.
     * @param options Options
     */
    publishToPyPi(options?: PyPiPublishOptions): void;
    /**
     * Adds a go publishing job.
     * @param options Options
     */
    publishToGo(options?: GoPublishOptions): void;
    private addPublishJob;
    private publibCommand;
    private githubReleaseCommand;
}
/**
 * Common publishing options
 */
export interface CommonPublishOptions {
    /**
     * Steps to execute before executing the publishing command. These can be used
     * to prepare the artifact for publishing if neede.
     *
     * These steps are executed after `dist/` has been populated with the build
     * output.
     *
     * Note that when using this in `publishToGitHubReleases` this will override steps added via `addGitHubPrePublishingSteps`.
     */
    readonly prePublishSteps?: JobStep[];
    /**
     * Additional tools to install in the publishing job.
     * @default - no additional tools are installed
     */
    readonly publishTools?: Tools;
}
/**
 * @deprecated Use `NpmPublishOptions` instead.
 */
export interface JsiiReleaseNpm extends NpmPublishOptions {
}
/**
 * Options for npm release
 */
export interface NpmPublishOptions extends CommonPublishOptions {
    /**
     * Tags can be used to provide an alias instead of version numbers.
     *
     * For example, a project might choose to have multiple streams of development
     * and use a different tag for each stream, e.g., stable, beta, dev, canary.
     *
     * By default, the `latest` tag is used by npm to identify the current version
     * of a package, and `npm install <pkg>` (without any `@<version>` or `@<tag>`
     * specifier) installs the latest tag. Typically, projects only use the
     * `latest` tag for stable release versions, and use other tags for unstable
     * versions such as prereleases.
     *
     * The `next` tag is used by some projects to identify the upcoming version.
     *
     * @default "latest"
     * @deprecated Use `npmDistTag` for each release branch instead.
     */
    readonly distTag?: string;
    /**
     * The domain name of the npm package registry.
     *
     * To publish to GitHub Packages, set this value to `"npm.pkg.github.com"`. In
     * this if `npmTokenSecret` is not specified, it will default to
     * `GITHUB_TOKEN` which means that you will be able to publish to the
     * repository's package store. In this case, make sure `repositoryUrl` is
     * correctly defined.
     *
     * @default "registry.npmjs.org"
     * @example "npm.pkg.github.com"
     */
    readonly registry?: string;
    /**
     * GitHub secret which contains the NPM token to use when publishing packages.
     * @default - "NPM_TOKEN" or "GITHUB_TOKEN" if `registry` is set to `npm.pkg.github.com`.
     */
    readonly npmTokenSecret?: string;
    /**
     * Options for publishing npm package to AWS CodeArtifact.
     *
     * @default - undefined
     */
    readonly codeArtifactOptions?: CodeArtifactOptions;
}
/**
 * Options for authorizing requests to a AWS CodeArtifact npm repository.
 */
export declare enum CodeArtifactAuthProvider {
    /**
     * Fixed credentials provided via Github secrets.
     */
    ACCESS_AND_SECRET_KEY_PAIR = "ACCESS_AND_SECRET_KEY_PAIR",
    /**
     * Ephemeral credentials provided via Github's OIDC integration with an IAM role.
     * See:
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_create_oidc.html
     * https://docs.github.com/en/actions/deployment/security-hardening-your-deployments/configuring-openid-connect-in-amazon-web-services
     */
    GITHUB_OIDC = "GITHUB_OIDC"
}
/**
 * Options for publishing packages to AWS CodeArtifact.
 */
export interface CodeArtifactOptions {
    /**
     * Provider to use for authorizing requests to AWS CodeArtifact.
     *
     * @default CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR
     */
    readonly authProvider?: CodeArtifactAuthProvider;
    /**
     * GitHub secret which contains the AWS access key ID to use when publishing packages to AWS CodeArtifact.
     * This property must be specified only when publishing to AWS CodeArtifact (`npmRegistryUrl` contains AWS CodeArtifact URL).
     *
     * @default - When the `authProvider` value is set to
     * `CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR`, the default is
     * "AWS_ACCESS_KEY_ID". For `CodeArtifactAuthProvider.GITHUB_OIDC`, this
     * value must be left undefined.
     */
    readonly accessKeyIdSecret?: string;
    /**
     * GitHub secret which contains the AWS secret access key to use when publishing packages to AWS CodeArtifact.
     * This property must be specified only when publishing to AWS CodeArtifact (`npmRegistryUrl` contains AWS CodeArtifact URL).
     *
     * @default - When the `authProvider` value is set to
     * `CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR`, the default is
     * "AWS_SECRET_ACCESS_KEY". For `CodeArtifactAuthProvider.GITHUB_OIDC`, this
     * value must be left undefined.
     */
    readonly secretAccessKeySecret?: string;
    /**
     * ARN of AWS role to be assumed prior to get authorization token from AWS CodeArtifact
     * This property must be specified only when publishing to AWS CodeArtifact (`registry` contains AWS CodeArtifact URL).
     * When using the `CodeArtifactAuthProvider.GITHUB_OIDC` auth provider, this value must be defined.
     *
     * @default undefined
     */
    readonly roleToAssume?: string;
}
/**
 * @deprecated Use `PyPiPublishOptions` instead.
 */
export interface JsiiReleasePyPi extends PyPiPublishOptions {
}
/**
 * Options for PyPI release
 */
export interface PyPiPublishOptions extends CommonPublishOptions {
    /**
     * The registry url to use when releasing packages.
     *
     * @default - twine default
     */
    readonly twineRegistryUrl?: string;
    /**
     * The GitHub secret which contains PyPI user name.
     * @default "TWINE_USERNAME"
     */
    readonly twineUsernameSecret?: string;
    /**
     * The GitHub secret which contains PyPI password.
     * @default "TWINE_PASSWORD"
     */
    readonly twinePasswordSecret?: string;
}
/**
 * @deprecated Use `NugetPublishOptions` instead.
 */
export interface JsiiReleaseNuget extends NugetPublishOptions {
}
/**
 * Options for NuGet releases
 */
export interface NugetPublishOptions extends CommonPublishOptions {
    /**
     * GitHub secret which contains the API key for NuGet.
     *
     * @default "NUGET_API_KEY"
     */
    readonly nugetApiKeySecret?: string;
    /**
     *  NuGet Server URL (defaults to nuget.org)
     */
    readonly nugetServer?: string;
}
/**
 * @deprecated Use `MavenPublishOptions` instead.
 */
export interface JsiiReleaseMaven extends MavenPublishOptions {
}
/**
 * Options for Maven releases
 */
export interface MavenPublishOptions extends CommonPublishOptions {
    /**
     * URL of Nexus repository. if not set, defaults to https://oss.sonatype.org
     *
     * @default "https://oss.sonatype.org"
     */
    readonly mavenEndpoint?: string;
    /**
     * Used in maven settings for credential lookup (e.g. use github when publishing to GitHub).
     *
     * @default "ossrh" (Maven Central) or "github" when using GitHub Packages
     */
    readonly mavenServerId?: string;
    /**
     * Deployment repository when not deploying to Maven Central
     *
     * @default - not set
     */
    readonly mavenRepositoryUrl?: string;
    /**
     * GitHub secret name which contains the GPG private key or file that includes
     * it. This is used to sign your Maven
     * packages. See instructions.
     *
     * @see https://github.com/aws/publib#maven
     * @default "MAVEN_GPG_PRIVATE_KEY" or not set when using GitHub Packages
     */
    readonly mavenGpgPrivateKeySecret?: string;
    /**
     * GitHub secret name which contains the GPG private key or file that includes
     * it. This is used to sign your Maven packages. See instructions.
     *
     * @see https://github.com/aws/publib#maven
     * @default "MAVEN_GPG_PRIVATE_KEY_PASSPHRASE" or not set when using GitHub Packages
     */
    readonly mavenGpgPrivateKeyPassphrase?: string;
    /**
     * GitHub secret name which contains the Username for maven repository.
     *
     * For Maven Central, you will need to Create JIRA account and then request a
     * new project (see links).
     *
     * @see https://issues.sonatype.org/secure/Signup
     * @see https://issues.sonatype.org/secure/CreateIssue.jspa?issuetype=21&pid=10134
     *
     * @default "MAVEN_USERNAME" or the GitHub Actor when using GitHub Packages
     */
    readonly mavenUsername?: string;
    /**
     * GitHub secret name which contains the Password for maven repository.
     *
     * For Maven Central, you will need to Create JIRA account and then request a
     * new project (see links).
     *
     * @see https://issues.sonatype.org/secure/Signup
     * @see https://issues.sonatype.org/secure/CreateIssue.jspa?issuetype=21&pid=10134
     *
     * @default "MAVEN_PASSWORD" or "GITHUB_TOKEN" when using GitHub Packages
     */
    readonly mavenPassword?: string;
    /**
     * GitHub secret name which contains the Maven Central (sonatype) staging
     * profile ID (e.g. 68a05363083174). Staging profile ID can be found in the
     * URL of the "Releases" staging profile under "Staging Profiles" in
     * https://oss.sonatype.org (e.g.
     * https://oss.sonatype.org/#stagingProfiles;11a33451234521)
  
     * @default "MAVEN_STAGING_PROFILE_ID" or not set when using GitHub Packages
     */
    readonly mavenStagingProfileId?: string;
}
/**
 * @deprecated Use `GoPublishOptions` instead.
 */
export interface JsiiReleaseGo extends GoPublishOptions {
}
/**
 * Options for Go releases.
 */
export interface GoPublishOptions extends CommonPublishOptions {
    /**
     * The name of the secret that includes a personal GitHub access token used to
     * push to the GitHub repository.
     *
     * Ignored if `githubUseSsh` is `true`.
     *
     * @default "GO_GITHUB_TOKEN"
     */
    readonly githubTokenSecret?: string;
    /**
     * The name of the secret that includes a GitHub deploy key used to push to the
     * GitHub repository.
     *
     * Ignored if `githubUseSsh` is `false`.
     *
     * @default "GO_GITHUB_DEPLOY_KEY"
     */
    readonly githubDeployKeySecret?: string;
    /**
     * Use SSH to push to GitHub instead of a personal accses token.
     *
     * @default false
     */
    readonly githubUseSsh?: boolean;
    /**
     * GitHub repository to push to.
     *
     * @default - derived from `moduleName`
     */
    readonly githubRepo?: string;
    /**
     * Branch to push to.
     *
     * @default "main"
     */
    readonly gitBranch?: string;
    /**
     * The user name to use for the release git commit.
     * @default "github-actions"
     */
    readonly gitUserName?: string;
    /**
     * The email to use in the release git commit.
     * @default "github-actions@github.com"
     */
    readonly gitUserEmail?: string;
    /**
     * The commit message.
     *
     * @default "chore(release): $VERSION"
     */
    readonly gitCommitMessage?: string;
}
interface VersionArtifactOptions {
    /**
     * The location of a text file (relative to `dist/`) that contains the version number.
     *
     * @example version.txt
     */
    readonly versionFile: string;
    /**
     * The location of a text file (relative to `dist/`) that contains the release tag.
     *
     * @example releasetag.txt
     */
    readonly releaseTagFile: string;
    /**
     * The location of an .md file (relative to `dist/`) that includes the changelog for the release.
     *
     * @example changelog.md
     */
    readonly changelogFile: string;
}
/**
 * Evaluates if the `registryUrl` is a AWS CodeArtifact registry.
 * @param registryUrl url of registry
 * @returns true for AWS CodeArtifact
 */
export declare function isAwsCodeArtifactRegistry(registryUrl: string | undefined): boolean | "" | undefined;
/**
 * Publishing options for GitHub releases.
 */
export interface GitHubReleasesPublishOptions extends VersionArtifactOptions, CommonPublishOptions {
}
/**
 * Publishing options for Git releases
 */
export interface GitPublishOptions extends VersionArtifactOptions {
    /**
     * The location of an .md file that includes the project-level changelog.
     */
    readonly projectChangelogFile?: string;
    /**
     * Branch to push to.
     *
     * @default "main"
     */
    readonly gitBranch?: string;
    /**
     * Override git-push command.
     *
     * Set to an empty string to disable pushing.
     */
    readonly gitPushCommand?: string;
}
export {};
