"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.isAwsCodeArtifactRegistry = exports.CodeArtifactAuthProvider = exports.Publisher = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const constants_1 = require("../github/constants");
const workflows_model_1 = require("../github/workflows-model");
const node_package_1 = require("../javascript/node-package");
const PUBLIB_VERSION = "latest";
const GITHUB_PACKAGES_REGISTRY = "npm.pkg.github.com";
const ARTIFACTS_DOWNLOAD_DIR = "dist";
const GITHUB_PACKAGES_MAVEN_REPOSITORY = "https://maven.pkg.github.com";
const GITHUB_PACKAGES_NUGET_REPOSITORY = "https://nuget.pkg.github.com";
const AWS_CODEARTIFACT_REGISTRY_REGEX = /.codeartifact.*.amazonaws.com/;
const PUBLIB_TOOLCHAIN = {
    js: {},
    java: { java: { version: "11.x" } },
    python: { python: { version: "3.x" } },
    go: { go: { version: "^1.16.0" } },
    dotnet: { dotnet: { version: "3.x" } },
};
/**
 * Implements GitHub jobs for publishing modules to package managers.
 *
 * Under the hood, it uses https://github.com/aws/publib
 */
class Publisher extends component_1.Component {
    constructor(project, options) {
        super(project);
        // functions that create jobs associated with a specific branch
        this._jobFactories = [];
        this._gitHubPrePublishing = [];
        this.buildJobId = options.buildJobId;
        this.artifactName = options.artifactName;
        this.publibVersion =
            options.publibVersion ?? options.jsiiReleaseVersion ?? PUBLIB_VERSION;
        this.jsiiReleaseVersion = this.publibVersion;
        this.condition = options.condition;
        this.dryRun = options.dryRun ?? false;
        this.workflowNodeVersion = options.workflowNodeVersion ?? "14.x";
        this.failureIssue = options.failureIssue ?? false;
        this.failureIssueLabel = options.failureIssueLabel ?? "failed-release";
        this.runsOn = options.workflowRunsOn ?? ["ubuntu-latest"];
        this.publishTasks = options.publishTasks ?? false;
    }
    /**
     * Called by `Release` to add the publishing jobs to a release workflow
     * associated with a specific branch.
     * @param branch The branch name
     * @param options Branch options
     *
     * @internal
     */
    _renderJobsForBranch(branch, options) {
        let jobs = {};
        for (const factory of this._jobFactories) {
            jobs = {
                ...jobs,
                ...factory(branch, options),
            };
        }
        return jobs;
    }
    /**
     * Adds pre publishing steps for the GitHub release job.
     *
     * @param steps The steps.
     */
    addGitHubPrePublishingSteps(...steps) {
        this._gitHubPrePublishing.push(...steps);
    }
    /**
     * Publish to git.
     *
     * This includes generating a project-level changelog and release tags.
     *
     * @param options Options
     */
    publishToGit(options) {
        const releaseTagFile = options.releaseTagFile;
        const versionFile = options.versionFile;
        const changelog = options.changelogFile;
        const projectChangelogFile = options.projectChangelogFile;
        const gitBranch = options.gitBranch ?? "main";
        const taskName = gitBranch === "main" || gitBranch === "master"
            ? Publisher.PUBLISH_GIT_TASK_NAME
            : `${Publisher.PUBLISH_GIT_TASK_NAME}:${gitBranch}`;
        const publishTask = this.project.addTask(taskName, {
            description: "Prepends the release changelog onto the project changelog, creates a release commit, and tags the release",
            env: {
                CHANGELOG: changelog,
                RELEASE_TAG_FILE: releaseTagFile,
                PROJECT_CHANGELOG_FILE: projectChangelogFile ?? "",
                VERSION_FILE: versionFile,
            },
            condition: '! git log --oneline -1 | grep -q "chore(release):"',
        });
        if (projectChangelogFile) {
            publishTask.builtin("release/update-changelog");
        }
        publishTask.builtin("release/tag-version");
        if (options.gitPushCommand !== "") {
            const gitPushCommand = options.gitPushCommand || `git push --follow-tags origin ${gitBranch}`;
            publishTask.exec(gitPushCommand);
        }
        return publishTask;
    }
    /**
     * Creates a GitHub Release.
     * @param options Options
     */
    publishToGitHubReleases(options) {
        this.addPublishJob((_branch, branchOptions) => {
            return {
                name: "github",
                registryName: "GitHub Releases",
                prePublishSteps: options.prePublishSteps ?? this._gitHubPrePublishing,
                publishTools: options.publishTools,
                permissions: {
                    contents: workflows_model_1.JobPermission.WRITE,
                },
                workflowEnv: {
                    GITHUB_TOKEN: "${{ secrets.GITHUB_TOKEN }}",
                    GITHUB_REPOSITORY: "${{ github.repository }}",
                    GITHUB_REF: "${{ github.ref }}",
                },
                run: this.githubReleaseCommand(options, branchOptions),
            };
        });
    }
    /**
     * Publishes artifacts from `js/**` to npm.
     * @param options Options
     */
    publishToNpm(options = {}) {
        const isGitHubPackages = options.registry?.startsWith(GITHUB_PACKAGES_REGISTRY);
        const isAwsCodeArtifact = isAwsCodeArtifactRegistry(options.registry);
        const isAwsCodeArtifactWithOidc = isAwsCodeArtifact &&
            options.codeArtifactOptions?.authProvider ===
                CodeArtifactAuthProvider.GITHUB_OIDC;
        const npmToken = node_package_1.defaultNpmToken(options.npmTokenSecret, options.registry);
        if (options.distTag) {
            this.project.logger.warn("The `distTag` option is deprecated. Use the npmDistTag option instead.");
        }
        const prePublishSteps = options.prePublishSteps ?? [];
        if (isAwsCodeArtifactWithOidc) {
            if (options.codeArtifactOptions?.accessKeyIdSecret ||
                options.codeArtifactOptions?.secretAccessKeySecret) {
                throw new Error("access and secret key pair should not be provided when using GITHUB_OIDC auth provider for AWS CodeArtifact");
            }
            else if (!options.codeArtifactOptions?.roleToAssume) {
                throw new Error('"roleToAssume" property is required when using GITHUB_OIDC for AWS CodeArtifact options');
            }
            const regionCaptureRegex = /codeartifact\.(.+)\.amazonaws\.com/;
            const region = options.registry?.match(regionCaptureRegex)?.[1];
            prePublishSteps.push({
                name: "Configure AWS Credentials via GitHub OIDC Provider",
                uses: "aws-actions/configure-aws-credentials@v1",
                with: {
                    "role-to-assume": options.codeArtifactOptions.roleToAssume,
                    "aws-region": region,
                },
            });
        }
        this.addPublishJob((_branch, branchOptions) => {
            if (branchOptions.npmDistTag && options.distTag) {
                throw new Error("cannot set branch-level npmDistTag and npmDistTag in publishToNpm()");
            }
            return {
                name: "npm",
                publishTools: PUBLIB_TOOLCHAIN.js,
                prePublishSteps,
                run: this.publibCommand("publib-npm"),
                registryName: "npm",
                env: {
                    NPM_DIST_TAG: branchOptions.npmDistTag ?? options.distTag ?? "latest",
                    NPM_REGISTRY: options.registry,
                },
                permissions: {
                    idToken: isAwsCodeArtifactWithOidc ? workflows_model_1.JobPermission.WRITE : undefined,
                    contents: workflows_model_1.JobPermission.READ,
                    packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
                },
                workflowEnv: {
                    NPM_TOKEN: npmToken ? secret(npmToken) : undefined,
                    // if we are publishing to AWS CodeArtifact, pass AWS access keys that will be used to generate NPM_TOKEN using AWS CLI.
                    AWS_ACCESS_KEY_ID: isAwsCodeArtifact && !isAwsCodeArtifactWithOidc
                        ? secret(options.codeArtifactOptions?.accessKeyIdSecret ??
                            "AWS_ACCESS_KEY_ID")
                        : undefined,
                    AWS_SECRET_ACCESS_KEY: isAwsCodeArtifact && !isAwsCodeArtifactWithOidc
                        ? secret(options.codeArtifactOptions?.secretAccessKeySecret ??
                            "AWS_SECRET_ACCESS_KEY")
                        : undefined,
                    AWS_ROLE_TO_ASSUME: isAwsCodeArtifact && !isAwsCodeArtifactWithOidc
                        ? options.codeArtifactOptions?.roleToAssume
                        : undefined,
                },
            };
        });
    }
    /**
     * Publishes artifacts from `dotnet/**` to NuGet Gallery.
     * @param options Options
     */
    publishToNuget(options = {}) {
        const isGitHubPackages = options.nugetServer?.startsWith(GITHUB_PACKAGES_NUGET_REPOSITORY);
        this.addPublishJob((_branch, _branchOptions) => ({
            name: "nuget",
            publishTools: PUBLIB_TOOLCHAIN.dotnet,
            prePublishSteps: options.prePublishSteps ?? [],
            run: this.publibCommand("publib-nuget"),
            registryName: "NuGet Gallery",
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
                packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
            },
            workflowEnv: {
                NUGET_API_KEY: secret(isGitHubPackages
                    ? "GITHUB_TOKEN"
                    : options.nugetApiKeySecret ?? "NUGET_API_KEY"),
                NUGET_SERVER: options.nugetServer ?? undefined,
            },
        }));
    }
    /**
     * Publishes artifacts from `java/**` to Maven.
     * @param options Options
     */
    publishToMaven(options = {}) {
        const isGitHubPackages = options.mavenRepositoryUrl?.startsWith(GITHUB_PACKAGES_MAVEN_REPOSITORY);
        const isGitHubActor = isGitHubPackages && options.mavenUsername == undefined;
        const mavenServerId = options.mavenServerId ?? (isGitHubPackages ? "github" : undefined);
        if (isGitHubPackages && mavenServerId != "github") {
            throw new Error('publishing to GitHub Packages requires the "mavenServerId" to be "github"');
        }
        this.addPublishJob((_branch, _branchOptions) => ({
            name: "maven",
            registryName: "Maven Central",
            publishTools: PUBLIB_TOOLCHAIN.java,
            prePublishSteps: options.prePublishSteps ?? [],
            run: this.publibCommand("publib-maven"),
            env: {
                MAVEN_ENDPOINT: options.mavenEndpoint,
                MAVEN_SERVER_ID: mavenServerId,
                MAVEN_REPOSITORY_URL: options.mavenRepositoryUrl,
            },
            workflowEnv: {
                MAVEN_GPG_PRIVATE_KEY: isGitHubPackages
                    ? undefined
                    : secret(options.mavenGpgPrivateKeySecret ?? "MAVEN_GPG_PRIVATE_KEY"),
                MAVEN_GPG_PRIVATE_KEY_PASSPHRASE: isGitHubPackages
                    ? undefined
                    : secret(options.mavenGpgPrivateKeyPassphrase ??
                        "MAVEN_GPG_PRIVATE_KEY_PASSPHRASE"),
                MAVEN_PASSWORD: secret(options.mavenPassword ??
                    (isGitHubPackages ? "GITHUB_TOKEN" : "MAVEN_PASSWORD")),
                MAVEN_USERNAME: isGitHubActor
                    ? "${{ github.actor }}"
                    : secret(options.mavenUsername ?? "MAVEN_USERNAME"),
                MAVEN_STAGING_PROFILE_ID: isGitHubPackages
                    ? undefined
                    : secret(options.mavenStagingProfileId ?? "MAVEN_STAGING_PROFILE_ID"),
            },
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
                packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
            },
        }));
    }
    /**
     * Publishes wheel artifacts from `python` to PyPI.
     * @param options Options
     */
    publishToPyPi(options = {}) {
        this.addPublishJob((_branch, _branchOptions) => ({
            name: "pypi",
            registryName: "PyPI",
            publishTools: PUBLIB_TOOLCHAIN.python,
            prePublishSteps: options.prePublishSteps ?? [],
            run: this.publibCommand("publib-pypi"),
            env: {
                TWINE_REPOSITORY_URL: options.twineRegistryUrl,
            },
            workflowEnv: {
                TWINE_USERNAME: secret(options.twineUsernameSecret ?? "TWINE_USERNAME"),
                TWINE_PASSWORD: secret(options.twinePasswordSecret ?? "TWINE_PASSWORD"),
            },
        }));
    }
    /**
     * Adds a go publishing job.
     * @param options Options
     */
    publishToGo(options = {}) {
        const prePublishSteps = options.prePublishSteps ?? [];
        const workflowEnv = {};
        if (options.githubUseSsh) {
            workflowEnv.GITHUB_USE_SSH = "true";
            workflowEnv.SSH_AUTH_SOCK = "/tmp/ssh_agent.sock";
            prePublishSteps.push({
                name: "Setup GitHub deploy key",
                run: 'ssh-agent -a ${SSH_AUTH_SOCK} && ssh-add - <<< "${GITHUB_DEPLOY_KEY}"',
                env: {
                    GITHUB_DEPLOY_KEY: secret(options.githubDeployKeySecret ?? "GO_GITHUB_DEPLOY_KEY"),
                    SSH_AUTH_SOCK: workflowEnv.SSH_AUTH_SOCK,
                },
            });
        }
        else {
            workflowEnv.GITHUB_TOKEN = secret(options.githubTokenSecret ?? "GO_GITHUB_TOKEN");
        }
        this.addPublishJob((_branch, _branchOptions) => ({
            name: "golang",
            publishTools: PUBLIB_TOOLCHAIN.go,
            prePublishSteps: prePublishSteps,
            run: this.publibCommand("publib-golang"),
            registryName: "GitHub Go Module Repository",
            env: {
                GITHUB_REPO: options.githubRepo,
                GIT_BRANCH: options.gitBranch,
                GIT_USER_NAME: options.gitUserName ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER.name,
                GIT_USER_EMAIL: options.gitUserEmail ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER.email,
                GIT_COMMIT_MESSAGE: options.gitCommitMessage,
            },
            workflowEnv: workflowEnv,
        }));
    }
    addPublishJob(factory) {
        this._jobFactories.push((branch, branchOptions) => {
            const opts = factory(branch, branchOptions);
            const jobname = `release_${opts.name}`;
            if (jobname in this._jobFactories) {
                throw new Error(`Duplicate job with name "${jobname}"`);
            }
            const commandToRun = this.dryRun
                ? `echo "DRY RUN: ${opts.run}"`
                : opts.run;
            const requiredEnv = new Array();
            // jobEnv is the env we pass to the github job (task environment + secrets/expressions).
            const jobEnv = { ...opts.env };
            const workflowEnvEntries = Object.entries(opts.workflowEnv ?? {}).filter(([_, value]) => value != undefined);
            for (const [name, expression] of workflowEnvEntries) {
                requiredEnv.push(name);
                jobEnv[name] = expression;
            }
            if (this.publishTasks) {
                const branchSuffix = branch === "main" || branch === "master" ? "" : `:${branch}`;
                // define a task which can be used through `projen publish:xxx`.
                const task = this.project.addTask(`publish:${opts.name.toLocaleLowerCase()}${branchSuffix}`, {
                    description: `Publish this package to ${opts.registryName}`,
                    env: opts.env,
                    requiredEnv: requiredEnv,
                });
                // first verify that we are on the correct branch
                task.exec(`test "$(git branch --show-current)" = "${branch}"`);
                // run commands
                task.exec(commandToRun);
            }
            const steps = [
                {
                    name: "Download build artifacts",
                    uses: "actions/download-artifact@v3",
                    with: {
                        name: constants_1.BUILD_ARTIFACT_NAME,
                        path: ARTIFACTS_DOWNLOAD_DIR,
                    },
                },
                {
                    name: "Restore build artifact permissions",
                    continueOnError: true,
                    run: [
                        `cd ${ARTIFACTS_DOWNLOAD_DIR} && setfacl --restore=${constants_1.PERMISSION_BACKUP_FILE}`,
                    ].join("\n"),
                },
                ...opts.prePublishSteps,
                {
                    name: "Release",
                    // it would have been nice if we could just run "projen publish:xxx" here but that is not possible because this job does not checkout sources
                    run: commandToRun,
                    env: jobEnv,
                },
            ];
            const perms = opts.permissions ?? { contents: workflows_model_1.JobPermission.READ };
            if (this.failureIssue) {
                steps.push(...[
                    {
                        name: "Extract Version",
                        if: "${{ failure() }}",
                        id: "extract-version",
                        run: 'echo "VERSION=$(cat dist/version.txt)" >> $GITHUB_OUTPUT',
                    },
                    {
                        name: "Create Issue",
                        if: "${{ failure() }}",
                        uses: "imjohnbo/issue-bot@v3",
                        with: {
                            labels: this.failureIssueLabel,
                            title: `Publishing v\${{ steps.extract-version.outputs.VERSION }} to ${opts.registryName} failed`,
                            body: "See https://github.com/${{ github.repository }}/actions/runs/${{ github.run_id }}",
                        },
                        env: {
                            GITHUB_TOKEN: "${{ secrets.GITHUB_TOKEN }}",
                        },
                    },
                ]);
                Object.assign(perms, { issues: workflows_model_1.JobPermission.WRITE });
            }
            return {
                [jobname]: {
                    tools: {
                        node: { version: this.workflowNodeVersion },
                        ...opts.publishTools,
                    },
                    name: `Publish to ${opts.registryName}`,
                    permissions: perms,
                    if: this.condition,
                    needs: [this.buildJobId],
                    runsOn: this.runsOn,
                    steps,
                },
            };
        });
    }
    publibCommand(command) {
        return `npx -p publib@${this.publibVersion} ${command}`;
    }
    githubReleaseCommand(options, branchOptions) {
        const changelogFile = options.changelogFile;
        const releaseTagFile = options.releaseTagFile;
        // create a github release
        const releaseTag = `$(cat ${releaseTagFile})`;
        const ghReleaseCommand = [
            `gh release create ${releaseTag}`,
            "-R $GITHUB_REPOSITORY",
            `-F ${changelogFile}`,
            `-t ${releaseTag}`,
            "--target $GITHUB_REF",
        ];
        if (branchOptions.prerelease) {
            ghReleaseCommand.push("-p");
        }
        const ghRelease = ghReleaseCommand.join(" ");
        // release script that does not error when re-releasing a given version
        const idempotentRelease = [
            "errout=$(mktemp);",
            `${ghRelease} 2> $errout && true;`,
            "exitcode=$?;",
            'if [ $exitcode -ne 0 ] && ! grep -q "Release.tag_name already exists" $errout; then',
            "cat $errout;",
            "exit $exitcode;",
            "fi",
        ].join(" ");
        return idempotentRelease;
    }
}
exports.Publisher = Publisher;
_a = JSII_RTTI_SYMBOL_1;
Publisher[_a] = { fqn: "projen.release.Publisher", version: "0.65.85" };
Publisher.PUBLISH_GIT_TASK_NAME = "publish:git";
function secret(secretName) {
    return `\${{ secrets.${secretName} }}`;
}
/**
 * Options for authorizing requests to a AWS CodeArtifact npm repository.
 */
var CodeArtifactAuthProvider;
(function (CodeArtifactAuthProvider) {
    /**
     * Fixed credentials provided via Github secrets.
     */
    CodeArtifactAuthProvider["ACCESS_AND_SECRET_KEY_PAIR"] = "ACCESS_AND_SECRET_KEY_PAIR";
    /**
     * Ephemeral credentials provided via Github's OIDC integration with an IAM role.
     * See:
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_create_oidc.html
     * https://docs.github.com/en/actions/deployment/security-hardening-your-deployments/configuring-openid-connect-in-amazon-web-services
     */
    CodeArtifactAuthProvider["GITHUB_OIDC"] = "GITHUB_OIDC";
})(CodeArtifactAuthProvider = exports.CodeArtifactAuthProvider || (exports.CodeArtifactAuthProvider = {}));
/**
 * Evaluates if the `registryUrl` is a AWS CodeArtifact registry.
 * @param registryUrl url of registry
 * @returns true for AWS CodeArtifact
 */
function isAwsCodeArtifactRegistry(registryUrl) {
    return registryUrl && AWS_CODEARTIFACT_REGISTRY_REGEX.test(registryUrl);
}
exports.isAwsCodeArtifactRegistry = isAwsCodeArtifactRegistry;
//# sourceMappingURL=data:application/json;base64,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