"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.multipleSelected = exports.anySelected = exports.getNodeMajorVersion = exports.isExecutable = exports.isWritable = exports.tryReadFileSync = exports.tryReadFile = exports.snakeCaseKeys = exports.kebabCaseKeys = exports.getGitVersion = exports.formatAsPythonModule = exports.sorted = exports.dedupArray = exports.deepMerge = exports.isObject = exports.isTruthy = exports.decamelizeKeysRecursively = exports.writeFile = exports.getFilePermissions = exports.execOrUndefined = exports.execCapture = exports.exec = void 0;
const child_process = require("child_process");
const path = require("path");
const Case = require("case");
const fs = require("fs-extra");
const logging = require("./logging");
const MAX_BUFFER = 10 * 1024 * 1024;
/**
 * Executes a command with STDOUT > STDERR.
 */
function exec(command, options) {
    logging.debug(command);
    child_process.execSync(command, {
        stdio: ["inherit", 2, "pipe"],
        maxBuffer: MAX_BUFFER,
        cwd: options.cwd,
    });
}
exports.exec = exec;
/**
 * Executes command and returns STDOUT. If the command fails (non-zero), throws an error.
 */
function execCapture(command, options) {
    logging.debug(command);
    return child_process.execSync(command, {
        stdio: ["inherit", "pipe", "pipe"],
        maxBuffer: MAX_BUFFER,
        cwd: options.cwd,
    });
}
exports.execCapture = execCapture;
/**
 * Executes `command` and returns its value or undefined if the command failed.
 */
function execOrUndefined(command, options) {
    try {
        const value = child_process
            .execSync(command, {
            stdio: ["inherit", "pipe", "pipe"],
            maxBuffer: MAX_BUFFER,
            cwd: options.cwd,
        })
            .toString("utf-8")
            .trim();
        if (!value) {
            return undefined;
        } // an empty string is the same as undefined
        return value;
    }
    catch {
        return undefined;
    }
}
exports.execOrUndefined = execOrUndefined;
function getFilePermissions(options) {
    const readonly = options.readonly ?? false;
    const executable = options.executable ?? false;
    if (readonly && executable) {
        return "544";
    }
    else if (readonly) {
        return "444";
    }
    else if (executable) {
        return "755";
    }
    else {
        return "644";
    }
}
exports.getFilePermissions = getFilePermissions;
function writeFile(filePath, data, options = {}) {
    if (fs.existsSync(filePath)) {
        fs.chmodSync(filePath, "600");
    }
    fs.mkdirpSync(path.dirname(filePath));
    fs.writeFileSync(filePath, data);
    fs.chmodSync(filePath, getFilePermissions(options));
}
exports.writeFile = writeFile;
function decamelizeKeysRecursively(input, opt) {
    const shouldAlwaysDecamelize = () => true;
    const shouldDecamelize = opt?.shouldDecamelize ?? shouldAlwaysDecamelize;
    const separator = opt?.separator ?? "_";
    const path_ = opt?.path ?? [];
    const maxDepth = opt?.maxDepth ?? 10;
    if (path_.length > maxDepth) {
        throw new Error("Decamelled too deeply - check that the input has no circular references");
    }
    if (Array.isArray(input)) {
        return input.map((k, i) => decamelizeKeysRecursively(k, {
            ...opt,
            path: [...path_, i.toString()],
        }));
    }
    if (typeof input === "object" && input !== null) {
        const mappedObject = {};
        for (const [key, value] of Object.entries(input)) {
            const transformedKey = shouldDecamelize([...path_, key], value)
                ? decamelize(key, separator)
                : key;
            mappedObject[transformedKey] = decamelizeKeysRecursively(value, {
                ...opt,
                path: [...path_, key],
            });
        }
        return mappedObject;
    }
    return input;
}
exports.decamelizeKeysRecursively = decamelizeKeysRecursively;
/**
 * Returns false if value is unset or a falsey value, and true otherwise.
 * @param value an environment variable
 */
function isTruthy(value) {
    return !(value === undefined ||
        ["null", "undefined", "0", "false", ""].includes(value.toLocaleLowerCase()));
}
exports.isTruthy = isTruthy;
/**
 * Return whether the given value is an object
 *
 * Even though arrays and instances of classes technically are objects, we
 * usually want to treat them differently, so we return false in those cases.
 */
function isObject(x) {
    return (x !== null &&
        typeof x === "object" &&
        !Array.isArray(x) &&
        x.constructor.name === "Object");
}
exports.isObject = isObject;
/**
 * Recursively merge objects together
 *
 * The leftmost object is mutated and returned. Arrays are not merged
 * but overwritten just like scalars.
 *
 * If an object is merged into a non-object, the non-object is lost.
 *
 * `undefined`s will cause a value to be deleted if destructive is enabled.
 */
function deepMerge(objects, destructive = false) {
    function mergeOne(target, source) {
        for (const key of Object.keys(source)) {
            const value = source[key];
            if (isObject(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (typeof target[key] !== "object") {
                    target[key] = value;
                }
                if ("__$APPEND" in value && Array.isArray(value.__$APPEND)) {
                    if (Array.isArray(target[key])) {
                        target[key].push(...value.__$APPEND);
                    }
                    else {
                        target[key] = value.__$APPEND;
                    }
                }
                mergeOne(target[key], value);
                // if the result of the merge is an empty object, it's because the
                // eventual value we assigned is `undefined`, and there are no
                // sibling concrete values alongside, so we can delete this tree.
                const output = target[key];
                if (typeof output === "object" &&
                    Object.keys(output).length === 0 &&
                    destructive) {
                    delete target[key];
                }
            }
            else if (value === undefined && destructive) {
                delete target[key];
            }
            else if (typeof value !== "undefined") {
                target[key] = value;
            }
        }
    }
    const others = objects.filter((x) => x != null);
    if (others.length === 0) {
        return {};
    }
    const into = others.splice(0, 1)[0];
    others.forEach((other) => mergeOne(into, other));
    return into;
}
exports.deepMerge = deepMerge;
/*
 * Deduplicate values in a list, returning a new array.
 * @param array list of values
 */
function dedupArray(array) {
    return array.filter((val, idx) => array.indexOf(val) === idx);
}
exports.dedupArray = dedupArray;
/**
 * Returns a sorted version of `x` or `undefined` if it is an empty array or object.
 */
function sorted(x) {
    if (x == null) {
        return undefined;
    }
    if (Array.isArray(x)) {
        if (x.length === 0) {
            return undefined;
        }
        return x.sort();
    }
    else if (typeof x === "object") {
        if (Object.keys(x).length === 0) {
            return undefined;
        }
        const result = {};
        for (const [key, value] of Object.entries(x).sort(([l], [r]) => l.localeCompare(r))) {
            result[key] = value;
        }
        return result;
    }
    else {
        return x;
    }
}
exports.sorted = sorted;
function formatAsPythonModule(name) {
    return name.replace(/-/g, "_").replace(/\./g, "_");
}
exports.formatAsPythonModule = formatAsPythonModule;
/**
 * Extract git version number from command line
 *
 * @param gitVersionOutput the output from `git version` CLI
 * @returns the version of git
 */
function getGitVersion(gitVersionOutput) {
    const match = gitVersionOutput.match(/\d+.\d+.\d+/);
    if (!match) {
        throw new Error("Unable to retrieve git version");
    }
    return match[0];
}
exports.getGitVersion = getGitVersion;
function kebabCaseKeys(obj, recursive = true) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        if (recursive) {
            obj = obj.map((v) => kebabCaseKeys(v, recursive));
        }
        return obj;
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (recursive) {
            v = kebabCaseKeys(v, recursive);
        }
        result[decamelize(k).replace(/_/gm, "-")] = v;
    }
    return result;
}
exports.kebabCaseKeys = kebabCaseKeys;
function snakeCaseKeys(obj, recursive = true, exclusiveForRecordKeys = []) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        if (recursive) {
            obj = obj.map((v) => snakeCaseKeys(v, recursive, exclusiveForRecordKeys));
        }
        return obj;
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (recursive) {
            v = snakeCaseKeys(v, recursive, exclusiveForRecordKeys);
        }
        const modifiedKey = exclusiveForRecordKeys.length == 0 || exclusiveForRecordKeys.includes(k)
            ? Case.snake(k)
            : k;
        result[modifiedKey] = v;
    }
    return result;
}
exports.snakeCaseKeys = snakeCaseKeys;
async function tryReadFile(file) {
    if (!(await fs.pathExists(file))) {
        return "";
    }
    return fs.readFile(file, "utf8");
}
exports.tryReadFile = tryReadFile;
function tryReadFileSync(file) {
    if (!fs.pathExistsSync(file)) {
        return undefined;
    }
    return fs.readFileSync(file, "utf8");
}
exports.tryReadFileSync = tryReadFileSync;
function isWritable(file) {
    try {
        fs.accessSync(file, fs.constants.W_OK);
        return true;
    }
    catch {
        return false;
    }
}
exports.isWritable = isWritable;
function isExecutable(file) {
    try {
        fs.accessSync(file, fs.constants.X_OK);
        return true;
    }
    catch {
        return false;
    }
}
exports.isExecutable = isExecutable;
function decamelize(s, sep = "_") {
    if (Case.of(s) === "camel") {
        return Case.lower(s, sep);
    }
    else {
        return s;
    }
}
function getNodeMajorVersion() {
    const match = process.version.match(/(\d+)\.(\d+)\.(\d+)/);
    if (match) {
        const [major] = match.slice(1).map((x) => parseInt(x));
        return major;
    }
    return undefined;
}
exports.getNodeMajorVersion = getNodeMajorVersion;
function anySelected(options) {
    return options.some((opt) => opt);
}
exports.anySelected = anySelected;
function multipleSelected(options) {
    return options.filter((opt) => opt).length > 1;
}
exports.multipleSelected = multipleSelected;
//# sourceMappingURL=data:application/json;base64,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