from dataclasses import field
from datetime import datetime
from typing import Dict, List, Optional, Union

from pydantic import Field, SecretStr, validator
from pydantic.dataclasses import dataclass
from typing_extensions import Literal

from kumoapi.common import StrEnum


@dataclass
class StartFileUploadRequest:
    r"""A request to begin an AWS S3 multipart file upload."""
    source_table_name: str
    connector_id: str
    num_parts: int = 1

    # AWS S3 multi-part upload only supports 1-1000 parts.
    # Refer: https://docs.aws.amazon.com/AmazonS3/latest/userguide/mpuoverview.html # noqa: E501
    @validator('num_parts')
    def check_num_parts(cls, num_parts: int) -> int:
        if num_parts < 1 or num_parts > 1000:
            raise ValueError("num_parts must be between 1 and 1000")
        return num_parts


@dataclass
class StartFileUploadResponse:
    r"""The server response to an initiation of a file upload."""
    # The temporary path where the file would be uploaded. This is required
    # to be sent back in CompleteFileUploadRequest after the parts are uploaded
    # using the below URLs.
    temp_upload_path: str
    # The upload ID is used as a backend reference for the file upload.
    upload_id: str
    # A mapping of part number to the presigned URLs for each part.
    # Each part must be uploaded using PUT request to these URL.
    presigned_part_urls: Dict[int, str] = field(default_factory=dict)


@dataclass
class PartUploadMetadata:
    r"""Metadata for a single part of file upload in S3."""
    # The AWS S3 part numbers can only range from 1-1000.
    part_number: int
    # The ETag of the uploaded part will be generated by AWS S3 SDK and
    # returned in the response headers.
    etag: str
    # More information like part checksum can be added in the future for
    # data validation


@dataclass
class CompleteFileUploadRequest:
    r"""Request to process an uploaded file for adding as a table."""
    source_table_name: str
    connector_id: str
    # Temporary upload path as returned in StartFileUploadResponse.
    temp_upload_path: str
    # Upload ID as returned in StartFileUploadResponse.
    upload_id: str
    parts_metadata: List[PartUploadMetadata] = field(default_factory=list)


@dataclass
class DeleteUploadedFileRequest:
    r"""Request to delete an uploaded file."""
    source_table_name: str
    connector_id: str


class DataSourceType(StrEnum):
    r"""The different data sources that Kumo can connect to."""
    SNOWFLAKE = "SNOWFLAKE"
    S3 = "S3"
    DATABRICKS = "DATABRICKS"
    BIGQUERY = "BIGQUERY"
    GLUE = "GLUE"
    UNAVAILABLE = "UNAVAILABLE"


class ConnectorType(StrEnum):
    r"""The different types of connectors that Kumo supports."""
    SNOWFLAKE = "snowflake"
    DATABRICKS = "databricks"
    BIGQUERY = "bigquery"
    FILE = "file"
    GLUE = "glue"


@dataclass
class ResourceTimestamps:
    created_at: datetime
    updated_at: datetime


@dataclass
class FileConnectorResourceConfig:
    name: str
    root_dir: str
    type: Literal[ConnectorType.FILE] = ConnectorType.FILE


@dataclass
class GlueConnectorResourceConfig:
    name: str
    account: str
    region: str
    database: str
    type: Literal[ConnectorType.GLUE] = ConnectorType.GLUE


@dataclass
class SnowflakeConnectorResourceConfig:
    r"""A configuration of a Snowflake connector."""
    name: str
    account: str
    warehouse: str
    database: str
    schema_name: str
    type: Literal[ConnectorType.SNOWFLAKE] = ConnectorType.SNOWFLAKE

    @validator('name', 'account', 'warehouse', 'database', 'schema_name')
    def strip_inputs(cls, value: str) -> str:
        return value.strip()


@dataclass
class DatabricksConnectorResourceConfig:
    r"""A configuration of a Databricks connector."""
    name: str
    host: str
    cluster_id: str
    warehouse_id: str
    catalog: str
    type: Literal[ConnectorType.DATABRICKS] = ConnectorType.DATABRICKS

    @validator('name', 'host', 'cluster_id', 'warehouse_id', 'catalog')
    def strip_inputs(cls, value: str) -> str:
        return value.strip()


@dataclass
class BigQueryConnectorResourceConfig:
    r"""A configuration of a Databricks connector."""
    name: str
    project_id: str
    dataset_id: str
    type: Literal[ConnectorType.BIGQUERY] = ConnectorType.BIGQUERY

    @validator('name', 'project_id', 'dataset_id')
    def strip_inputs(cls, value: str) -> str:
        return value.strip()


ConnectorResourceConfig = Union[
    SnowflakeConnectorResourceConfig,
    DatabricksConnectorResourceConfig,
    BigQueryConnectorResourceConfig,
    FileConnectorResourceConfig,
    GlueConnectorResourceConfig,
]


@dataclass(frozen=True)
class UsernamePassword:
    r"""Basic authentication with username/password

    Attributes:
        username (SecretStr): Account user name
        password (SecretStr): Account password

    """
    username: SecretStr
    password: SecretStr


@dataclass(frozen=True)
class KeyPair:
    r"""Basic authentication with user/private key pair
    More info when using it for snowflake can be found at:
    https://docs.snowflake.com/en/user-guide/key-pair-auth

    Attributes:
        user (SecretStr): Account user name
        private_key (SecretStr): Private key
        private_key_passphrase (Optional[SecretStr]): Private key passphrase
        to regulate access to the key.

    """
    user: SecretStr
    private_key: SecretStr
    private_key_passphrase: Optional[SecretStr] = None


@dataclass(frozen=True)
class BigQueryCredentials:
    r"""Credentials required for connecting to BigQuery. All these attributes
    are obtained from the service_key_account.json file generated by Google
    Cloud SDK. Refer to this Notion page for more info:
    https://www.notion.so/PRD-GCP-BigQuery-Data-Warehouse-Connector-5402ad1a630141419637b77b0e8b6fab?pvs=4#41586168c02f44f99d4ca4213d1cb736 # noqa: E501

    Attributes:
    'private_key_id', 'private_key', 'client_email', 'token_uri'
        private_key_id (SecretStr): Private key ID of the service account
            in Google Cloud.
        private_key (SecretStr): Private key
        client_email (SecretStr): Client email
        token_uri (SecretStr): Token URI.
            E.g. https://oauth2.googleapis.com/token
        auth_uri (SecretStr): Auth URI.
            E.g. https://accounts.google.com/o/oauth2/auth
    """
    private_key_id: SecretStr
    private_key: SecretStr
    client_id: SecretStr
    client_email: SecretStr
    token_uri: SecretStr
    auth_uri: SecretStr


@dataclass(frozen=True)
class DatabricksCredentials:
    r"""Authentication with Client Id, Client Secret and Personal Access Token

    Attributes:
        client_id (SecretStr): Client ID
        client_secret (SecretStr): Client Secret
        pat (SecretStr): Personal access token

    """
    client_id: SecretStr
    client_secret: SecretStr
    pat: SecretStr


Credentials = Union[UsernamePassword, KeyPair, BigQueryCredentials,
                    DatabricksCredentials]


@dataclass
class CreateConnectorArgs:
    config: ConnectorResourceConfig = Field(discriminator='type')
    credentials: Optional[Credentials] = None
    description: str = ''


@dataclass
class ConnectorResponse:
    id: str
    timestamps: ResourceTimestamps
    config: ConnectorResourceConfig = Field(discriminator='type')
