"""
HTTP client for the Enode Quant SDK.

This module provides a single entrypoint, `run_query`, which sends SQL queries
to the configured Enode API Gateway endpoint and returns the decoded JSON rows.

All networking concerns (timeouts, auth headers, error mapping) are centralized
here, so the rest of the SDK never needs to know about HTTP, requests, or JSON.
"""

from __future__ import annotations

import json
from typing import Any, Dict, List

import requests


from enode_quant.config import get_api_key, get_api_url
from enode_quant.errors import (
    APIConnectionError,
    AuthenticationError,
    InvalidQueryError,
    MissingCredentialsError,
    ServerError,
)

# Default timeout for HTTP requests (in seconds).
_DEFAULT_TIMEOUT = 10.0


def _build_headers(api_key: str) -> Dict[str, str]:
    """
    Internal helper to build HTTP headers for the Enode API.
    """
    return {
        "x-api-key": api_key,
        "Content-Type": "application/json",
    }


def run_query(sql: str, timeout: float = _DEFAULT_TIMEOUT) -> List[Dict[str, Any]]:
    """
    Execute a SQL query against the Enode Quant API.

    Parameters
    ----------
    sql : str
        The SQL query string to execute. This should be generated by the SDK's
        SQL builder functions and should always fully-qualify tables
        (e.g. `market.stock_quotes`).
    timeout : float, optional
        Timeout in seconds for the HTTP request. Defaults to 10 seconds.

    Returns
    -------
    List[Dict[str, Any]]
        A list of row objects decoded from the JSON response. Each row is
        represented as a dictionary mapping column names to values.

    Raises
    ------
    MissingCredentialsError
        If no credentials are configured (user has not run `enode login`).
    APIConnectionError
        If the request fails due to network issues, timeouts, or DNS errors.
    AuthenticationError
        If the API responds with 401/403 (invalid or unauthorized API key).
    InvalidQueryError
        If the API responds with a 4xx status that indicates a bad query or
        invalid request parameters.
    ServerError
        If the API responds with 5xx or returns malformed/unexpected JSON.
    """
    if not isinstance(sql, str) or not sql.strip():
        raise InvalidQueryError("SQL query must be a non-empty string.")

    try:
        api_url = get_api_url()
        api_key = get_api_key()
    except MissingCredentialsError:
        # Let this bubble up: callers should know they need to run `enode login`.
        raise

    headers = _build_headers(api_key)
    payload = {"query": sql}

    try:
        response = requests.post(
            api_url,
            headers=headers,
            data=json.dumps(payload),
            timeout=timeout,
        )
    except requests.exceptions.RequestException as e:
        # This covers all network-related errors: timeouts, connection errors, etc.
        raise APIConnectionError(f"Failed to connect to Enode API: {e}") from e

    # Map HTTP status codes to SDK-specific exceptions.
    status = response.status_code

    if status == 401 or status == 403:
        raise AuthenticationError(
            f"Authentication failed with status code {status}. "
            "Check your API key and run `enode login` again if needed."
        )
    elif 400 <= status < 500:
        # Client-side error: typically invalid SQL or parameters.
        # Include a truncated response body for easier debugging.
        body_preview = response.text[:200].strip()
        msg = f"API returned {status} for the query. "
        if body_preview:
            msg += f"Response: {body_preview}"
        raise InvalidQueryError(msg)
    elif 500 <= status < 600:
        # Server-side error: Lambda / DB / internal.
        body_preview = response.text[:200].strip()
        msg = f"Server error from Enode API (status {status})."
        if body_preview:
            msg += f" Response: {body_preview}"
        raise ServerError(msg)

    # At this point we expect a successful 2xx response.
    try:
        data = response.json()
    except ValueError as e:
        # Response was not valid JSON.
        raise ServerError("Enode API returned a non-JSON response.") from e

    # Normalize the response format.
    # We expect either:
    #   - a list of row dicts
    #   - or a dict with a 'result' key containing such a list
    #   - or a dict with a 'rows' key containing such a list
    if isinstance(data, list):
        rows = data
    elif isinstance(data, dict):
        if "result" in data and isinstance(data["result"], list):
            rows = data["result"]
        elif "rows" in data and isinstance(data["rows"], list):
            rows = data["rows"]
        else:
            # Some APIs return a single object instead of a list.
            # We normalize this to a list of one row to keep the API stable.
            rows = [data]
    else:
        raise ServerError(
            f"Unexpected response format from Enode API: {type(data).__name__}"
        )

    # Ensure the final return type is always List[Dict[str, Any]].
    normalized_rows: List[Dict[str, Any]] = []
    for row in rows:
        if isinstance(row, dict):
            normalized_rows.append(row)
        else:
            # If the API ever returns non-dict rows, we wrap them in a dict.
            normalized_rows.append({"value": row})

    return normalized_rows
